document.addEventListener("DOMContentLoaded", function () {

(function (wp) {

    if (!wp) return;

    const el = wp.element.createElement;
    const { registerPlugin } = wp.plugins || {};
    const { PluginSidebar } = wp.editPost || {};
    const { PanelBody, Button, Spinner } = wp.components || {};
    const { useState } = wp.element || {};
    const { select, dispatch } = wp.data || {};

    if (!registerPlugin || !PluginSidebar) return;

    function buildFAQHTML(faqs) {
        let html = '<h2>Frequently Asked Questions</h2>';
        html += '<div class="miro-faq">';
        faqs.forEach(f => {
            html += `
            <div class="miro-faq-item">
                <h3>${f.q}</h3>
                <p>${f.a}</p>
            </div>`;
        });
        html += '</div>';
        return html;
    }

    const FAQPanel = () => {

        const [loading, setLoading] = useState(false);
        const [faqs, setFaqs] = useState([]);
        const [error, setError] = useState("");

        const insertFAQ = () => {
            try {
                const html = buildFAQHTML(faqs);
                const block = wp.blocks.createBlock("core/html", { content: html });
                dispatch("core/editor").insertBlocks(block);
            } catch (err) {
                // Silent error handling
            }
        };

        const generateFAQ = async () => {
            setLoading(true);
            setError("");
            setFaqs([]);

            const post_id = select("core/editor").getCurrentPostId();
            const title = select("core/editor").getEditedPostAttribute("title") || '';
            const content = select("core/editor").getEditedPostContent() || '';

            if (!title || title.trim() === '') {
                setError("Please enter a post title first.");
                setLoading(false);
                return;
            }

            try {
                const res = await fetch(MiroFAQ.rest, {
                    method: "POST",
                    headers: {
                        "Content-Type": "application/json",
                        "X-WP-Nonce": MiroFAQ.nonce
                    },
                    body: JSON.stringify({ post_id, title, content })
                });

                const j = await res.json();

                // Check for WordPress REST API error
                if (j.code && j.message) {
                    setError(j.message);
                } else if (j.faqs) {
                    setFaqs(j.faqs);
                } else {
                    setError("No FAQ returned.");
                }

            } catch (e) {
                setError("Error generating FAQ: " + e.message);
                console.error('FAQ Error:', e);
            }

            setLoading(false);
        };

        return el(
            PluginSidebar,
            { name: "miro-ai-faq-panel", title: "AI FAQ Generator", icon: "editor-help" },

            el(PanelBody, { title: "Generate FAQ", initialOpen: true },

                el(Button, {
                    isPrimary: true,
                    type: "button",
                    disabled: loading,
                    onClick: (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        generateFAQ();
                    }
                }, loading ? "Generating…" : "Generate FAQ"),

                loading && el(Spinner, {}),

                error && el("p", { style: { color: "red", marginTop: "10px" } }, error),

                faqs.length > 0 && el("div", { style: { marginTop: "15px" } },

                    el("h3", {}, "Preview"),

                    faqs.map((f, i) =>
                        el("div", { key: i, style: { marginBottom: "10px" } },
                            el("strong", {}, "Q: "), f.q, el("br"),
                            el("strong", {}, "A: "), f.a
                        )
                    ),

                    el(Button, {
                        isSecondary: true,
                        type: "button",
                        style: { marginTop: "15px" },
                        onClick: (e) => {
                            e.preventDefault();
                            e.stopPropagation();
                            insertFAQ();
                        }
                    }, "Insert FAQ into Post")
                )
            )
        );
    };

    registerPlugin("miro-ai-faq", { render: FAQPanel });

})(window.wp || {});

});
