/* global MiroAIKeywordLab, wp */
(function(){
  const { createElement: h, render, Fragment, useState, useEffect } = wp.element;

  const HeaderCell = ({label, field, sort, setSort}) => {
    const active = sort.field === field;
    const dir = active ? (sort.asc ? '↑' : '↓') : '↕';
    return h('th',
      {
        onClick: () => {
          if (active) setSort({ field, asc: !sort.asc });
          else setSort({ field, asc: true });
        },
        title: 'Click to sort',
      },
      h('span', {style:{color:'#2271b1'}}, label), h('span', {className:'arrow'}, ' ' + dir)
    );
  };

  const Badge = ({children, tone}) => {
    const cls = 'miro-kws-badge ' + (tone||'');
    return h('span', {className: cls}, children);
  };

  const App = () => {
    const [loading, setLoading] = useState(false);
    const [items, setItems] = useState([]);
    const [sort, setSort] = useState({ field: 'opportunity', asc: false });
    const [error, setError] = useState('');

    const sorted = [...items].sort((a,b) => {
      const f = sort.field;
      const av = a[f], bv = b[f];
      // numeric first
      const an = (typeof av === 'number') ? av : (av==null ? -Infinity : av);
      const bn = (typeof bv === 'number') ? bv : (bv==null ? -Infinity : bv);
      if (typeof av === 'number' || typeof bv === 'number') {
        return sort.asc ? (an - bn) : (bn - an);
      }
      // string
      const as = (av ?? '').toString().toLowerCase();
      const bs = (bv ?? '').toString().toLowerCase();
      return sort.asc ? as.localeCompare(bs) : bs.localeCompare(as);
    });

    const fetchSuggestions = async () => {
      setError('');
      setLoading(true);
      try {
        const res = await wp.apiFetch({
          path: MiroAIKeywordLab.base + 'ai-lab/suggest',
          method: 'POST',
          headers: { 'X-WP-Nonce': MiroAIKeywordLab.nonce }
        });
        setItems(res.items || []);
      } catch(e) {
        setError(e && e.message ? e.message : 'Failed to generate');
      } finally {
        setLoading(false);
      }
    };

    const addToTracker = async (row) => {
      try {
        const res = await wp.apiFetch({
          path: MiroAIKeywordLab.base + 'ai-lab/add-to-tracker',
          method: 'POST',
          headers: { 'X-WP-Nonce': MiroAIKeywordLab.nonce },
          data: {
            items: [{
              keyword: row.keyword,
              target_url: row.page || '',
              device: 'desktop',
              country: 'US',
              tags: 'ai-lab'
            }]
          }
        });
        alert('Added to Rank Tracker: ' + (res.added||0) + ' (skipped: ' + (res.skipped||0) + ')');
      } catch(e) {
        alert('Failed to add: ' + (e && e.message ? e.message : String(e)));
      }
    };

    const createDraft = async (row) => {
      try {
        const res = await wp.apiFetch({
          path: MiroAIKeywordLab.base + 'ai-lab/create-draft',
          method: 'POST',
          headers: { 'X-WP-Nonce': MiroAIKeywordLab.nonce },
          data: {
            title: row.title || row.keyword,
            slug: row.slug || (row.keyword || '').toLowerCase().replace(/[^a-z0-9]+/g,'-').replace(/(^-|-$)/g,''),
            post_type: 'post',
            status: 'draft'
          }
        });
        const go = confirm('Draft created. Open editor?');
        if (go && res.edit) window.location = res.edit;
      } catch(e) {
        alert('Failed to create draft: ' + (e && e.message ? e.message : String(e)));
      }
    };

    useEffect(() => { fetchSuggestions(); }, []);

    return h(Fragment, null,
      h('div', {style:{margin:'12px 0'}},
        h('button', {
          className:'button button-primary',
          onClick: fetchSuggestions,
          disabled: loading
        }, loading ? 'Generating…' : 'Generate Suggestions'),
        h('span', {style:{marginLeft:10}}, MiroAIKeywordLab.hasKey ? '' :
          h('span', {className:'miro-muted'}, 'AI Keyword Lab needs your OpenAI key (Miro AI → Settings).'))
      ),
      error ? h('div', {style:{color:'#b00020', marginBottom:10}}, error) : null,
      h('table', {className:'widefat striped miro-kws-table'},
        h('thead', null,
          h('tr', null,
            h(HeaderCell, {label:'Keyword', field:'keyword', sort, setSort}),
            h(HeaderCell, {label:'Intent', field:'intent', sort, setSort}),
            h(HeaderCell, {label:'KD', field:'kd', sort, setSort}),
            h(HeaderCell, {label:'Opp.', field:'opportunity', sort, setSort}),
            h(HeaderCell, {label:'Clicks', field:'clicks', sort, setSort}),
            h(HeaderCell, {label:'Impr.', field:'impressions', sort, setSort}),
            h(HeaderCell, {label:'Last Pos', field:'position', sort, setSort}),
            h(HeaderCell, {label:'Suggested URL', field:'page', sort, setSort}),
            h('th', null, 'Actions')
          )
        ),
        h('tbody', null,
          (sorted.length ? sorted : []).map((r, i) => {
            const posBadgeTone = (r.position==null) ? '' : (r.position <= 10 ? 'green' : (r.position <= 20 ? 'blue' : 'red'));
            return h('tr', {key:i},
              h('td', null, r.keyword || '—'),
              h('td', null, r.intent || '—'),
              h('td', null, r.kd != null ? r.kd : '—'),
              h('td', null, r.opportunity != null ? r.opportunity : '—'),
              h('td', null, r.clicks != null ? r.clicks : '—'),
              h('td', null, r.impressions != null ? r.impressions : '—'),
              h('td', null,
                r.position!=null ? h(Badge, {tone:posBadgeTone}, r.position) : '—'
              ),
              h('td', null, r.page ? h('a', {href:r.page, target:'_blank'}, r.page) : h('span', {className:'miro-muted'}, '—')),
              h('td', {className:'miro-actions'},
                h('button', {className:'button', onClick: () => addToTracker(r)}, 'Add to Rank Tracker'),
                h('button', {className:'button', onClick: () => createDraft(r)}, 'Create Draft')
              )
            );
          })
        )
      )
    );
  };

  render(h(App), document.getElementById('miro-ai-keyword-lab-app'));
})();
