/* ============================================================
   Miro AI SEO Suite — Frontend JS (FINAL CLEAN VERSION)
   All tabs fixed & Optimizer PRO enabled
============================================================ */

(function(){

    function $(sel){ return document.querySelector(sel); }
    function $all(sel){ return document.querySelectorAll(sel); }
    function create(tag){ return document.createElement(tag); }

    const REST  = window.MiroAISuite?.rest || '/wp-json/miro/v1/';
    const NONCE = window.MiroAISuite?.nonce || '';

    /* ==========================================
       TAB SWITCHING
    ========================================== */
    document.addEventListener("click", (e)=>{
        if (!e.target.classList.contains("miro-tab-btn")) return;

        // hide all
        $all(".miro-tab-panel").forEach(p=>p.style.display="none");

        // show target
        const target = e.target.getAttribute("href");
        const panel  = document.querySelector(target);
        if (panel) panel.style.display = "block";

        // active button
        $all(".miro-tab-btn").forEach(b=>b.classList.remove("active"));
        e.target.classList.add("active");

        e.preventDefault();
    });

    /* ==========================================
       Unified AI Generator
    ========================================== */
    async function generate(prompt, context="ai-suite"){
        try {
            const res = await fetch(REST + "ai/complete", {
                method: "POST",
                headers: {
                    "X-WP-Nonce": NONCE,
                    "Content-Type": "application/json"
                },
                body: JSON.stringify({ prompt, context })
            });

            return await res.json();
        } catch (err){
            return { error: "AI request failed: " + err.message };
        }
    }

    function loading(box){
        box.innerHTML = `<div class="miro-loading">⏳ Generating…</div>`;
    }

    function copyButton(text){
        const btn = create("button");
        btn.className = "miro-copy-btn";
        btn.textContent = "Copy";

        btn.onclick = ()=>{
            navigator.clipboard.writeText(text);
            btn.textContent = "Copied!";
            setTimeout(()=> btn.textContent="Copy",1500);
        };

        return btn;
    }

/* ============================================================
   Title & Meta Generator — PRO VERSION
============================================================ */
const btnTD = $("#miro-td-generate");

if (btnTD) {
    btnTD.onclick = async () => {

        const input = $("#miro-td-input").value.trim();
        const out   = $("#miro-td-results");

        if (!input) {
            out.textContent = "Enter a topic or content.";
            return;
        }

        loading(out);

        const prompt = `
You are an elite SEO strategist.

Generate **5 SEO titles** and **5 meta descriptions** for this topic:

"${input}"

STRICT RULES:
- Titles: 45–65 characters
- No emojis
- Click-worthy, benefit focused
- Natural SEO (no stuffing)
- Descriptions: 130–155 characters
- High CTR style
- Clear, human, clean English

Return JSON ONLY:
{
  "titles": ["...", "...", "..."],
  "descriptions": ["...", "...", "..."]
}
        `;

        const j = await generate(prompt, "titlemeta");

        if (j.error) { out.textContent = j.error; return; }

        let txt = j.text.replace(/```json|```/g, "").trim();
        let data;

        try { data = JSON.parse(txt); }
        catch(e){
            out.textContent = "Invalid JSON from AI.";
            return;
        }

        const titles = data.titles || [];
        const descs  = data.descriptions || [];

        let html = `
            <div class="miro-card">
                <h3>SEO Titles</h3>
                <ul class="miro-list">
        `;

        titles.forEach(t=>{
            html += `
                <li>
                    <div class="miro-output">${t}</div>
                    <button class="miro-copy-btn"
                        onclick="navigator.clipboard.writeText('${t.replace(/'/g, "\\'")}')">Copy</button>
                </li>
            `;
        });

        html += `
                </ul>
                <h3 style="margin-top:20px;">Meta Descriptions</h3>
                <ul class="miro-list">
        `;

        descs.forEach(d=>{
            html += `
                <li>
                    <div class="miro-output">${d}</div>
                    <button class="miro-copy-btn"
                        onclick="navigator.clipboard.writeText('${d.replace(/'/g, "\\'")}')">Copy</button>
                </li>
            `;
        });

        html += `</ul></div>`;

        out.innerHTML = html;
    };
}



    /* ============================================================
       2) Bulk Rewriter
    ============================================================ */

    const btnBulk = $("#miro-bulk-generate");
    if (btnBulk){
        btnBulk.onclick = async ()=>{
            const raw = $("#miro-bulk-input").value.trim();
            const out = $("#miro-bulk-results");

            if (!raw){
                out.textContent = "Enter one sentence per line.";
                return;
            }

            loading(out);

            const lines = raw.split("\n").map(x=>x.trim()).filter(Boolean);

            const prompt = `
Rewrite each line to be clearer.

Return JSON:
{
  "rewritten": ["...", "..."]
}
Lines:
${lines.join("\n")}
            `;

            const j = await generate(prompt,"bulk_rewriter");
            if (j.error){ out.textContent = j.error; return; }

            let clean = j.text.replace(/```json|```/g,"");

            let data;
            try { data = JSON.parse(clean); }
            catch(e){ out.textContent="Invalid JSON"; return; }

            const rewritten = data.rewritten || [];

            out.innerHTML = "";
            const ul = create("ul");

            rewritten.forEach(r=>{
                const li = create("li");
                li.textContent = r;
                ul.append(li);
            });

            out.append(ul);
            out.append(copyButton(rewritten.join("\n")));
        };
    }


    /* ============================================================
       3) Simple Rewriter
    ============================================================ */

    const btnRewrite = $("#miro-rewrite-go");
    if (btnRewrite){
        btnRewrite.onclick = async ()=>{
            const text = $("#miro-rewrite-input").value.trim();
            const out  = $("#miro-rewrite-results");

            if (!text){
                out.textContent = "Enter text.";
                return;
            }

            loading(out);

            const prompt = `
Rewrite this text to be clearer and more SEO friendly.
Return only the rewritten text.

${text}
            `;

            const j = await generate(prompt,"rewriter");

            if (j.error){ out.textContent = j.error; return; }

            const output = j.text || "";

            out.innerHTML = `
                <div class="miro-card">
                    <h3>Rewritten Text</h3>
                    <div class="miro-output">${output}</div>
                </div>
            `;

            out.append(copyButton(output));
        };
    }


    /* ============================================================
       4) Keyword Suggestions
    ============================================================ */

    const btnKW = $("#miro-kw-generate");
    if (btnKW){
        btnKW.onclick = async ()=>{
            const topic = $("#miro-kw-input").value.trim();
            const out   = $("#miro-kw-results");

            if (!topic){
                out.textContent = "Enter topic.";
                return;
            }

            loading(out);

            const prompt = `
Generate 20 SEO keyword suggestions for:
"${topic}"
Return as plain list separated by newlines.
            `;

            const j = await generate(prompt,"keywords");

            if (j.error){ out.textContent=j.error; return; }

            const kws = (j.text||"").split("\n").map(s=>s.trim()).filter(Boolean);

            out.innerHTML = "";
            const ul = create("ul");

            kws.forEach(k=>{
                const li = create("li");
                li.textContent = k;
                ul.append(li);
            });

            out.append(ul);
            out.append(copyButton(kws.join("\n")));
        };
    }


    /* ============================================================
       5) FAQ Generator
    ============================================================ */

    const btnFAQ = $("#miro-faq-generate");
    if (btnFAQ){
        btnFAQ.onclick = async ()=>{
            const text = $("#miro-faq-input").value.trim();
            const out  = $("#miro-faq-results");

            if (!text){
                out.textContent="Enter content.";
                return;
            }

            loading(out);

            const prompt = `
Generate FAQs and Schema.

Return JSON:
{
  "faqs":[{"q":"...","a":"..."}],
  "schema": {...}
}

Topic:
${text}
            `;

            const j = await generate(prompt,"faq");

            if (j.error){ out.textContent=j.error; return; }

            let data;
            try { data = JSON.parse(j.text); }
            catch(e){ out.textContent="Invalid JSON"; return; }

            const faqs   = data.faqs || [];
            const schema = JSON.stringify(data.schema || {}, null, 2);

            out.innerHTML = "";

            const wrap = create("div");

            faqs.forEach(f=>{
                const box = create("div");
                box.className="miro-faq-item";
                box.innerHTML = `<strong>Q:</strong> ${f.q}<br><strong>A:</strong> ${f.a}`;
                wrap.append(box);
            });

            out.append(wrap);

            const pre = create("pre");
            pre.className="miro-schema-block";
            pre.textContent = schema;
            out.append(pre);
            out.append(copyButton(schema));
        };
    }


/* ============================================================
   Bulk Rewriter — PRO VERSION (Unified /ai/complete)
============================================================ */
const btnRW = $("#miro-rw-generate");

if (btnRW) {
    btnRW.onclick = async () => {

        const input  = $("#miro-rw-input").value.trim();
        const unique = $("#miro-rw-unique").checked;
        const out    = $("#miro-rw-results");

        if (!input) {
            out.textContent = "Enter paragraphs to rewrite.";
            return;
        }

        loading(out);

        // Split paragraphs cleanly
        const blocks = input
            .split(/\n\s*\n/)
            .map(s => s.trim())
            .filter(Boolean);

        /* -------------------------------
           Rewrite MODE
        --------------------------------*/
        const mode = unique
            ? "Rewrite each paragraph in a fully unique style while preserving meaning. Change structure, vocabulary, tone — avoid semantic similarity."
            : "Rewrite each paragraph professionally, longer, richer, cleaner, more fluent, more descriptive, and SEO-friendly. Add helpful detail if missing.";

        /* -------------------------------
           Full PRO Prompt
        --------------------------------*/
        const prompt = `
You are a senior content editor.

You must REWRITE each paragraph **aggressively**.

🔥 RULES:
- ALWAYS transform the sentences (never return original or similar sentences)
- Improve clarity, flow, grammar, structure
- Add mild depth or details only if helpful (do NOT change meaning)
- Use natural, human, premium writing style
- Improve SEO naturally (readability, context, quality)
- DO NOT keyword stuff
- DO NOT shorten; prefer longer & richer content
- DO NOT merge paragraphs
- OUTPUT must have EXACT same number of paragraphs as input

Mode: ${mode}

Input paragraphs:
${blocks.map((b, i) => `PARAGRAPH ${i + 1}:\n${b}`).join("\n\n")}

Return JSON ONLY:
{
  "rewritten": ["...", "...", ...]
}
        `;

        /* -------------------------------
           AI CALL
        --------------------------------*/
        const j = await generate(prompt, "bulk_rewriter");

        if (j.error) {
            out.textContent = j.error;
            return;
        }

        let clean = j.text.replace(/```json|```/g, "").trim();

        let data;
        try {
            data = JSON.parse(clean);
        } catch (e) {
            out.textContent = "⚠️ Invalid JSON returned by AI.";
            return;
        }

        const arr = data.rewritten || [];

        /* -------------------------------
           Build PRO UI Output
        --------------------------------*/
        let html = `
            <div class="miro-card">
                <h3>Rewritten Paragraphs</h3>
                <p class="miro-note">Improved style • Better clarity • SEO optimized</p>
        `;

        arr.forEach((p, i) => {
            html += `
                <div class="miro-block" style="margin-bottom:18px;">
                    <strong style="font-size:14px; color:#444;">Paragraph ${i + 1}</strong>
                    <div class="miro-output" style="margin-top:6px;">${p}</div>
                    <button class="miro-copy-btn" style="margin-top:6px;"
                        onclick="navigator.clipboard.writeText(\`${p.replace(/`/g, "\\`")}\`)">
                        Copy
                    </button>
                </div>
            `;
        });

        html += `</div>`;
        out.innerHTML = html;
    };
}

    /* ============================================================
       6) OPTIMIZER PRO — FINAL VERSION
    ============================================================ */

    const btnOpt = $("#miro-opt-score");
    if (btnOpt){
        btnOpt.onclick = async ()=>{

            const text     = $("#miro-opt-input").value.trim();
            const boxScore = $("#miro-opt-scorebox");
            const boxFixes = $("#miro-opt-fixes");
            const boxFinal = $("#miro-opt-final");

            if (!text){
                boxScore.textContent = "Enter your article.";
                return;
            }

            boxScore.innerHTML="";
            boxFixes.innerHTML="";
            boxFinal.innerHTML="";

            loading(boxScore);

            /* 🔥 ADVANCED PROMPT */
            const prompt = `
You are a senior SEO auditor.
Analyze content. Return JSON ONLY.

{
  "overall_score": 0-100,
  "scores": {
    "readability": {"value":0-100,"why":"..."},
    "structure":   {"value":0-100,"why":"..."},
    "keywords":    {"value":0-100,"why":"..."},
    "engagement":  {"value":0-100,"why":"..."},
    "clarity":     {"value":0-100,"why":"..."},
    "grammar":     {"value":0-100,"why":"..."},
    "seo":         {"value":0-100,"why":"..."}
  },
  "fixes":[
    {
      "area":"...",
      "what_to_fix":"- bullet list",
      "example":"..."
    }
  ]
}

Content:
${text}
            `;

            const j = await generate(prompt,"optimizer");

            if (j.error){
                boxScore.textContent=j.error;
                return;
            }

            let clean = j.text.replace(/```json|```/g,"");
            let data;

            try { data = JSON.parse(clean); }
            catch(e){ boxScore.textContent="Invalid JSON"; return; }

            const scores = data.scores || {};
            const fixes  = data.fixes  || [];
            const overall = data.overall_score || 0;

            /* SCORE UI */
            const cats = ["readability","structure","keywords","engagement","clarity","grammar","seo"];

            let htmlScores = `
                <div class="miro-card">
                    <h3>Overall Score: ${overall}/100</h3>
                    <div class="miro-score-grid">
            `;

            cats.forEach(cat=>{
                const v = scores[cat]?.value || 0;
                const why = scores[cat]?.why || "No explanation.";

                let color = "#3CB371";
                if (v < 80) color="#f7b733";
                if (v < 50) color="#ff5b5b";

                htmlScores += `
                    <div class="miro-score-item">
                        <div class="miro-score-label">${cat.toUpperCase()}</div>
                        <div class="miro-score-bar">
                            <div class="miro-score-fill" style="width:${v}%; background:${color};"></div>
                        </div>
                        <div class="miro-score-value">${v}/100</div>
                        <div class="miro-score-why">${why}</div>
                    </div>
                `;
            });

            htmlScores += `</div></div>`;
            boxScore.innerHTML = htmlScores;

            /* FIXES UI */
            let htmlFix = `
                <div class="miro-card">
                    <h3>Recommended Fixes</h3>
            `;

            fixes.forEach(f=>{
                htmlFix += `
                    <div class="miro-fix-block">
                        <strong>${f.area.toUpperCase()}</strong>
                        <p><strong>What to fix:</strong><br>${f.what_to_fix.replace(/\n/g,"<br>")}</p>
                        <p><strong>Example:</strong><br>${f.example}</p>
                    </div>
                `;
            });

            htmlFix += `</div>`;
            boxFixes.innerHTML = htmlFix;

            /* AUTO IMPROVE */
            const btn = create("button");
            btn.className="button button-secondary";
            btn.textContent="Auto-Improve Article";

            btn.onclick = async ()=>{
                loading(boxFinal);

                const fixPrompt = `
Improve the content using ONLY these FIX instructions:

${fixes.map(f=>`- ${f.area}: ${f.what_to_fix}`).join("\n")}

Return JSON:
{ "improved": "..." }

Original:
${text}
                `;

                const j2 = await generate(fixPrompt,"optimizer_fix");

                if (j2.error){
                    boxFinal.textContent=j2.error;
                    return;
                }

                let clean2 = j2.text.replace(/```json|```/g,"");

                let parsed;
                try { parsed = JSON.parse(clean2); }
                catch(e){ boxFinal.textContent="Invalid JSON"; return; }

                boxFinal.innerHTML = `
                    <div class="miro-card">
                        <h3>Improved Article</h3>
                        <div class="miro-output">${parsed.improved || ""}</div>
                    </div>
                `;
            };

            boxFinal.append(btn);
        };
    }

})();


