(function () {
  const { registerPlugin } = wp.plugins;
  const { PluginDocumentSettingPanel } = wp.editPost;
  const { PanelBody, Button, TextControl, Notice, Spinner } = wp.components;
  const { useSelect, dispatch, select } = wp.data;
  const { createElement: h, useState, useEffect } = wp.element;

  // --- SAME weights as your PHP (so we can color yellow client-side if server didn't send level/color)
  const weightMap = {
    // Core
    title_kw: 8, content_kw: 8, faq: 8,
    meta_kw: 7, length: 7, kw_first_10: 6, title_len: 5, meta_len: 5, slug_kw: 4,
    // Structure & Links
    h1_only_one: 4, heading_order: 4, first_link_internal: 4, internal_density: 5,
    ext_link: 3, ext_dofollow: 3, internal_links: 5,
    // Media
    image_count: 4, figcaption: 2, lazy_images: 2, rich_media: 2,
    // Keyword & slug hygiene
    kw_density: 5, kw_unique: 6, slug_len: 3, slug_stopwords: 3,
    // Title extras
    title_begin: 3, title_sentiment: 2, title_power: 2, title_number: 2,
    // Readability
    toc: 1, short_paras: 1, sentence_len: 2,
    // Misc
    subhead_kw: 3, img_alt_kw: 3
  };

  const palette = {
    green:  '#16a34a',
    yellow: '#f59e0b',
    red:    '#ef4444'
  };


  function insertLinkAtCursor(link) {
    try {
      const current = select('core/editor').getEditedPostContent() || '';
      const anchor = `<p><a href="${link.url}">${link.title || link.url}</a></p>`;
      dispatch('core/editor').editPost({ content: current + '\n' + anchor });
    } catch (_) {}
  }

  // Decide the bullet color with strong fallbacks:
  // 1) server-provided color
  // 2) server level → yellow if "warning"
  // 3) client weight map → yellow for 4–7, red for 8+, green if passed
  function colorForItem(it) {
    if (it && it.color) return it.color;
    if (it && it.level === 'warning') return palette.yellow;
    if (it && it.passed) return palette.green;

    const w = weightMap[it?.id] ?? 8; // default heavy if unknown
    if (w >= 8) return palette.red;
    if (w >= 4) return palette.yellow;
    return palette.red;
  }

  function Group({ title, items, postTitle, content, kw, excerpt, slug }) {
    const fixOne = async (it) => {
      try {
        const r = await wp.apiFetch({
          path: '/miro/v1/seo/fix',
          method: 'POST',
          data: { issue: it.id, title: postTitle, content, focus_keyword: kw, meta_desc: excerpt, slug }
        });
        if (r?.title)     dispatch('core/editor').editPost({ title: r.title });
        if (r?.meta_desc) dispatch('core/editor').editPost({ excerpt: r.meta_desc });
        if (r?.slug)      dispatch('core/editor').editPost({ slug: r.slug });
        if (r?.body_html) dispatch('core/editor').editPost({ content: r.body_html });
      } catch (e) {
        // Fix failed silently
      }
    };

    return h(
      'div',
      { style: { marginTop: '10px' } },
      h('h4', { style: { margin: '6px 0' } }, title),
      h(
        'ul',
        { style: { paddingLeft: '18px', margin: 0 } },
        (items || []).map((it, i) =>
          h(
            'li',
            { key: it.id || i, style: { marginBottom: '6px', listStyle: 'none' } },
            h('span', {
              style: {
                display: 'inline-block',
                width: '16px',
                height: '16px',
                borderRadius: '50%',
                background: colorForItem(it),
                marginRight: '8px',
                verticalAlign: 'middle'
              }
            }),
            h('span', { style: { fontWeight: it.passed ? '500' : '600' } }, it.label),
            !it.passed &&
              h(Button, { size: 'small', style: { marginLeft: '8px' }, onClick: () => fixOne(it) }, 'Fix with AI'),
            !it.passed && it.advice &&
              h('div', { style: { fontSize: '11px', color: '#555', marginLeft: '24px', marginTop: '2px' } }, it.advice)
          )
        )
      )
    );
  }

  function GSCQueries({ postId }) {
    const [rows, setRows] = useState(null);
    const [err, setErr] = useState('');

    useEffect(() => {
      let alive = true;
      async function load() {
        try {
          const res = await wp.apiFetch({
            path: `/miro/v1/gsc/queries?post_id=${postId}&days=28`,
            method: 'GET',
            headers: { 'X-WP-Nonce': (window.MiroAI && MiroAI.nonce) || '' }
          });
          if (alive) setRows(res.rows || []);
        } catch (e) {
          if (alive) setErr(e?.message || String(e));
        }
      }
      if (postId) load();
      return () => { alive = false; };
    }, [postId]);

    if (!postId) return h('div', null, 'Loading post…');
    if (err) return h('div', { style:{marginTop:'6px', color:'#b00'} }, 'GSC error: ' + err);
    if (!rows) return h('div', { style:{marginTop:'6px'} }, 'Loading GSC keywords…');
    if (rows.length === 0) return h('div', { style:{marginTop:'6px'} }, 'No GSC data yet.');

    return h('div', { style:{marginTop:'6px'} },
      h('table', { style:{width:'100%', borderCollapse:'collapse'} },
        h('thead', null,
          h('tr', null,
            h('th', { style:{textAlign:'left', borderBottom:'1px solid #ddd', padding:'4px'} }, 'Query'),
            h('th', { style:{textAlign:'right', borderBottom:'1px solid #ddd', padding:'4px'} }, 'Clicks'),
            h('th', { style:{textAlign:'right', borderBottom:'1px solid #ddd', padding:'4px'} }, 'Impr.'),
            h('th', { style:{textAlign:'right', borderBottom:'1px solid #ddd', padding:'4px'} }, 'CTR'),
            h('th', { style:{textAlign:'right', borderBottom:'1px solid #ddd', padding:'4px'} }, 'Pos.')
          )
        ),
        h('tbody', null,
          rows.slice(0, 10).map((r, i) =>
            h('tr', { key: i },
              h('td', { style:{ padding:'4px', borderBottom:'1px solid #f1f1f1' } }, r.query),
              h('td', { style:{ padding:'4px', textAlign:'right', borderBottom:'1px solid #f1f1f1' } }, r.clicks),
              h('td', { style:{ padding:'4px', textAlign:'right', borderBottom:'1px solid #f1f1f1' } }, r.impressions),
              h('td', { style:{ padding:'4px', textAlign:'right', borderBottom:'1px solid #f1f1f1' } }, ((r.ctr || 0) * 100).toFixed(2) + '%'),
              h('td', { style:{ padding:'4px', textAlign:'right', borderBottom:'1px solid #f1f1f1' } }, (r.position || 0).toFixed(1))
            )
          )
        )
      )
    );
  }

  function InternalLinks({ postId, onInsert }) {
    const [rows, setRows] = useState(null);
    const [err, setErr] = useState('');

    useEffect(() => {
      let alive = true;
      async function load() {
        try {
          const res = await wp.apiFetch({ path: `/miro/v1/links/suggest?post_id=${postId}&limit=10` });
          if (!alive) return;
          setRows(res.items || []);
        } catch (e) {
          if (!alive) return;
          setErr(e?.message || String(e));
        }
      }
      if (postId) load();
      return () => { alive = false; };
    }, [postId]);

    if (err) return h(Notice, { status: 'error', isDismissible: false }, 'Links failed: ' + err);
    if (!rows) return h('div', null, 'Loading suggestions…');
    if (rows.length === 0) return h('div', null, 'No suggestions yet (save the post to build embeddings).');

    return h('div', null,
      rows.map((r, i) => h('div', { key: i, style: { display:'flex', alignItems:'center', margin:'6px 0' } },
        h('div', { style:{flex:'1'} },
          h('div', { style:{fontWeight:'600'} }, r.title),
          h('div', { style:{fontSize:'11px', color:'#666'} }, r.url, ' • conf ', r.confidence)
        ),
        h(Button, { size:'small', onClick: () => onInsert(r) }, 'Insert at cursor')
      ))
    );
  }

  function Panel() {
    const postTitle = useSelect((s) => s('core/editor').getEditedPostAttribute('title'));
    const content = useSelect((s) => s('core/editor').getEditedPostContent());
    const slug = useSelect((s) => s('core/editor').getEditedPostAttribute('slug'));
    const excerpt = useSelect((s) => s('core/editor').getEditedPostAttribute('excerpt'));
    const postId = useSelect((s) => s('core/editor').getCurrentPostId(), []);
    const meta = useSelect((s) => s('core/editor').getEditedPostAttribute('meta') || {}, []);
    const { editPost } = dispatch('core/editor');
    const kw = meta.miro_focus_keyword || '';


    return h(
      PluginDocumentSettingPanel,
      { name: 'miro-ai-seo-panel', title: 'Miro AI SEO', className: 'miro-ai-seo-panel' },
      h(
        PanelBody,
        { initialOpen: true, title: 'SEO Assistant (Live)' },
        h('div', null,
          h(TextControl, {
            label: 'Focus Keyword',
            value: kw,
            onChange: (v) => editPost({ meta: { ...meta, miro_focus_keyword: v } }),
            placeholder: 'e.g. best wordpress seo'
          })
        )
      ),
      h(PanelBody, { initialOpen: true, title: 'GSC Keywords (28d)' }, h(GSCQueries, { postId })),
      h(PanelBody, { initialOpen: true, title: 'Internal Links' }, h(InternalLinks, { postId, onInsert: insertLinkAtCursor }))
    );
  }

  registerPlugin('miro-ai-seo', { render: Panel, icon: 'chart-area' });
})();
