(function(){
  const { registerPlugin } = wp.plugins;
  const { PluginSidebar } = wp.editPost;
  const { PanelBody, Button, CheckboxControl, Notice } = wp.components;
  const { useState } = wp.element;
  const apiFetch = wp.apiFetch;
  const select = wp.data.select;

  apiFetch.use( apiFetch.createNonceMiddleware(MiroIL.nonce) );

  function App(){
    const postId = select("core/editor").getCurrentPostId();
    const [items,setItems] = useState([]);
    const [picked,setPicked] = useState({});
    const [msg,setMsg] = useState("");

    async function load(){
      setMsg("Loading…");
      try{
        const res = await apiFetch({ path: `/miro/v1/il/suggest?post_id=${postId}` });
        setItems(res.items || []);
        setPicked({});
        setMsg(res.items?.length ? "" : "No safe anchors found in your content yet.");
      }catch(e){
        setMsg("Failed: " + (e?.message || e));
      }
    }

    async function apply(){
      const chosen = items.filter((it,idx)=> picked[idx]);
      if(!chosen.length){ setMsg("Pick at least 1 suggestion."); return; }
      setMsg("Applying…");
      try{
        const payload = { post_id: postId, items: chosen.map(x=>({anchor:x.anchor,url:x.url})) };
        const res = await apiFetch({ path:'/miro/v1/il/apply', method:'POST', data: payload });
        setMsg(`Inserted ${res.inserted||0} link(s).`);
      }catch(e){
        setMsg("Apply failed: " + (e?.message || e));
      }
    }

    async function undo(){
      setMsg("Undo…");
      try{
        await apiFetch({ path:'/miro/v1/il/undo', method:'POST', data:{post_id:postId} });
        setMsg("Undo done.");
      }catch(e){
        setMsg("Undo failed: " + (e?.message || e));
      }
    }

    return wp.element.createElement(PluginSidebar, { name:"miro-il", title:"Miro Internal Links" },
      wp.element.createElement(PanelBody, { title:"Suggestions", initialOpen:true },
        msg ? wp.element.createElement(Notice, { status:"info", isDismissible:false }, msg) : null,
        wp.element.createElement("div", { style:{display:"flex", gap:"8px", marginBottom:"10px"} },
          wp.element.createElement(Button, { variant:"primary", onClick: load }, "Find Suggestions"),
          wp.element.createElement(Button, { onClick: apply }, "Apply Selected"),
          wp.element.createElement(Button, { onClick: undo, isDestructive:true }, "Undo")
        ),
        items.map((it,idx)=> wp.element.createElement(CheckboxControl, {
          key: idx,
          label: `${it.anchor} → ${it.title}`,
          help: it.url,
          checked: !!picked[idx],
          onChange: (v)=> setPicked(Object.assign({}, picked, {[idx]: v}))
        }))
      )
    );
  }

  registerPlugin("miro-internal-links", { render: App });
})();
