(function(){

  const H = {
    "X-WP-Nonce": miroGSCKeywords.nonce,
    "Content-Type": "application/json"
  };

  const kwURL     = miroGSCKeywords.kwURL;
  const aiURL     = miroGSCKeywords.aiURL;
  const trURL     = miroGSCKeywords.trackURL;
  const syncStart = miroGSCKeywords.syncStart;
  const syncStatus= miroGSCKeywords.syncStatus;
  const siteName  = miroGSCKeywords.site;

  function $(sel){ return jQuery(sel); }
  function fmtInt(n){ return (n||0).toLocaleString(); }
  function fmtPct(x){ return ((Math.round((x||0)*1000)/10).toFixed(1))+"%"; }
  function fmtPos(p){ return (p>0? Number(p).toFixed(1) : "—"); }

  function esc(s){
    s = (s==null? "" : String(s));
    return s.replace(/[&<>\"\']/g, m => ({ "&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;" }[m]));
  }
  function jsonTry(text){ try{ return JSON.parse(text); }catch(_){ return null; } }
  function clamp(v,a,b){ return Math.max(a, Math.min(b, v)); }

  function metricLabel(m){
    if(m==="impressions") return "Impressions";
    if(m==="position") return "Top URL Position";
    if(m==="ctr") return "CTR";
    return "Clicks";
  }

  function metricFormat(m, v){
    if(v==null || Number.isNaN(v)) return "—";
    if(m==="ctr") return fmtPct(v);
    if(m==="position") return fmtPos(v);
    if(m==="impressions") return fmtInt(Math.round(v));
    return fmtInt(Math.round(v));
  }

  function normalizeSeries(m, arr){
    arr = Array.isArray(arr) ? arr.slice() : [];
    if(!arr.length) return arr;

    if(m==="position" || m==="ctr"){
      let last = null;
      for(let i=0;i<arr.length;i++){
        let v = arr[i];
        if(v==null || Number.isNaN(v)){
          arr[i] = (last==null ? null : last);
        } else {
          last = v;
        }
      }
      if(arr.every(v => v==null)) return arr.map(_=>0);
      const first = arr.find(v => v!=null);
      return arr.map(v => (v==null ? first : v));
    }

    return arr.map(v => (v==null || Number.isNaN(v)) ? 0 : Number(v));
  }

  function computeTrend(m, series){
    series = normalizeSeries(m, series);
    const n = series.length;
    if(n < 4) return 0;

    const slice = Math.min(5, Math.max(2, Math.floor(n/3)));
    const start = series.slice(0, slice);
    const end   = series.slice(n - slice);

    const avg = (a)=> a.reduce((s,x)=>s+(Number(x)||0),0)/Math.max(1,a.length);

    let t = avg(end) - avg(start);
    if(m==="position") t = -t; // improvement if position goes down
    if(m==="ctr") t = t * 100; // percentage points

    return t;
  }

  // draw position so "better" goes up
  function toDrawValues(metric, seriesOriginal){
    const s = normalizeSeries(metric, seriesOriginal);
    if(metric !== "position") return s;
    return s.map(v=>{
      const x = Number(v);
      if(!x || x <= 0) return 0;
      return 1 / x;
    });
  }

  function sparkBuild(metric, seriesOriginal){
    const seriesRaw = Array.isArray(seriesOriginal) ? seriesOriginal.slice() : [];
    const series    = normalizeSeries(metric, seriesRaw);
    const drawVals  = toDrawValues(metric, seriesOriginal);

    const W=160, Hh=40, P=4; // Match RankFlow size
    const n = drawVals.length || 0;

    if(!n){
      return { path:"", pts:[], series, seriesOriginal: seriesRaw, gradientId: "" };
    }

    let min = Infinity, max = -Infinity;
    for(const v of drawVals){
      const x = Number(v);
      if(Number.isFinite(x)){
        if(x < min) min = x;
        if(x > max) max = x;
      }
    }
    if(!Number.isFinite(min) || !Number.isFinite(max)){ min=0; max=1; }
    if(max === min){ max = min + 1; }

    // For position metric, flip values (lower position = better = higher on chart)
    let processedVals = drawVals.slice();
    let color = "#6b7280"; // subtle gray (matches design system)
    if(metric === "position"){
      const maxPos = max;
      processedVals = drawVals.map(v => maxPos - v);
      const fmax = Math.max(...processedVals);
      const fmin = Math.min(...processedVals);
      min = fmin;
      max = fmax;
      if(max === min) max = min + 1;
    }

    const step = (W-2*P)/Math.max(1,n-1);
    const pts=[];

    for(let i=0;i<n;i++){
      const v = Number(processedVals[i]) || 0;
      const x = P + i*step;
      const k = (v - min) / (max - min);
      const y = Hh-P-(k*(Hh-2*P));
      pts.push([x,y]);
    }

    // Build path through all points (lineTo ensures full range to last date)
    let d = `M${pts[0][0]},${pts[0][1]}`;
    for(let i=1;i<n;i++){
      d += ` L${pts[i][0]},${pts[i][1]}`;
    }

    // Create unique gradient ID
    const gradientId = "rfKwGrad_" + Math.random().toString(16).slice(2);

    return { path:d, pts, series, seriesOriginal: seriesRaw, gradientId, color };
  }

  function trendHTML(metric, trend){
    const t = parseFloat(trend||0);
    let cls = "rf-flat";
    if(t > 0) cls = "rf-up";
    if(t < 0) cls = "rf-down";

    const sym = t > 0 ? "▲" : (t < 0 ? "▼" : "—");
    const val = Math.abs(t).toFixed(1);
    const suf = (metric==="ctr") ? "pp" : "";

    return `<div class="rf-trend ${cls}">${sym} ${val}${suf?` ${suf}`:""}</div>`;
  }

  function arrowDeltaHTML(curr, prev, kind){
    const a = parseFloat(curr)||0, b = parseFloat(prev)||0;
    let d = a - b;
    if(kind === "pos"){ d = (a>0&&b>0) ? (b - a) : 0; } // positive = improvement
    const cls = d>0 ? "miro-delta-up" : (d<0 ? "miro-delta-down" : "miro-delta-zero");
    const sym = d>0 ? "▲" : (d<0 ? "▼" : "—");
    let text = "";
    if(kind === "ctr") text = Math.abs(d*100).toFixed(1)+"%";
    else if(kind==="pos") text = Math.abs(d).toFixed(1);
    else text = Math.abs(d).toLocaleString();
    return `<div class="miro-delta ${cls}">${sym} ${text}</div>`;
  }

  // === NO-CACHE BANNER / QUICK SYNC ===
  const $alert = $("#kwAlert");
  const $syncBtn = $("#kwQuickSync");
  const $syncTxt = $("#kwQuickSyncStatus");

  async function quickSync(){
    $syncTxt.show().text("Starting…");
    try{
      const r = await fetch(syncStart + "?days=90", { method:"POST", headers:H, credentials:"same-origin" });
      const t = await r.text(); const j = jsonTry(t) || {};
      if(!r.ok || !j.job_id){
        const msg = (j && (j.message || j.code)) ? (j.message || j.code) : "Error starting sync";
        $syncTxt.text(msg);
        return;
      }
      poll(j.job_id);
    }catch(e){ $syncTxt.text("Error: " + (e?.message||"unknown")); }
  }

  async function poll(job){
    const iv = setInterval(async ()=>{
      try{
        const r = await fetch(syncStatus + "?job=" + encodeURIComponent(job) + "&_t=" + Date.now(), { headers:H, credentials:"same-origin", cache: "no-store" });
        const t = await r.text(); const s = jsonTry(t) || {};
        if(s.error){ $syncTxt.text("Error: " + s.error); clearInterval(iv); return; }
        $syncTxt.text((s.step||"Syncing") + " — " + (s.progress||0) + "%");
        if(s.done){
          clearInterval(iv);
          $syncTxt.text("Done");
          $alert.hide();
          kwPage = 1;
          await loadKeywords();
          await loadAIPlaybook();
          $syncTxt.text("Ready");
        }
      }catch(_){}
    }, 2500);
  }
  $syncBtn.on("click", e=>{ e.preventDefault(); quickSync(); });

  // === AI PLAYBOOK ===
  const aiBtn   = $("#aiBuild");
  const aiDev   = $("#aiDevice");
  const aiCtry  = $("#aiCountry");
  const aiDays  = $("#aiDays");
  const aiBox   = $("#kwAI");
  const aiStat  = $("#aiStatus");

  async function loadAIPlaybook(){
    aiStat.show(); aiBox.text("");
    const qs = new URLSearchParams({
      device: aiDev.val()||"",
      country: aiCtry.val()||"",
      days: aiDays.val()||"30",
      site: siteName,
      _t: Date.now()
    });
    try{
      const res = await fetch(aiURL + "?" + qs.toString(), { headers:H, credentials:"same-origin", cache: "no-store" });
      const txt = await res.text(); const data = jsonTry(txt);
      if(!res.ok){
        if(data?.code==="no_cache"){ $alert.show(); aiBox.html("No GSC cache yet. Run a sync."); }
        else aiBox.html("Couldn't generate AI suggestions. " + esc(data?.message || txt.slice(0,200)));
        return;
      }
      aiBox.text(data?.summary || "No suggestions.");
    }catch(e){
      aiBox.html("Couldn't generate AI suggestions. " + esc(e?.message||e));
    }
    finally{ aiStat.hide(); }
  }

  aiBtn.on("click", ()=>{ loadAIPlaybook(); });

  // === KEYWORDS TABLE ===
  let kwPage = 1;
  let lastItems = [];
  let sparkDates = [];
  let sparkCache = {}; // idx|metric => {pts, seriesOriginal}

  const kwSearch   = $("#kwSearch");
  const kwDevice   = $("#kwDevice");
  const kwCountry  = $("#kwCountry");
  const kwDays     = $("#kwDays");
  const kwPer      = $("#kwPer");
  const kwSort     = $("#kwSort");
  const kwSegment  = $("#kwSegment");
  const kwTrendMetric = $("#kwTrendMetric");
  const kwTotalB   = $("#kwTotal");
  const kwPrevBtn  = $("#kwPrev")[0];
  const kwNextBtn  = $("#kwNext")[0];
  const kwPageLbl  = $("#kwPage")[0];
  const kwExport   = $("#kwExport");
  const tbody      = $("#kwBody");

  // Populate country dropdowns from server (guarantees list is never empty)
  function fillCountrySelects(){
    let list = (typeof miroGSCKeywords !== "undefined" && Array.isArray(miroGSCKeywords.countries)) ? miroGSCKeywords.countries : [];
    if (!list.length) {
      list = [
        { code: "", name: "All countries" },
        { code: "US", name: "United States" }, { code: "GB", name: "United Kingdom" }, { code: "CA", name: "Canada" },
        { code: "AU", name: "Australia" }, { code: "DE", name: "Germany" }, { code: "FR", name: "France" }, { code: "IN", name: "India" }
      ];
    }
    [kwCountry[0], $("#aiCountry")[0]].forEach(function(sel){
      if (!sel) return;
      sel.innerHTML = "";
      list.forEach(function(c){
        const opt = document.createElement("option");
        opt.value = c.code || "";
        opt.textContent = c.name || c.code || "All countries";
        sel.appendChild(opt);
      });
    });
  }
  fillCountrySelects();

  $("#kwReload").on("click", ()=>{ kwPage = 1; loadKeywords(); });
  kwPrevBtn.addEventListener("click", ()=>{ if(kwPage>1){ kwPage--; loadKeywords(); } });
  kwNextBtn.addEventListener("click", ()=>{ kwPage++; loadKeywords(); });

  kwTrendMetric.on("change", ()=>{
    sparkCache = {};
    rerenderTrendCells();
  });

  kwExport.on("click", async ()=>{
    const per = parseInt(kwPer.val()||"50",10);
    const qs = new URLSearchParams({
      q: kwSearch.val()||"",
      device: kwDevice.val()||"",
      country: kwCountry.val()||"",
      days: kwDays.val()||"30",
      page: 1,
      per_page: per,
      sort: kwSort.val()||"clicks_desc",
      segment: kwSegment.val()||"",
      csv: "1"
    });
    const res = await fetch(kwURL+"?"+qs.toString(), {headers:H, credentials:"same-origin"});
    const blob = await res.blob();
    const a = document.createElement("a");
    a.href = URL.createObjectURL(blob);
    a.download = "keywords.csv";
    a.click();
    URL.revokeObjectURL(a.href);
  });

  let trackedSet = new Set();

  async function fetchTrackedSet(){
    try {
      const sep = trURL.indexOf("?") >= 0 ? "&" : "?";
      const res = await fetch(trURL + sep + "list=1&_t=" + Date.now(), {
        headers: H,
        credentials: "same-origin",
        cache: "no-store"
      });
      const j = jsonTry(await res.text()) || {};
      const kw = j.keywords || [];
      trackedSet = new Set(kw.map(function(x){ return (x||"").toLowerCase(); }));
    } catch (_) { trackedSet = new Set(); }
  }

  function isTracked(q){
    return q && trackedSet.has(String(q).toLowerCase());
  }

  async function trackKeyword(q){
    const res = await fetch(trURL, {
      method:"POST", headers:H, credentials:"same-origin",
      body:JSON.stringify({ q })
    });
    const t = await res.text(); const j=jsonTry(t)||{};
    if(!res.ok || j.code) throw new Error(j.message||t||"Track failed");
    trackedSet.add(String(q).toLowerCase());
    return true;
  }

  function rowActionsHTML(q){
    if (isTracked(q)) {
      return `<span class="muted" style="font-size:11px;">Tracked</span>`;
    }
    return `<button class="btn tiny" data-act="track" data-q="${esc(q)}">Track</button>`;
  }

  function pageCellHTML(row){
    const url   = row.page_url || "";
    const title = row.page_title || "";
    const score = row.page_score;
    const idx   = row.page_index || "";
    const words = row.page_words || 0;

    if(!url) return "—";

    let top = esc(title || url);
    let link = `<a href="${esc(url)}" target="_blank" rel="noopener noreferrer">${top}</a>`;

    const badges = [];
    if(typeof score === "number" && !isNaN(score) && score > 0){
      const sRounded = Math.round(score);
      let sClass = "miro-badge-chip miro-badge-score";
      if(sRounded < 50) sClass += " low";
      else if(sRounded < 80) sClass += " mid";
      else sClass += " high";
      badges.push(`<span class="${sClass}">Score ${sRounded}</span>`);
    }
    if(idx) badges.push(`<span class="miro-badge-chip miro-badge-index">${esc(idx)}</span>`);
    if(words) badges.push(`<span class="miro-badge-chip miro-badge-words">${words.toLocaleString()} words</span>`);

    return `<div class="miro-page-cell" style="display:flex;align-items:center;gap:8px;flex-wrap:nowrap;white-space:nowrap;">
      <div style="flex:1;min-width:0;overflow:hidden;text-overflow:ellipsis;">${link}</div>
      ${badges.length ? `<div class="miro-page-badges" style="display:flex;align-items:center;gap:6px;flex-shrink:0;">${badges.join("")}</div>` : ""}
    </div>`;
  }

  function getSeries(row, metric){
    if(row && row.series && row.series[metric] && Array.isArray(row.series[metric])){
      return row.series[metric];
    }
    return row.spark || [];
  }

  function buildTrendCellHTML(row, idx){
    const metric = kwTrendMetric.val() || "clicks";
    const seriesOriginal = getSeries(row, metric);

    const t = computeTrend(metric, seriesOriginal);
    const { path, pts, seriesOriginal: originalForTip, gradientId, color } = sparkBuild(metric, seriesOriginal);

    // keep stroke neutral; CSS can style it
    sparkCache[idx+"|"+metric] = { pts, seriesOriginal: normalizeSeries(metric, originalForTip) };

    return `
      <div class="rf-trend-cell-wrapper" style="display:flex;flex-direction:column;gap:6px;align-items:flex-start;">
        <div class="rf-spark-wrap" data-idx="${idx}">
          <svg class="rf-spark" viewBox="0 0 160 40" data-metric="${esc(metric)}" data-row-idx="${idx}" preserveAspectRatio="none">
            <path d="${path}" fill="none" stroke="${color}" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
            <circle class="rf-spark-dot" r="3.5" cx="0" cy="0" fill="${color}" opacity="0" style="transition: opacity 0.12s ease;"/>
          </svg>
        </div>
        <div class="rf-trend-details" style="display:flex;flex-direction:column;gap:2px;align-items:flex-start;width:100%;">
          <div class="rf-metric" style="font-size:11px;white-space:nowrap;">${esc(metricLabel(metric))}</div>
          ${trendHTML(metric, t)}
        </div>
      </div>
    `;
  }

  function rerenderTrendCells(){
    if(!lastItems.length) return;
    for(let i=0;i<lastItems.length;i++){
      const tr = tbody.find(`tr[data-idx="${i}"]`)[0];
      if(!tr) continue;
      const td = tr.querySelector("td.kwTrendCell");
      if(!td) continue;
      td.innerHTML = buildTrendCellHTML(lastItems[i], i);
    }
  }

  async function loadKeywords(){
    await fetchTrackedSet();
    const per = parseInt(kwPer.val()||"25",10);
    const qs = new URLSearchParams({
      q: kwSearch.val() || "",
      device: kwDevice.val() || "",
      country: kwCountry.val() || "",
      days: kwDays.val() || "30",
      page: kwPage,
      per_page: per,
      sort: kwSort.val() || "clicks_desc",
      segment: kwSegment.val() || "",
      _t: Date.now()
    });

    tbody.html("");
    lastItems = [];
    sparkDates = [];
    sparkCache = {};

    let res, text, data;
    try{
      res  = await fetch(kwURL + "?" + qs.toString(), { headers:H, credentials:"same-origin", cache: "no-store" });
      text = await res.text();
      data = jsonTry(text);
    }catch(e){
      alert("Network error loading keywords: " + (e?.message||"unknown"));
      kwPageLbl.textContent = "Page " + kwPage;
      kwTotalB.text("Total: 0");
      kwPrevBtn.disabled = (kwPage<=1);
      kwNextBtn.disabled = true;
      return;
    }

    if(!res.ok){
      if(data?.code==="no_cache" || data?.data?.status===400){
        const msgEl = document.getElementById("kwAlertMessage");
        if (msgEl && data.message) msgEl.textContent = data.message;
        $alert.show();
      } else alert(data?.message || "Error loading keywords.");
      kwPageLbl.textContent = "Page " + kwPage;
      kwTotalB.text("Total: 0");
      kwPrevBtn.disabled = (kwPage<=1);
      kwNextBtn.disabled = true;
      return;
    }

    if (data?.code === 'no_cache') {
      const msgEl = document.getElementById("kwAlertMessage");
      if (msgEl && data.message) msgEl.textContent = data.message;
      $alert.show();
    } else {
      $alert.hide();
    }

    const items = (data.items||[]);
    lastItems = items;

    sparkDates = Array.isArray(data.spark_dates) ? data.spark_dates : [];

    items.forEach((row, idx)=>{
      const clicks = fmtInt(row.clicks);
      const impr   = fmtInt(row.impressions);
      const ctr    = fmtPct(row.ctr);

      // ✅ table position = TOP URL position (now matches drawer top URL)
      const pos    = fmtPos(row.position);

      const clicksPrev = row.clicks_prev||0;
      const imprPrev   = row.impressions_prev||0;
      const ctrPrev    = row.ctr_prev||0;
      const posPrev    = row.position_prev||0;

      tbody.append(`
        <tr data-idx="${idx}">
          <td class="kw-q">${esc(row.query)}</td>
          <td>${esc(row.intent||"—")}${row.is_brand? " <span class='miro-badge lite'>Brand</span>": ""}</td>
          <td>${pageCellHTML(row)}</td>
          <td><div class="kw-num">${clicks}</div>${arrowDeltaHTML(row.clicks, clicksPrev, "clicks")}</td>
          <td><div class="kw-num">${impr}</div>${arrowDeltaHTML(row.impressions, imprPrev, "impr")}</td>
          <td><div class="kw-num">${ctr}</div>${arrowDeltaHTML(row.ctr, ctrPrev, "ctr")}</td>
          <td><div class="kw-num">${pos}</div>${arrowDeltaHTML(row.position, posPrev, "pos")}</td>
          <td>${fmtInt(Math.round(row.opportunity||0))}</td>
          <td>${row.pages||1}</td>
          <td class="kwTrendCell">${buildTrendCellHTML(row, idx)}</td>
          <td>${rowActionsHTML(row.query)}</td>
        </tr>
      `);
    });

    tbody.off("click", "button[data-act=track]").on("click","button[data-act=track]", async function(e){
      e.stopPropagation();
      const q = this.getAttribute("data-q");
      this.disabled = true; const old = this.textContent; this.textContent = "Adding…";
      try{
        await trackKeyword(q);
        this.outerHTML = "<span class='muted' style='font-size:11px;'>Tracked</span>";
      }catch(e){
        alert("Could not add: " + (e?.message||"Unknown"));
        this.disabled = false; this.textContent = old;
      }
    });

    tbody.off("click", "tr").on("click","tr", function(e){
      if (e.target.closest("button")) return;
      const idx = parseInt(this.getAttribute("data-idx") || "-1", 10);
      if (!isNaN(idx) && idx >= 0 && idx < lastItems.length){
        openDrawer(lastItems[idx]);
      }
    });

    kwPageLbl.textContent = "Page " + (data.page||1);
    kwTotalB.text("Total: " + fmtInt(data.total||0));

    kwPrevBtn.disabled = (kwPage<=1);
    const lastPage = data.total ? Math.ceil(data.total / per) : null;
    kwNextBtn.disabled = lastPage ? (kwPage >= lastPage) : ((items||[]).length < per);
  }

  // === Spark hover tooltip (EXACT DAY + VALUE) ===
  const kwTooltip = document.getElementById("rfKwTooltip");
  
  tbody.off("mousemove", ".rf-spark-wrap").on("mousemove", ".rf-spark-wrap", function(e){
    if(!kwTooltip) return;
    const svg = this.querySelector("svg.rf-spark");
    if(!svg) return;

    const rowIdx = parseInt(svg.getAttribute("data-row-idx") || "-1", 10);
    const metric = (kwTrendMetric.val() || "clicks");
    const key = rowIdx + "|" + metric;

    const cached = sparkCache[key];
    if(!cached || !cached.pts || !cached.pts.length) return;

    const rect = svg.getBoundingClientRect();
    const n = cached.pts.length;
    if (rect.width <= 0) return;

    const W = 160, P = 4;
    const relX = e.clientX - rect.left;
    const viewBoxX = relX * (W / rect.width);
    let t = (viewBoxX - P) / (W - 2 * P);
    t = Math.max(0, Math.min(1, t));
    const pointIdx = clamp(Math.round(t * (n - 1)), 0, n - 1);

    const pt = cached.pts[pointIdx];
    const dot = svg.querySelector(".rf-spark-dot");

    if(dot && pt){
      dot.setAttribute("cx", pt[0]);
      dot.setAttribute("cy", pt[1]);
      dot.setAttribute("opacity", "1");
    }

    const date = sparkDates[pointIdx] || ("Day " + (pointIdx + 1));
    const val = cached.seriesOriginal[pointIdx];
    const row = lastItems[rowIdx];
    const query = row ? (row.query || "") : "";

    // Format value based on metric
    let valText = "";
    if(metric === "ctr"){
      valText = ((val || 0) * 100).toFixed(1) + "%";
    } else if(metric === "position"){
      valText = (val || 0).toFixed(1);
    } else {
      valText = (val || 0).toLocaleString();
    }

    // Set tooltip content
    kwTooltip.innerHTML =
      '<div style="font-weight:600;margin-bottom:2px;">' + esc(query) + '</div>' +
      '<div>Date: ' + date + '</div>' +
      '<div>' + esc(metricLabel(metric)) + ': ' + valText + '</div>' +
      '<div style="color:#64748b;">Old → New (left to right)</div>';

    kwTooltip.style.display = "block";
    kwTooltip.style.position = "fixed";
    kwTooltip.style.visibility = "hidden"; // Hidden to measure

    // Get tooltip dimensions
    const tipRect = kwTooltip.getBoundingClientRect();
    const tipWidth = tipRect.width || 220;
    const tipHeight = tipRect.height || 120;

    kwTooltip.style.visibility = "visible"; // Make visible

    // Position tooltip (to the right of cursor)
    let tx = e.clientX + 12;
    let ty = e.clientY - 40;

    // Boundary checks
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;

    if(tx + tipWidth > viewportWidth - 8){
      tx = e.clientX - tipWidth - 12; // Show on left if not enough space on right
    }
    if(tx < 8) tx = 8;
    if(ty < 8) ty = 8;
    if(ty + tipHeight > viewportHeight - 8){
      ty = viewportHeight - tipHeight - 8;
    }

    kwTooltip.style.left = tx + "px";
    kwTooltip.style.top = ty + "px";
  });

  tbody.off("mouseleave", ".rf-spark-wrap").on("mouseleave", ".rf-spark-wrap", function(){
    const dot = this.querySelector(".rf-spark-dot");
    if(dot) dot.setAttribute("opacity", "0");
    if(kwTooltip) kwTooltip.style.display = "none";
  });

  // === Drawer logic ===
  const drawer      = document.getElementById("kwDrawer");
  const drawerMeta  = document.getElementById("kwDrawerMeta");
  const drawerTitle = document.getElementById("kwDrawerTitle");
  const drawerMain  = document.getElementById("kwDrawerMain");
  const drawerPages = document.getElementById("kwDrawerPages");
  const drawerClose = drawer ? drawer.querySelector(".miro-kw-close") : null;

  function closeDrawer(){ if(drawer) drawer.classList.remove("open"); }

  function openDrawer(row){
    if(!drawer) return;
    const days = kwDays.val() || "30";
    const seg  = kwSegment.val() || "";

    drawerTitle.textContent = row.query || "";
    drawerMeta.textContent = `Window: last ${days} days` + (seg ? ` • Segment: ${seg}` : "");

    let html = "";
    if (row.page_url){
      html += `<div class="kw-drawer-block">
        <div class="kw-drawer-label">Primary page</div>
        ${pageCellHTML(row)}
      </div>`;
    } else {
      html += `<div class="kw-drawer-label">No primary page detected for this query.</div>`;
    }

    // ✅ show BOTH positions so it’s not “fake”
    const topPos = row.position || 0;
    const kwPos  = row.position_keyword || 0;
    const topPrev= row.position_prev || 0;
    const kwPrev = row.position_keyword_prev || 0;

    html += `<div class="kw-drawer-stats">
      <div><strong>Clicks:</strong> ${fmtInt(row.clicks||0)} <span class="muted">(prev ${fmtInt(row.clicks_prev||0)})</span></div>
      <div><strong>Impressions:</strong> ${fmtInt(row.impressions||0)} <span class="muted">(prev ${fmtInt(row.impressions_prev||0)})</span></div>
      <div><strong>Top URL pos:</strong> ${fmtPos(topPos)} <span class="muted">(prev ${fmtPos(topPrev)})</span></div>
      <div><strong>Keyword avg pos:</strong> ${fmtPos(kwPos)} <span class="muted">(prev ${fmtPos(kwPrev)})</span></div>
      <div><strong>CTR:</strong> ${fmtPct(row.ctr||0)} <span class="muted">(prev ${fmtPct(row.ctr_prev||0)})</span></div>
    </div>`;

    drawerMain.innerHTML = html;

    const pages = row.pages_list || [];
    if (pages.length){
      let table = `<div class="kw-drawer-block">
        <div class="kw-drawer-label">Ranking URLs for this keyword</div>
        <table>
          <thead>
            <tr><th>URL</th><th>Clicks</th><th>Impr</th><th>Pos</th></tr>
          </thead>
          <tbody>`;
      pages.forEach(p=>{
        table += `<tr>
          <td><a href="${esc(p.url||"")}" target="_blank" rel="noopener noreferrer">${esc(p.url||"")}</a></td>
          <td>${fmtInt(p.clicks||0)}</td>
          <td>${fmtInt(p.impressions||0)}</td>
          <td>${fmtPos(p.position||0)}</td>
        </tr>`;
      });
      table += `</tbody></table></div>`;
      drawerPages.innerHTML = table;
    } else {
      drawerPages.innerHTML = "";
    }

    drawer.classList.add("open");
  }

  if(drawerClose) drawerClose.addEventListener("click", closeDrawer);
  if(drawer) drawer.addEventListener("click", (e)=>{ if(e.target === drawer) closeDrawer(); });

  window.addEventListener("miro-tab-activated", function(e){
    if (e.detail && e.detail.tab === "tab-keywords") {
      loadKeywords();
      loadAIPlaybook();
    }
  });

  loadKeywords();
  loadAIPlaybook();

})();
