(function(){
  if (!window.miroFixCenter) return;

  const NONCE = miroFixCenter.nonce;
  const API   = miroFixCenter.rest;

  function $(sel){ return jQuery(sel); }
  function esc(s){
    return String(s||"").replace(/[&<>"']/g, m => ({
      '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'
    }[m]));
  }
  function fmtInt(n){ return (n||0).toLocaleString(); }
  function fmtPos(p){ return (p>0 ? Number(p).toFixed(1) : "—"); }
  function fmtPct(x){ return (Number(x||0).toFixed(2)) + "%"; }

  function keyOf(page){
    try { return encodeURIComponent(String(page||"")); }
    catch(e){ return String(page||""); }
  }

  // Safe: append query params whether URL has ? or not
  function withParams(url, params){
    url = String(url||"");
    const sep = url.indexOf("?") === -1 ? "?" : "&";
    const q = Object.keys(params||{}).map(k => (
      encodeURIComponent(k) + "=" + encodeURIComponent(String(params[k]))
    )).join("&");
    return url + (q ? (sep + q) : "");
  }

  // Headers
  function H_get(){
    return { "X-WP-Nonce": NONCE, "Accept": "application/json" };
  }
  function H_post(){
    return { "X-WP-Nonce": NONCE, "Content-Type": "application/json", "Accept": "application/json" };
  }

  // Flag labels (must match REST output keys)
  const ISSUE_LABEL = {
    missing_title: "Missing SEO title",
    missing_meta: "Missing meta description",
    title_too_long: "Title too long",
    title_too_short: "Title too short",
    meta_too_long: "Meta too long",
    meta_too_short: "Meta too short",
    top_query_missing_in_title: "Top query missing in title",
    top_query_missing_in_meta: "Top query missing in meta",
    brand_takes_space: "Brand takes space"
  };

  let STATE = {
    rows: [],
    filtered: [],
    selected: new Set(),   // key(page)
    open: null,
    map: new Map(),        // key -> row
    lastScan: { from:"", to:"", days:30 }
  };

  async function getJSON(url){
    const r = await fetch(url, {
      method:"GET",
      headers: H_get(),
      credentials:"same-origin"   // ✅ keep cookies
    });
    const t = await r.text();
    let j=null; try{ j=JSON.parse(t); }catch(e){}
    if (!r.ok) throw (j && j.message) ? j.message : t;
    return j;
  }

  async function postJSON(url, body){
    const r = await fetch(url, {
      method:"POST",
      headers: H_post(),
      credentials:"same-origin",  // ✅ critical for WP auth cookies
      body: JSON.stringify(body||{})
    });
    const t = await r.text();
    let j=null; try{ j=JSON.parse(t); }catch(e){}
    if (!r.ok) throw (j && j.message) ? j.message : t;
    return j;
  }

  function setStatus(msg){ $("#mfc-status").text(msg); }
  function setProgress(pct){
    pct = Math.max(0, Math.min(100, pct||0));
    $("#mfc-progress-bar").css("width", pct + "%");
  }

  function applySearchFilter(){
    const q = String($("#mfc-search").val() || "").trim().toLowerCase();
    if (!q) { STATE.filtered = STATE.rows.slice(); return; }

    STATE.filtered = STATE.rows.filter(r => {
      const a = String(r.wp_title || "").toLowerCase();
      const b = String(r.page || "").toLowerCase();
      const c = String(r.top_query || "").toLowerCase();
      return a.includes(q) || b.includes(q) || c.includes(q);
    });
  }

  // clean flags for table: show 2 chips + +N
  function renderFlagChips(row){
    const issues = Array.isArray(row.issues) ? row.issues : [];
    if (!issues.length) return `<span class="mfc-flag mfc-flag-ok">OK</span>`;

    const show = issues.slice(0,2);
    const more = issues.length - show.length;

    const chips = show.map(k => {
      const label = ISSUE_LABEL[k] || k;

      let cls = "mfc-flag";
      if (k.includes("missing")) cls += " is-bad";
      else if (k.includes("too_long") || k.includes("too_short") || k.includes("missing_in_") || k.includes("brand_")) cls += " is-warn";
      else cls += " is-info";

      return `<span class="${cls}" title="${esc(label)}">${esc(label)}</span>`;
    }).join(" ");

    const moreChip = more > 0
      ? ` <span class="mfc-flag is-more" title="${more} more">+${more}</span>`
      : "";

    return chips + moreChip;
  }

  function renderIssues(row){
    const issues = Array.isArray(row.issues) ? row.issues : [];
    if (!issues.length) return `<div class="af-muted">No issues detected. This is a pure CTR opportunity.</div>`;

    return `<ul style="margin:0;padding-left:18px;">
      ${issues.slice(0,10).map(x => `<li>${esc(ISSUE_LABEL[x] || x)}</li>`).join("")}
    </ul>`;
  }

  function renderPresets(row){
    const p = row.presets || {};
    const parts = [];

    ["safe","medium","aggressive"].forEach(k => {
      if (!p[k]) return;
      parts.push(`
        <button type="button" class="button mfc-preset" data-preset="${k}">
          ${esc(p[k].label || k)}
          <span class="af-muted" style="margin-left:6px;">Pred CTR: <strong>${fmtPct(p[k].pred_ctr||0)}</strong> · +${fmtInt(p[k].extra_clicks||0)} clicks</span>
        </button>
      `);
    });

    if (!parts.length) return `<div class="af-muted">No presets.</div>`;
    return parts.join(" ");
  }

  function renderTable(){
    const $tb = $("#mfc-table tbody");
    $tb.empty();

    const rows = STATE.filtered;
    if (!rows.length){
      $tb.append(`<tr><td colspan="10" class="af-muted">No results. Try lower min impressions or higher max position.</td></tr>`);
      $("#mfc-bulk-apply").prop("disabled", true);
      return;
    }

    rows.forEach((r) => {
      const key = keyOf(r.page);
      const checked = STATE.selected.has(key) ? 'checked' : '';
      const actionText = (r.post_id || (r.term_id && r.taxonomy)) ? 'Open' : 'View';
      const flagsHtml = renderFlagChips(r);

      const tag = r.post_id ? 'Post' : (r.term_id ? (String(r.taxonomy||'').replace('_',' ') || 'Term') : 'URL');
      const tagHtml = `<span class="mfc-flag is-info" style="margin-right:6px;">${esc(tag)}</span>`;

      $tb.append(`
        <tr data-key="${key}" class="mfc-row">
          <td><input class="mfc-check" type="checkbox" data-key="${key}" ${checked} /></td>
          <td>
            <div style="font-weight:650;">${esc(r.wp_title || r.page)}</div>
            <div class="af-muted" style="font-size:12px;word-break:break-all;margin-top:2px;">${esc(r.page)}</div>
            <div class="af-muted" style="font-size:12px;margin-top:4px;">
              ${tagHtml}
              Top query: <strong>${esc(r.top_query || "—")}</strong>
            </div>
          </td>
          <td>${fmtPos(r.pos)}</td>
          <td>${fmtPct(r.ctr)}</td>
          <td>${fmtPct(r.expected_ctr)}</td>
          <td><strong>${fmtInt(r.extra_clicks||0)}</strong></td>
          <td>${fmtInt(r.impressions||0)}</td>
          <td>${fmtInt(r.clicks||0)}</td>
          <td>${flagsHtml}</td>
          <td><button type="button" class="button mfc-open" data-key="${key}">${actionText}</button></td>
        </tr>
      `);
    });

    $("#mfc-bulk-apply").prop("disabled", STATE.selected.size === 0);
  }

  function getRowByKey(key){
    return STATE.map.get(String(key||"")) || null;
  }

  function updateAfterPreview(){
    const t = $("#mfc-d-title-in").val() || "";
    const d = $("#mfc-d-desc-in").val() || "";
    $("#mfc-d-after").html(`
      <div style="font-weight:650;margin-bottom:4px;">${esc(t || "(empty)")}</div>
      <div class="af-muted" style="word-break:break-all;margin-bottom:6px;">${esc(STATE.open ? STATE.open.page : "")}</div>
      <div>${esc(d || "(empty)")}</div>
    `);
  }

  function openDrawer(row){
    STATE.open = row;

    const $drawer = $("#mfc-drawer");
    $drawer.attr("aria-hidden", "false").addClass("open is-open");
    $drawer.css("pointer-events","auto");

    $("#mfc-d-title").text(row.wp_title || row.page);
    $("#mfc-d-url").text(row.page);

    const explain = `You rank around ${fmtPos(row.pos)} but CTR is ${fmtPct(row.ctr)}. Expected is ${fmtPct(row.expected_ctr)}. Fixing snippet can win ~${fmtInt(row.extra_clicks||0)} extra clicks in this window.`;
    $("#mfc-d-explain").text(explain);

    $("#mfc-d-metrics").html(
      `Impr: <strong>${fmtInt(row.impressions||0)}</strong> · Clicks: <strong>${fmtInt(row.clicks||0)}</strong> · Window: <strong>${esc(STATE.lastScan.from)}</strong> → <strong>${esc(STATE.lastScan.to)}</strong>`
    );

    $("#mfc-d-query").text(row.top_query || "—");

    if ($("#mfc-d-issues").length) {
      $("#mfc-d-issues").html(renderIssues(row));
    }

    const curTitle = (row.current && row.current.title) ? row.current.title : "";
    const curDesc  = (row.current && row.current.desc) ? row.current.desc : "";
    const curSource= (row.current && row.current.source) ? row.current.source : "";

    const entityLabel = row.post_id ? "Post" : (row.term_id ? ("Term: " + (row.taxonomy || "")) : "URL");
    const entityExtra = row.term_id ? ` · term_id: <strong>${esc(row.term_id)}</strong>` : "";

    $("#mfc-d-before").html(`
      <div style="font-weight:650;margin-bottom:4px;">${esc(curTitle || "(no SEO title set)")}</div>
      <div class="af-muted" style="word-break:break-all;margin-bottom:6px;">${esc(row.page)}</div>
      <div>${esc(curDesc || "(no meta description set)")}</div>
      <div class="af-muted" style="margin-top:8px;">
        Source: <strong>${esc(curSource||"")}</strong> · ${esc(entityLabel)}${entityExtra}
      </div>
    `);

    if ($("#mfc-d-presets").length){
      $("#mfc-d-presets").html(renderPresets(row));
    }

    // default inputs
    const safe = row.presets && row.presets.safe ? row.presets.safe : null;
    if (safe){
      $("#mfc-d-title-in").val(safe.title || "");
      $("#mfc-d-desc-in").val(safe.meta || "");
    } else {
      $("#mfc-d-title-in").val(curTitle || "");
      $("#mfc-d-desc-in").val(curDesc || "");
    }

    updateAfterPreview();
    $("#mfc-d-msg").text("");
  }

  function closeDrawer(){
    STATE.open = null;
    $("#mfc-drawer").attr("aria-hidden", "true").removeClass("open is-open");
  }

  async function doScan(){
    setProgress(5);
    setStatus("Scanning…");

    const days    = parseInt($("#mfc-days").val() || "30", 10);
    const minImpr = parseInt($("#mfc-min-impr").val() || "0", 10);
    const maxPos  = parseInt($("#mfc-max-pos").val() || "12", 10);

    const url = withParams(API.scan, { days, min_impr: minImpr, max_pos: maxPos });

    try{
      const j = await getJSON(url);
      const rows = Array.isArray(j.rows) ? j.rows : [];

      STATE.lastScan = { from: j.from || "", to: j.to || "", days };
      STATE.rows = rows;
      STATE.selected.clear();

      STATE.map = new Map();
      rows.forEach(r => STATE.map.set(keyOf(r.page), r));

      applySearchFilter();
      renderTable();

      setProgress(100);
      setStatus(`Done. Found ${rows.length} opportunities.`);
      setTimeout(()=>setProgress(0), 800);
    } catch(err){
      setProgress(0);
      setStatus("Scan failed: " + err);
    }
  }

  async function doApply(row, title, meta, alsoWP){
    const body = {
      // post OR term support ✅
      post_id: row.post_id || 0,
      term_id: row.term_id || 0,
      taxonomy: row.taxonomy || "",

      title: title || "",
      meta: meta || "",
      also_wp_title: alsoWP ? 1 : 0,
      scope: "detected",
      also_miro: 1,
      baseline: {
        from: STATE.lastScan.from,
        to: STATE.lastScan.to,
        pos: row.pos,
        ctr: row.ctr,
        expected_ctr: row.expected_ctr,
        impressions: row.impressions,
        clicks: row.clicks,
        extra_clicks: row.extra_clicks
      }
    };
    return await postJSON(API.apply, body);
  }

  async function doUndo(row, idx){
    // NOTE: if you later add term undo route, you can extend this.
    return await postJSON(API.undo, { post_id: row.post_id, backup_index: idx || 0 });
  }

  async function doBulkApply(){
    const keys = Array.from(STATE.selected);
    if (!keys.length) return;

    setStatus("Bulk apply started…");
    setProgress(0);

    let ok = 0, fail = 0;

    for (let i=0;i<keys.length;i++){
      const key = keys[i];
      const row = getRowByKey(key);

      const isTerm = (!row?.post_id && row?.term_id && row?.taxonomy);
      if (!row || (!row.post_id && !isTerm)) { fail++; continue; }

      const safe = row.presets && row.presets.safe ? row.presets.safe : null;
      const title = safe ? (safe.title || "") : "";
      const meta  = safe ? (safe.meta  || "") : "";

      try{
        await postJSON(API.apply, {
          post_id: row.post_id || 0,
          term_id: row.term_id || 0,
          taxonomy: row.taxonomy || "",

          title,
          meta,
          also_wp_title: 0,
          scope: "detected",
          also_miro: 1,
          baseline: {
            from: STATE.lastScan.from,
            to: STATE.lastScan.to,
            pos: row.pos,
            ctr: row.ctr,
            expected_ctr: row.expected_ctr,
            impressions: row.impressions,
            clicks: row.clicks,
            extra_clicks: row.extra_clicks
          }
        });
        ok++;
      } catch(e){
        fail++;
      }

      setProgress(Math.round(((i+1)/keys.length)*100));
      setStatus(`Bulk applying… ${i+1}/${keys.length} (OK: ${ok}, Fail: ${fail})`);
    }

    setStatus(`Bulk done. OK: ${ok}, Fail: ${fail}.`);
    setTimeout(()=>setProgress(0), 1200);
  }

  // -------------------------
  // Events
  // -------------------------
  $(document).on("click", "#mfc-scan", function(e){ e.preventDefault(); doScan(); });

  $(document).on("input", "#mfc-search", function(){
    applySearchFilter();
    renderTable();
  });

  $(document).on("change", "#mfc-check-all", function(){
    const on = $(this).is(":checked");
    STATE.selected.clear();
    if (on){
      (STATE.filtered || []).forEach(r => STATE.selected.add(keyOf(r.page)));
    }
    renderTable();
  });

  $(document).on("change", ".mfc-check", function(){
    const key = String($(this).attr("data-key") || "");
    if (!key) return;

    if ($(this).is(":checked")) STATE.selected.add(key);
    else STATE.selected.delete(key);

    $("#mfc-bulk-apply").prop("disabled", STATE.selected.size === 0);
  });

  // Open drawer
  $(document).on("click", ".mfc-open", function(e){
    e.preventDefault();

    let key = String($(this).attr("data-key") || "");
    if (!key){
      const $tr = $(this).closest("tr[data-key]");
      key = String($tr.attr("data-key") || "");
    }

    if (!key){
      setStatus("Open failed: no key found. Re-scan.");
      return;
    }

    const row = getRowByKey(key);
    if (!row){
      setStatus("Row not found. Re-scan.");
      return;
    }

    openDrawer(row);
  });

  $(document).on("click", "#mfc-d-close", function(e){ e.preventDefault(); closeDrawer(); });

  $(document).on("input", "#mfc-d-title-in, #mfc-d-desc-in", function(){
    updateAfterPreview();
  });

  $(document).on("click", ".mfc-preset", function(e){
    e.preventDefault();
    if (!STATE.open) return;
    const k = String($(this).data("preset") || "");
    const p = STATE.open.presets && STATE.open.presets[k] ? STATE.open.presets[k] : null;
    if (!p) return;

    $("#mfc-d-title-in").val(p.title || "");
    $("#mfc-d-desc-in").val(p.meta || "");
    updateAfterPreview();
    $("#mfc-d-msg").text(`Selected: ${p.label || k}`);
  });

  // Apply (POST or TERM)
  $(document).on("click", "#mfc-d-apply", async function(e){
    e.preventDefault();
    if (!STATE.open) return;

    const row = STATE.open;
    const isTerm = (!row.post_id && row.term_id && row.taxonomy);

    if (!row.post_id && !isTerm){
      $("#mfc-d-msg").text("This URL is not mapped to a WordPress post or term.");
      return;
    }

    const title = String($("#mfc-d-title-in").val() || "").trim();
    const meta  = String($("#mfc-d-desc-in").val() || "").trim();
    const alsoWP = $("#mfc-d-also-wp").is(":checked");

    $("#mfc-d-msg").text("Applying…");

    try{
      const res = await doApply(row, title, meta, alsoWP);
      $("#mfc-d-msg").text("Applied. Backup saved.");

      row.current = row.current || {};
      row.current.title = title;
      row.current.desc  = meta;
      row.current.source = "updated";

      // Refresh before panel
      $("#mfc-d-before").html(`
        <div style="font-weight:650;margin-bottom:4px;">${esc(title || "(no SEO title set)")}</div>
        <div class="af-muted" style="word-break:break-all;margin-bottom:6px;">${esc(row.page)}</div>
        <div>${esc(meta || "(no meta description set)")}</div>
        <div class="af-muted" style="margin-top:8px;">Updated.</div>
      `);

    } catch(err){
      $("#mfc-d-msg").text("Apply failed: " + err);
    }
  });

  // Undo (posts only right now unless you add term undo endpoint)
  $(document).on("click", "#mfc-d-undo", async function(e){
    e.preventDefault();
    if (!STATE.open) return;

    const row = STATE.open;
    if (!row.post_id){
      $("#mfc-d-msg").text("Undo currently supports posts only.");
      return;
    }

    $("#mfc-d-msg").text("Undoing…");

    try{
      const j = await doUndo(row, 0);
      $("#mfc-d-msg").text("Undo complete.");

      if (j && j.now){
        row.current = row.current || {};
        row.current.title = j.now.title || "";
        row.current.desc  = j.now.desc || "";
        row.current.source = j.now.source || "";

        $("#mfc-d-title-in").val(row.current.title);
        $("#mfc-d-desc-in").val(row.current.desc);
        updateAfterPreview();

        $("#mfc-d-before").html(`
          <div style="font-weight:650;margin-bottom:4px;">${esc(row.current.title || "(no SEO title set)")}</div>
          <div class="af-muted" style="word-break:break-all;margin-bottom:6px;">${esc(row.page)}</div>
          <div>${esc(row.current.desc || "(no meta description set)")}</div>
          <div class="af-muted" style="margin-top:8px;">Source: <strong>${esc(row.current.source||"")}</strong></div>
        `);
      }
    } catch(err){
      $("#mfc-d-msg").text("Undo failed: " + err);
    }
  });

  $(document).on("click", "#mfc-bulk-apply", function(e){
    e.preventDefault();
    doBulkApply();
  });

})();
