(function(){
  const cfg = window.MIRO_GSC_OVERVIEW || {};
  const H   = {"X-WP-Nonce": cfg.nonce || ""},
        sumURL = cfg.sumURL || "",
        winURL = cfg.winURL || "",
        losURL = cfg.losURL || "",
        aiURL  = cfg.aiURL || "",
        countriesURL = cfg.countriesURL || "";

  let days         = 28;
  let trendMetric  = 'impressions';
  let winnersData  = [];
  let losersData   = [];
  let winnersRange = null;
  let losersRange  = null;

  const fmt = {
    int: n => (n||0).toLocaleString(),
    pct: x => ((x||0)*100).toFixed(1) + '%',
    pos: x => (x||0).toFixed(1)
  };

  function fmtTs(ts){
    if(!ts) return '—';
    try{
      const d = new Date(ts*1000);
      return d.toLocaleString();
    }catch(e){ return '—'; }
  }

  function normalize(arr, invert=false){
    if(!arr || arr.length===0) return [];
    let src = arr.slice();
    if(invert){
      const mx = Math.max(...src);
      src = src.map(v => mx - v);
    }
    const mn = Math.min(...src), mx = Math.max(...src);
    if(mx === mn) return src.map(_=>0.5);
    return src.map(v => (v - mn) / (mx - mn));
  }

  function smooth(arr){
    if(!arr || arr.length<3) return arr.slice();
    let out = [];
    for(let i=0;i<arr.length;i++){
      const a = arr[i-1] ?? arr[i];
      const b = arr[i];
      const c = arr[i+1] ?? arr[i];
      out.push((a+b+c)/3);
    }
    return out;
  }

  function deltaClass(delta, invert=false){
    if(Math.abs(delta) < 1e-9) return 'flat';
    return invert ? (delta < 0 ? 'up' : 'down') : (delta > 0 ? 'up' : 'down');
  }

  function setKPI(valId, deltaId, val, delta, type, invert=false){
    const vEl = document.getElementById(valId);
    const dEl = document.getElementById(deltaId);
    if(!vEl || !dEl) return;
    vEl.textContent = type==='pct' ? fmt.pct(val) : type==='pos' ? fmt.pos(val) : fmt.int(val);
    const cls  = deltaClass(delta, invert);
    const sign = cls==='up' ? '+' : cls==='down' ? '-' : '';
    const abs  = Math.abs(delta);
    const dTxt = type==='pct' ? fmt.pct(abs) : type==='pos' ? fmt.pos(abs) : fmt.int(abs);
    dEl.innerHTML = 'Δ <span class="'+cls+'">'+sign+dTxt+'</span>';
  }

  const METRIC_GOLD = '#D8B15A';
  const METRIC_GOLD_HOVER = '#C49A4A';
  const METRIC_RED = '#ef4444';
  const METRIC_FLAT = '#94a3b8';

  function drawDonut(svgId, ratio, delta, goodUp=true){
    const el = document.getElementById(svgId);
    if(!el) return;
    const pct = Math.max(0,Math.min(1,ratio||0));
    const size = 100, cx = size/2, cy = size/2, r = size/2 - 10;
    const angle   = pct * 360;
    const largeArc= angle > 180 ? 1 : 0;
    const rad     = (angle - 90) * Math.PI / 180;
    const x       = cx + r * Math.cos(rad);
    const y       = cy + r * Math.sin(rad);
    const strokeW = 10;

    const colorGood = METRIC_GOLD;
    const colorBad  = METRIC_RED;
    const colorFlat = METRIC_FLAT;
    const color = (delta > 0 ? (goodUp ? colorGood : colorBad) : (delta < 0 ? (goodUp ? colorBad : colorGood) : colorFlat));

    const bgCircle = '<circle cx="'+cx+'" cy="'+cy+'" r="'+r+'" fill="none" stroke="#f1f5f9" stroke-width="'+strokeW+'" stroke-linecap="round"/>';
    const mainArc = '<path d="M'+cx+' '+(cy-r)+' A'+r+' '+r+' 0 '+largeArc+' 1 '+x+' '+y+'" '+
      'stroke="'+color+'" stroke-width="'+strokeW+'" fill="none" stroke-linecap="round"/>';

    el.innerHTML = bgCircle + mainArc;

    const valId = svgId + 'Val';
    const valEl = document.getElementById(valId);
    if(valEl) valEl.textContent = Math.round(pct * 100) + '%';
  }

  function setMetricTrend(trendId, delta, type){
    const el = document.getElementById(trendId);
    if(!el) return;
    if (delta === undefined || delta === null || (typeof delta === 'number' && Math.abs(delta) < 1e-9)) {
      el.textContent = 'Flat';
      el.className = 'ov-metric-trend flat';
      return;
    }
    const isUp = delta > 0;
    const sign = isUp ? '+' : '';
    const txt = type === 'pct' ? sign + ((delta || 0) * 100).toFixed(1) + '%' : sign + fmt.int(Math.abs(delta));
    el.textContent = (isUp ? '↑ ' : '↓ ') + txt;
    el.className = 'ov-metric-trend ' + (isUp ? 'up' : 'down');
  }

  function sparkline(vals, color){
    if(!vals || !vals.length) return '';
    color = color || '#2563eb';
    const W=120,H=26,P=2;
    const mn = Math.min(...vals);
    const mx = Math.max(...vals);
    const norm = (mx===mn) ? vals.map(_=>0.5) : vals.map(v=>(v-mn)/(mx-mn));
    const step = (W-2*P)/Math.max(1,norm.length-1);

    const pts = norm.map((n,i)=>{
      const x = P + i*step;
      const y = H-P-n*(H-2*P);
      return {x,y};
    });

    let line = '';
    pts.forEach((p,i)=>{ line += (i===0?'M':' L')+p.x+' '+p.y; });

    return ''
      + '<path class="spark-line" d="'+line+'"/>'
      + '<circle class="s-dot" r="2.5" fill="'+color+'" stroke="#ffffff" stroke-width="1" opacity="0"></circle>';
  }

  function updateDonutNote(){
    const note = document.getElementById('donutNote');
    if(!note) return;
    if (days === 7) note.textContent = '7-day view – good for spikes, more volatile. Use 28d or 90d for stable decisions.';
    else if (days === 28) note.textContent = '28-day window – recommended for weekly/monthly SEO decisions.';
    else if (days === 90) note.textContent = '90-day window – best for long-term SEO trends and seasonality.';
    else note.textContent = '';
  }

  function trendLabel(metric){
    switch(metric){
      case 'impressions': return 'Trend (impressions/day)';
      case 'ctr':         return 'Trend (CTR over time)';
      case 'position':    return 'Trend (avg position over time)';
      case 'clicks':
      default:            return 'Trend (clicks/day)';
    }
  }
  function updateTrendHeader(){
    const wHead = document.getElementById('wTrendHead');
    const lHead = document.getElementById('lTrendHead');
    const label = trendLabel(trendMetric);
    if (wHead) wHead.textContent = label;
    if (lHead) lHead.textContent = label;
  }
  function seriesForMetric(item){
    if (!item) return [];
    switch(trendMetric){
      case 'impressions': return item.spark_impr || item.spark || [];
      case 'ctr':         return item.spark_ctr || item.spark || [];
      case 'position':    return item.spark_pos || item.spark || [];
      case 'clicks':
      default:            return item.spark_clicks || item.spark || [];
    }
  }

  function attachSparkHandlers(){
    const tip = document.getElementById('ovSparkTooltip');
    if (!tip) return;

    const sparks = document.querySelectorAll('#tab-overview svg.spark');
    sparks.forEach(svg=>{
      svg.addEventListener('mousemove', function(e){
        const kind   = svg.getAttribute('data-kind') || 'winner';
        const rowIdx = parseInt(svg.getAttribute('data-idx') || '0', 10);

        const list  = (kind === 'loser') ? losersData : winnersData;
        const item  = list[rowIdx];
        if (!item) return;

        const vals  = seriesForMetric(item);
        const n     = vals.length;
        if (!n) return;

        const rect  = svg.getBoundingClientRect();
        const W=120,H=26,P=2;
        const relX  = e.clientX - rect.left;
        let t       = (relX - P) / (W - 2*P);
        t           = Math.max(0, Math.min(1, t));
        const pointIdx = Math.round(t * (n-1));

        const mn = Math.min(...vals);
        const mx = Math.max(...vals);
        const norm = (mx===mn) ? vals.map(_=>0.5) : vals.map(v=>(v-mn)/(mx-mn));
        const step = (W-2*P)/Math.max(1,n-1);
        const x = P + pointIdx * step;
        const y = H - P - norm[pointIdx]*(H-2*P);

        const dot = svg.querySelector('.s-dot');
        if (dot){
          dot.setAttribute('cx', x);
          dot.setAttribute('cy', y);
          dot.setAttribute('opacity', '1');
        }

        let range   = (kind === 'loser') ? losersRange : winnersRange;
        let dateStr = '';
        if (range && range.start){
          const d0 = new Date(range.start + 'T00:00:00Z');
          d0.setDate(d0.getDate() + pointIdx);
          dateStr = d0.toISOString().slice(0,10);
        }

        let metricLabel = (trendMetric==='impressions') ? 'Impressions'
                        : (trendMetric==='ctr')         ? 'CTR'
                        : (trendMetric==='position')    ? 'Avg position'
                        : 'Clicks';

        const v = vals[pointIdx] || 0;
        let valText = (trendMetric==='ctr') ? fmt.pct(v)
                   : (trendMetric==='position') ? fmt.pos(v)
                   : fmt.int(v);

        // Set content first so we can measure tooltip dimensions
        tip.innerHTML =
          '<div style="font-weight:600;margin-bottom:2px;">'+(item.query||'')+'</div>' +
          (dateStr ? '<div>Date: '+dateStr+'</div>' : '') +
          '<div>'+metricLabel+': '+valText+'</div>' +
          '<div style="color:#64748b;">Old → New (left to right)</div>';

        tip.style.display  = 'block';
        tip.style.position = 'fixed';
        tip.style.visibility = 'hidden'; // Hidden to measure without flashing
        
        // Get tooltip dimensions
        const tipRect = tip.getBoundingClientRect();
        const tipWidth = tipRect.width || 220;
        const tipHeight = tipRect.height || 120;
        
        tip.style.visibility = 'visible'; // Make visible now
        
        // Position based on column: Winners (right) = tooltip on left, Losers (left) = tooltip on right
        const isWinner = (kind !== 'loser');
        let tx, ty;
        const offset = 12;
        
        if (isWinner) {
          // Winners column (right side) - show tooltip on the left side of cursor
          tx = e.clientX - tipWidth - offset;
        } else {
          // Losers column (left side) - show tooltip on the right side of cursor
          tx = e.clientX + offset;
        }
        ty = e.clientY - 40;
        
        // Boundary checks to keep tooltip on screen
        const viewportWidth = window.innerWidth;
        const viewportHeight = window.innerHeight;
        
        if (tx < 8) tx = 8;
        if (tx + tipWidth > viewportWidth - 8) tx = viewportWidth - tipWidth - 8;
        if (ty < 8) ty = 8;
        if (ty + tipHeight > viewportHeight - 8) ty = viewportHeight - tipHeight - 8;

        tip.style.left = tx + 'px';
        tip.style.top  = ty + 'px';
      });

      svg.addEventListener('mouseleave', function(){
        const dot = svg.querySelector('.s-dot');
        if (dot) dot.setAttribute('opacity','0');
        if (tip) tip.style.display = 'none';
      });
    });
  }

  function renderMovers(){
    updateTrendHeader();
    const wb = document.getElementById('winnersBody');
    const lb = document.getElementById('losersBody');

    if (wb){
      wb.innerHTML = '';
      (winnersData || []).forEach((it,idx)=>{
        const vals = seriesForMetric(it);
        wb.innerHTML += '<tr>'+
          '<td>'+it.query+'</td>'+
          '<td><span class="badge '+(it.delta_clicks>=0?'up':'down')+'">'+(it.delta_clicks>=0?'+':'')+fmt.int(it.delta_clicks)+'</span></td>'+
          '<td>'+fmt.pct(it.ctr)+'</td>'+
          '<td>'+fmt.pos(it.position)+'</td>'+
          '<td><svg class="spark" data-kind="winner" data-idx="'+idx+'">'+sparkline(vals, '#2563eb')+'</svg></td>'+
        '</tr>';
      });
    }

    if (lb){
      lb.innerHTML = '';
      (losersData || []).forEach((it,idx)=>{
        const vals = seriesForMetric(it);
        lb.innerHTML += '<tr>'+
          '<td>'+it.query+'</td>'+
          '<td><span class="badge '+(it.delta_clicks>=0?'up':'down')+'">'+(it.delta_clicks>=0?'+':'')+fmt.int(it.delta_clicks)+'</span></td>'+
          '<td>'+fmt.pct(it.ctr)+'</td>'+
          '<td>'+fmt.pos(it.position)+'</td>'+
          '<td><svg class="spark" data-kind="loser" data-idx="'+idx+'">'+sparkline(vals, '#dc2626')+'</svg></td>'+
        '</tr>';
      });
    }

    const wCountEl = document.getElementById('winnersCount');
    if (wCountEl) wCountEl.textContent = (winnersData || []).length;
    const lCountEl = document.getElementById('losersCount');
    if (lCountEl) lCountEl.textContent = (losersData || []).length;

    attachSparkHandlers();
  }

  function renderTrust(sum){
    const pills = document.getElementById('dataTrustPills');
    if(!pills) return;

    const s = sum && sum.trust ? sum.trust : null;
    if(!s){
      pills.innerHTML = '<span class="trust-pill">Totals: property-level date series</span>';
      return;
    }

    const siteOk = s.site_ok ? 'trust-ok' : 'trust-warn';
    const qOk    = s.query_ok ? 'trust-ok' : 'trust-warn';

    pills.innerHTML = ''
      + '<span class="trust-pill"><span class="'+siteOk+'">Totals</span>: updated '+fmtTs(s.site_updated_at)+' • range '+(s.site_range?.start||'—')+' → '+(s.site_range?.end||'—')+'</span>'
      + '<span class="trust-pill"><span class="'+qOk+'">Query cache</span>: rows '+(s.query_rows||0)+' • updated '+fmtTs(s.query_updated_at)+'</span>';
  }

  async function loadAll(){
    updateDonutNote();

    const q = new URLSearchParams({days:String(days)});
    const [sum,wins,los] = await Promise.all([
      fetch(sumURL+'?'+q,{headers:H,credentials:'same-origin'}).then(r=>r.json()),
      fetch(winURL+'?window='+days,{headers:H,credentials:'same-origin'}).then(r=>r.json()),
      fetch(losURL+'?window='+days,{headers:H,credentials:'same-origin'}).then(r=>r.json()),
    ]);

    const wt = document.getElementById('winnersTitle');
    if (wt) wt.textContent = 'Winners ('+days+'d)';
    const lt = document.getElementById('losersTitle');
    if (lt) lt.textContent = 'Losers ('+days+'d)';

    if(sum && sum.ok){
      renderTrust(sum);

      const traffic = (typeof sum.search_traffic === 'number') ? sum.search_traffic : sum.clicks;
      setKPI('kpiClicks','kpiClicksDelta',traffic, sum.delta.clicks,'int');
      setKPI('kpiImpr','kpiImprDelta',sum.impressions, sum.delta.impressions,'int');
      setKPI('kpiKw','kpiKwDelta',sum.total_keywords || 0, sum.delta_keywords || 0,'int');
      setKPI('kpiVis','kpiVisDelta',sum.visibility, sum.delta.visibility,'int');

      const dVis = (sum.delta && typeof sum.delta.visibility==='number') ? sum.delta.visibility : 0;
      const dCtr = (sum.delta && typeof sum.delta.ctr==='number') ? sum.delta.ctr : 0;
      const dPage1 = (sum.delta && typeof sum.delta.page1_share==='number') ? sum.delta.page1_share : 0;
      const dBrand = (sum.delta && typeof sum.delta.brand_share==='number') ? sum.delta.brand_share : 0;
      const dNonBrand = (sum.delta && typeof sum.delta.nonbrand_share==='number') ? sum.delta.nonbrand_share : 0;

      drawDonut('donutVis',      sum.visibility_norm,  dVis, true);
      setMetricTrend('donutVisTrend',  dVis, 'int');
      drawDonut('donutCTR',      sum.ctr_norm,        dCtr, true);
      setMetricTrend('donutCTRTrend',  dCtr, 'pct');
      drawDonut('donutRank',     sum.page1_share || 0, dPage1, true);
      setMetricTrend('donutRankTrend', dPage1, 'pct');
      drawDonut('donutBrand',    sum.brand_share || 0, dBrand, false);
      setMetricTrend('donutBrandTrend', dBrand, 'pct');
      drawDonut('donutNonBrand', sum.nonbrand_share || 0, dNonBrand, true);
      setMetricTrend('donutNonBrandTrend', dNonBrand, 'pct');

      loadAI(sum);
      loadCountries();
    }

    winnersData  = (wins && wins.ok) ? (wins.items || []) : [];
    losersData   = (los && los.ok) ? (los.items || []) : [];
    winnersRange = (wins && wins.ok) ? (wins.range || null) : null;
    losersRange  = (los && los.ok) ? (los.range || null) : null;

    renderMovers();
  }

  async function loadAI(sum){
    const box = document.getElementById('aiInsights');
    if(!box) return;
    try{
      const payload = {
        visibility: sum.visibility,
        clicks: sum.clicks,
        impressions: sum.impressions,
        ctr: sum.ctr,
        position: sum.position,
        delta: sum.delta
      };
      const res = await fetch(aiURL,{
        method:'POST',
        headers:{...H,'Content-Type':'application/json'},
        credentials:'same-origin',
        body:JSON.stringify(payload)
      }).then(r=>r.json());
      box.innerHTML = res && res.ok && res.html ? res.html : 'No insights available.';
    }catch(e){
      box.innerHTML = 'Error generating insights.';
    }
  }

  async function loadCountries(){
    const box = document.getElementById('topCountries');
    if(!box || !countriesURL) return;
    
    // Update title to show selected days
    const card = box.closest('.card');
    if(card){
      const titleEl = card.querySelector('.title');
      if(titleEl){
        titleEl.textContent = 'Top Countries (' + days + 'd)';
      }
    }
    
    try{
      const url = countriesURL + (countriesURL.includes('?') ? '&' : '?') + 'days=' + days;
      const res = await fetch(url, {
        headers: H,
        credentials: 'same-origin'
      }).then(r=>r.json());
      
      if(res && res.ok && res.countries && res.countries.length > 0){
        // Filter out UNKNOWN countries
        const validCountries = res.countries.filter(c => c.country && c.country !== 'UNKNOWN' && c.country !== '');
        
        if(validCountries.length > 0){
          // Calculate total clicks for proper bar percentage
          const totalClicks = validCountries.reduce((sum, c) => sum + (c.clicks || 0), 0);
          
          // Try to get country name using Intl.DisplayNames if available
          let countryNames = {};
          try {
            if (typeof Intl !== 'undefined' && Intl.DisplayNames) {
              const dn = new Intl.DisplayNames(['en'], { type: 'region' });
              validCountries.forEach(c => {
                try {
                  const name = dn.of(c.country);
                  if (name) countryNames[c.country] = name;
                } catch(e) {}
              });
            }
          } catch(e) {}
          
          // ISO-3 to ISO-2 mapper (GSC returns ISO-3, flags need ISO-2)
          function iso3ToIso2(code){
            if(!code || typeof code !== 'string') return '';
            const upper = code.trim().toUpperCase();
            // If already 2 letters, just lowercase
            if(upper.length === 2 && /^[A-Z]{2}$/.test(upper)){
              return upper.toLowerCase();
            }
            // Complete ISO-3 to ISO-2 mapping (all countries)
            const map = {
              'AFG': 'af', 'ALB': 'al', 'DZA': 'dz', 'ASM': 'as', 'AND': 'ad', 'AGO': 'ao', 'AIA': 'ai', 'ATA': 'aq',
              'ATG': 'ag', 'ARG': 'ar', 'ARM': 'am', 'ABW': 'aw', 'AUS': 'au', 'AUT': 'at', 'AZE': 'az', 'BHS': 'bs',
              'BHR': 'bh', 'BGD': 'bd', 'BRB': 'bb', 'BLR': 'by', 'BEL': 'be', 'BLZ': 'bz', 'BEN': 'bj', 'BMU': 'bm',
              'BTN': 'bt', 'BOL': 'bo', 'BES': 'bq', 'BIH': 'ba', 'BWA': 'bw', 'BVT': 'bv', 'BRA': 'br', 'IOT': 'io',
              'BRN': 'bn', 'BGR': 'bg', 'BFA': 'bf', 'BDI': 'bi', 'CPV': 'cv', 'KHM': 'kh', 'CMR': 'cm', 'CAN': 'ca',
              'CYM': 'ky', 'CAF': 'cf', 'TCD': 'td', 'CHL': 'cl', 'CHN': 'cn', 'CXR': 'cx', 'CCK': 'cc', 'COL': 'co',
              'COM': 'km', 'COG': 'cg', 'COD': 'cd', 'COK': 'ck', 'CRI': 'cr', 'HRV': 'hr', 'CUB': 'cu', 'CUW': 'cw',
              'CYP': 'cy', 'CZE': 'cz', 'CIV': 'ci', 'DNK': 'dk', 'DJI': 'dj', 'DMA': 'dm', 'DOM': 'do', 'ECU': 'ec',
              'EGY': 'eg', 'SLV': 'sv', 'GNQ': 'gq', 'ERI': 'er', 'EST': 'ee', 'SWZ': 'sz', 'ETH': 'et', 'FLK': 'fk',
              'FRO': 'fo', 'FJI': 'fj', 'FIN': 'fi', 'FRA': 'fr', 'GUF': 'gf', 'PYF': 'pf', 'ATF': 'tf', 'GAB': 'ga',
              'GMB': 'gm', 'GEO': 'ge', 'DEU': 'de', 'GHA': 'gh', 'GIB': 'gi', 'GRC': 'gr', 'GRL': 'gl', 'GRD': 'gd',
              'GLP': 'gp', 'GUM': 'gu', 'GTM': 'gt', 'GGY': 'gg', 'GIN': 'gn', 'GNB': 'gw', 'GUY': 'gy', 'HTI': 'ht',
              'HMD': 'hm', 'VAT': 'va', 'HND': 'hn', 'HKG': 'hk', 'HUN': 'hu', 'ISL': 'is', 'IND': 'in', 'IDN': 'id',
              'IRN': 'ir', 'IRQ': 'iq', 'IRL': 'ie', 'IMN': 'im', 'ISR': 'il', 'ITA': 'it', 'JAM': 'jm', 'JPN': 'jp',
              'JEY': 'je', 'JOR': 'jo', 'KAZ': 'kz', 'KEN': 'ke', 'KIR': 'ki', 'PRK': 'kp', 'KOR': 'kr', 'KWT': 'kw',
              'KGZ': 'kg', 'LAO': 'la', 'LVA': 'lv', 'LBN': 'lb', 'LSO': 'ls', 'LBR': 'lr', 'LBY': 'ly', 'LIE': 'li',
              'LTU': 'lt', 'LUX': 'lu', 'MAC': 'mo', 'MDG': 'mg', 'MWI': 'mw', 'MYS': 'my', 'MDV': 'mv', 'MLI': 'ml',
              'MLT': 'mt', 'MHL': 'mh', 'MTQ': 'mq', 'MRT': 'mr', 'MUS': 'mu', 'MYT': 'yt', 'MEX': 'mx', 'FSM': 'fm',
              'MDA': 'md', 'MCO': 'mc', 'MNG': 'mn', 'MNE': 'me', 'MSR': 'ms', 'MAR': 'ma', 'MOZ': 'mz', 'MMR': 'mm',
              'NAM': 'na', 'NRU': 'nr', 'NPL': 'np', 'NLD': 'nl', 'NCL': 'nc', 'NZL': 'nz', 'NIC': 'ni', 'NER': 'ne',
              'NGA': 'ng', 'NIU': 'nu', 'NFK': 'nf', 'MKD': 'mk', 'MNP': 'mp', 'NOR': 'no', 'OMN': 'om', 'PAK': 'pk',
              'PLW': 'pw', 'PSE': 'ps', 'PAN': 'pa', 'PNG': 'pg', 'PRY': 'py', 'PER': 'pe', 'PHL': 'ph', 'PCN': 'pn',
              'POL': 'pl', 'PRT': 'pt', 'PRI': 'pr', 'QAT': 'qa', 'REU': 're', 'ROU': 'ro', 'RUS': 'ru', 'RWA': 'rw',
              'BLM': 'bl', 'SHN': 'sh', 'KNA': 'kn', 'LCA': 'lc', 'MAF': 'mf', 'SPM': 'pm', 'VCT': 'vc', 'WSM': 'ws',
              'SMR': 'sm', 'STP': 'st', 'SAU': 'sa', 'SEN': 'sn', 'SRB': 'rs', 'SYC': 'sc', 'SLE': 'sl', 'SGP': 'sg',
              'SXM': 'sx', 'SVK': 'sk', 'SVN': 'si', 'SLB': 'sb', 'SOM': 'so', 'ZAF': 'za', 'SGS': 'gs', 'SSD': 'ss',
              'ESP': 'es', 'LKA': 'lk', 'SDN': 'sd', 'SUR': 'sr', 'SJM': 'sj', 'SWE': 'se', 'CHE': 'ch', 'SYR': 'sy',
              'TWN': 'tw', 'TJK': 'tj', 'TZA': 'tz', 'THA': 'th', 'TLS': 'tl', 'TGO': 'tg', 'TKL': 'tk', 'TON': 'to',
              'TTO': 'tt', 'TUN': 'tn', 'TUR': 'tr', 'TKM': 'tm', 'TCA': 'tc', 'TUV': 'tv', 'UGA': 'ug', 'UKR': 'ua',
              'ARE': 'ae', 'GBR': 'gb', 'UMI': 'um', 'USA': 'us', 'URY': 'uy', 'UZB': 'uz', 'VUT': 'vu', 'VEN': 've',
              'VNM': 'vn', 'VGB': 'vg', 'VIR': 'vi', 'WLF': 'wf', 'ESH': 'eh', 'YEM': 'ye', 'ZMB': 'zm', 'ZWE': 'zw',
              'ALA': 'ax'
            };
            return map[upper] || '';
          }
          
          // Helper to get flag URL or placeholder
          function getFlagUrl(code){
            const iso2 = iso3ToIso2(code);
            if(iso2 && /^[a-z]{2}$/.test(iso2)){
              return 'https://flagcdn.com/24x18/' + iso2 + '.png';
            }
            return null;
          }
          
          function buildCountryLi(c){
            const pct = totalClicks > 0 ? Math.min(100, ((c.clicks / totalClicks) * 100)) : 0;
            const pctStr = pct.toFixed(1);
            const titleAttr = countryNames[c.country] ? countryNames[c.country] : (c.country || '');
            const flagUrl = getFlagUrl(c.country);
            const flagImg = flagUrl
              ? '<img src="' + flagUrl + '" alt="' + (c.country || '') + '" loading="lazy" width="24" height="18" style="border:1px solid #e5e7eb;border-radius:2px;object-fit:cover;flex-shrink:0;" onerror="this.style.display=\'none\';this.nextElementSibling.style.display=\'flex\';" />'
              : '';
            const flagPlaceholder = '<div style="width:24px;height:18px;background:#e5e7eb;border-radius:2px;flex-shrink:0;display:' + (flagUrl ? 'none' : 'flex') + ';align-items:center;justify-content:center;font-size:10px;color:#6b7280;" title="' + (c.country || '') + '">?</div>';
            return '<li class="ov-country-row">' +
              '<div class="ov-country-main"><div class="ov-country-flag">' + flagImg + flagPlaceholder + '<span class="ov-country-code" title="' + (titleAttr || '—') + '">' + (c.country || '—') + '</span></div>' +
              '<div class="ov-country-bar-wrap"><div class="ov-country-bar" style="width:' + pctStr + '%"></div><span class="ov-country-pct">' + pctStr + '%</span></div></div>' +
              '<div class="ov-country-stats"><div class="ov-country-stat"><div class="ov-country-num">' + fmt.int(c.clicks) + '</div><div class="ov-country-lbl">clicks</div></div>' +
              '<div class="ov-country-stat"><div class="ov-country-num">' + fmt.int(c.impressions) + '</div><div class="ov-country-lbl">impr</div></div>' +
              '<div class="ov-country-stat"><div class="ov-country-num">' + fmt.pct(c.ctr/100) + '</div><div class="ov-country-lbl">CTR</div></div></div></li>';
          }

          const top5 = validCountries.slice(0, 5);
          const rest = validCountries.slice(5);
          let html = '<ul class="ov-countries-list">' + top5.map(buildCountryLi).join('') + '</ul>';
          if (rest.length > 0) {
            html += '<div class="ov-countries-more" id="ovCountriesMore"><ul class="ov-countries-list">' + rest.map(buildCountryLi).join('') + '</ul></div>';
            html += '<button type="button" class="ov-countries-toggle" id="ovCountriesToggle">View more (' + rest.length + ')</button>';
          }
          box.innerHTML = html;

          const toggleBtn = document.getElementById('ovCountriesToggle');
          const moreEl = document.getElementById('ovCountriesMore');
          if (toggleBtn && moreEl) {
            toggleBtn.addEventListener('click', function(){
              const expanded = moreEl.classList.toggle('expanded');
              toggleBtn.textContent = expanded ? 'View less' : 'View more (' + rest.length + ')';
            });
          }
        } else {
          box.innerHTML = '<p style="color:#6b7280;font-size:13px;margin:8px 0;">No country data available. Please sync data from Google Search Console first.</p>';
        }
      } else {
        box.innerHTML = '<p style="color:#6b7280;font-size:13px;margin:8px 0;">No country data available. Sync data first.</p>';
      }
    }catch(e){
      box.innerHTML = '<p style="color:#dc2626;font-size:13px;margin:8px 0;">Error loading countries.</p>';
    }
  }

  document.querySelectorAll('[data-ov-days]').forEach(btn=>{
    btn.addEventListener('click',()=>{
      days = parseInt(btn.getAttribute('data-ov-days'), 10) || 28;
      document.querySelectorAll('[data-ov-days]').forEach(b=>b.classList.remove('active'));
      btn.classList.add('active');
      loadAll();
    });
  });

  const defDaysBtn = document.querySelector('[data-ov-days="28"]');
  if (defDaysBtn) defDaysBtn.classList.add('active');

  document.querySelectorAll('.ov-trend-metric').forEach(btn=>{
    btn.addEventListener('click',()=>{
      trendMetric = btn.getAttribute('data-trend') || 'clicks';
      document.querySelectorAll('.ov-trend-metric').forEach(b=>b.classList.remove('active'));
      btn.classList.add('active');
      renderMovers();
    });
  });

  loadAll();
})();

