(function () {
    const nonce = MiroGSC_Rank.nonce;
    const trkURL = MiroGSC_Rank.url;
    const HEAD = { "X-WP-Nonce": nonce };

    const daysSel    = document.getElementById("trkDays");
    const devSel     = document.getElementById("trkDevice");
    const countryInp = document.getElementById("trkCountry");
    const bodyEl     = document.getElementById("trkBody");
    const reloadBtn  = document.getElementById("trkReload");

    if (!daysSel || !devSel || !countryInp || !bodyEl || !reloadBtn) return;

    reloadBtn.addEventListener("click", load);

    window.addEventListener("miro-tab-activated", function(e){
        if (e.detail && e.detail.tab === "tab-rank") load();
    });

    const fmtInt = n => (n || 0).toLocaleString();
    const fmtPos = x => (x || 0).toFixed(1);

    const esc = s =>
        String(s || "").replace(/[&<>"]/g, c => ({
            "&": "&amp;",
            "<": "&lt;",
            ">": "&gt;",
            "\"": "&quot;"
        }[c]));

    let rowData = []; // Store row data for tooltips

    /* ======================================
       SPARKLINE (position series) - Match Keywords tab exactly
       - vals = positions per day (same scale as Keywords: 1/x then max-v)
       - lower position = better; same y-axis as Keywords position sparkline
       - delta = position change (negative = improved, positive = worse)
       - rowIdx = row index for tooltip data
    ====================================== */
    function spark(vals, delta, rowIdx) {
        if (!vals || !Array.isArray(vals) || vals.length < 2) return "";

        const clean = vals.map(v => Number(v));
        const drawVals = clean.map(v => (v > 0 && Number.isFinite(v)) ? 1 / v : 0);
        const n = drawVals.length;
        let maxDraw = Math.max(...drawVals.filter(v => v > 0));
        if (!Number.isFinite(maxDraw) || maxDraw <= 0) maxDraw = 1;
        const processedVals = drawVals.map(v => maxDraw - v);
        let fmin = Math.min(...processedVals);
        let fmax = Math.max(...processedVals);
        if (fmax === fmin) fmax = fmin + 1;
        const range = fmax - fmin;

        const W = 160, H = 40, P = 4;
        const step = (W - 2 * P) / Math.max(1, n - 1);
        const pts = [];
        for (let i = 0; i < n; i++) {
            const x = P + i * step;
            const k = range > 0 ? (processedVals[i] - fmin) / range : 0.5;
            const y = H - P - k * (H - 2 * P);
            pts.push({ x, y });
        }

        let d = `M${pts[0].x},${pts[0].y}`;
        for (let i = 1; i < pts.length; i++) d += ` L${pts[i].x},${pts[i].y}`;

        const color = "#6b7280";
        return `
            <div class="rf-spark-wrap">
                <svg class="rf-spark" viewBox="0 0 ${W} ${H}" preserveAspectRatio="none" data-row-idx="${rowIdx || ''}">
                    <path d="${d}" stroke="${color}" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round"/>
                    <circle class="rf-spark-dot" r="3.5" cx="0" cy="0" fill="${color}" opacity="0" style="transition: opacity 0.12s ease;"/>
                </svg>
            </div>
        `;
    }

    /* ============ BADGES (same as before) ============ */
    function badgePos(dp) {
        dp = Number(dp);
        if (dp < 0) return `<span class="rf-badge rf-badge-up">▲ ${Math.abs(dp).toFixed(1)}</span>`;
        if (dp > 0) return `<span class="rf-badge rf-badge-down">▼ ${dp.toFixed(1)}</span>`;
        return `<span class="rf-badge rf-badge-flat">• 0.0</span>`;
    }

    function badgeDelta(v) {
        if (v > 0) return `<span style="color:#10b981">+${fmtInt(v)}</span>`;
        if (v < 0) return `<span style="color:#ef4444">${fmtInt(v)}</span>`;
        return `<span class="muted">0</span>`;
    }

    /* ======================================
       LOAD TRACKED KEYWORDS
    ====================================== */
    async function load() {
        const qs = new URLSearchParams({
            days: daysSel.value || "28",
            device: devSel.value || "",
            country: (countryInp.value || "").toUpperCase(),
            _t: Date.now()
        });

        let res, json;
        try {
            res = await fetch(trkURL + "?" + qs.toString(), {
                headers: HEAD,
                cache: "no-store"
            });
        } catch (e) {
            bodyEl.innerHTML = `<tr><td colspan="10">Network error: ${e.message || e}</td></tr>`;
            return;
        }

        try { json = await res.json(); } catch { json = null; }

        bodyEl.innerHTML = "";

        if (!res.ok) {
            bodyEl.innerHTML = `<tr><td colspan="10">${json?.message || "Error loading data"}</td></tr>`;
            return;
        }

        rowData = []; // Reset row data

        (json.items || []).forEach((row, idx) => {
            const qStr  = esc(row.query);

            const newP  = Number(row.position);
            const lastP = Number(row.last_position);

            const dPos  = newP - lastP;
            const dClk  = Number(row.clicks - row.last_clicks);
            const dImp  = Number(row.impressions - row.last_impressions);

            // Position history for sparkline (already built in PHP)
            const sp = Array.isArray(row.spark) ? row.spark : [];
            const spDates = Array.isArray(row.spark_dates) ? row.spark_dates : [];

            // Store row data for tooltips
            rowData.push({
                query: qStr,
                spark: sp,
                spark_dates: spDates
            });

            const tr = document.createElement("tr");

            tr.innerHTML = `
                <td>${qStr}</td>
                <td>${isNaN(newP) ? "—" : newP.toFixed(1)}</td>
                <td>${isNaN(lastP) ? "—" : lastP.toFixed(1)}</td>
                <td>${badgePos(dPos)}</td>
                <td>${fmtInt(row.clicks)}</td>
                <td>${badgeDelta(dClk)}</td>
                <td>${fmtInt(row.impressions)}</td>
                <td>${badgeDelta(dImp)}</td>

                <td>
                    ${spark(sp, dPos, idx)}
                </td>

                <td style="text-align:right;white-space:nowrap;">
                    <button class="rf-del-btn" data-q="${qStr}">
                        Remove
                    </button>
                </td>
            `;

            bodyEl.appendChild(tr);
        });

        // Attach tooltip handlers after rendering
        attachSparkTooltips();
    }

    /* ======================================
       SPARKLINE TOOLTIP HANDLERS
    ====================================== */
    function attachSparkTooltips() {
        const tip = document.getElementById('rfRankTooltip');
        if (!tip) return;

        const wraps = document.querySelectorAll('#tab-rank .rf-spark-wrap');
        wraps.forEach(wrap => {
            const svg = wrap.querySelector('.rf-spark');
            if (!svg) return;
            wrap.addEventListener('mousemove', function(e) {
                const rowIdx = parseInt(svg.getAttribute('data-row-idx') || '0', 10);
                const item = rowData[rowIdx];
                if (!item) return;

                const vals = item.spark || [];
                const dates = item.spark_dates || [];
                const n = vals.length;
                if (!n || n < 2) return;

                const rect = svg.getBoundingClientRect();
                if (rect.width <= 0) return;

                const W = 160, H = 40, P = 4;
                const relX = e.clientX - rect.left;
                const viewBoxX = relX * (W / rect.width);
                let t = (viewBoxX - P) / (W - 2 * P);
                t = Math.max(0, Math.min(1, t));
                const pointIdx = Math.max(0, Math.min(n - 1, Math.round(t * (n - 1))));

                // Same scale as spark(): 1/x then max-v (match Keywords)
                const drawVals = vals.map(v => (v > 0 && Number.isFinite(Number(v))) ? 1 / Number(v) : 0);
                const maxDraw = Math.max(...drawVals.filter(x => x > 0)) || 1;
                const processedVals = drawVals.map(v => maxDraw - v);
                let fmin = Math.min(...processedVals);
                let fmax = Math.max(...processedVals);
                if (fmax === fmin) fmax = fmin + 1;
                const range = fmax - fmin;
                const norm = processedVals.map(v => (v - fmin) / range);
                const step = (W - 2 * P) / Math.max(1, n - 1);
                const x = P + pointIdx * step;
                const y = H - P - norm[pointIdx] * (H - 2 * P);

                // Show dot
                const dot = svg.querySelector('.rf-spark-dot');
                if (dot) {
                    dot.setAttribute('cx', x);
                    dot.setAttribute('cy', y);
                    dot.setAttribute('opacity', '1');
                }

                // Get date
                let dateStr = '';
                if (dates[pointIdx]) {
                    dateStr = dates[pointIdx];
                }

                // Get position value
                const posValue = vals[pointIdx] || 0;

                // Set tooltip content
                tip.innerHTML =
                    '<div style="font-weight:600;margin-bottom:2px;">' + esc(item.query) + '</div>' +
                    (dateStr ? '<div>Date: ' + dateStr + '</div>' : '') +
                    '<div>Position: ' + fmtPos(posValue) + '</div>' +
                    '<div style="color:#64748b;">Old → New (left to right)</div>';

                tip.style.display = 'block';
                tip.style.position = 'fixed';
                tip.style.visibility = 'hidden'; // Hidden to measure

                // Get tooltip dimensions
                const tipRect = tip.getBoundingClientRect();
                const tipWidth = tipRect.width || 220;
                const tipHeight = tipRect.height || 120;

                tip.style.visibility = 'visible'; // Make visible

                // Position tooltip (to the right of cursor)
                let tx = e.clientX + 12;
                let ty = e.clientY - 40;

                // Boundary checks
                const viewportWidth = window.innerWidth;
                const viewportHeight = window.innerHeight;

                if (tx + tipWidth > viewportWidth - 8) {
                    tx = e.clientX - tipWidth - 12; // Show on left if not enough space on right
                }
                if (tx < 8) tx = 8;
                if (ty < 8) ty = 8;
                if (ty + tipHeight > viewportHeight - 8) {
                    ty = viewportHeight - tipHeight - 8;
                }

                tip.style.left = tx + 'px';
                tip.style.top = ty + 'px';
            });

            wrap.addEventListener('mouseleave', function() {
                const dot = wrap.querySelector('.rf-spark-dot');
                if (dot) {
                    dot.setAttribute('opacity', '0');
                }
                if (tip) {
                    tip.style.display = 'none';
                }
            });
        });
    }

    /* ======================================
       REMOVE TRACKED KEYWORD
    ====================================== */
    bodyEl.addEventListener("click", async e => {
        const btn = e.target.closest("button[data-q]");
        if (!btn) return;

        const q = btn.dataset.q;
        btn.disabled = true;
        btn.textContent = "Removing…";

        let res, json;
        try {
            res = await fetch(trkURL, {
                method: "DELETE",
                headers: { ...HEAD, "Content-Type": "application/json" },
                body: JSON.stringify({ q })
            });
        } catch (e2) {
            alert("Remove failed: " + (e2.message || e2));
            btn.disabled = false;
            btn.textContent = "Remove";
            return;
        }

        try { json = await res.json(); } catch { json = null; }

        if (!res.ok) {
            alert(json?.message || "Remove failed");
            btn.disabled = false;
            btn.textContent = "Remove";
            return;
        }

        const tr = btn.closest("tr");
        if (tr) tr.remove();
    });

    load();
})();
