/* ============================================================
   Miro GSC Utils
   Shared helpers for RankFlow, Keywords, QuickWins, RankTracker
   ============================================================ */

window.MiroGSC = window.MiroGSC || {};

/* --------------------------
   Number + string utilities
--------------------------- */
MiroGSC.fmtInt = function (n) {
    return (n || 0).toLocaleString();
};

MiroGSC.fmtPct = function (x) {
    return ((Math.round((x || 0) * 1000) / 10).toFixed(1)) + "%";
};

MiroGSC.fmtPos = function (p) {
    return p > 0 ? Number(p).toFixed(1) : "—";
};

MiroGSC.esc = function (s) {
    s = (s == null ? "" : String(s));
    return s.replace(/[&<>\"\']/g, m => ({
        "&": "&amp;",
        "<": "&lt;",
        ">": "&gt;",
        "\"": "&quot;",
        "'": "&#39;"
    }[m]));
};

MiroGSC.jsonTry = function (txt) {
    try { return JSON.parse(txt); }
    catch (_) { return null; }
};

/* --------------------------
   Delta Arrows
   ▲ up / ▼ down / — flat
--------------------------- */
MiroGSC.arrowDelta = function (curr, prev, type) {
    const a = parseFloat(curr) || 0;
    const b = parseFloat(prev) || 0;
    let d = a - b;

    // Position: lower = better
    if (type === "pos") {
        d = (a > 0 && b > 0) ? (b - a) : 0;
    }

    const cls = d > 0 ? "miro-delta-up" :
                d < 0 ? "miro-delta-down" :
                        "miro-delta-zero";

    const sym = d > 0 ? "▲" :
                d < 0 ? "▼" :
                        "—";

    let val = "";
    if (type === "ctr") val = Math.abs(d * 100).toFixed(1) + "%";
    else if (type === "pos") val = Math.abs(d).toFixed(1);
    else val = Math.abs(d).toLocaleString();

    return `<div class="miro-delta ${cls}" title="Current vs previous">${sym} ${val}</div>`;
};

/* --------------------------
   Trend Arrow (for sparkline)
--------------------------- */
MiroGSC.trend = function (trend) {
    const t = parseFloat(trend || 0);
    const cls = t > 0 ? "up" : t < 0 ? "down" : "flat";
    const sym = t > 0 ? "▲" : t < 0 ? "▼" : "—";
    const val = Math.abs(t).toFixed(1);
    return `<div class="miro-trend ${cls}">${sym} ${val}</div>`;
};

/* --------------------------
   Sparkline Generator (blue)
--------------------------- */
MiroGSC.spark = function (arr) {
    if (!arr || !arr.length) return "";

    const W = 140, H = 28, P = 2;
    const maxY = Math.max(1, ...arr);
    const n = arr.length;
    const step = (W - 2 * P) / Math.max(1, n - 1);

    // Points
    const pts = [];
    for (let i = 0; i < n; i++) {
        const v = arr[i];
        pts.push([P + i * step, H - P - ((v / maxY) * (H - 2 * P))]);
    }

    // Path
    let d = "M" + pts[0][0] + " " + pts[0][1];
    for (let i = 1; i < n; i++) {
        d += " L" + pts[i][0] + " " + pts[i][1];
    }

    return `
        <svg class="spark" viewBox="0 0 ${W} ${H}">
            <path d="${d}" fill="none" stroke="#2563eb" stroke-width="1.5"/>
        </svg>
    `;
};

/* --------------------------
   CSV Export Helper
--------------------------- */
MiroGSC.downloadCSV = function (blob, filename = "export.csv") {
    const a = document.createElement("a");
    a.href = URL.createObjectURL(blob);
    a.download = filename;
    a.click();
    URL.revokeObjectURL(a.href);
};
