/* global window */
( function(wp) {
  const { registerPlugin } = wp.plugins;
  const { PluginDocumentSettingPanel } = wp.editPost;
  const { useState, Fragment } = wp.element;
  const { TextControl, Button, Notice, Spinner, Badge } = wp.components;
  const { select, dispatch } = wp.data;

  const cfg = window.MIRO_INTENT_CFG || {};
  const restBase = cfg.restBase || '';
  const nonce = cfg.nonce || '';

  function call(endpoint, body) {
    return fetch(restBase + endpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-WP-Nonce': nonce
      },
      body: JSON.stringify(body || {})
    }).then(r => r.json());
  }

  const Panel = () => {
    const post = select('core/editor').getCurrentPost();
    const title = select('core/editor').getEditedPostAttribute('title') || post?.title || '';
    const content = select('core/editor').getEditedPostAttribute('content') || post?.content || '';
    const [kw, setKw] = useState( (post?.meta && post.meta._miro_focus_keyword) || '' );
    const [state, setState] = useState({ loading:false, result:null, error:'' });

    const onCheck = async () => {
      setState({ loading:true, result:null, error:'' });
      try {
        const res = await call('intent/check', { focus_keyword: kw, title, content });
        if (res.code) throw new Error(res.message || 'API error');
        setState({ loading:false, result:res, error:'' });
      } catch (e) {
        setState({ loading:false, result:null, error:e.message || 'Failed' });
      }
    };

    const onPatch = async () => {
      if (!state.result) return;
      const target = state.result.expected_intent || 'informational';
      setState(s => ({ ...s, loading:true, error:'' }));
      try {
        const res = await call('intent/patch_intro', { focus_keyword: kw, target_intent: target, title, content });
        if (res.code) throw new Error(res.message || 'API error');
        // Update editor content
        dispatch('core/editor').editPost({ content: res.body_html });
        setState(s => ({ ...s, loading:false }));
      } catch (e) {
        setState(s => ({ ...s, loading:false, error:e.message || 'Patch failed' }));
      }
    };

    const Result = () => {
      if (state.loading) return <Spinner />;
      if (state.error) return <Notice status="error" isDismissible={false}>{state.error}</Notice>;
      if (!state.result) return null;

      const { expected_intent, content_intent, match, advice } = state.result;
      const badgeColor = match ? 'green' : 'red';
      return (
        <Fragment>
          <div style={{display:'flex', gap:8, marginBottom:8}}>
            <Badge status={badgeColor}>Expected: {expected_intent}</Badge>
            <Badge status="warning">Content: {content_intent}</Badge>
          </div>
          <div style={{fontSize:'12px', lineHeight:1.45, marginBottom:8}}>{advice}</div>
          {!match && (
            <Button variant="primary" onClick={onPatch}>Rewrite Intro to {expected_intent}</Button>
          )}
        </Fragment>
      );
    };

    return (
      <PluginDocumentSettingPanel
        name="miro-intent"
        title="Miro AI — Intent"
        className="miro-intent-panel"
      >
        <TextControl
          label="Focus Keyword"
          value={kw}
          onChange={setKw}
          placeholder="e.g. best noise cancelling headphones"
        />
        <div style={{display:'flex', gap:8, marginTop:8}}>
          <Button variant="secondary" onClick={onCheck} disabled={!kw || state.loading}>
            {state.loading ? 'Checking…' : 'Check Intent'}
          </Button>
        </div>
        <div style={{marginTop:10}}>
          <Result />
        </div>
      </PluginDocumentSettingPanel>
    );
  };

  registerPlugin('miro-intent-plugin', { render: Panel });
} )(window.wp);
