/* AI Keyword Playbook — admin UI (no React; simple DOM) */
(function(){
  const mount = document.getElementById('miro-playbook-app');
  if (!mount) return;

  const restScan     = mount.dataset.restScan;
  const restIdeas    = mount.dataset.restIdeas;
  const restClusters = mount.dataset.restClusters;
  const restLinks    = mount.dataset.restLinks;
  const nonce        = mount.dataset.nonce;

  function el(tag, attrs, ...kids){
    const n = document.createElement(tag);
    if (attrs) Object.entries(attrs).forEach(([k,v])=>{
      if (k === 'class') n.className = v;
      else if (k === 'style') n.setAttribute('style', v);
      else n.setAttribute(k, v);
    });
    kids.forEach(k => {
      if (k == null) return;
      if (typeof k === 'string') n.appendChild(document.createTextNode(k));
      else n.appendChild(k);
    });
    return n;
  }

  const tabs = el('div', {class: 'miro-tabs'},
    el('button', {class:'miro-tab is-active', 'data-tab':'overview'}, 'Overview'),
    el('button', {class:'miro-tab', 'data-tab':'ideas'}, 'Keyword Ideas'),
    el('button', {class:'miro-tab', 'data-tab':'clusters'}, 'Clusters'),
    el('button', {class:'miro-tab', 'data-tab':'links'}, 'Link Suggestions'),
    el('div', {style:'flex:1'}),
    el('button', {id:'miro-playbook-scan', class:'button button-primary'}, 'Scan Now')
  );

  const view = el('div', {class:'miro-tabviews'},
    el('div', {class:'miro-tabview is-active', 'data-view':'overview'},
      el('p', {class:'description'}, 'Click "Scan Now" to analyze your content and cached GSC keywords.'),
      el('div', {id:'miro-pb-summary'})
    ),
    el('div', {class:'miro-tabview', 'data-view':'ideas'},
      el('div', {id:'miro-pb-ideas'})
    ),
    el('div', {class:'miro-tabview', 'data-view':'clusters'},
      el('div', {id:'miro-pb-clusters'})
    ),
    el('div', {class:'miro-tabview', 'data-view':'links'},
      el('div', {id:'miro-pb-links'})
    ),
  );

  mount.appendChild(tabs);
  mount.appendChild(view);

  function setActive(which){
    document.querySelectorAll('.miro-tab').forEach(b=>b.classList.remove('is-active'));
    document.querySelector('.miro-tab[data-tab="'+which+'"]').classList.add('is-active');
    document.querySelectorAll('.miro-tabview').forEach(v=>v.classList.remove('is-active'));
    document.querySelector('.miro-tabview[data-view="'+which+'"]').classList.add('is-active');
  }

  tabs.addEventListener('click', (e)=>{
    const b = e.target.closest('.miro-tab'); if (!b) return;
    setActive(b.dataset.tab);
    if (b.dataset.tab === 'ideas') loadIdeas();
    if (b.dataset.tab === 'clusters') loadClusters();
    if (b.dataset.tab === 'links') loadLinks();
  });

  async function scan(){
    document.getElementById('miro-playbook-scan').disabled = true;
    document.getElementById('miro-playbook-scan').textContent = 'Scanning…';
    try{
      const r = await fetch(restScan, {
        method: 'POST',
        headers: {'Content-Type':'application/json','X-WP-Nonce':nonce},
        body: JSON.stringify({ post_types:['post'], limit: 800 })
      });
      const j = await r.json();
      const s = j && j.summary ? j.summary : {};
      document.getElementById('miro-pb-summary').innerHTML =
        '<div class="miro-grid">'+
        '<div class="miro-card"><b>Posts scanned</b><div>'+ (s.posts||0) +'</div></div>'+
        '<div class="miro-card"><b>Easy-win keywords</b><div>'+ (s.opps||0) +'</div></div>'+
        '<div class="miro-card"><b>Topic clusters</b><div>'+ (s.clusters||0) +'</div></div>'+
        '</div>';
      setActive('ideas'); // jump to ideas
      await loadIdeas();
    } catch(e){
      alert('Scan failed: ' + (e && e.message ? e.message : e));
    } finally {
      document.getElementById('miro-playbook-scan').disabled = false;
      document.getElementById('miro-playbook-scan').textContent = 'Scan Now';
    }
  }

  async function loadIdeas(){
    const box = document.getElementById('miro-pb-ideas');
    box.textContent = 'Loading…';
    try{
      const j = await fetch(restIdeas, { headers:{'X-WP-Nonce':nonce} }).then(r=>r.json());
      const rows = (j && j.ideas) ? j.ideas : [];
      const tbl = el('table', {class:'widefat striped'});
      const thead = el('thead', null, el('tr', null,
        el('th', null, 'Score'),
        el('th', null, 'Title (suggested)'),
        el('th', null, 'Keyword'),
        el('th', null, 'Note')
      ));
      const tbody = el('tbody');
      rows.forEach(r=>{
        tbody.appendChild(el('tr', null,
          el('td', null, String(r.score||'')),
          el('td', null, r.title||''),
          el('td', null, r.keyword||''),
          el('td', null, r.note||'')
        ));
      });
      tbl.appendChild(thead); tbl.appendChild(tbody);
      box.innerHTML = '';
      box.appendChild(tbl);
    }catch(e){
      box.textContent = 'Failed to load ideas.';
    }
  }

  async function loadClusters(){
    const box = document.getElementById('miro-pb-clusters');
    box.textContent = 'Loading…';
    try{
      const j = await fetch(restClusters, { headers:{'X-WP-Nonce':nonce} }).then(r=>r.json());
      const rows = (j && j.clusters) ? j.clusters : [];
      const frag = document.createDocumentFragment();
      rows.forEach(c=>{
        const card = el('div', {class:'miro-card'}, 
          el('div', {class:'miro-card-title'}, c.topic + ' ('+c.count+')')
        );
        const ul = el('ul');
        c.items.slice(0,10).forEach(i=> ul.appendChild(el('li', null, i.title)) );
        card.appendChild(ul);
        frag.appendChild(card);
      });
      box.innerHTML='';
      box.appendChild(frag);
    }catch(e){
      box.textContent = 'Failed to load clusters.';
    }
  }

  async function loadLinks(){
    const box = document.getElementById('miro-pb-links');
    box.textContent = 'Loading…';
    try{
      const j = await fetch(restLinks, { headers:{'X-WP-Nonce':nonce} }).then(r=>r.json());
      const rows = (j && j.pairs) ? j.pairs : [];
      const tbl = el('table', {class:'widefat striped'});
      const thead = el('thead', null, el('tr', null,
        el('th', null, 'From'),
        el('th', null, '→'),
        el('th', null, 'To'),
        el('th', null, 'Reason')
      ));
      const tbody = el('tbody');
      rows.forEach(r=>{
        tbody.appendChild(el('tr', null,
          el('td', null, r.from_title),
          el('td', null, '→'),
          el('td', null, r.to_title),
          el('td', null, r.reason||'')
        ));
      });
      tbl.appendChild(thead); tbl.appendChild(tbody);
      box.innerHTML = '';
      box.appendChild(tbl);
    }catch(e){
      box.textContent = 'Failed to load link suggestions.';
    }
  }

  document.getElementById('miro-playbook-scan').addEventListener('click', scan);
})();