(function () {
    const { nonce, endpoint } = MiroGSC_RankTracker;

    const HEAD = { "X-WP-Nonce": nonce };

    const daysSel    = document.getElementById("trkDays");
    const devSel     = document.getElementById("trkDevice");
    const countryInp = document.getElementById("trkCountry");
    const reloadBtn  = document.getElementById("trkReload");

    const tableBody  = document.getElementById("rfTable");
    const pagination = document.getElementById("rfPagination");

    reloadBtn.addEventListener("click", load);

    /* ========= Helpers ========= */
    const esc = s =>
        String(s || "").replace(/[&<>"]/g, c => ({
            "&":"&amp;", "<":"&lt;", ">":"&gt;", "\"":"&quot;"
        }[c]));

    const fmt = n => (n || 0).toLocaleString();

    /* ========= Pagination Setup ========= */
    let currentPage = 1;
    const perPage = 20;
    let fullRows = [];

    function renderPagination() {
        const total = fullRows.length;
        const pages = Math.max(1, Math.ceil(total / perPage));

        let html = "";
        for (let i=1; i<=pages; i++) {
            html += `<button class="rf-page-btn ${i===currentPage ? "active":""}" data-p="${i}">${i}</button>`;
        }
        pagination.innerHTML = html;

        pagination.querySelectorAll("[data-p]").forEach(btn=>{
            btn.addEventListener("click", ()=>{
                currentPage = parseInt(btn.dataset.p);
                renderTable();
            });
        });
    }

    function renderTable() {
        tableBody.innerHTML = "";

        const start = (currentPage - 1) * perPage;
        const show  = fullRows.slice(start, start + perPage);

        show.forEach(r => {
            const tr = document.createElement("tr");

            // trend arrow logic
            let trend = 0;
            if (r.trend && r.trend.length >= 2) {
                const first = r.trend[0];
                const last  = r.trend[r.trend.length - 1];

                // SEO logic: smaller = better
                if (last < first) trend = 1;   // improved
                else if (last > first) trend = -1; // worse
                else trend = 0;
            }

            tr.innerHTML = `
                <td>${esc(r.keyword)}</td>
                <td>${r.new_pos.toFixed(1)}</td>
                <td>${r.old_pos.toFixed(1)}</td>
                <td>${
                    r.delta > 0
                        ? `<span class="rf-badge-up">▲ ${r.delta.toFixed(1)}</span>`
                        : r.delta < 0
                            ? `<span class="rf-badge-down">▼ ${Math.abs(r.delta).toFixed(1)}</span>`
                            : `<span class="rf-badge-flat">—</span>`
                }</td>

                <td>${fmt(r.clicks)}</td>
                <td>${fmt(r.impr)}</td>

                <td>
                    ${RankFlow.sparkline(r.trend, trend)}
                </td>

                <td>
                    <button class="btn tiny rf-remove" data-q="${esc(r.keyword)}">Remove</button>
                </td>
            `;

            tableBody.appendChild(tr);
        });

        bindRemove();
    }

    /* ========= Remove keyword ========= */
    function bindRemove() {
        document.querySelectorAll(".rf-remove").forEach(btn=>{
            btn.addEventListener("click", async () => {
                const q = btn.dataset.q;
                if (!confirm("Remove: " + q + "?")) return;

                const res = await fetch(endpoint, {
                    method:"DELETE",
                    headers:{ ...HEAD, "Content-Type":"application/json" },
                    body: JSON.stringify({ q })
                });

                currentPage = 1;
                load();
            });
        });
    }

    /* ========= Main LOAD ========= */
    async function load() {
        tableBody.innerHTML = `<tr><td colspan="10" style="padding:20px;text-align:center;">Loading…</td></tr>`;

        const qs = new URLSearchParams({
            days: daysSel.value || "28",
            device: devSel.value || "",
            country: (countryInp.value || "").toUpperCase()
        });

        const res = await fetch(`${endpoint}?${qs.toString()}`, { headers: HEAD });

        let json = null;
        try { json = await res.json(); } catch (e) {}

        if (!res.ok || !json || !json.ok) {
            tableBody.innerHTML = `<tr><td colspan="10">Error loading data</td></tr>`;
            return;
        }

        fullRows = json.rows || [];

        if (fullRows.length === 0) {
            tableBody.innerHTML = `<tr><td colspan="10">No tracked keywords found.</td></tr>`;
            pagination.innerHTML = "";
            return;
        }

        currentPage = 1;
        renderPagination();
        renderTable();
    }

    load();
})();
