(function () {
  function ready(fn){
    if (document.readyState === "loading") {
      document.addEventListener("DOMContentLoaded", fn);
    } else fn();
  }

  function escHtml(s){
    return String(s ?? "")
      .replaceAll("&","&amp;")
      .replaceAll("<","&lt;")
      .replaceAll(">","&gt;")
      .replaceAll('"',"&quot;")
      .replaceAll("'","&#039;");
  }

  function clampNum(v, def=0){
    const n = Number(v);
    return Number.isFinite(n) ? n : def;
  }

  // Badge
  function badge(delta) {
    delta = clampNum(delta, 0);
    if (delta > 0) return `<span class="rf-badge rf-up">▲ ${delta.toFixed(2)}</span>`;
    if (delta < 0) return `<span class="rf-badge rf-down">▼ ${Math.abs(delta).toFixed(2)}</span>`;
    return `<span class="rf-badge rf-zero">• 0</span>`;
  }

  // Sparkline (position) — same scale as Keywords tab: 1/x then max-v, same y-axis
  function spark(trend = [], delta = 0, gid = "rfGrad") {
    if (!Array.isArray(trend) || trend.length < 2) return "";

    const clean = trend.map(v => clampNum(v, 0));
    const drawVals = clean.map(v => (v > 0) ? 1 / v : 0);
    let maxDraw = Math.max(...drawVals.filter(v => v > 0));
    if (!Number.isFinite(maxDraw) || maxDraw <= 0) maxDraw = 1;
    const processedVals = drawVals.map(v => maxDraw - v);
    let fmin = Math.min(...processedVals);
    let fmax = Math.max(...processedVals);
    if (fmax === fmin) fmax = fmin + 1;
    const range = fmax - fmin;
    const n = processedVals.length;

    const W = 160, H = 40, P = 4;
    const step = (W - 2 * P) / Math.max(1, n - 1);
    const pts = processedVals.map((v, i) => {
      const x = P + i * step;
      const k = (v - fmin) / range;
      const y = H - P - k * (H - 2 * P);
      return { x, y };
    });

    let d = `M${pts[0].x},${pts[0].y}`;
    for (let i = 1; i < pts.length; i++) {
      d += ` L${pts[i].x},${pts[i].y}`;
    }

    let color = "#2563eb";
    if (delta > 0) color = "#16a34a";
    if (delta < 0) color = "#dc2626";

    return `
      <div class="rf-spark-wrap">
        <svg class="rf-spark" viewBox="0 0 ${W} ${H}" preserveAspectRatio="none">
          <path d="${d}" stroke="${color}" stroke-width="2" fill="none"/>
          <circle class="rf-spark-dot" r="3.5" cx="0" cy="0" fill="${color}" opacity="0" style="transition: opacity 0.12s ease;"/>
        </svg>
      </div>
    `;
  }

  // Expose for Rank Tracker tab if it uses RankFlow.sparkline
  window.RankFlow = window.RankFlow || {};
  window.RankFlow.sparkline = spark;

  function fmtPos(v) {
    const n = clampNum(v, 0);
    return n > 0 ? n.toFixed(2) : "—";
  }

  function setupSparkTooltips(tableBodyRef, currentSliceRef, tipRef) {
    if (!tableBodyRef || !tipRef) return;
    tableBodyRef.addEventListener("mousemove", function (e) {
      const wrap = e.target.closest ? e.target.closest(".rf-spark-wrap") : null;
      if (!wrap) {
        tipRef.style.display = "none";
        tableBodyRef.querySelectorAll(".rf-spark-dot").forEach(function (d) { d.setAttribute("opacity", "0"); });
        return;
      }
      const tr = wrap.closest("tr");
      if (!tr) return;
      const rowIdx = parseInt(tr.getAttribute("data-row-idx"), 10);
      if (isNaN(rowIdx) || rowIdx < 0) return;
      const row = (currentSliceRef && currentSliceRef.current) ? currentSliceRef.current[rowIdx] : null;
      if (!row) return;
      const vals = Array.isArray(row.trend) ? row.trend : [];
      const dates = Array.isArray(row.spark_dates) ? row.spark_dates : [];
      const n = vals.length;
      if (n < 2) return;
      const svg = wrap.querySelector(".rf-spark");
      if (!svg) return;

      const rect = svg.getBoundingClientRect();
      if (rect.width <= 0) return;

      const W = 160, H = 40, P = 4;
      const relX = e.clientX - rect.left;
      const viewBoxX = relX * (W / rect.width);
      let t = (viewBoxX - P) / (W - 2 * P);
      t = Math.max(0, Math.min(1, t));
      const pointIdx = Math.max(0, Math.min(n - 1, Math.round(t * (n - 1))));

      const clean = vals.map(function (v) { return clampNum(v, 0); });
      const drawVals = clean.map(function (v) { return (v > 0) ? 1 / v : 0; });
      let maxDraw = Math.max.apply(null, drawVals.filter(function (v) { return v > 0; }));
      if (!Number.isFinite(maxDraw) || maxDraw <= 0) maxDraw = 1;
      const processedVals = drawVals.map(function (v) { return maxDraw - v; });
      let fmin = Math.min.apply(null, processedVals);
      let fmax = Math.max.apply(null, processedVals);
      if (fmax === fmin) fmax = fmin + 1;
      const range = fmax - fmin;
      const v = processedVals[pointIdx];
      const k = (v - fmin) / range;
      const x = P + pointIdx * ((W - 2 * P) / Math.max(1, n - 1));
      const y = H - P - k * (H - 2 * P);

      const dot = svg.querySelector(".rf-spark-dot");
      if (dot) {
        dot.setAttribute("cx", x);
        dot.setAttribute("cy", y);
        dot.setAttribute("opacity", "1");
      }

      const dateStr = dates[pointIdx] || ("Day " + (pointIdx + 1));
      const posValue = vals[pointIdx] != null ? vals[pointIdx] : 0;
      tipRef.innerHTML =
        "<div style=\"font-weight:600;margin-bottom:2px;\">" + escHtml(row.keyword || "") + "</div>" +
        "<div>Date: " + escHtml(dateStr) + "</div>" +
        "<div>Position: " + fmtPos(posValue) + "</div>" +
        "<div style=\"color:#64748b;\">Old → New (left to right)</div>";
      tipRef.style.display = "block";
      tipRef.style.position = "fixed";
      tipRef.style.visibility = "hidden";
      const tipRect = tipRef.getBoundingClientRect();
      const tipWidth = tipRect.width || 220;
      const tipHeight = tipRect.height || 120;
      tipRef.style.visibility = "visible";
      var viewportWidth = window.innerWidth;
      var viewportHeight = window.innerHeight;
      var tx = e.clientX + 12;
      var ty = e.clientY - 40;
      if (tx + tipWidth > viewportWidth - 8) tx = e.clientX - tipWidth - 12;
      if (tx < 8) tx = 8;
      if (ty < 8) ty = e.clientY + 10;
      if (ty + tipHeight > viewportHeight - 8) ty = e.clientY - tipHeight - 10;
      if (ty < 8) ty = 8;
      if (ty + tipHeight > viewportHeight - 8) ty = viewportHeight - tipHeight - 8;
      tipRef.style.left = tx + "px";
      tipRef.style.top = ty + "px";
    });
    tableBodyRef.addEventListener("mouseleave", function (e) {
      const wrap = e.target.closest ? e.target.closest(".rf-spark-wrap") : null;
      if (!wrap) return;
      if (e.relatedTarget && e.relatedTarget.closest && e.relatedTarget.closest(".rf-spark-wrap")) return;
      tipRef.style.display = "none";
      tableBodyRef.querySelectorAll(".rf-spark-dot").forEach(function (d) { d.setAttribute("opacity", "0"); });
    });
  }

  function buildPages(current, total){
    // first, last, current±2 with ellipsis
    const pages = new Set([1, total, current-2, current-1, current, current+1, current+2]);
    const list = [...pages].filter(p => p >= 1 && p <= total).sort((a,b)=>a-b);

    const out = [];
    let prev = 0;
    for (const p of list){
      if (prev && p - prev > 1) out.push("…");
      out.push(p);
      prev = p;
    }
    return out;
  }

  ready(async function () {

    const config = window.miroRankflow || {};
    const endpoint = config.url;
    const nonce    = config.nonce;

    // If globals missing, just stop quietly
    if (!endpoint || !nonce) return;

    const HEAD = { "X-WP-Nonce": nonce };

    // DOM
    const elUpdated  = document.getElementById("rfUpdated");
    const elKpiTotal = document.getElementById("rfKpiTotal");
    const elKpiWins  = document.getElementById("rfKpiWins");
    const elKpiLoss  = document.getElementById("rfKpiLoss");
    const elTopPos   = document.getElementById("rfTopPos");
    const elWinners  = document.getElementById("rfWinners");
    const elLosers   = document.getElementById("rfLosers");
    const elCann     = document.getElementById("rfCann");
    const elVol      = document.getElementById("rfVol");

    const tableBody = document.getElementById("rfTable");
    const pagerWrap = document.getElementById("rfPagination");

    // Filters
    const fSearch = document.getElementById("rfSearch");
    const fMode   = document.getElementById("rfMode");
    const fSort   = document.getElementById("rfSort");
    const fMaxPos = document.getElementById("rfMaxPos");
    const fMinImpr = document.getElementById("rfMinImpr");
    const fMinAbsDelta = document.getElementById("rfMinAbsDelta");
    const fPerPage = document.getElementById("rfPerPage");
    const fReset = document.getElementById("rfReset");

    // If tab not printed yet, stop
    if (!tableBody || !pagerWrap || !elUpdated) return;

    let allRows = [];
    let rows = [];
    let page = 1;
    let perPage = 20;
    var currentSliceRef = { current: [] };

    var tip = document.getElementById("rfSparkTooltip");
    if (!tip) {
      tip = document.createElement("div");
      tip.id = "rfSparkTooltip";
      tip.className = "rf-spark-tooltip";
      tip.setAttribute("aria-hidden", "true");
      document.body.appendChild(tip);
    } else if (tip.parentNode !== document.body) {
      document.body.appendChild(tip);
    }
    tip.style.zIndex = "999999";
    setupSparkTooltips(tableBody, currentSliceRef, tip);

    function applyFilters(){
      const q = (fSearch?.value || "").trim().toLowerCase();
      const mode = fMode?.value || "all";
      const sort = fSort?.value || "new_pos_asc";

      const maxPos = fMaxPos?.value ? clampNum(fMaxPos.value, 0) : 0;
      const minImpr = fMinImpr?.value ? clampNum(fMinImpr.value, 0) : 0;
      const minAbsDelta = fMinAbsDelta?.value ? clampNum(fMinAbsDelta.value, 0) : 0;

      rows = allRows.filter(r => {
        const kw = String(r.keyword || "");
        const newPos = clampNum(r.new_pos, 0);
        const delta = clampNum(r.delta, 0);
        const impr = clampNum(r.impr, 0);

        if (q && !kw.toLowerCase().includes(q)) return false;

        if (maxPos > 0 && newPos > maxPos) return false;
        if (minImpr > 0 && impr < minImpr) return false;
        if (minAbsDelta > 0 && Math.abs(delta) < minAbsDelta) return false;

        if (mode === "winners" && delta <= 0) return false;
        if (mode === "losers" && delta >= 0) return false;
        if (mode === "movers" && delta === 0) return false;
        if (mode === "top" && newPos > 10) return false;

        return true;
      });

      // sorting
      rows.sort((a,b) => {
        const an = clampNum(a.new_pos, 9999), bn = clampNum(b.new_pos, 9999);
        const ad = clampNum(a.delta, 0), bd = clampNum(b.delta, 0);
        const ai = clampNum(a.impr, 0), bi = clampNum(b.impr, 0);
        const ac = clampNum(a.clicks, 0), bc = clampNum(b.clicks, 0);

        if (sort === "new_pos_asc") return an - bn || (bi - ai);
        if (sort === "delta_desc")  return bd - ad || (an - bn);
        if (sort === "delta_asc")   return ad - bd || (an - bn);
        if (sort === "impr_desc")   return bi - ai || (an - bn);
        if (sort === "clicks_desc") return bc - ac || (an - bn);
        return 0;
      });

      page = 1;
      renderTable();
    }

    function renderTable() {
      perPage = clampNum(fPerPage?.value, perPage) || perPage;

      const total = rows.length;
      const totalPages = Math.max(1, Math.ceil(total / perPage));
      if (page > totalPages) page = totalPages;

      const start = (page - 1) * perPage;
      const slice = rows.slice(start, start + perPage);

      currentSliceRef.current = slice;
      tableBody.innerHTML = slice.map((r, idx) => {
        const kw = escHtml(r.keyword);
        const newPos = clampNum(r.new_pos, 0).toFixed(2);
        const oldPos = clampNum(r.old_pos, 0).toFixed(2);
        const d = clampNum(r.delta, 0);
        const clicks = clampNum(r.clicks, 0);
        const impr = clampNum(r.impr, 0);

        const gid = `rfGrad_${page}_${idx}_${Math.random().toString(16).slice(2)}`;

        return `
          <tr data-row-idx="${idx}">
            <td>${kw}</td>
            <td>${newPos}</td>
            <td>${oldPos}</td>
            <td>${badge(d)}</td>
            <td>${clicks}</td>
            <td>${impr}</td>
            <td>${spark(r.trend, d, gid)}</td>
          </tr>
        `;
      }).join("");

      renderPagination(totalPages);
    }

    function renderPagination(totalPages){
      const parts = buildPages(page, totalPages);

      let html = `
        <span class="rf-btn" data-act="prev">Prev</span>
      `;

      for (const p of parts){
        if (p === "…") {
          html += `<span class="rf-page rf-ellipsis">…</span>`;
        } else {
          html += `<span class="rf-page ${p === page ? "rf-page-active" : ""}" data-page="${p}">${p}</span>`;
        }
      }

      html += `<span class="rf-btn" data-act="next">Next</span>`;
      pagerWrap.innerHTML = html;

      pagerWrap.querySelectorAll("[data-page]").forEach(btn => {
        btn.addEventListener("click", () => {
          page = Number(btn.dataset.page);
          renderTable();
        });
      });

      const prev = pagerWrap.querySelector("[data-act=prev]");
      const next = pagerWrap.querySelector("[data-act=next]");

      if (prev) prev.onclick = () => { if (page > 1) { page--; renderTable(); } };
      if (next) next.onclick = () => { if (page < totalPages) { page++; renderTable(); } };
    }

    async function load() {
      const res = await fetch(endpoint, { headers: HEAD, credentials: "same-origin" });
      const j = await res.json();

      if (!j.ok) {
        elUpdated.textContent = j.msg || "No snapshot available.";
        if (elKpiTotal) elKpiTotal.textContent = "—";
        if (elKpiWins) elKpiWins.textContent = "—";
        if (elKpiLoss) elKpiLoss.textContent = "—";
        if (elVol) elVol.textContent = "—";
        if (elTopPos) elTopPos.innerHTML = "";
        if (elWinners) elWinners.innerHTML = "<li class=\"muted\">No snapshot</li>";
        if (elLosers) elLosers.innerHTML = "<li class=\"muted\">No snapshot</li>";
        if (elCann) elCann.innerHTML = "";
        return;
      }

      var totalKw = j.total != null ? j.total : (j.rows ? j.rows.length : 0);
      var winsList = j.winners || [];
      var lossList = j.losers || [];

      elUpdated.textContent = "Last updated: " + (j.last_updated || "—") + " • Keywords: " + totalKw;
      if (elKpiTotal) elKpiTotal.textContent = totalKw.toLocaleString();
      if (elKpiWins) elKpiWins.textContent = winsList.length;
      if (elKpiLoss) elKpiLoss.textContent = lossList.length;
      if (elVol) elVol.textContent = (j.volatility != null && j.volatility !== "") ? String(j.volatility) : "—";

      // Overview boxes
      const topPos = (j.rows || [])
        .slice()
        .sort((a,b) => clampNum(a.new_pos,9999) - clampNum(b.new_pos,9999) || clampNum(b.impr,0) - clampNum(a.impr,0))
        .slice(0, 10);

      if (elTopPos) {
        elTopPos.innerHTML = topPos.length
          ? topPos.map(function(r){
              var kw = escHtml(r.keyword);
              var p = clampNum(r.new_pos, 0).toFixed(2);
              var imp = clampNum(r.impr, 0);
              return "<li><strong>" + kw + "</strong> <span class=\"muted\">#" + p + "</span> <span class=\"muted\">(" + imp + " impr)</span></li>";
            }).join("")
          : "<li class=\"muted\">No data in snapshot</li>";
      }

      if (elWinners) elWinners.innerHTML = winsList.length
        ? winsList.map(function(w){ return "<li><strong>" + escHtml(w.keyword) + "</strong> " + badge(w.delta) + "</li>"; }).join("")
        : "<li class=\"muted\">No 7d movers</li>";
      if (elLosers) elLosers.innerHTML = lossList.length
        ? lossList.map(function(w){ return "<li><strong>" + escHtml(w.keyword) + "</strong> " + badge(w.delta) + "</li>"; }).join("")
        : "<li class=\"muted\">No 7d movers</li>";

      if (elCann) {
        var cannList = j.cannibal || [];
        elCann.innerHTML = cannList.length
          ? cannList.map(function(c){ return "<li><strong>" + escHtml(c.keyword) + "</strong> (" + clampNum(c.pages, 0) + " pages)</li>"; }).join("")
          : "<li class=\"muted\">None</li>";
      }

      if (elVol) elVol.textContent = String(j.volatility ?? "");

      allRows = (j.rows || []).map(r => ({
        keyword:     r.keyword || "",
        old_pos:     clampNum(r.old_pos, 0),
        new_pos:     clampNum(r.new_pos, 0),
        delta:       clampNum(r.delta, 0),
        clicks:      clampNum(r.clicks, 0),
        impr:        clampNum(r.impr, 0),
        trend:       Array.isArray(r.trend) ? r.trend : [],
        spark_dates: Array.isArray(r.spark_dates) ? r.spark_dates : []
      }));

      // Default view: position 1 first (what you asked)
      if (fSort) fSort.value = "new_pos_asc";

      applyFilters();
    }

    // Filters events
    [fSearch, fMode, fSort, fMaxPos, fMinImpr, fMinAbsDelta].forEach(el => {
      if (!el) return;
      el.addEventListener("input", applyFilters);
      el.addEventListener("change", applyFilters);
    });

    if (fPerPage) fPerPage.addEventListener("change", applyFilters);

    if (fReset) fReset.addEventListener("click", () => {
      if (fSearch) fSearch.value = "";
      if (fMode) fMode.value = "all";
      if (fSort) fSort.value = "new_pos_asc";
      if (fMaxPos) fMaxPos.value = "";
      if (fMinImpr) fMinImpr.value = "";
      if (fMinAbsDelta) fMinAbsDelta.value = "";
      if (fPerPage) fPerPage.value = "20";
      applyFilters();
    });

    // Rebuild button
    const rebuildBtn = document.getElementById("rfRebuild");
    if (rebuildBtn) {
      rebuildBtn.addEventListener("click", async () => {
        rebuildBtn.disabled = true;
        rebuildBtn.textContent = "Rebuilding…";
        try {
          const r = await fetch(config.rebuild, {
            method: "POST",
            credentials: "same-origin",
            headers: { "X-WP-Nonce": nonce }
          });
          await r.json().catch(()=>{});
          location.reload();
        } catch(e){
          rebuildBtn.disabled = false;
          rebuildBtn.textContent = "Rebuild Snapshot";
          alert("Rebuild failed: " + (e.message || e));
        }
      });
    }

    load();
  });
})();
