(function(){
  function el(tag, attrs, children){
    var e = document.createElement(tag);
    if (attrs) Object.keys(attrs).forEach(function(k){
      if (k === 'class') e.className = attrs[k];
      else if (k === 'html') e.innerHTML = attrs[k];
      else e.setAttribute(k, attrs[k]);
    });
    (children || []).forEach(function(c){
      if (typeof c === 'string') e.appendChild(document.createTextNode(c));
      else if (c) e.appendChild(c);
    });
    return e;
  }

  function status(text){
    var s = document.getElementById('pp-trends-status');
    if (s) s.textContent = text || '';
  }

  function copyToClipboard(text){
    if (navigator.clipboard && navigator.clipboard.writeText) {
      navigator.clipboard.writeText(text).then(function(){
        status('Copied!');
        setTimeout(function(){ status(''); }, 800);
      }).catch(function(){
        fallbackCopy(text);
      });
    } else {
      fallbackCopy(text);
    }
  }

  function fallbackCopy(text){
    var ta = el('textarea', {style:'position:fixed;left:-9999px;top:-9999px;'}, [text]);
    document.body.appendChild(ta);
    ta.select();
    try { document.execCommand('copy'); } catch(e) {}
    document.body.removeChild(ta);
    status('Copied!');
    setTimeout(function(){ status(''); }, 800);
  }

  function renderList(items){
    var wrap = document.getElementById('pp-trends-list');
    if (!wrap) return;
    wrap.innerHTML = '';
    if (!items || !items.length) {
      wrap.appendChild(el('p', null, ['No trends found yet.']));
      return;
    }

    var table = el('table', {class:'widefat fixed striped'});
    var thead = el('thead', null, [
      el('tr', null, [
        el('th', null, ['Title']),
        el('th', null, ['Source']),
        el('th', null, ['Actions'])
      ])
    ]);
    table.appendChild(thead);

    var tbody = el('tbody');
    items.forEach(function(it){
      var tr = el('tr');

      // Title cell
      tr.appendChild(el('td', null, [
        el('a', {href: it.url || '#', target:'_blank'}, [it.title || '(no title)'])
      ]));

      // Source cell
      tr.appendChild(el('td', null, [it.source || '']));

      // Actions cell
      var btnCopy = el('button', {class:'button'}, ['Copy Title']);
      btnCopy.addEventListener('click', function(){
        copyToClipboard(it.title || '');
      });

      var btnGen = el('button', {class:'button button-primary'}, ['Generate With AI']);
      btnGen.addEventListener('click', function(){
        var topic = it.title || '';
        if (!topic) return;

        // Build the generate endpoint from the localized base (/miro/v1/trends)
var genUrl = (PP_TRENDS.rest_url || '').replace(/\/trends(?:\?.*)?$/, '/trends/generate_draft');

fetch(genUrl, {
  method: 'POST',
  headers: {
    'X-WP-Nonce': PP_TRENDS.nonce,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({ topic: it.title || '', use_ai: true })
})

        .then(function(r){ return r.json(); })
        .then(function(res){
          if (res && res.ok) {
            status('Draft created!');
            if (res.edit_link) window.open(res.edit_link, '_blank');
          } else {
            status('Failed to generate');
          }
        })
        .catch(function(e){
          status('Error');
        });
      });

      var tdAct = el('td', null, [
        btnCopy,
        document.createTextNode(' '),
        btnGen
      ]);
      tr.appendChild(tdAct);

      tbody.appendChild(tr);
    });

    table.appendChild(tbody);
    wrap.appendChild(table);
  }

  function fetchTrends(force){
    status('Loading...');
    var url = PP_TRENDS.rest_url + (force ? '?force=1' : '');
    var headers = {};
    if (force && PP_TRENDS.nonce) {
      headers['X-WP-Nonce'] = PP_TRENDS.nonce;
    }
    fetch(url, {
      method: 'GET',
      headers: headers
    })
    .then(function(r){ return r.json(); })
    .then(function(json){
      renderList(json.items || []);
      status('Loaded '+(json.count||0)+' items'+(json.country ? ' ('+json.country+')' : ''));
    })
    .catch(function(err){
      status('Error loading trends');
    });
  }

  document.addEventListener('DOMContentLoaded', function(){
    var btn = document.getElementById('pp-trends-refresh');
    if (btn) btn.addEventListener('click', function(){ fetchTrends(true); });
    fetchTrends(false);
  });
})();
