<?php
namespace Miro_AI_SEO;
if (!defined('ABSPATH')) exit;

/**
 * AI Fix — Drawer UI (no popup), SERP preview, tone/focus controls
 * - Works in Classic + Gutenberg
 * - No global CSS frameworks (prevents admin layout issues)
 * - Proper REST nonce: wp_rest via X-WP-Nonce
 * - Keyboard shortcut: Ctrl+Shift+K to open/close
 */
class AI_Fix {

    const REST_NS = 'miro/v1';

    public static function init() {
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue']);
        add_action('admin_footer', [__CLASS__, 'print_drawer']);
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
        add_action('add_meta_boxes', [__CLASS__, 'add_metabox'], 20);
    }

    /* ---------------- Metabox (Classic) ---------------- */
    public static function add_metabox() {
        $screen = get_current_screen();
        if (!$screen || $screen->base !== 'post') return;
        add_meta_box(
            'miro_ai_fix_metabox',
            'AI Fix SEO',
            function($post){
                echo '<button type="button" class="button button-primary" id="miro-ai-fix-open">Open AI Fix</button>';
                echo '<p class="description" style="margin-top:8px;">Improve Title, Meta Description, and Slug with AI.</p>';
                echo '<p class="description">Shortcut: <code>Ctrl+Shift+K</code></p>';
            },
            null,
            'side',
            'high'
        );
    }

    public static function enqueue($hook) {
        if (!in_array($hook, ['post.php','post-new.php'], true)) return;
        // no global CSS libs to avoid layout shifts
    }

    /* ---------------- Drawer markup + JS ---------------- */
    public static function print_drawer() {
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, ['post','post-new'], true)) return;

        $rest       = esc_url_raw(rest_url());
        $rest_nonce = wp_create_nonce('wp_rest');
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Post ID for editor context, not form processing.
        $post_id    = isset($_GET['post']) ? absint(wp_unslash($_GET['post'])) : 0;
        if ($screen->base === 'post-new') $post_id = 0;

        $site_host  = wp_parse_url(home_url(), PHP_URL_HOST) ?: '';
        ?>
        <style>
          /* Drawer container (right side), no overlay, non-blocking */
          #miro-ai-fix-drawer {
            position: fixed;
            top: 32px; /* below admin bar */
            right: 0;
            width: 440px;
            max-width: 92vw;
            height: calc(100vh - 32px);
            background: #fff;
            border-left: 1px solid #e5e7eb;
            box-shadow: -6px 0 24px rgba(0,0,0,.08);
            transform: translateX(110%);
            transition: transform .22s ease;
            z-index: 100000; /* above editor header */
            display: flex;
            flex-direction: column;
          }
          body.wp-admin.folded #miro-ai-fix-drawer {
            top: 32px;
            height: calc(100vh - 32px);
          }
          #miro-ai-fix-drawer.open { transform: translateX(0); }

          /* Header, footer, scrollable content */
          .miro-d-head {
            padding: 10px 12px;
            border-bottom: 1px solid #e5e7eb;
            display: flex; align-items: center; justify-content: space-between;
            background: #fff;
          }
          .miro-d-title { margin: 0; font-size: 15px; font-weight: 600; }
          .miro-d-body {
            padding: 12px; overflow: auto; flex: 1 1 auto;
          }
          .miro-d-foot {
            padding: 10px 12px;
            border-top: 1px solid #e5e7eb;
            display: flex; gap: 8px; justify-content: flex-end;
            background: #fff;
          }

          /* Inputs / layout */
          .miro-row{ display: flex; gap: 10px; }
          .miro-col{ flex: 1; min-width: 0; }
          .miro-label{ display:block; font-weight:600; color:#0f172a; margin-bottom:4px; }
          .miro-input{ width: 100%; border:1px solid #e5e7eb; border-radius:8px; padding:8px; font-size:13px; }
          .miro-help{ color:#6b7280; font-size:11.5px; }
          .miro-badge{ display:inline-block; background:#eef2ff; color:#1e40af; border-radius:10px; padding:2px 8px; font-size:11px; }
          .miro-kbd{ font-family:monospace; background:#f1f5f9; border-radius:4px; padding:1px 6px; }

          /* SERP preview */
          .miro-serp { border:1px solid #e5e7eb; border-radius:10px; padding:12px; }
          .miro-serp-url { color:#166534; font-size:12px; margin-bottom:2px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap; }
          .miro-serp-title { color:#1a0dab; font-size:18px; line-height:1.25; margin-bottom:4px; }
          .miro-serp-desc { color:#4b5563; font-size:13px; }
          .miro-counts { font-size:11px; color:#6b7280; margin-top:6px; display:flex; gap:10px; }
          .miro-counts b { color:#111827; }
          .miro-bad { color:#b91c1c; }
          .miro-good{ color:#065f46; }

          @media (max-width: 782px){
            #miro-ai-fix-drawer { top: 46px; height: calc(100vh - 46px); width: 100vw; }
            .miro-row { flex-direction: column; }
          }
        </style>

        <!-- Drawer -->
        <div id="miro-ai-fix-drawer" aria-hidden="true">
          <div class="miro-d-head">
            <h2 class="miro-d-title">AI Fix SEO</h2>
            <div style="display:flex;align-items:center;gap:8px;color:#6b7280;font-size:12px;">
              <span>Shortcut <span class="miro-kbd">Ctrl+Shift+K</span></span>
              <button type="button" class="button" id="miro-ai-fix-close">Close</button>
            </div>
          </div>

          <div class="miro-d-body">
            <div id="miro-ai-fix-alert" class="hidden" style="margin-bottom:8px; font-size:13px;"></div>

            <!-- Controls -->
            <div class="miro-row" style="margin-bottom:10px;">
              <div class="miro-col">
                <label class="miro-label">Focus Keyword (optional)</label>
                <input type="text" id="miro-ai-fix-focus" class="miro-input" placeholder="e.g. best running shoes for flat feet">
                <div class="miro-help">We’ll bias Title/Description to include this.</div>
              </div>
              <div class="miro-col">
                <label class="miro-label">Tone</label>
                <select id="miro-ai-fix-tone" class="miro-input">
                  <option value="neutral" selected>Neutral</option>
                  <option value="friendly">Friendly</option>
                  <option value="punchy">Punchy</option>
                  <option value="formal">Formal</option>
                  <option value="confident">Confident</option>
                </select>
              </div>
            </div>

            <div class="miro-row" style="margin-bottom:10px;">
              <div class="miro-col">
                <label class="miro-label">Title Target Length</label>
                <select id="miro-ai-fix-title-len" class="miro-input">
                  <option value="55-60" selected>~55–60 chars</option>
                  <option value="50-55">~50–55 chars</option>
                  <option value="60-65">~60–65 chars</option>
                </select>
              </div>
              <div class="miro-col">
                <label class="miro-label">Description Target Length</label>
                <select id="miro-ai-fix-desc-len" class="miro-input">
                  <option value="150-160" selected>~150–160 chars</option>
                  <option value="140-155">~140–155 chars</option>
                  <option value="160-170">~160–170 chars</option>
                </select>
              </div>
            </div>

            <!-- Context -->
            <label class="miro-label">Context used by AI <span class="miro-badge">read-only</span></label>
            <textarea id="miro-ai-fix-context" class="miro-input" rows="3" readonly></textarea>

            <!-- Title -->
            <div class="miro-row" style="margin-top:10px;">
              <div class="miro-col">
                <label class="miro-label">Current Title</label>
                <input type="text" id="miro-ai-fix-title-current" class="miro-input" readonly>
              </div>
              <div class="miro-col">
                <label class="miro-label">Suggested Title</label>
                <input type="text" id="miro-ai-fix-title-new" class="miro-input" placeholder="~55–60 chars">
              </div>
            </div>

            <!-- Description -->
            <div class="miro-row" style="margin-top:10px;">
              <div class="miro-col">
                <label class="miro-label">Current Meta Description</label>
                <textarea id="miro-ai-fix-desc-current" class="miro-input" rows="3" readonly></textarea>
              </div>
              <div class="miro-col">
                <label class="miro-label">Suggested Meta Description</label>
                <textarea id="miro-ai-fix-desc-new" class="miro-input" rows="3" placeholder="~150–160 chars"></textarea>
              </div>
            </div>

            <!-- Slug -->
            <div style="margin-top:10px;">
              <label class="miro-label">Suggested Slug</label>
              <input type="text" id="miro-ai-fix-slug-new" class="miro-input" placeholder="post-url-slug">
            </div>

            <!-- SERP Preview -->
            <div style="margin-top:12px;">
              <label class="miro-label">SERP Preview</label>
              <div class="miro-serp" id="miro-serp">
                <div class="miro-serp-url"><?php echo esc_html($site_host); ?> › <span id="miro-serp-path">…</span></div>
                <div class="miro-serp-title" id="miro-serp-title">—</div>
                <div class="miro-serp-desc" id="miro-serp-desc">—</div>
                <div class="miro-counts">
                  <span>Title: <b id="miro-count-title">0</b> chars</span>
                  <span>Description: <b id="miro-count-desc">0</b> chars</span>
                </div>
              </div>
              <div class="miro-help" style="margin-top:6px;">Green counts ≈ good range; red = too short/long.</div>
            </div>
          </div>

          <div class="miro-d-foot">
            <button type="button" class="button" id="miro-ai-fix-regenerate">Regenerate</button>
            <button type="button" class="button button-primary" id="miro-ai-fix-apply">Apply</button>
          </div>
        </div>

        <script>
        (function(){
          // Add header button if not present (Gutenberg)
          if (!document.getElementById('miro-ai-fix-open')) {
            const hdr = document.querySelector('.edit-post-header__toolbar') || document.querySelector('#wpbody-content .wrap h1');
            if (hdr) {
              const btn = document.createElement('button');
              btn.id = 'miro-ai-fix-open';
              btn.className = 'button button-primary';
              btn.textContent = 'AI Fix SEO';
              btn.style.marginLeft = '8px';
              hdr.appendChild(btn);
            }
          }

          const postId    = <?php echo (int)$post_id; ?>;
          const restRoot  = <?php echo json_encode($rest); ?>;
          const restNonce = <?php echo json_encode($rest_nonce); ?>;

          const drawer  = document.getElementById('miro-ai-fix-drawer');
          const openBtn = document.getElementById('miro-ai-fix-open');
          const closeBtn= document.getElementById('miro-ai-fix-close');
          const alertEl = document.getElementById('miro-ai-fix-alert');

          const focusEl = document.getElementById('miro-ai-fix-focus');
          const toneEl  = document.getElementById('miro-ai-fix-tone');
          const tLenEl  = document.getElementById('miro-ai-fix-title-len');
          const dLenEl  = document.getElementById('miro-ai-fix-desc-len');

          const ctxEl  = document.getElementById('miro-ai-fix-context');
          const tCurEl = document.getElementById('miro-ai-fix-title-current');
          const tNewEl = document.getElementById('miro-ai-fix-title-new');
          const dCurEl = document.getElementById('miro-ai-fix-desc-current');
          const dNewEl = document.getElementById('miro-ai-fix-desc-new');
          const sNewEl = document.getElementById('miro-ai-fix-slug-new');

          const serpTitle = document.getElementById('miro-serp-title');
          const serpDesc  = document.getElementById('miro-serp-desc');
          const serpPath  = document.getElementById('miro-serp-path');
          const cntTitle  = document.getElementById('miro-count-title');
          const cntDesc   = document.getElementById('miro-count-desc');

          function openDrawer(){ drawer.classList.add('open'); }
          function closeDrawer(){ drawer.classList.remove('open'); }
          function note(msg, ok){
            alertEl.textContent = msg;
            alertEl.style.color = ok ? '#065f46' : '#b91c1c';
            alertEl.classList.remove('hidden');
            setTimeout(()=>alertEl.classList.add('hidden'), 3500);
          }

          // keyboard shortcut Ctrl+Shift+K
          document.addEventListener('keydown', function(e){
            if ((e.ctrlKey || e.metaKey) && e.shiftKey && e.key.toLowerCase() === 'k') {
              e.preventDefault();
              if (drawer.classList.contains('open')) closeDrawer(); else { openDrawer(); fetchSuggestions(); }
            }
            if (e.key === 'Escape' && drawer.classList.contains('open')) closeDrawer();
          });

          function getTitle(){
            var el = document.querySelector('#post-title-0') || document.getElementById('title');
            return el ? el.value.trim() : '';
          }
          function getContent(){
            var classic = document.getElementById('content');
            if (classic) return classic.value.trim();
            var canvas = document.querySelector('.editor-styles-wrapper');
            return canvas ? (canvas.innerText || '').slice(0, 5000) : '';
          }
          function getExcerpt(){
            var ex = document.getElementById('excerpt');
            return ex ? ex.value.trim() : '';
          }
          function slugify(s){
            return (s||'').toLowerCase().replace(/[^a-z0-9\- ]+/g,'').trim().replace(/\s+/g,'-').replace(/\-+/g,'-').replace(/^\-+|\-+$/g,'');
          }

          function populateFromEditor(){
            const t = getTitle();
            const c = getContent();
            const ex= getExcerpt();
            tCurEl.value = t || '';
            dCurEl.value = ex || '';
            ctxEl.value  = (t?(t+'\\n\\n'):'') + (ex?(ex+'\\n\\n'):'') + c;
            if (!sNewEl.value) sNewEl.value = slugify(t);
            updatePreview();
          }

          function inRange(n, lo, hi){ return n>=lo && n<=hi; }
          function updateCounts(){
            const t = (tNewEl.value||'').trim();
            const d = (dNewEl.value||'').trim();
            const tt = t.length, dd = d.length;
            cntTitle.textContent = tt;
            cntDesc.textContent  = dd;
            const [tLo,tHi] = (tLenEl.value||'55-60').split('-').map(Number);
            const [dLo,dHi] = (dLenEl.value||'150-160').split('-').map(Number);
            cntTitle.className = inRange(tt, tLo, tHi) ? 'miro-good' : 'miro-bad';
            cntDesc.className  = inRange(dd, dLo, dHi) ? 'miro-good' : 'miro-bad';
          }
          function updatePreview(){
            updateCounts();
            serpTitle.textContent = (tNewEl.value||'').trim() || '—';
            serpDesc.textContent  = (dNewEl.value||'').trim() || '—';
            serpPath.textContent  = slugify(sNewEl.value||'') || '…';
          }
          ['input','change'].forEach(evt=>{
            tNewEl.addEventListener(evt, updatePreview);
            dNewEl.addEventListener(evt, updatePreview);
            sNewEl.addEventListener(evt, updatePreview);
            tLenEl.addEventListener(evt, updatePreview);
            dLenEl.addEventListener(evt, updatePreview);
          });

          function fetchSuggestions(){
            populateFromEditor();
            note('Generating suggestions…', true);
            const payload = {
              post_id: postId,
              context: ctxEl.value,
              current_title: tCurEl.value,
              current_desc: dCurEl.value,
              focus: (focusEl.value||'').trim(),
              tone: (toneEl.value||'neutral').trim(),
              title_len: (tLenEl.value||'55-60'),
              desc_len: (dLenEl.value||'150-160')
            };
            return fetch(restRoot + 'miro/v1/ai/fix/suggest', {
              method: 'POST',
              headers: {'Content-Type':'application/json','X-WP-Nonce': restNonce},
              body: JSON.stringify(payload),
              credentials: 'same-origin'
            })
            .then(r => r.json())
            .then(j => {
              if (!j || j.code) { throw new Error(j && j.message ? j.message : 'AI error'); }
              if (j.title) tNewEl.value = j.title;
              if (j.description) dNewEl.value = j.description;
              if (j.slug) sNewEl.value = j.slug;
              updatePreview();
              note('Suggestions ready', true);
            })
            .catch(err => note(err.message || 'Failed to generate', false));
          }

          function applySuggestions(){
            if (!postId) { note('Save draft first to get a post ID.', false); return; }
            const payload = {
              post_id: postId,
              title: (tNewEl.value||'').trim(),
              description: (dNewEl.value||'').trim(),
              slug: slugify((sNewEl.value||'').trim())
            };
            if (!payload.title && !payload.description && !payload.slug) {
              note('Nothing to apply.', false); return;
            }
            note('Applying…', true);
            fetch(restRoot + 'miro/v1/ai/fix/apply', {
              method:'POST',
              headers:{'Content-Type':'application/json','X-WP-Nonce': restNonce},
              body: JSON.stringify(payload),
              credentials: 'same-origin'
            })
            .then(r=>r.json())
            .then(j=>{
              if (!j || j.code) throw new Error(j && j.message ? j.message : 'Apply failed');
              note('Applied. Reloading…', true);
              setTimeout(()=>{ location.reload(); }, 700);
            })
            .catch(err=> note(err.message||'Failed to apply', false));
          }

          // Wire
          if (openBtn) openBtn.addEventListener('click', function(){
            if (!drawer.classList.contains('open')) { openDrawer(); fetchSuggestions(); }
          });
          if (closeBtn) closeBtn.addEventListener('click', closeDrawer);
          document.addEventListener('click', function(e){
            // click outside on right edge closes (small UX nicety)
            if (drawer.classList.contains('open')) {
              const within = drawer.contains(e.target) || (openBtn && openBtn.contains(e.target));
              if (!within && e.clientX < (window.innerWidth - drawer.offsetWidth)) {
                closeDrawer();
              }
            }
          });
          document.getElementById('miro-ai-fix-regenerate').addEventListener('click', fetchSuggestions);
          document.getElementById('miro-ai-fix-apply').addEventListener('click', applySuggestions);
        })();
        </script>
        <?php
    }

    /* ---------------- REST ---------------- */
    public static function register_routes() {
        register_rest_route(self::REST_NS, '/ai/fix/suggest', [
            'methods'  => 'POST',
            'permission_callback' => function(\WP_REST_Request $r){
                return current_user_can('edit_posts');
            },
            'callback' => [__CLASS__, 'rest_suggest'],
        ]);

        register_rest_route(self::REST_NS, '/ai/fix/apply', [
            'methods'  => 'POST',
            'permission_callback' => function(\WP_REST_Request $r){
                $p = $r->get_json_params();
                $post_id = isset($p['post_id']) ? (int)$p['post_id'] : 0;
                return $post_id ? current_user_can('edit_post', $post_id) : current_user_can('edit_posts');
            },
            'callback' => [__CLASS__, 'rest_apply'],
        ]);
    }

    public static function rest_suggest(\WP_REST_Request $r) {
        $nonce = $r->get_header('X-WP-Nonce');
        if (!$nonce || !wp_verify_nonce($nonce, 'wp_rest')) {
            return new \WP_Error('rest_cookie_invalid_nonce','Invalid REST nonce.', ['status'=>403]);
        }

        $p = $r->get_json_params();
        $context = wp_strip_all_tags((string)($p['context'] ?? ''), true);
        $cur_title = sanitize_text_field((string)($p['current_title'] ?? ''));
        $cur_desc  = sanitize_text_field((string)($p['current_desc'] ?? ''));
        $focus     = sanitize_text_field((string)($p['focus'] ?? ''));
        $tone      = sanitize_text_field((string)($p['tone'] ?? 'neutral'));
        $title_len = preg_replace('/[^0-9\-]/','', (string)($p['title_len'] ?? '55-60'));
        $desc_len  = preg_replace('/[^0-9\-]/','', (string)($p['desc_len'] ?? '150-160'));

        $api = self::get_api_key();
        $model = self::get_model();
        if (!$api) {
            return new \WP_Error('no_key','OpenAI key missing. Set it in Miro AI → Settings.', ['status'=>400]);
        }

        $sys = "You are a senior SEO copywriter. Return ONLY JSON with: {title:string, description:string, slug:string}.
Constraints:
- Tone: {$tone}
- Focus keyword: ".($focus ?: 'none')."
- Title target length: {$title_len} chars (approx), concise, compelling, include focus if provided.
- Meta description target length: {$desc_len} chars (approx), natural, include focus if provided, value-driven.
- Slug: lowercase, hyphenated, <= 7 words, remove stopwords. No extra text.";

        $usr = "Current Title: {$cur_title}
Current Description: {$cur_desc}
Context:
{$context}

Return improved values.";

        $out = self::call_openai_json($api, $model, $sys, $usr);
        if (!is_array($out)) $out = [];
        $title = sanitize_text_field($out['title'] ?? '');
        $desc  = sanitize_text_field($out['description'] ?? '');
        $slug  = sanitize_title($out['slug'] ?? '');

        return ['title'=>$title, 'description'=>$desc, 'slug'=>$slug];
    }

    public static function rest_apply(\WP_REST_Request $r) {
        $nonce = $r->get_header('X-WP-Nonce');
        if (!$nonce || !wp_verify_nonce($nonce, 'wp_rest')) {
            return new \WP_Error('rest_cookie_invalid_nonce','Invalid REST nonce.', ['status'=>403]);
        }

        $p = $r->get_json_params();
        $post_id = isset($p['post_id']) ? (int)$p['post_id'] : 0;
        if (!$post_id || !current_user_can('edit_post', $post_id)) {
            return new \WP_Error('bad_post','Missing or invalid post_id', ['status'=>400]);
        }

        $title = sanitize_text_field((string)($p['title'] ?? ''));
        $desc  = sanitize_text_field((string)($p['description'] ?? ''));
        $slug  = sanitize_title((string)($p['slug'] ?? ''));

        $upd = ['ID'=>$post_id];
        if ($title !== '') $upd['post_title'] = $title;
        if ($slug  !== '') $upd['post_name']  = wp_unique_post_slug($slug, $post_id, get_post_status($post_id), get_post_type($post_id), wp_get_post_parent_id($post_id));

        if (count($upd) > 1) {
            $res = wp_update_post($upd, true);
            if (is_wp_error($res)) return new \WP_Error('update_failed', $res->get_error_message(), ['status'=>500]);
        }

        if ($desc !== '') update_post_meta($post_id, 'miro_meta_description', $desc);

        // Yoast
        if (defined('WPSEO_VERSION')) {
            if ($title !== '') update_post_meta($post_id, '_yoast_wpseo_title', $title);
            if ($desc  !== '') update_post_meta($post_id, '_yoast_wpseo_metadesc', $desc);
        }
        // Rank Math
        if (defined('RANK_MATH_VERSION')) {
            if ($title !== '') update_post_meta($post_id, 'rank_math_title', $title);
            if ($desc  !== '') update_post_meta($post_id, 'rank_math_description', $desc);
        }

        return ['ok'=>true];
    }

    /* ---------------- Helpers ---------------- */
    private static function get_api_key(): string {
        $key = get_option('miro_ai_openai_key', '');
        if (!empty($key)) return trim($key);
        foreach (['MIRO_AI_OPENAI_API_KEY','OPENAI_API_KEY'] as $c) {
            if (defined($c) && constant($c)) return trim(constant($c));
        }
        foreach (['MIRO_OPENAI_API_KEY','OPENAI_API_KEY'] as $e) {
            $v = getenv($e); if ($v) return trim($v);
        }
        $v = apply_filters('miro_ai/ai_key', '');
        return is_string($v) ? trim($v) : '';
    }
    private static function get_model(): string {
        $model = apply_filters('miro_ai/ai_model', '');
        return is_string($model) && $model !== '' ? trim($model) : 'gpt-4o-mini';
    }
    private static function call_openai_json($api, $model, $sys, $usr) {
        $res = wp_remote_post('https://api.openai.com/v1/chat/completions', [
            'timeout' => 45,
            'headers' => [
                'Authorization' => 'Bearer '.$api,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode([
                'model' => $model,
                'messages' => [
                    ['role'=>'system','content'=>$sys],
                    ['role'=>'user','content'=>$usr],
                ],
                'response_format' => ['type'=>'json_object'],
                'temperature' => 0.35,
            ]),
        ]);
        if (is_wp_error($res)) return [];
        $json = json_decode(wp_remote_retrieve_body($res), true);
        $content = $json['choices'][0]['message']['content'] ?? '';
        if (!$content) return [];
        $parsed = json_decode($content, true);
        return is_array($parsed) ? $parsed : [];
    }
}
