<?php
if (!defined('ABSPATH')) exit;

/**
 * ALT Fix — Self-contained, premium FS-style UI
 * - Full-width hero under admin notices
 * - Scans posts page-by-page (Next / Previous)
 * - Shows ALT coverage per post with colored pills + status badges
 * - Bulk-generate ALT tags for selected posts
 *
 * REST endpoints expected:
 *   POST /miro/v1/altfix/scan
 *   POST /miro/v1/altfix/apply
 */
class Miro_Alt_Fix_Admin {
    const DEFAULT_PARENT_SLUG = 'miro-ai-seo';
    private static $hook_suffix = '';

    public static function init() {
        add_action('admin_menu', [__CLASS__, 'menu'], 99);
        add_action('admin_enqueue_scripts', [__CLASS__, 'assets']);
    }

    public static function menu() {
        $parent_slug = apply_filters('miro/parent_slug', self::DEFAULT_PARENT_SLUG);
        global $admin_page_hooks;
        if (!isset($admin_page_hooks[$parent_slug])) {
            $parent_slug = 'options-general.php';
        }

        self::$hook_suffix = add_submenu_page(
            $parent_slug,
            'ALT Fix',
            'ALT Fix',
            'manage_options',
            'miro-alt-fix',
            [__CLASS__, 'render_page'],
            40
        );
    }

    public static function assets($hook) {
        if ($hook !== self::$hook_suffix) return;

        wp_enqueue_script('wp-api-fetch');
        wp_enqueue_script('jquery');

        wp_localize_script('jquery', 'MiroAltFixCfg', [
            'nonce'    => wp_create_nonce('wp_rest'),
            'scanURL'  => rest_url('miro/v1/altfix/scan'),
            'applyURL' => rest_url('miro/v1/altfix/apply'),
        ]);
    }

    public static function render_page() {
        // logo (optional, safe)
        $af_logo = (defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/img/miro-logo.webp' : '');
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">ALT Fix</h1>
        </div>

        <div class="miro-alt-wrap">
<style>
  /* ===== Page wrapper ===== */
  .miro-alt-wrap{
    margin: 10px 20px 20px 0;
  }
  .miro-alt-wrap *{
    box-sizing: border-box;
  }

  /* =========================
     HERO (soft pro gray — not white)
     ========================= */

  .af-hero{
    margin: 0 0 18px;
    border-radius: 12px;
    padding: 14px 16px;

    background: linear-gradient(180deg, #f3f4f6 0%, #eef2f7 100%);
    border: 1px solid #d6d9de;
    border-left: 6px solid #f59e0b;
    box-shadow: 0 8px 20px rgba(15,23,42,0.07);

    display: grid;
    grid-template-columns: 52px 1fr auto;
    gap: 14px;
    align-items: center;

    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
  }

  /* Logo box blends with hero */
  .af-hero-pill{
    width: 48px;
    height: 48px;
    border-radius: 12px;
    background: rgba(255,255,255,0.55);
    border: 1px solid rgba(214,217,222,0.95);
    display:flex;
    align-items:center;
    justify-content:center;
    overflow: hidden;
    box-shadow: 0 4px 12px rgba(15,23,42,0.06);
  }
  .af-hero-pill-inner{
    width: 100%;
    height: 100%;
    display:flex;
    align-items:center;
    justify-content:center;
    background: transparent;
  }
  .af-hero-pill-inner img{
    width: 112%;
    height: 112%;
    object-fit: contain;
    display:block;
  }
  .af-hero-pill-fallback{
    font-weight: 600;
    letter-spacing: .04em;
    font-size: 12px;
    color: #111827;
  }

  /* Main */
  .af-hero-main{ min-width: 0; }

  .af-hero-title-row{
    display:flex;
    align-items:center;
    gap: 8px;
    flex-wrap: wrap;
    margin-bottom: 2px;
  }
  .af-hero-title{
    margin: 0;
    font-size: 15px;
    font-weight: 500;
    color: #111827;
    letter-spacing: normal;
  }

  /* Tags */
  .af-hero-tag{
    padding: 2px 8px;
    border-radius: 999px;
    font-size: 10px;
    text-transform: uppercase;
    letter-spacing: .06em;
    background: rgba(255,255,255,0.55);
    border: 1px solid rgba(214,217,222,0.95);
    color: #374151;
    font-weight: 500;
  }
  .af-hero-tag.alt{
    background: rgba(238,242,255,0.70);
    border-color: rgba(199,210,254,0.95);
    color: #1e3a8a;
  }

  .af-hero-sub{
    margin: 0;
    font-size: 13px;
    font-weight: 400;
    color: #4b5563;
    line-height: 1.5;
  }

  /* Chips row */
  .af-hero-chips{
    display:flex;
    flex-wrap:wrap;
    gap: 8px;
    margin-top: 10px;
  }

  .af-chip,
  .af-chip-pill{
    padding: 5px 10px;
    border-radius: 999px;
    font-size: 12px;
    border: 1px solid rgba(214,217,222,0.95);
    background: rgba(255,255,255,0.55);
    color: #111827;
    display:inline-flex;
    align-items:center;
    gap: 6px;
    white-space: nowrap;
    font-weight: 400;
  }

  .af-chip strong{
    font-weight: 500;
    color: #111827;
  }
  .af-chip span{
    color: #6b7280;
    font-weight: 400;
  }

  /* Safe mode pill (darker green + white text) */
  .af-chip-pill.on{
    border-color: rgba(22,163,74,0.70);
    background: rgba(22,163,74,0.55);
    color: #ffffff;
    font-weight: 600;
  }

  .af-chip-pill.off{
    border-color: rgba(239,68,68,0.45);
    background: rgba(239,68,68,0.10);
    color: #7f1d1d;
    font-weight: 600;
  }

  /* Right tip box */
 /* ===== Tip box (pro) ===== */
.af-hero-metrics{
  justify-self: end;
  align-self: stretch;
  min-width: 260px;

  background: rgba(255,255,255,0.58);
  border: 1px solid rgba(214,217,222,0.95);
  border-radius: 12px;
  padding: 10px 12px;

  display:flex;
  flex-direction: column;
  gap: 8px;

  box-shadow: 0 4px 14px rgba(15,23,42,0.06);
  position: relative;
  overflow: hidden;
}

/* small top accent */
.af-hero-metrics::before{
  content:"";
  position:absolute;
  left:0; top:0; right:0;
  height: 2px;
  background: linear-gradient(90deg, rgba(245,158,11,0.8), rgba(37,99,235,0.55));
  opacity: .9;
}

.af-tip-title{
  display:flex;
  align-items:center;
  gap: 8px;
  font-size: 13px;
  font-weight: 400;
  color: #111827;
}
.af-tip-title strong{
  font-weight: 500;
}

.af-tip-ico{
  width: 24px;
  height: 24px;
  border-radius: 8px;
  display:flex;
  align-items:center;
  justify-content:center;

  background: rgba(245,158,11,0.12);
  border: 1px solid rgba(245,158,11,0.22);
  font-size: 14px;
  line-height: 1;
}

.af-tip-sub{
  font-size: 12px;
  font-weight: 400;
  color: #6b7280;
  line-height: 1.35;
}

.af-tip-hl{
  display:inline-block;
  padding: 1px 6px;
  border-radius: 999px;
  background: rgba(245,158,11,0.14);
  border: 1px solid rgba(245,158,11,0.22);
  color: #7c4a03;
  font-weight: 600;
}


  /* =========================
     Cards
     ========================= */
  .af-card {
    background: #ffffff;
    border-radius: 16px;
    border: 1px solid #e5e7eb;
    padding: 18px 20px;
    margin-bottom: 16px;
    box-shadow: 0 10px 30px rgba(15,23,42,0.06);
  }
  .af-card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 10px;
    margin-bottom: 12px;
  }
  .af-card-title {
    font-size: 15px;
    font-weight: 600;
    color: #111827;
  }
  .af-card-sub {
    font-size: 12px;
    font-weight: 400;
    color: #6b7280;
  }

  .af-form-grid {
    display: grid;
    grid-template-columns: minmax(0, 1.2fr) minmax(0, 1fr);
    gap: 14px 24px;
  }
  @media (max-width: 900px){
    .af-form-grid { grid-template-columns: minmax(0, 1fr); }
  }

  .af-field { display: flex; flex-direction: column; gap: 5px; font-size: 13px; }
  .af-label { font-weight: 600; color: #111827; }
  .af-label small { font-weight: 400; color: #6b7280; margin-left: 4px; }

  .af-input, .af-select {
    width: 100%;
    padding: 6px 8px;
    border-radius: 8px;
    border: 1px solid #d1d5db;
    font-size: 13px;
    color: #111827;
    background: #ffffff;
  }
  .af-input:focus, .af-select:focus {
    outline: none;
    border-color: #2563eb;
    box-shadow: 0 0 0 1px rgba(37,99,235,0.25);
  }

  .af-radio-row, .af-checkbox-row {
    display: flex;
    flex-wrap: wrap;
    gap: 10px 18px;
    font-size: 13px;
    color: #111827;
  }
  .af-radio-row label, .af-checkbox-row label {
    display: inline-flex;
    align-items: center;
    gap: 6px;
  }
  .af-radio-row strong { font-weight: 600; }

  /* Toolbar / status */
  .af-toolbar {
    display: flex;
    flex-wrap: wrap;
    align-items: center;
    gap: 10px;
    margin-top: 12px;
  }
  .af-toolbar .button-primary {
    border-radius: 999px;
    padding: 6px 16px;
  }
  .af-toolbar .button { font-size: 13px; }
  .af-pager {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    border-radius: 999px;
    padding: 3px 8px;
    background: #f9fafb;
    border: 1px solid #e5e7eb;
  }
  .af-pager span { font-size: 12px; color: #6b7280; }
  .af-toolbar-status { font-size: 12px; color: #6b7280; }

  .af-hidden { display: none !important; }

  /* KPI summary */
  .af-kpi-grid {
    display: grid;
    grid-template-columns: repeat(4, minmax(0, 1fr));
    gap: 10px;
    margin-bottom: 18px;
  }
  @media (max-width: 900px) {
    .af-kpi-grid { grid-template-columns: repeat(2, minmax(0, 1fr)); }
  }
  .af-kpi {
    border-radius: 12px;
    padding: 10px 12px;
    background: #f9fafb;
    border: 1px solid #e5e7eb;
  }
  .af-kpi-label { font-size: 11px; color: #6b7280; margin-bottom: 3px; }
  .af-kpi-value { font-size: 17px; font-weight: 600; color: #111827; }

  /* Results table */
  .miro-alt-wrap table.af-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
    border-radius: 14px;
    overflow: hidden;
    border: 1px solid #e5e7eb;
    background: #ffffff;
  }
  .miro-alt-wrap table.af-table thead { background: #f3f4f6; }
  .miro-alt-wrap table.af-table thead th {
    padding: 8px 10px;
    font-size: 12px;
    font-weight: 600;
    color: #4b5563;
    border-bottom: 1px solid #e5e7eb;
  }
  .miro-alt-wrap table.af-table tbody td {
    padding: 7px 10px;
    font-size: 13px;
    color: #111827;
    border-bottom: 1px solid #edf2f7;
    vertical-align: middle;
  }
  .miro-alt-wrap table.af-table tbody tr:nth-child(even) td { background: #fafafa; }
  .miro-alt-wrap table.af-table tbody tr.af-hidden-row td { display: none; }

  .af-actions .button.button-small {
    margin-right: 4px;
    margin-bottom: 2px;
  }

  /* Pills + status */
  .af-img-pill {
    display: inline-flex;
    align-items: center;
    padding: 2px 10px;
    border-radius: 999px;
    font-size: 11px;
    font-weight: 600;
    letter-spacing: 0.01em;
    white-space: nowrap;
  }
  .af-img-good { background: #e6f6f0; color: #047857; }
  .af-img-mid  { background: #fff7e6; color: #92400e; }
  .af-img-bad  { background: #fee2e2; color: #b91c1c; }

  .af-status-badge {
    display: inline-flex;
    align-items: center;
    padding: 2px 9px;
    border-radius: 999px;
    font-size: 11px;
    font-weight: 600;
  }
  .af-status-critical { background:#fee2e2; color:#b91c1c; }
  .af-status-needs-fix { background:#fff7e6; color:#92400e; }
  .af-status-good     { background:#e6f6f0; color:#047857; }
  .af-status-none     { background:#e5e7eb; color:#374151; }

  /* Filters */
  .af-results-head {
    display: flex;
    flex-wrap: wrap;
    justify-content: space-between;
    align-items: center;
    gap: 10px;
    margin-bottom: 6px;
  }
  .af-filters {
    display: inline-flex;
    flex-wrap: wrap;
    gap: 8px;
    align-items: center;
    font-size: 12px;
    color: #4b5563;
  }
  .af-filters label {
    display: inline-flex;
    align-items: center;
    gap: 4px;
  }
  .af-search {
    padding: 5px 8px;
    border-radius: 999px;
    border: 1px solid #d1d5db;
    font-size: 12px;
    min-width: 200px;
  }
  .af-search:focus {
    outline: none;
    border-color: #2563eb;
    box-shadow: 0 0 0 1px rgba(37,99,235,0.25);
  }

  /* Inspector */
  .af-inspector-title {
    font-size: 14px;
    font-weight: 600;
    margin-bottom: 6px;
  }
  .af-inspector-desc {
    font-size: 12px;
    font-weight: 400;
    color: #6b7280;
    margin-bottom: 10px;
  }
  .af-inspector-table {
    width: 100%;
    border-collapse: collapse;
    border-radius: 10px;
    overflow: hidden;
    border: 1px solid #e5e7eb;
    background: #ffffff;
  }
  .af-inspector-table th,
  .af-inspector-table td {
    padding: 6px 10px;
    font-size: 12px;
    border-bottom: 1px solid #edf2f7;
  }
  .af-inspector-table thead { background: #f9fafb; }

  .af-muted { font-size: 12px; color: #6b7280; }
  /* ===== Colored feature chips (soft + pro) ===== */
.af-chip-blue{
  background: rgba(37,99,235,0.12) !important;
  border-color: rgba(37,99,235,0.28) !important;
  color: #0b2a6f !important;
}
.af-chip-blue span{ color: rgba(11,42,111,0.75) !important; }

.af-chip-purple{
  background: rgba(124,58,237,0.12) !important;
  border-color: rgba(124,58,237,0.26) !important;
  color: #3b1a8a !important;
}
.af-chip-purple span{ color: rgba(59,26,138,0.72) !important; }

/* Optional: make Safe pill slightly richer */
.af-chip-pill.on{
  background: rgba(22,163,74,0.58) !important;
  border-color: rgba(22,163,74,0.72) !important;
  color: #fff !important;
}
/* ===== Pro chips ===== */
.af-hero-chips{ gap:10px; }

.af-chip-pro{
  position: relative;
  padding: 6px 12px;
  border-radius: 999px;
  background: rgba(255,255,255,0.55);
  border: 1px solid rgba(214,217,222,0.95);
  box-shadow: 0 2px 8px rgba(15,23,42,0.05);
  display: inline-flex;
  align-items: center;
  gap: 8px;
  font-weight: 400;
  color: #111827;
}

/* left accent line (the “pro” touch) */
.af-chip-pro::before{
  content:"";
  width: 8px;
  height: 8px;
  border-radius: 999px;
  background: #94a3b8;
  box-shadow: 0 0 0 3px rgba(148,163,184,0.18);
}

/* small sub text */
.af-chip-sub{
  color: #6b7280;
  font-weight: 400;
  font-size: 12px;
}

/* keep strong medium weight */
.af-chip-pro strong{
  font-weight: 500;
  color: #111827;
}

/* variant tints (subtle) */
.af-chip-alt{
  background: rgba(245,158,11,0.10);               /* Miro yellow */
  border-color: rgba(245,158,11,0.22);
}
.af-chip-alt::before{
  background: #f59e0b;
  box-shadow: 0 0 0 3px rgba(245,158,11,0.18);
}

.af-chip-scan{
  background: rgba(37,99,235,0.08);                /* blue */
  border-color: rgba(37,99,235,0.18);
}
.af-chip-scan::before{
  background: #2563eb;
  box-shadow: 0 0 0 3px rgba(37,99,235,0.16);
}

.af-chip-safe{
  background: rgba(22,163,74,0.10);                /* green */
  border-color: rgba(22,163,74,0.20);
}
.af-chip-safe::before{
  background: #16a34a;
  box-shadow: 0 0 0 3px rgba(22,163,74,0.16);
}

</style>



            <!-- HERO (new) -->
          <div class="af-hero">
    <div class="af-hero-pill">
        <div class="af-hero-pill-inner">
            <?php if (!empty($af_logo)) : ?>
                <img src="<?php echo esc_url($af_logo); ?>" alt="<?php echo esc_attr__('Miro AI SEO logo','miro-ai-seo-free'); ?>">
            <?php else: ?>
                <div class="af-hero-pill-fallback">ALT</div>
            <?php endif; ?>
        </div>
    </div>

    <div class="af-hero-main">
        <div class="af-hero-title-row">
            <div class="af-hero-title">Fix missing ALT text in bulk.</div>
            <span class="af-hero-tag">Media SEO</span>
        </div>

        <p class="af-hero-sub">
            Scan posts page by page, see which images are missing ALT,
            and auto-generate descriptive ALT text with a single click.
        </p>

<div class="af-hero-chips">
  <div class="af-chip af-chip-pro af-chip-alt"><span class="af-dot"></span>ALT Doctor<span class="af-chip-sub">module</span></div>
  <div class="af-chip af-chip-pro af-chip-scan"><span class="af-dot"></span>Page-by-page<span class="af-chip-sub">scanner</span></div>
  <div class="af-chip af-chip-pro af-chip-safe"><span class="af-dot"></span>Safe mode<span class="af-chip-sub">supported</span></div>
</div>


    </div>

<div class="af-hero-metrics">
  <div class="af-tip-title">
    <span class="af-tip-ico">💡</span>
    <span>Tip: Run monthly</span>
  </div>

  <div class="af-tip-sub">
    Best practice: use <span class="af-tip-hl">Fill only missing ALT</span> unless you really want to overwrite.
  </div>
</div>

</div>


            <div class="af-card">
                <div class="af-card-header">
                    <div>
                        <div class="af-card-title">Scan settings</div>
                        <div class="af-card-sub">
                            Pick which content to scan and how aggressive the ALT replacement should be.
                        </div>
                    </div>
                </div>

                <div class="af-form-grid">
                    <div class="af-field">
                        <div class="af-label">Post type</div>
                        <select id="af-posttype" class="af-select">
                            <option value="post">Posts</option>
                            <option value="page">Pages</option>
                        </select>
                    </div>

                    <div class="af-field">
                        <div class="af-label">
                            Include featured image
                            <small>Count and fix featured image ALT too.</small>
                        </div>
                        <div class="af-checkbox-row">
                            <label>
                                <input type="checkbox" id="af-featured" checked>
                                Yes, include featured image
                            </label>
                        </div>
                    </div>

                    <div class="af-field">
                        <div class="af-label">Mode</div>
                        <div class="af-radio-row">
                            <label>
                                <input type="radio" name="af-mode" value="fill" checked>
                                <span><strong>Fill only missing ALT</strong> (safe)</span>
                            </label>
                            <label>
                                <input type="radio" name="af-mode" value="overwrite">
                                <span><strong>Overwrite all ALT</strong> (aggressive)</span>
                            </label>
                        </div>
                    </div>

                    <div class="af-field">
                        <div class="af-label">Apply to</div>
                        <div class="af-radio-row">
                            <label>
                                <input type="radio" name="af-apply" value="content" checked>
                                Post content only
                            </label>
                            <label>
                                <input type="radio" name="af-apply" value="library">
                                Content + Media Library
                            </label>
                        </div>
                    </div>

                    <div class="af-field">
                        <div class="af-label">Media title sync</div>
                        <div class="af-checkbox-row">
                            <label>
                                <input type="checkbox" id="af-settitle" checked>
                                Also update attachment Title from generated ALT
                            </label>
                        </div>
                    </div>
                </div>

                <div class="af-toolbar" style="margin-top:16px;">
                    <button type="button" class="button button-primary" id="af-scan">
                        Scan current page
                    </button>

                    <span id="af-scan-progress" class="af-muted af-hidden">Scanning…</span>
                    <span id="af-scan-status" class="af-toolbar-status"></span>
                </div>
            </div>

            <div id="af-summary" class="af-card af-hidden">
                <div class="af-card-header">
                    <div>
                        <div class="af-card-title">Current page overview</div>
                        <div class="af-card-sub">These stats are only for the posts shown on this page.</div>
                    </div>
                </div>
                <div class="af-kpi-grid">
                    <div class="af-kpi">
                        <div class="af-kpi-label">Posts on this page</div>
                        <div class="af-kpi-value" id="af-sum-posts">0</div>
                    </div>
                    <div class="af-kpi">
                        <div class="af-kpi-label">Images (this page)</div>
                        <div class="af-kpi-value" id="af-sum-images">0</div>
                    </div>
                    <div class="af-kpi">
                        <div class="af-kpi-label">Missing ALT</div>
                        <div class="af-kpi-value" id="af-sum-missing">0</div>
                    </div>
                    <div class="af-kpi">
                        <div class="af-kpi-label">ALT coverage</div>
                        <div class="af-kpi-value" id="af-sum-coverage">0%</div>
                    </div>
                </div>
            </div>

            <div id="af-results" class="af-card af-hidden">
                <div class="af-results-head">
                    <div>
                        <div class="af-card-title">Scan results</div>
                        <div class="af-card-sub">Each row shows ALT coverage for one post.</div>
                    </div>
                    <div class="af-filters">
                        <label>
                            <input type="checkbox" id="af-only-missing">
                            Only posts with missing ALT
                        </label>
                        <label>
                            <input type="checkbox" id="af-only-noalt">
                            Only posts where all images have no ALT
                        </label>
                        <label>
                            Status:
                            <select id="af-status-filter" class="af-select" style="width:auto;padding:3px 6px;font-size:12px;">
                                <option value="all">All</option>
                                <option value="critical">Critical</option>
                                <option value="needs-fix">Needs fix</option>
                                <option value="good">Good</option>
                                <option value="none">No images</option>
                            </select>
                        </label>
                        <input type="search" id="af-search" class="af-search" placeholder="Search by title…">
                    </div>
                </div>

                <p class="af-muted" style="margin-bottom:10px;">
                    <strong>Critical</strong> = most images missing ALT,
                    <strong>Needs fix</strong> = some missing,
                    <strong>Good</strong> = all ALT set,
                    <strong>No images</strong> = nothing to fix.
                </p>

                <table class="af-table" id="af-table">
                    <thead>
                    <tr>
                        <th style="width:30px;"><input type="checkbox" id="af-check-all"></th>
                        <th>Post</th>
                        <th>Focus</th>
                        <th>Images &amp; ALT</th>
                        <th>Coverage</th>
                        <th>Status</th>
                        <th style="width:210px;">Actions</th>
                    </tr>
                    </thead>
                    <tbody></tbody>
                </table>

                <div class="af-toolbar" style="margin-top:12px;">
                    <button type="button" class="button button-primary" id="af-apply">
                        Generate ALT for selected
                    </button>

                    <div class="af-pager">
                        <button type="button" class="button button-small" id="af-page-prev" disabled>&laquo;</button>
                        <span id="af-page-label">Page 0</span>
                        <button type="button" class="button button-small" id="af-page-next" disabled>&raquo;</button>
                    </div>

                    <span id="af-apply-progress" class="af-muted af-hidden">Applying…</span>
                </div>

                <div id="af-log" class="af-muted" style="margin-top:10px;"></div>
            </div>

            <div id="af-inspector" class="af-card af-hidden">
                <div class="af-inspector-title" id="af-inspector-title">Images in post</div>
                <div class="af-inspector-desc">
                    Quick view of the images detected in this post and whether they already have ALT text.
                </div>
                <table class="af-inspector-table">
                    <thead>
                    <tr>
                        <th>Attachment ID</th>
                        <th>Featured?</th>
                        <th>ALT present?</th>
                    </tr>
                    </thead>
                    <tbody id="af-inspector-body"></tbody>
                </table>
            </div>
        </div>

        <script>
        jQuery(function($){
            let currentPage      = 0;
            let currentDone      = false;
            let currentPageItems = [];

            function escapeHtml(s){
                return (s||"").toString().replace(/[&<>"']/g, function(m){
                    return ({ "&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#039;" })[m];
                });
            }

            function counts(item){
                const total   = (item.images || []).length;
                const missing = (item.images || []).filter(i => !i.has_alt).length;
                return { missing, total };
            }

            function statusFor(item){
                const c = counts(item);
                if (c.total === 0) return { key: "none",      label: "No images" };
                if (c.missing === 0) return { key: "good",    label: "Good" };

                const ratio = c.missing / c.total;
                if (ratio >= 0.6) return { key: "critical",   label: "Critical" };
                if (ratio >= 0.2) return { key: "needs-fix",  label: "Needs fix" };
                return { key: "good", label: "Good" };
            }

            function coverageFor(item){
                const c = counts(item);
                if (c.total === 0) return 100;
                return ((c.total - c.missing) / c.total) * 100;
            }

            function coverageClass(cov){
                if (cov >= 90) return "af-img-good";
                if (cov >= 60) return "af-img-mid";
                return "af-img-bad";
            }

            function imagesText(c){
                if (c.total === 0) return "No images found";
                if (c.missing === 0) {
                    if (c.total === 1) return "All ALT set (1 image)";
                    return "All ALT set (" + c.total + " images)";
                }
                if (c.missing === 1) return "1 image missing ALT of " + c.total;
                return c.missing + " images missing ALT of " + c.total;
            }

            function rowHtml(item){
                const c      = counts(item);
                const st     = statusFor(item);
                const cov    = coverageFor(item);
                const covCls = coverageClass(cov);

                const edit   = item.edit_link ? "<a href=\""+escapeHtml(item.edit_link)+"\" class=\"button button-small\">Edit</a>" : "";
                const view   = item.view_link ? "<a href=\""+escapeHtml(item.view_link)+"\" class=\"button button-small\" target=\"_blank\" rel=\"noopener\">View</a>" : "";
                const inspectBtn = "<button type=\"button\" class=\"button button-small af-inspect\">View images</button>";
                const statusBadge = "<span class=\"af-status-badge af-status-"+st.key+"\">"+escapeHtml(st.label)+"</span>";

                const title = escapeHtml(item.title || ("#"+item.post_id));
                const type  = escapeHtml(item.post_type || item.type || "");

                return "" +
                    "<tr data-post=\""+item.post_id+"\" " +
                        "data-missing=\""+c.missing+"\" " +
                        "data-total=\""+c.total+"\" " +
                        "data-status=\""+st.key+"\" " +
                        "data-title=\""+title+"\" " +
                        "data-type=\""+type+"\">" +
                        "<td><input type=\"checkbox\" class=\"af-row\"></td>" +
                        "<td><strong>"+title+"</strong></td>" +
                        "<td>"+escapeHtml(item.focus || "")+"</td>" +
                        "<td><span class=\"af-img-pill "+covCls+"\" title=\"ALT coverage for this post\">"+
                            escapeHtml(imagesText(c))+
                        "</span></td>" +
                        "<td>"+cov.toFixed(0)+"%</td>" +
                        "<td>"+statusBadge+"</td>" +
                        "<td class=\"af-actions\">"+inspectBtn+" "+edit+" "+view+"</td>" +
                    "</tr>";
            }

            function updatePagerUI(){
                $("#af-page-label").text("Page " + (currentPage || 0));
                $("#af-page-prev").prop("disabled", currentPage <= 1);
                $("#af-page-next").prop("disabled", currentDone || currentPage === 0);
            }

            function updateSummary(){
                const items = currentPageItems;
                if (!items.length) {
                    $("#af-summary").addClass("af-hidden");
                    return;
                }

                let totalImages  = 0;
                let totalMissing = 0;

                items.forEach(function(item){
                    const c = counts(item);
                    totalImages  += c.total;
                    totalMissing += c.missing;
                });

                const postsOnPage = items.length;
                const covered     = totalImages > 0
                    ? ((totalImages - totalMissing) / totalImages) * 100
                    : 0;

                $("#af-sum-posts").text(postsOnPage.toLocaleString());
                $("#af-sum-images").text(totalImages.toLocaleString());
                $("#af-sum-missing").text(totalMissing.toLocaleString());
                $("#af-sum-coverage").text(covered.toFixed(1)+"%");

                $("#af-summary").removeClass("af-hidden");
            }

            function applyFilters(){
                const onlyMissing  = $("#af-only-missing").is(":checked");
                const onlyNoAlt    = $("#af-only-noalt").is(":checked");
                const statusFilter = $("#af-status-filter").val() || "all";
                const q            = ($("#af-search").val() || "").toLowerCase();
                const chosenType   = $("#af-posttype").val();

                $("#af-table tbody tr").each(function(){
                    const $tr    = $(this);
                    const miss   = parseInt($tr.attr("data-missing"), 10) || 0;
                    const total  = parseInt($tr.attr("data-total"), 10) || 0;
                    const status = $tr.attr("data-status") || "good";
                    const title  = ($tr.attr("data-title") || "").toLowerCase();
                    const type   = ($tr.attr("data-type") || "").toLowerCase();

                    let hide = false;

                    if (chosenType && type && type !== chosenType) hide = true;
                    if (onlyMissing && miss === 0) hide = true;

                    if (onlyNoAlt) {
                        const allNoAlt = (total > 0 && miss === total);
                        if (!allNoAlt) hide = true;
                    }

                    if (statusFilter !== "all" && status !== statusFilter) hide = true;
                    if (q && title.indexOf(q) === -1) hide = true;

                    if (hide) $tr.addClass("af-hidden-row"); else $tr.removeClass("af-hidden-row");
                });
            }

            function renderPage(items, page, doneFlag){
                currentPage      = page;
                currentDone      = !!doneFlag;

                const chosenType = $("#af-posttype").val();

                const filtered = (items || []).filter(function(it){
                    const t = (it.post_type || it.type || "").toLowerCase();
                    if (!chosenType) return true;
                    if (!t) return true;
                    return t === chosenType;
                });

                currentPageItems = filtered;

                const tbody = $("#af-table tbody");
                tbody.empty();

                currentPageItems.forEach(function(it){
                    tbody.append(rowHtml(it));
                });

                if (currentPageItems.length) {
                    $("#af-results").removeClass("af-hidden");
                } else {
                    $("#af-results").addClass("af-hidden");
                }

                updateSummary();
                applyFilters();
                updatePagerUI();
            }

            async function fetchPage(page){
                if (page < 1) return;

                $("#af-scan-progress").removeClass("af-hidden");
                $("#af-log").text("");
                $("#af-inspector").addClass("af-hidden");

                try {
                    const args = {
                        method: "POST",
                        headers: {
                            "Content-Type": "application/json",
                            "X-WP-Nonce": MiroAltFixCfg.nonce
                        },
                        body: JSON.stringify({
                            post_type: $("#af-posttype").val(),
                            include_featured: $("#af-featured").is(":checked"),
                            page: page,
                            per_page: 25
                        })
                    };
                    const res   = await fetch(MiroAltFixCfg.scanURL, args).then(r => r.json());
                    const rawItems = res.items || [];
                    const done  = !!res.done;

                    renderPage(rawItems, res.page || page, done);

                    const txt = done
                        ? "This is the last page of results."
                        : "Showing page " + (res.page || page) + ".";
                    $("#af-scan-status").text(txt);

                } catch(e) {
                    console.error(e);
                    alert("Scan failed. Check console.");
                } finally {
                    $("#af-scan-progress").addClass("af-hidden");
                }
            }

            async function doApply(){
                const rows = $("#af-table tbody tr").not(".af-hidden-row").filter(function(){
                    return $(this).find(".af-row").is(":checked");
                });

                if (!rows.length) {
                    alert("Select at least one post to apply ALT fixes.");
                    return;
                }

                const mode      = $("input[name=\"af-mode\"]:checked").val();
                const applyTo   = $("input[name=\"af-apply\"]:checked").val();
                const setTitle  = $("#af-settitle").is(":checked");
                const inclFeat  = $("#af-featured").is(":checked");

                const items = [];

                rows.each(function(){
                    const postId = parseInt($(this).attr("data-post"), 10);
                    const item   = currentPageItems.find(x => x.post_id === postId);
                    if (!item) return;

                    let ids;
                    if (mode === "fill") {
                        ids = (item.images || []).filter(i => !i.has_alt).map(i => i.id);
                    } else {
                        ids = (item.images || []).map(i => i.id);
                    }

                    if (ids.length) {
                        items.push({ post_id: postId, image_ids: ids });
                    }
                });

                if (!items.length) {
                    alert("Nothing to apply based on your selection and mode.");
                    return;
                }

                $("#af-apply-progress").removeClass("af-hidden");

                try {
                    const res = await fetch(MiroAltFixCfg.applyURL, {
                        method: "POST",
                        headers: {
                            "Content-Type": "application/json",
                            "X-WP-Nonce": MiroAltFixCfg.nonce
                        },
                        body: JSON.stringify({
                            mode: mode,
                            apply_to: applyTo,
                            set_title: setTitle,
                            include_featured: inclFeat,
                            items: items
                        })
                    }).then(r => r.json());

                    const log = $("#af-log");
                    (res.messages || []).forEach(function(m){
                        log.append($("<div>").text(m));
                    });
                    log.append($("<div>").text("Changed ALT on: " + (res.alt_changed || 0) + " images."));

                    if (currentPage > 0) {
                        await fetchPage(currentPage);
                    }
                } catch(e) {
                    console.error(e);
                    alert("Apply failed. Check console.");
                } finally {
                    $("#af-apply-progress").addClass("af-hidden");
                }
            }

            function openInspector(postId){
                const item = currentPageItems.find(x => x.post_id === postId);
                if (!item) return;

                const tbody = $("#af-inspector-body");
                tbody.empty();

                (item.images || []).forEach(function(img){
                    const hasAlt = img.has_alt ? "Yes" : "Missing";
                    const feat   = img.is_featured ? "Yes" : "No";
                    tbody.append(
                        "<tr>" +
                            "<td>"+img.id+"</td>" +
                            "<td>"+feat+"</td>" +
                            "<td>"+hasAlt+"</td>" +
                        "</tr>"
                    );
                });

                $("#af-inspector-title").text("Images in: " + (item.title || ("#"+item.post_id)));
                $("#af-inspector").removeClass("af-hidden");
            }

            // ===== Events =====
            $("#af-scan, #af-scan-hero").on("click", function(){
                fetchPage(1);
            });

            $("#af-page-prev").on("click", function(){
                if (currentPage > 1) fetchPage(currentPage - 1);
            });

            $("#af-page-next").on("click", function(){
                if (!currentDone && currentPage >= 1) fetchPage(currentPage + 1);
            });

            $("#af-apply").on("click", function(){
                doApply();
            });

            $("#af-check-all").on("click", function(){
                $(".af-row").prop("checked", this.checked);
            });

            $("#af-only-missing").on("change", applyFilters);
            $("#af-only-noalt").on("change", applyFilters);
            $("#af-status-filter").on("change", applyFilters);
            $("#af-search").on("input", applyFilters);
            $("#af-posttype").on("change", function(){
                if (currentPage > 0) {
                    fetchPage(1);
                }
            });

            $("#af-table").on("click", ".af-inspect", function(e){
                e.preventDefault();
                const tr     = $(this).closest("tr");
                const postId = parseInt(tr.attr("data-post"), 10);
                openInspector(postId);
            });
        });
        </script>
        <?php
    }
}

Miro_Alt_Fix_Admin::init();
