<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

class AI_Faq {

    public static function init() {
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    public static function register_routes() {
        register_rest_route('miro/v1', '/ai/faq', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'generate'],
            'permission_callback' => function () {
                return current_user_can('edit_posts');
            }
        ]);
    }

    public static function generate($req) {
        try {

            $p        = $req->get_json_params();
            $post_id  = intval($p['post_id'] ?? 0);
            $title    = sanitize_text_field($p['title'] ?? '');
            $content  = wp_kses_post($p['content'] ?? '');

            if (!$title) {
                return new \WP_Error('missing_title', 'Missing post title.', ['status' => 400]);
            }

            // Get AI Key
            $key = get_option('miro_ai_openai_key', '');
            if (!$key) {
                return new \WP_Error('missing_key', 'OpenAI key missing. Set it in Miro AI → Settings.', ['status' => 400]);
            }

            /* -----------------------------------------------------------
             *  IMPROVED PROMPT (clean, stable, and always JSON-safe)
             * ---------------------------------------------------------*/
            $prompt = '
You are an SEO expert. Generate 5–10 high-quality FAQ questions and answers based ONLY on:

TITLE:
' . $title . '

CONTENT:
' . wp_strip_all_tags($content) . '

RULES:
- Keep questions short, clear, and helpful.
- Keep answers 1–3 sentences.
- Avoid repeating the title.
- DO NOT include emojis.
- DO NOT include markdown.
- Return STRICT JSON only.

JSON FORMAT (MANDATORY):
{
  "faqs": [
    {"q": "...", "a": "..."}
  ],
  "schema": {
    "@context": "https://schema.org",
    "@type": "FAQPage",
    "mainEntity": [
      {
        "@type": "Question",
        "name": "...",
        "acceptedAnswer": { "@type": "Answer", "text": "..." }
      }
    ]
  }
}
            ';

            /* -----------------------------------------------------------
             *  CALL OPENAI USING JSON MODE
             * ---------------------------------------------------------*/
            $res = wp_remote_post(
                'https://api.openai.com/v1/chat/completions',
                [
                    'timeout' => 60,
                    'headers' => [
                        'Content-Type'  => 'application/json',
                        'Authorization' => 'Bearer ' . $key,
                    ],
                    'body' => wp_json_encode([
                        'model' => 'gpt-4o-mini',
                        'response_format' => ['type' => 'json_object'],
                        'messages' => [
                            ['role' => 'system', 'content' => 'Return ONLY VALID JSON.'],
                            ['role' => 'user',   'content' => $prompt],
                        ]
                    ])
                ]
            );

            if (is_wp_error($res)) {
                return new \WP_Error('api_error', 'OpenAI API error: ' . $res->get_error_message(), ['status' => 500]);
            }

            $code = wp_remote_retrieve_response_code($res);
            $raw = wp_remote_retrieve_body($res);
            $json = json_decode($raw, true);

            if ($code < 200 || $code >= 300) {
                $error_msg = $json['error']['message'] ?? 'Unknown API error';
                return new \WP_Error('api_error', 'OpenAI API error: ' . $error_msg, ['status' => 500]);
            }

            $content_json = $json['choices'][0]['message']['content'] ?? '';
            if (!$content_json) {
                return new \WP_Error('empty_response', 'OpenAI returned empty response.', ['status' => 500]);
            }

            $parsed = json_decode($content_json, true);

            if (!isset($parsed['faqs'], $parsed['schema'])) {
                return new \WP_Error('bad_json', 'AI returned invalid JSON. Response: ' . substr($content_json, 0, 200), ['status' => 500]);
            }

            // Sanitize FAQ output
            $faqs = [];
            foreach ($parsed['faqs'] as $f) {
                $faqs[] = [
                    'q' => sanitize_text_field($f['q']),
                    'a' => wp_kses_post($f['a'])
                ];
            }

            // Ensure the schema is clean JSON-LD
            $schema = $parsed['schema'];

            /* -----------------------------------------------------------
             *  SAVE SCHEMA FOR GOOGLE RICH RESULTS
             * ---------------------------------------------------------*/
            if ($post_id) {
                update_post_meta($post_id, '_miro_schema_jsonld', wp_json_encode($schema, JSON_UNESCAPED_UNICODE));
                update_post_meta($post_id, '_miro_schema_type', 'FAQPage');
                update_post_meta($post_id, '_miro_schema_auto_print', 1);
            }

            /* -----------------------------------------------------------
             *  FINAL RESPONSE
             * ---------------------------------------------------------*/
            return [
                'faqs'   => $faqs,
                'schema' => $schema,
                'saved'  => (bool)$post_id
            ];

        } catch (\Throwable $e) {
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional debug logging
                error_log('Miro FAQ Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
            }
            return new \WP_Error('fatal', 'Server error: ' . $e->getMessage(), ['status' => 500]);
        }
    }
}
