<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * 404 Monitor — Export CSV & Clear Log + quick summary
 * SS STYLE (ALT Fix FS-style): scoped under .miro-alt-wrap
 */
class Miro_404_Monitor {
    const MENU_PARENT   = 'miro-ai-seo';
    const MENU_SLUG     = 'miro-404-monitor';
    const TABLE_OPT     = 'miro_404_db_version';
    const TABLE_VER     = '1';
    const USE_DB        = true;

    const NONCE_ACTION  = 'miro_404_actions';
    const NONCE_NAME    = 'miro_404_nonce';
    const EXPORT_LIMIT  = 10000; // CSV max rows

    public static function init(): void {
        add_action('admin_menu', [__CLASS__, 'menu'], 12);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
        add_action('template_redirect', [__CLASS__, 'capture_404'], 1);

        add_action('admin_post_miro_404_export', [__CLASS__, 'handle_export']);
        add_action('admin_post_miro_404_clear',  [__CLASS__, 'handle_clear']);
        // register_activation_hook(MIRO_AI_SEO_FILE, [__CLASS__, 'install']); // optional
    }

    /** View cap (can be Editor+) */
    protected static function cap_view(): string {
        return function_exists('\\miro_ai_cap') ? (string)\miro_ai_cap() : 'manage_options';
    }

    /** Manage cap (Admin-only) */
    protected static function cap_manage(): string {
        return function_exists('\\miro_ai_manage_cap') ? (string)\miro_ai_manage_cap() : 'manage_options';
    }

    /** Admin menu */
    public static function menu(): void {
        $cap = self::cap_view();

        if (function_exists('miro_ai_add_submenu_once')) {
            \miro_ai_add_submenu_once(
                self::MENU_PARENT,
                '404 Monitor',
                '404 Monitor',
                $cap,
                self::MENU_SLUG,
                [__CLASS__, 'render_page'],
                33
            );
        } else {
            add_submenu_page(
                self::MENU_PARENT,
                '404 Monitor',
                '404 Monitor',
                $cap,
                self::MENU_SLUG,
                [__CLASS__, 'render_page'],
                33
            );
        }
    }

    /** Optional external CSS */
    public static function enqueue_assets($hook): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Page detection for asset loading, not form processing.
        $page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : '';
        if ($page !== self::MENU_SLUG) return;
        wp_enqueue_style('miro-404-css', MIRO_AI_SEO_URL . 'assets/css/miro-404.css', [], MIRO_AI_SEO_VERSION);
    }

    /** Capture 404s */
    public static function capture_404(): void {
        if (!is_404()) return;

        // Skip admin / AJAX / cron / REST
        if (is_admin()) return;
        if (defined('DOING_AJAX') && DOING_AJAX) return;
        if (defined('DOING_CRON') && DOING_CRON) return;
        if (defined('REST_REQUEST') && REST_REQUEST) return;

        $req_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '/';
        if ($req_uri === '') $req_uri = '/';

        // Ignore common noise probes
        $lower = strtolower($req_uri);
        $noise = [
            'wp-admin', 'wp-login', 'xmlrpc.php', '.php',
            '.env', '.git', 'cgi-bin', 'vendor/', 'wp-content/uploads',
        ];
        foreach ($noise as $needle) {
            if (strpos($lower, $needle) !== false) return;
        }

        // Normalize + store URL WITHOUT query string (better grouping, less spam)
        $path = wp_parse_url($req_uri, PHP_URL_PATH);
        $path = is_string($path) && $path !== '' ? $path : '/';
        $path = '/' . ltrim($path, '/');
        $path = preg_replace('#/+#', '/', $path);

        // Clamp to avoid huge rows
        if (strlen($path) > 2048) $path = substr($path, 0, 2048);

        $url = home_url($path);

        $ref_raw = isset($_SERVER['HTTP_REFERER']) ? sanitize_url(wp_unslash($_SERVER['HTTP_REFERER'])) : '';
        $ref     = $ref_raw !== '' ? self::sanitize_referrer($ref_raw) : '';

        $ua_raw = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $ua     = $ua_raw !== '' ? substr($ua_raw, 0, 200) : '';

        $ip_raw = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $ip     = $ip_raw !== '' ? self::anonymize_ip($ip_raw) : '';

        // Rate limit: same URL + IP only logs once per 60s
        $rl_key = 'miro_404_rl_' . substr(md5($path . '|' . $ip), 0, 16);
        if (get_transient($rl_key)) return;
        set_transient($rl_key, 1, 60);

        // DB first
        if (self::USE_DB && isset($GLOBALS['wpdb'])) {
            global $wpdb;
            $table = self::table_name();
            if (self::table_exists($table)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->insert($table, [
                    'url'        => $url,
                    'referrer'   => $ref,
                    'user_agent' => $ua,
                    'ip'         => $ip,
                    'created_at' => current_time('mysql'),
                ], ['%s','%s','%s','%s','%s']);
                return;
            }
        }

        // Fallback option log (keep last 500)
        $log = get_option('miro_404_fallback_log', []);
        if (!is_array($log)) $log = [];

        $log[] = [
            'url' => $url, 'ref' => $ref, 'ua' => $ua, 'ip' => $ip, 'ts' => current_time('mysql'),
        ];

        if (count($log) > 500) $log = array_slice($log, -500);
        update_option('miro_404_fallback_log', $log, false);
    }

    /** Install DB table */
    public static function install(): void {
        if (!self::USE_DB) return;
        if (!function_exists('dbDelta')) {
            require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        }
        global $wpdb;
        $table   = self::table_name();
        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            url TEXT NOT NULL,
            referrer TEXT NULL,
            user_agent VARCHAR(200) NULL,
            ip VARCHAR(64) NULL,
            created_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY created_at (created_at)
        ) {$charset};";

        dbDelta($sql);
        update_option(self::TABLE_OPT, self::TABLE_VER, false);
    }

    /** Admin page */
    public static function render_page(): void {
        if (!current_user_can(self::cap_view())) {
            wp_die(esc_html__('You do not have permission.', 'miro-ai-seo'));
        }

        $using_db = self::using_db();
        $raw_rows = self::get_rows(300);

        // Aggregate by URL
        $by_url = [];
        foreach ($raw_rows as $r) {
            $url = trim((string)($r['url'] ?? ''));
            if ($url === '') continue;

            $ref = $r['referrer'] ?? ($r['ref'] ?? '');
            $ua  = $r['user_agent'] ?? ($r['ua'] ?? '');
            $ip  = $r['ip'] ?? '';
            $ts  = $r['created_at'] ?? ($r['ts'] ?? '');

            if (!isset($by_url[$url])) {
                $by_url[$url] = [
                    'when' => $ts,
                    'url'  => $url,
                    'ref'  => $ref,
                    'ua'   => $ua,
                    'ip'   => $ip,
                    'hits' => 1,
                ];
            } else {
                $by_url[$url]['hits']++;
                // Keep last seen
                $by_url[$url]['when'] = $ts;
                $by_url[$url]['ref']  = $ref;
                $by_url[$url]['ua']   = $ua;
                $by_url[$url]['ip']   = $ip;
            }
        }

        $rows = array_values($by_url);

        usort($rows, function ($a, $b) {
            $cmp = ($b['hits'] <=> $a['hits']);
            if ($cmp !== 0) return $cmp;
            return strcmp((string)$b['when'], (string)$a['when']);
        });

        if (count($rows) > 100) $rows = array_slice($rows, 0, 100);

        $summary = self::summarize(1000);

        $export_url = esc_url(admin_url('admin-post.php?action=miro_404_export'));
        $clear_url  = esc_url(admin_url('admin-post.php?action=miro_404_clear'));

        $af_logo = (defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/img/miro-logo.webp' : '');

        echo '<div class="wrap"><h1 class="wp-heading-inline">404 Monitor</h1></div>';
        echo '<div class="miro-alt-wrap">';

        // HERO
        echo '<div class="af-hero">';
        echo '  <div class="af-hero-pill"><div class="af-hero-pill-inner">';
        if (!empty($af_logo)) {
            echo '<img src="'.esc_url($af_logo).'" alt="'.esc_attr__('Miro AI SEO logo','miro-ai-seo').'">';
        } else {
            echo '<div class="af-hero-pill-fallback">404</div>';
        }
        echo '  </div></div>';

        echo '  <div class="af-hero-main">';
        echo '    <div class="af-hero-title-row">';
        echo '      <div class="af-hero-title">Find & fix broken URLs before they waste crawl budget.</div>';
        echo '      <span class="af-hero-tag alt">Monitor</span>';
        echo '    </div>';
        echo '    <p class="af-hero-sub">See which paths return 404, where they come from, and fix the ones that matter.</p>';
        echo '    <div class="af-hero-chips">';
        echo '      <div class="af-chip-pro af-chip-alt"><strong>'.intval($summary['unique_urls']).'</strong><span class="af-chip-sub">unique URLs</span></div>';
        echo '      <div class="af-chip-pro af-chip-scan"><strong>'.intval($summary['count']).'</strong><span class="af-chip-sub">hits scanned</span></div>';
        echo '      <div class="af-chip-pro af-chip-purple"><strong>'.($using_db ? 'DB' : 'Fallback').'</strong><span class="af-chip-sub">storage</span></div>';
        echo '      <div class="af-chip-pro af-chip-safe"><strong>Top first</strong><span class="af-chip-sub">by hits</span></div>';
        echo '    </div>';
        echo '  </div>';

        echo '  <div class="af-hero-metrics">';
        echo '    <div class="af-tip-title"><span class="af-tip-ico">💡</span><span><strong>Tip:</strong> fix the top URLs</span></div>';
        echo '    <div class="af-tip-sub">Start with URLs that have the most hits → add a redirect, fix internal links, or update campaigns. Bad 404s can hurt <span class="af-tip-hl">crawl efficiency</span>.</div>';
        echo '    <div class="af-toolbar miro-404-toolbar-mt">';

        echo '      <form method="post" action="'.esc_attr($export_url).'" class="miro-404-form-m0">';
        wp_nonce_field(self::NONCE_ACTION, self::NONCE_NAME);
        echo '        <button type="submit" class="button button-primary">Export CSV</button>';
        echo '      </form>';

        echo '      <form method="post" action="'.esc_attr($clear_url).'" onsubmit="return confirm(\'Clear the entire 404 log?\');" class="miro-404-form-m0">';
        wp_nonce_field(self::NONCE_ACTION, self::NONCE_NAME);
        echo '        <button type="submit" class="button">Clear Log</button>';
        echo '      </form>';

        echo '    </div>';
        echo '  </div>';
        echo '</div>';

        // Summary card
        echo '<div class="af-card">';
        echo '  <div class="af-card-header">';
        echo '    <div>';
        echo '      <div class="af-card-title">Summary</div>';
        echo '      <div class="af-card-sub">Based on the last '.intval($summary['count']).' logged hits.</div>';
        echo '    </div>';
        echo '  </div>';

        echo '  <div class="af-grid">';
        echo '    <div class="af-kpi miro-404-kpi-span3"><small>Total hits</small><strong>404 hits</strong><div class="af-val">'.intval($summary['count']).'</div><div class="af-sub">Spikes can be bots or broken links.</div></div>';
        echo '    <div class="af-kpi miro-404-kpi-span3"><small>Distinct URLs</small><strong>Unique URLs</strong><div class="af-val">'.intval($summary['unique_urls']).'</div><div class="af-sub">Fix highest-hit URLs first.</div></div>';
        echo '    <div class="af-kpi miro-404-kpi-span3"><small>Referrers</small><strong>Unique referrers</strong><div class="af-val">'.intval($summary['unique_refs']).'</div><div class="af-sub">Campaigns & external sites matter.</div></div>';
        echo '    <div class="af-kpi miro-404-kpi-span3"><small>IPs</small><strong>Unique IPs</strong><div class="af-val">'.intval($summary['unique_ips']).'</div><div class="af-sub">IPs are anonymized.</div></div>';

        echo '    <div class="af-box miro-404-box-span6"><small>Most frequently broken pages</small><strong class="miro-404-strong-block">Top URLs</strong><ol>';
        if (!empty($summary['top_urls'])) {
            foreach ($summary['top_urls'] as $u => $c) {
                echo '<li><code>'.esc_html($u).'</code> — '.intval($c).' hits</li>';
            }
        } else {
            echo '<li>No 404 URLs recorded yet.</li>';
        }
        echo '    </ol></div>';

        echo '    <div class="af-box miro-404-box-span6"><small>Where the bad links come from</small><strong class="miro-404-strong-block">Top referrers</strong><ol>';
        if (!empty($summary['top_refs'])) {
            foreach ($summary['top_refs'] as $r => $c) {
                $label = $r === '' ? '(direct / no referrer)' : $r;
                echo '<li><code>'.esc_html($label).'</code> — '.intval($c).' hits</li>';
            }
        } else {
            echo '<li>No referrer data yet.</li>';
        }
        echo '    </ol></div>';

        echo '  </div>';

        echo '  <div class="af-help-row">';
        echo '    <div class="af-help af-help-green"><strong>Safe fixes</strong><ul><li>Fix internal links</li><li>Update menus + sitemap links</li><li>Correct typos in slugs</li></ul></div>';
        echo '    <div class="af-help af-help-blue"><strong>When to redirect</strong><ul><li>Old URL has backlinks</li><li>Old campaign URLs</li><li>Moved content</li></ul></div>';
        echo '    <div class="af-help af-help-orange"><strong>Ignore noise</strong><ul><li>Random bot scans</li><li>wp-admin probes</li><li>Non-existent files</li></ul></div>';
        echo '  </div>';

        echo '</div>';

        // Table card
        echo '<div class="af-card">';
        echo '  <div class="af-card-header">';
        echo '    <div><div class="af-card-title">404 URLs (unique)</div><div class="af-card-sub">Grouped by URL with total hit count (showing up to 100 rows).</div></div>';
        echo '    <div class="af-muted">'.($using_db ? 'Logging: DB' : 'Logging: Fallback option').' </div>';
        echo '  </div>';

        echo '  <div class="af-table-wrap">';
        echo '    <div class="af-table-head"><h2>Results</h2><span>Drag header edges to resize columns</span></div>';
        echo '    <table class="widefat striped miro-404-resizable-table">';
        echo '      <thead><tr>
                    <th class="miro-404-th-width">When (last hit)</th>
                    <th>URL</th>
                    <th class="miro-404-th-width-90">Hits</th>
                    <th class="miro-404-th-width-240">Referrer (last)</th>
                    <th class="miro-404-th-width-260">User Agent (last)</th>
                    <th class="miro-404-th-width">IP (last)</th>
                  </tr></thead><tbody>';

        if (empty($rows)) {
            echo '<tr><td colspan="6">No 404 entries recorded yet.</td></tr>';
        } else {
            foreach ($rows as $r) {
                $when    = (string)($r['when'] ?? '');
                $url_raw = (string)($r['url'] ?? '');
                $ref     = (string)($r['ref'] ?? '');
                $ua      = (string)($r['ua'] ?? '');
                $ip      = (string)($r['ip'] ?? '');
                $hits    = intval($r['hits'] ?? 1);

                echo '<tr>';
                echo '<td class="af-nowrap">'.esc_html($when).'</td>';
                echo '<td class="af-ellipsis" title="'.esc_attr($url_raw).'">'.esc_html($url_raw).'</td>';
                echo '<td class="af-nowrap miro-404-td-center">'.esc_html((string)$hits).'</td>';
                echo '<td class="af-break">'.esc_html($ref).'</td>';
                echo '<td class="af-break">'.esc_html($ua).'</td>';
                echo '<td class="af-nowrap">'.esc_html($ip).'</td>';
                echo '</tr>';
            }
        }

        echo '      </tbody></table>';
        echo '  </div>';
        echo '</div>';

        // Column resize JS (your original)
        echo '<script>
        (function(){
            var table = document.querySelector(".miro-alt-wrap .miro-404-resizable-table");
            if (!table) return;
            var ths = table.querySelectorAll("thead th");
            if (!ths || !ths.length) return;

            var startX, startWidth, colIndex, dragging = false;

            ths.forEach(function(th, index){
                var resizer = document.createElement("div");
                resizer.className = "miro-404-resizer";
                th.appendChild(resizer);

                resizer.addEventListener("mousedown", function(e){
                    e.preventDefault();
                    startX = e.pageX;
                    startWidth = th.offsetWidth;
                    colIndex = index;
                    dragging = true;
                    th.classList.add("miro-404-resizing");
                    document.addEventListener("mousemove", onMouseMove);
                    document.addEventListener("mouseup", onMouseUp);
                });
            });

            function onMouseMove(e){
                if (!dragging) return;
                var dx = e.pageX - startX;
                var newWidth = startWidth + dx;
                if (newWidth < 80) newWidth = 80;

                var header = ths[colIndex];
                header.style.width = newWidth + "px";
                header.style.minWidth = newWidth + "px";

                var rows = table.querySelectorAll("tbody tr");
                rows.forEach(function(row){
                    var cell = row.cells[colIndex];
                    if (cell) {
                        cell.style.width = newWidth + "px";
                        cell.style.minWidth = newWidth + "px";
                    }
                });
            }

            function onMouseUp(){
                if (!dragging) return;
                dragging = false;
                ths[colIndex].classList.remove("miro-404-resizing");
                document.removeEventListener("mousemove", onMouseMove);
                document.removeEventListener("mouseup", onMouseUp);
            }
        })();
        </script>';

        echo '</div>'; // .miro-alt-wrap
    }

    /* ===== Export & Clear ===== */

    public static function handle_export(): void {
        // ✅ Admin-only
        if (!current_user_can(self::cap_manage())) {
            wp_die(esc_html__('Insufficient permissions.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        check_admin_referer(self::NONCE_ACTION, self::NONCE_NAME);

        $rows = self::get_rows(self::EXPORT_LIMIT, true);
        $filename = 'miro-404-export-' . gmdate('Y-m-d-His') . '.csv';

        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="'.$filename.'"');

        // php://output is a stream; WP_Filesystem does not apply.
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        $out = fopen('php://output', 'w');
        fputcsv($out, ['when','url','referrer','user_agent','ip']);

        foreach ($rows as $r) {
            $when = $r['created_at'] ?? ($r['ts'] ?? '');
            $url  = $r['url'] ?? '';
            $ref  = $r['referrer'] ?? ($r['ref'] ?? '');
            $ua   = $r['user_agent'] ?? ($r['ua'] ?? '');
            $ip   = $r['ip'] ?? '';
            fputcsv($out, [$when, $url, $ref, $ua, $ip]);
        }
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose($out);
        exit;
    }

    public static function handle_clear(): void {
        // ✅ Admin-only
        if (!current_user_can(self::cap_manage())) {
            wp_die(esc_html__('Insufficient permissions.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        check_admin_referer(self::NONCE_ACTION, self::NONCE_NAME);

        $cleared = false;

        if (self::using_db()) {
            global $wpdb;
            $table = self::table_name();
            $table_safe = self::safe_table($table);
            if ($table_safe) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name validated by safe_table(); identifiers cannot be parameterized.
                $wpdb->query("TRUNCATE TABLE `{$table_safe}`");
                $cleared = true;
            }
        }

        delete_option('miro_404_fallback_log');

        wp_safe_redirect(add_query_arg([
            'page'    => self::MENU_SLUG,
            'cleared' => $cleared ? '1' : '0'
        ], admin_url('admin.php')));
        exit;
    }

    /* ====================== helpers ====================== */

    protected static function using_db(): bool {
        if (!self::USE_DB || !isset($GLOBALS['wpdb'])) return false;
        $table = self::table_name();
        return self::table_exists($table);
    }

    protected static function table_name(): string {
        global $wpdb;
        return $wpdb->prefix . 'miro_404_log';
    }

    /** Return safe table name or false */
    protected static function safe_table(string $table) {
        if (!isset($GLOBALS['wpdb'])) return false;
        global $wpdb;
        $prefix = (string)$wpdb->prefix;

        if (strpos($table, $prefix) !== 0) return false;

        $suffix = substr($table, strlen($prefix));
        if (!preg_match('/^[a-zA-Z0-9_]+$/', (string)$suffix)) return false;

        return $table;
    }

    protected static function table_exists(string $table): bool {
        if (!isset($GLOBALS['wpdb'])) return false;
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
        return $exists === $table;
    }

    protected static function get_rows(int $limit = 100, bool $raw = false): array {
        $limit = max(1, $limit);

        if (self::using_db()) {
            global $wpdb;
            $table = self::safe_table(self::table_name());
            if (!$table) return [];

            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name validated by safe_table(); identifiers cannot be parameterized.
            $sql = $wpdb->prepare(
                "SELECT id, url, referrer, user_agent, ip, created_at
                 FROM `{$table}`
                 ORDER BY id DESC
                 LIMIT %d",
                $limit
            );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

            // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $rows = $wpdb->get_results($sql, ARRAY_A);
            return is_array($rows) ? $rows : [];
        }

        $rows = get_option('miro_404_fallback_log', []);
        if (!is_array($rows)) $rows = [];
        $rows = array_reverse($rows);
        if (count($rows) > $limit) $rows = array_slice($rows, 0, $limit);
        return $rows;
    }

    protected static function summarize(int $scanN = 1000): array {
        $scanN = max(10, $scanN);
        $rows  = self::get_rows($scanN, true);

        $count = count($rows);
        $urls = [];
        $refs = [];
        $ips  = [];

        foreach ($rows as $r) {
            $u = trim((string)($r['url'] ?? ''));
            $f = trim((string)($r['referrer'] ?? ($r['ref'] ?? '')));
            $i = trim((string)($r['ip'] ?? ''));

            if ($u !== '') $urls[$u] = ($urls[$u] ?? 0) + 1;
            $refs[$f] = ($refs[$f] ?? 0) + 1;
            if ($i !== '') $ips[$i] = ($ips[$i] ?? 0) + 1;
        }

        arsort($urls);
        arsort($refs);

        return [
            'count'       => $count,
            'unique_urls' => count($urls),
            'unique_refs' => count($refs),
            'unique_ips'  => count($ips),
            'top_urls'    => array_slice($urls, 0, 5, true),
            'top_refs'    => array_slice($refs, 0, 5, true),
        ];
    }

    protected static function anonymize_ip(string $ip): string {
        $ip = trim($ip);
        // ipv4: mask last octet
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $parts = explode('.', $ip);
            if (count($parts) === 4) {
                $parts[3] = '0';
                return implode('.', $parts);
            }
            return $ip;
        }
        // ipv6: keep first 4 blocks
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $parts = explode(':', $ip);
            $parts = array_slice($parts, 0, 4);
            return implode(':', $parts) . '::';
        }
        return sanitize_text_field($ip);
    }

    protected static function sanitize_referrer(string $ref): string {
        $ref = trim($ref);
        $p = wp_parse_url($ref);
        if (!is_array($p)) return '';
        $host = isset($p['host']) ? (string)$p['host'] : '';
        $path = isset($p['path']) ? (string)$p['path'] : '';
        if ($host === '' && $path === '') return '';
        $out = ($host !== '' ? $host : '') . ($path !== '' ? $path : '');
        if (strlen($out) > 400) $out = substr($out, 0, 400);
        return sanitize_text_field($out);
    }
}
