<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) { exit; }

class Admin {

    private static $booted = false;

    public static function init() {
        if (self::$booted) return;
        self::$booted = true;

        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin']);
        add_action('admin_init', [__CLASS__, 'register_settings']);

        add_action('admin_post_miro_suite_full_reset',   [__CLASS__, 'handle_full_reset']);
        add_action('admin_post_miro_suite_custom_reset', [__CLASS__, 'handle_custom_reset']);
        add_action('admin_post_miro_gsc_reset',          [__CLASS__, 'handle_gsc_reset']);

        add_filter('miro/parent_slug', [__CLASS__, 'parent_slug']);

        add_filter('miro/ai/complete', [__CLASS__, 'ai_complete'], 10, 2);
    }

    public static function parent_slug() { return 'miro-ai-seo'; }

    public static function register_settings() {
        register_setting('miro_ai_seo_settings', 'miro_ai_openai_key', [
            'sanitize_callback' => function ($v) { return is_string($v) ? sanitize_text_field(trim($v)) : ''; },
        ]);
        register_setting('miro_ai_seo_settings', 'miro_ai_gemini_key', [
            'sanitize_callback' => function ($v) { return is_string($v) ? sanitize_text_field(trim($v)) : ''; },
        ]);
        register_setting('miro_ai_seo_settings', 'miro_ai_focus_language', [
            'default' => 'en',
            'sanitize_callback' => function ($v) {
                $v = is_string($v) ? sanitize_text_field(trim($v)) : '';
                return $v !== '' ? $v : 'en';
            },
        ]);
    }

    public static function enqueue_admin($hook) {
        $is_miro_screen = (strpos($hook, '_page_miro-') !== false) || (strpos($hook, 'toplevel_page_miro-ai-seo') !== false);
        if (!$is_miro_screen) return;

        wp_enqueue_style('miro-ai-admin', MIRO_AI_SEO_URL . 'assets/css/admin.css', [], MIRO_AI_SEO_VERSION);

        wp_enqueue_script(
            'miro-ai-admin-js',
            MIRO_AI_SEO_URL . 'assets/js/admin.js',
            ['wp-element', 'wp-api-fetch', 'wp-components'],
            MIRO_AI_SEO_VERSION,
            true
        );

        wp_localize_script('miro-ai-admin-js', 'MiroAI', [
            'rest'  => esc_url_raw(rest_url('miro/v1/')),
            'nonce' => wp_create_nonce('wp_rest'),
            'opts'  => ['lang' => get_option('miro_ai_focus_language', 'en')]
        ]);

        // Trends asset only on trends page
        $is_trends = (strpos($hook, '_page_miro-ai-trends') !== false);
        if ($is_trends) {
            wp_enqueue_script(
                'miro-trends-admin',
                MIRO_AI_SEO_URL . 'assets/js/trends-admin.js',
                [],
                MIRO_AI_SEO_VERSION,
                true
            );
            // Use current request protocol (fixes localhost SSL issues)
            $rest_url = rest_url('miro/v1/trends');
            $host = wp_parse_url(home_url(), PHP_URL_HOST);
            if ($host === 'localhost' || strpos($host, '127.0.0.1') !== false || strpos($host, '.local') !== false) {
                $rest_url = set_url_scheme($rest_url, 'http');
            } else {
                $rest_url = set_url_scheme($rest_url, null); // Use current request scheme
            }
            wp_localize_script('miro-trends-admin', 'PP_TRENDS', [
                'rest_url' => esc_url_raw($rest_url),
                'nonce'    => wp_create_nonce('wp_rest'),
            ]);
        }

        // Overview quick GSC sync (only if button exists)
        $is_overview = (strpos($hook, 'toplevel_page_miro-ai-seo') !== false) || (strpos($hook, '_page_miro-ai-seo') !== false);
        if ($is_overview) {
            $rest  = esc_url_raw(rest_url('miro/v1/'));
            $nonce = wp_create_nonce('wp_rest');
            wp_add_inline_script('miro-ai-admin-js', '
            (function(){
              var btn = document.getElementById("miro-gsc-quick-sync");
              var status = document.getElementById("miro-gsc-quick-status");
              if(!btn || !status) return;
              function setS(t){ status.textContent = t; }
              async function start(){
                try{
                  setS("Starting…");
                  var r = await fetch("'.$rest.'gsc/sync/start", { method:"POST", headers:{"X-WP-Nonce":"'.$nonce.'"} });
                  var j = await r.json();
                  if(!j.job_id){ setS("Error: no job id"); return; }
                  poll(j.job_id);
                }catch(e){ setS("Error: " + e.message); }
              }
              function poll(job){
                var iv = setInterval(async function(){
                  try{
                    var r = await fetch("'.$rest.'gsc/sync/status?job=" + encodeURIComponent(job), { headers:{"X-WP-Nonce":"'.$nonce.'"} });
                    var s = await r.json();
                    setS( (s.error ? ("Error: " + s.error) : (s.step + " — " + (s.progress||0) + "%")) );
                    if(s.done) clearInterval(iv);
                  }catch(e){}
                }, 3000);
              }
              btn.addEventListener("click", function(e){ e.preventDefault(); start(); });
            })();', 'after');
        }
    }

    // Editor panels and metaboxes removed for CodeCanyon release

    // ----- RENDER METHODS (these prevent “undefined method” fatals) -----

    public static function render_dashboard() {
        echo '<div class="wrap"><h1>Miro AI SEO — Dashboard</h1>';

        echo '<div class="miro-grid" style="display:grid;grid-template-columns:repeat(auto-fit,minmax(320px,1fr));gap:16px;margin-top:12px;">';

        echo '<div class="card" style="background:#fff;border:1px solid #e2e8f0;border-radius:8px;padding:16px;">';
        echo '<h2 style="margin-top:0;">Google Search Console</h2>';
        echo '<p>Use background sync to avoid admin freezes.</p>';
        echo '<p>';
        echo '<a href="'.esc_url(admin_url('admin.php?page=miro-gsc')).'" class="button">Open GSC</a> ';
        echo '<button id="miro-gsc-quick-sync" class="button button-primary">Sync Now (Background)</button> ';
        echo '<span id="miro-gsc-quick-status" class="miro-gsc-status-ml">Idle</span>';
        echo '</p>';
        echo '<p class="description">Requires connection & selected property in GSC page.</p>';
        echo '</div>';

        echo '<div class="miro-card">';
        echo '<h2>Quick Tips</h2>';
        echo '<ul class="miro-list-disc">';
        echo '<li>Keep long actions async (REST + cron), never block admin.</li>';
        echo '<li>Load assets only on plugin pages.</li>';
        echo '<li>Render UI from cache; never call live APIs on page load.</li>';
        echo '</ul>';
        echo '</div>';

        echo '</div>';

        echo '<h2 style="margin-top:24px;">Recent Sync Log</h2>';
        $log = get_option('miro_ai_seo_sync_log', []);
        if (!$log) {
            echo '<p>No entries yet.</p>';
        } else {
            echo '<ul>';
            foreach (array_slice(array_reverse($log), 0, 10) as $row) {
                echo '<li><code>' . esc_html($row['ts']) . '</code> — ' . esc_html($row['note']) . '</li>';
            }
            echo '</ul>';
        }
        echo '</div>';
    }

    public static function render_generator() {
        // Hidden from release build - stub feature
        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('AI Post Generator', 'miro-ai-seo') . '</h1>';
        echo '<p>' . esc_html__('This feature is currently under development.', 'miro-ai-seo') . '</p>';
        echo '</div>';
    }

    public static function render_settings() {
        $af_logo = defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/img/miro-logo.webp' : '';
        ?>
        <div class="wrap">
        <h1 class="wp-heading-inline"><?php echo esc_html__('Settings', 'miro-ai-seo'); ?></h1>
        </div>
        
        <div class="miro-alt-wrap">
        <!-- Hero Banner (Fix Center style) -->
        <div class="af-hero">
            <div class="af-hero-pill">
                <div class="af-hero-pill-inner">
                    <?php if (!empty($af_logo)): ?>
                        <img src="<?php echo esc_url($af_logo); ?>" alt="<?php echo esc_attr__('Miro AI SEO logo', 'miro-ai-seo'); ?>">
                    <?php else: ?>
                        <div class="af-hero-pill-fallback">⚙️</div>
                    <?php endif; ?>
                </div>
            </div>
            <div class="af-hero-main">
                <div class="af-hero-title-row">
                    <div class="af-hero-title">Plugin Configuration</div>
                    <span class="af-hero-tag">Settings</span>
                </div>
                <p class="af-hero-sub">
                    Configure your API keys, language settings, and manage plugin data. BYOK (Bring Your Own Key) for AI services.
                </p>
                <div class="af-hero-chips">
                    <div class="af-chip af-chip-pro af-chip-alt"><span class="af-dot"></span>BYOK<span class="af-chip-sub">Bring Your Own Key</span></div>
                    <div class="af-chip af-chip-pro af-chip-scan"><span class="af-dot"></span>AI Ready<span class="af-chip-sub">OpenAI / Gemini</span></div>
                    <div class="af-chip af-chip-pro af-chip-safe"><span class="af-dot"></span>Secure<span class="af-chip-sub">Keys stored locally</span></div>
                </div>
            </div>
        </div>

        <div class="af-card" style="margin-bottom: 16px;">
        <div class="af-card-header">
            <div class="af-card-title">API Keys & Language</div>
        </div>
        <form method="post" action="options.php">
            <?php settings_fields('miro_ai_seo_settings'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="miro_ai_openai_key"><?php echo esc_html__('OpenAI API Key', 'miro-ai-seo'); ?></label></th>
                    <td><input type="text" id="miro_ai_openai_key" name="miro_ai_openai_key" value="<?php echo esc_attr( get_option('miro_ai_openai_key', '') ); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="miro_ai_gemini_key"><?php echo esc_html__('Gemini API Key', 'miro-ai-seo'); ?></label></th>
                    <td><input type="text" id="miro_ai_gemini_key" name="miro_ai_gemini_key" value="<?php echo esc_attr( get_option('miro_ai_gemini_key', '') ); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="miro_ai_focus_language"><?php echo esc_html__('Default Language', 'miro-ai-seo'); ?></label></th>
                    <td><input type="text" id="miro_ai_focus_language" name="miro_ai_focus_language" value="<?php echo esc_attr( get_option('miro_ai_focus_language', 'en') ); ?>" class="regular-text" /></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
        </div>

        <div class="af-card">
        <div class="af-card-header">
            <div class="af-card-title" style="color:#b91c1c;">Danger Zone</div>
        </div>
        <p style="margin-bottom:12px;">Reset plugin data. Each form posts to <code>admin-post.php</code> with its own action and nonce.</p>

        <p style="margin-bottom:8px;"><strong>Reset GSC</strong> — removes GSC connection and cached data only:</p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block; margin-bottom:16px;">
            <input type="hidden" name="action" value="miro_gsc_reset">
            <?php wp_nonce_field('miro_gsc_reset'); ?>
            <button type="submit" class="button button-secondary miro-danger-border" onclick="return confirm('Remove all GSC options and cached data?');">Reset GSC</button>
        </form>

        <p style="margin-bottom:8px;"><strong>Reset All</strong> — removes all plugin options, transients, and drops plugin tables:</p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block; margin-bottom:16px;">
            <input type="hidden" name="action" value="miro_suite_full_reset">
            <?php wp_nonce_field('miro_suite_full_reset'); ?>
            <button type="submit" class="button button-secondary miro-danger-border" onclick="return confirm('Remove ALL plugin data? This cannot be undone.');">Reset All</button>
        </form>

        <p style="margin-bottom:8px;"><strong>Or select items to reset:</strong></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>"
            onsubmit="
                const boxes = this.querySelectorAll('input[type=checkbox]:checked');
                if (boxes.length === 0) { alert('Please select at least one item to reset.'); return false; }
                return confirm('Are you sure? This cannot be undone?');
            ">
            <input type="hidden" name="action" value="miro_suite_custom_reset">
            <?php wp_nonce_field('miro_suite_custom_reset'); ?>
            <ul class="miro-list-disc" style="margin-left:20px;">
                <li><label><input type="checkbox" name="reset[index]" value="1"> Index Monitor</label></li>
                <li><label><input type="checkbox" name="reset[rank]" value="1"> Rank Tracker</label></li>
                <li><label><input type="checkbox" name="reset[gsc]" value="1"> GSC Tokens</label></li>
                <li><label><input type="checkbox" name="reset[focus]" value="1"> Focus Keywords</label></li>
                <li><label><input type="checkbox" name="reset[ai]" value="1"> AI Keys</label></li>
                <li><label><input type="checkbox" name="reset[all]" value="1"><strong> FULL RESET</strong></label></li>
            </ul>
            <p><button type="submit" class="button button-secondary miro-danger-border">Run Selected Reset</button></p>
        </form>
        </div>

        <?php
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Redirect params for message display only, not form processing.
        if (!empty($_GET['miro_reset_error']) && $_GET['miro_reset_error'] === 'expired'): ?>
        <div class="notice notice-warning is-dismissible" style="margin-top: 16px;">
            <p><?php esc_html_e('The verification link expired. Please try "Reset All" again.', 'miro-ai-seo'); ?></p>
        </div>
        <?php endif;
        if (!empty($_GET['miro_reset_done'])): ?>
        <div class="notice notice-success is-dismissible" style="margin-top: 16px;">
            <p>✅ Reset complete. Plugin data has been removed. You can now start clean.</p>
        </div>
        <?php endif; ?>
        <p class="description" style="margin-top:8px;">Verification: <strong>Reset GSC</strong> deletes pp_miro_gsc_* and miro_gsc_* options + <code>_transient_miro_gsc_%</code>. <strong>Reset All</strong> deletes all plugin options and transients, post meta (<code>miro_focus_keyword</code>, <code>_miro_index_*</code>), and drops tables: <code>miro_404_log</code>, <code>miro_rt_keywords</code>, <code>miro_rt_snapshots</code>, <code>miro_rt_competitors</code>, <code>miro_ilx_index</code>, <code>miro_ilx_edges</code>, <code>ppmiro_index_log</code>.</p>
        </div> <!-- /miro-alt-wrap -->
        <?php
    }

    public static function render_trends() {
        if (class_exists('\\Miro_AI_SEO\\Trends') && method_exists('\\Miro_AI_SEO\\Trends', 'render_admin_page')) {
            \Miro_AI_SEO\Trends::render_admin_page();
        } else {
            echo '<div class="wrap"><h1>Trend Finder</h1>';
            echo '<p class="miro-trends-error-color">Trends module is not loaded.</p>';
            echo '</div>';
        }
    }

    // Keep this page if your menu uses it
    public static function render_links() {
        ?>
        <div class="wrap">
          <h1><?php esc_html_e('Internal Links', 'miro-ai-seo'); ?></h1>
          <p><?php esc_html_e('If you have a link index builder, run it here.', 'miro-ai-seo'); ?></p>

          <p class="description">
            This page is optional. The Internal Links PRO tool runs inside the post editor metabox now.
          </p>
        </div>
        <?php
    }

    // ----- Reset handlers -----

    /**
     * Reset GSC only: pp_miro_gsc_* and miro_gsc_* options + GSC transients.
     * Verification: options deleted = pp_miro_gsc_settings, pp_miro_gsc_status, pp_miro_gsc_property, pp_miro_gsc_properties,
     * miro_gsc_* (settings, status, property, properties, last_queries, sync_status, sites_cache, meta, cached_sites,
     * site_series, country_series, queries, last_queries_cache, cache). Transients: _transient_miro_gsc_% and _transient_timeout_miro_gsc_%.
     */
    public static function handle_gsc_reset() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        check_admin_referer('miro_gsc_reset');

        $gsc_opts = [
            'pp_miro_gsc_settings', 'pp_miro_gsc_status', 'pp_miro_gsc_property', 'pp_miro_gsc_properties',
            'miro_gsc_settings', 'miro_gsc_status', 'miro_gsc_property', 'miro_gsc_properties',
            'miro_gsc_last_queries', 'miro_gsc_sync_status', 'miro_gsc_sites_cache', 'miro_gsc_meta',
            'miro_gsc_cached_sites', 'miro_gsc_site_series', 'miro_gsc_country_series', 'miro_gsc_queries',
            'miro_gsc_last_queries_cache', 'miro_gsc_cache', 'ppmiro_gsc_last_queries',
        ];
        foreach ($gsc_opts as $opt) {
            delete_option($opt);
        }

        global $wpdb;
        $gsc_transient_pattern = $wpdb->esc_like('_transient_miro_gsc_') . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Reset: one-time transient cleanup; no caching for delete.
        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $gsc_transient_pattern));
        $gsc_transient_timeout_pattern = $wpdb->esc_like('_transient_timeout_miro_gsc_') . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Reset: one-time transient cleanup; no caching for delete.
        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $gsc_transient_timeout_pattern));

        wp_safe_redirect(add_query_arg('miro_reset_done', '1', admin_url('admin.php?page=miro-ai-settings')));
        exit;
    }

    /**
     * Full reset: all plugin options, transients, post meta, and plugin-owned tables.
     * Verification — options: miro_ai_*, miro_rt_*, pp_miro_gsc_*, miro_gsc_*, miro_index_*, miro_broker_secret, miro_site_id, etc.
     * Tables dropped: wp_miro_404_log, wp_miro_rt_keywords, wp_miro_rt_snapshots, wp_miro_rt_competitors, wp_miro_ilx_index, wp_miro_ilx_edges, wp_ppmiro_index_log.
     */
    public static function handle_full_reset() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        $nonce = isset($_REQUEST['_wpnonce']) ? sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])) : '';
        if (!wp_verify_nonce($nonce, 'miro_suite_full_reset')) {
            wp_safe_redirect(add_query_arg('miro_reset_error', 'expired', admin_url('admin.php?page=miro-ai-settings')));
            exit;
        }

        $opts = [
            'miro_ai_openai_key', 'miro_ai_gemini_key', 'miro_ai_focus_language', 'miro_ai_seo_sync_log',
            'miro_rt_settings', 'miro_rt_status', 'miro_rt_keywords', 'miro_rt_snapshots', 'miro_rt_competitors',
            'pp_miro_gsc_settings', 'pp_miro_gsc_status', 'pp_miro_gsc_property', 'pp_miro_gsc_properties',
            'miro_gsc_settings', 'miro_gsc_status', 'miro_gsc_property', 'miro_gsc_properties',
            'miro_gsc_last_queries', 'miro_gsc_sync_status', 'miro_gsc_sites_cache', 'miro_gsc_meta',
            'miro_gsc_cached_sites', 'miro_gsc_site_series', 'miro_gsc_country_series', 'miro_gsc_queries',
            'miro_gsc_last_queries_cache', 'miro_gsc_cache', 'ppmiro_gsc_last_queries',
            'miro_index_settings', 'miro_index_backfill_state', 'miro_broker_secret', 'miro_site_id',
            'miro_redirect_settings', 'miro_redirect_rules', 'miro_404_db_version', 'miro_404_fallback_log',
            'miro_instant_index_settings', 'miro_il_index_version', 'miro_il_graph_version', 'miro_ilx_dbv', 'miro_ilx_df',
        ];
        foreach ($opts as $opt) {
            delete_option($opt);
        }

        global $wpdb;
        $gsc_transient_pattern = $wpdb->esc_like('_transient_miro_gsc_') . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Full reset: one-time transient cleanup; no caching for delete.
        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $gsc_transient_pattern));
        $gsc_transient_timeout_pattern = $wpdb->esc_like('_transient_timeout_miro_gsc_') . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Full reset: one-time transient cleanup; no caching for delete.
        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $gsc_transient_timeout_pattern));
        $pattern = $wpdb->esc_like('_transient_miro') . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Full reset: one-time transient cleanup; no caching for delete.
        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $pattern));
        $pattern_timeout = $wpdb->esc_like('_transient_timeout_miro') . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Full reset: one-time transient cleanup; no caching for delete.
        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $pattern_timeout));

        $meta_keys = [
            'miro_focus_keyword',
            '_miro_index_status', '_miro_index_details', '_miro_index_last_check',
            '_miro_index_first_seen_notidx', '_miro_index_retry_count',
        ];
        foreach ($meta_keys as $mk) {
            delete_post_meta_by_key($mk);
        }

        wp_clear_scheduled_hook('miro_index_monitor_daily');
        wp_clear_scheduled_hook('miro_index_backfill_tick');
        wp_clear_scheduled_hook('miro_rt_hourly');

        $prefix = $wpdb->prefix;
        $table_suffixes = ['miro_404_log', 'miro_rt_keywords', 'miro_rt_snapshots', 'miro_rt_competitors', 'miro_ilx_index', 'miro_ilx_edges', 'ppmiro_index_log'];
        foreach ($table_suffixes as $suffix) {
            if (!preg_match('/^[a-zA-Z0-9_]+$/', $suffix)) {
                continue;
            }
            $tbl = $prefix . $suffix;
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Full reset: table from whitelist; direct DROP required.
            $wpdb->query("DROP TABLE IF EXISTS `{$tbl}`");
        }

        wp_safe_redirect(add_query_arg('miro_reset_done', '1', admin_url('admin.php?page=miro-ai-settings')));
        exit;
    }

    public static function handle_custom_reset() {
        if (!current_user_can('manage_options')) wp_die(esc_html__('You do not have permission to perform this action.', 'miro-ai-seo'), '', ['response' => 403]);
        check_admin_referer('miro_suite_custom_reset');

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Keys sanitized via sanitize_key below; values are checkbox "1" only.
        $reset_raw = isset($_POST['reset']) && is_array($_POST['reset']) ? wp_unslash($_POST['reset']) : [];
        if (empty($reset_raw)) {
            wp_safe_redirect(admin_url('admin.php?page=miro-ai-seo&reset=none'));
            exit;
        }

        $choices = array_map('sanitize_key', array_keys($reset_raw));

        if (in_array('all', $choices, true)) {
            self::handle_full_reset();
            return;
        }

        if (in_array('index', $choices, true)) {
            delete_option('miro_index_monitor_data');
            delete_option('miro_index_scan_state');
        }
        if (in_array('rank', $choices, true)) {
            delete_option('miro_rt_keywords');
            delete_option('miro_rt_snapshots');
        }
        if (in_array('gsc', $choices, true)) {
            // Delete all GSC-related options
            $gsc_opts = [
                'pp_miro_gsc_settings','pp_miro_gsc_property','pp_miro_gsc_status','pp_miro_gsc_properties',
                'miro_gsc_last_queries','miro_gsc_property','miro_gsc_sync_status','miro_gsc_sites_cache',
                'miro_gsc_meta','miro_gsc_cached_sites','miro_gsc_site_series','miro_gsc_country_series',
                'miro_gsc_queries','miro_gsc_last_queries_cache','ppmiro_gsc_last_queries','miro_gsc_cache',
                'miro_gsc_settings','miro_gsc_status','miro_gsc_properties',
            ];
            foreach ($gsc_opts as $opt) delete_option($opt);
            
            // Delete all GSC-related transients
            global $wpdb;
            $gsc_transient_pattern = $wpdb->esc_like('_transient_miro_gsc_') . '%';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom reset: one-time transient cleanup; no caching for delete.
            $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $gsc_transient_pattern));
            $gsc_transient_timeout_pattern = $wpdb->esc_like('_transient_timeout_miro_gsc_') . '%';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom reset: one-time transient cleanup; no caching for delete.
            $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $gsc_transient_timeout_pattern));
        }
        if (in_array('focus', $choices, true)) {
            delete_post_meta_by_key('miro_focus_keyword');
            delete_option('miro_titles_focuskw');
        }
        if (in_array('ai', $choices, true)) {
            delete_option('miro_ai_openai_key');
            delete_option('miro_ai_gemini_key');
        }

        wp_safe_redirect(add_query_arg('miro_reset_done', '1', admin_url('admin.php?page=miro-ai-settings')));
        exit;
    }

    // ----- AI bridge (same as your original) -----

    public static function ai_complete($unused, $args = []) {
        $prompt  = isset($args['prompt'])  ? (string)$args['prompt']  : '';
        $context = isset($args['context']) ? (string)$args['context'] : 'default';
        $opts    = isset($args['opts'])    ? (array)$args['opts']     : [];

        if ($prompt === '') return '';

        $openai = trim((string) get_option('miro_ai_openai_key', ''));
        if ($openai === '') return 'AI error: OpenAI key missing. Set it in Miro AI → Settings.';

        return self::ai_openai($prompt, $context, $opts, $openai);
    }

    private static function ai_openai($prompt, $context, $opts, $key) {
        $model = isset($opts['model']) ? (string)$opts['model'] : 'gpt-4o-mini';
        $temperature = isset($opts['temperature']) ? floatval($opts['temperature']) : 0.2;
        $max_tokens  = isset($opts['max_tokens']) ? intval($opts['max_tokens']) : 600;
        $lang        = get_option('miro_ai_focus_language', 'en');

        // Increase timeout for long-form content generation
        $long_contexts = ['trend_article', 'ai_post_generator', 'ai_post_generator_expand', 'ai_post_generator_balance'];
        $timeout = (in_array($context, $long_contexts, true) && $max_tokens > 2000) ? 60 : 30;

        $endpoint = 'https://api.openai.com/v1/chat/completions';
        $payload  = [
            'model'    => $model,
            'messages' => [
                ['role'=>'system','content'=>"You are Miro AI ({$context}). Respond in {$lang}. Be concise and actionable."],
                ['role'=>'user','content'=>$prompt],
            ],
            'temperature' => $temperature,
            'max_tokens'  => $max_tokens,
        ];

        $res = wp_remote_post($endpoint, [
            'timeout' => $timeout,
            'headers' => [
                'Authorization' => 'Bearer ' . $key,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode($payload),
        ]);

        if (is_wp_error($res)) return 'AI error: ' . $res->get_error_message();

        $code = wp_remote_retrieve_response_code($res);
        $body = json_decode(wp_remote_retrieve_body($res), true);

        if ($code >= 200 && $code < 300 && !empty($body['choices'][0]['message']['content'])) {
            $raw = trim($body['choices'][0]['message']['content']);

            // JSON extraction fix
            if (strpos($raw, '{') !== false) {
                $start = strpos($raw, '{');
                $end   = strrpos($raw, '}');
                if ($start !== false && $end !== false && $end > $start) {
                    $json = substr($raw, $start, $end - $start + 1);
                    if (json_decode($json, true) !== null) return $json;
                }
            }

            return $raw;
        }

        return 'AI error: ' . ($body['error']['message'] ?? 'Unknown');
    }
}
