<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) { exit; }

class Feeds {
    // Map categories to trusted RSS feeds
    private static $SOURCES = [
        'tech' => [
            'https://news.google.com/rss/search?q=technology',
            'https://techcrunch.com/feed/',
            'https://www.theverge.com/rss/index.xml',
            'https://www.engadget.com/rss.xml',
            'https://arstechnica.com/feed/',
        ],
        'business' => [
            'https://news.google.com/rss/search?q=business',
            'https://www.marketwatch.com/rss/topstories',
            'https://www.cnbc.com/id/100003114/device/rss/rss.html',
        ],
        'health' => [
            'https://news.google.com/rss/search?q=health',
            'https://www.healthline.com/rss',
            'https://www.medicalnewstoday.com/rss',
        ],
        'travel' => [
            'https://news.google.com/rss/search?q=travel',
            'https://www.cntraveler.com/feed/rss',
            'https://www.lonelyplanet.com/news/feed',
        ],
        'gaming' => [
            'https://news.google.com/rss/search?q=gaming',
            'https://www.ign.com/rss.xml',
            'https://www.gamespot.com/feeds/mashup/',
            'https://www.pcgamer.com/rss/',
        ],
        'science' => [
            'https://news.google.com/rss/search?q=science',
            'https://www.sciencedaily.com/rss/top/science.xml',
        ],
    ];

    // Basic English stopwords for lightweight keyword extraction
    private static $STOP = [
        'the','a','an','and','or','but','if','then','else','for','to','of','in','on','with','by','at','from',
        'is','are','was','were','be','been','being','as','it','its','this','that','these','those',
        'you','your','we','our','they','their','he','she','his','her','i','me','my',
        'about','into','over','under','after','before','during','without','within',
        'new','latest','breaking','update','report','how','why','what'
    ];

    public static function categories() {
        return array_keys(self::$SOURCES);
    }

    public static function get_trends(string $category, int $limit = 10): array {
        $category = strtolower(trim($category));
        if (!isset(self::$SOURCES[$category])) {
            $category = 'tech';
        }
        $limit = max(1, min(20, $limit));

        $cache_key = 'miro_trends_' . $category;
        $cached = get_transient($cache_key);
        if (is_array($cached) && !empty($cached)) {
            return array_slice($cached, 0, $limit);
        }

        // WordPress SimplePie wrapper
        require_once ABSPATH . WPINC . '/feed.php';

        $items = [];
        $seen  = [];
        foreach (self::$SOURCES[$category] as $url) {
            $feed = fetch_feed($url);
            if (is_wp_error($feed)) { continue; }
            $maxitems   = $feed->get_item_quantity(8);
            $feed_items = $feed->get_items(0, $maxitems);
            foreach ($feed_items as $fi) {
                $title = (string) wp_strip_all_tags($fi->get_title());
                $link  = (string) esc_url_raw($fi->get_permalink());
                if (!$title || !$link) continue;
                if (isset($seen[$link])) continue;
                $seen[$link] = true;

                $date = $fi->get_date('U');
                $desc = $fi->get_description();
                if (!$desc && method_exists($fi,'get_content')) {
                    $desc = $fi->get_content();
                }
                $summary = wp_strip_all_tags((string)$desc);
                $summary = mb_substr($summary, 0, 400);

                $items[] = [
                    'title'   => $title,
                    'url'     => $link,
                    'summary' => $summary,
                    'source'  => wp_parse_url($url, PHP_URL_HOST),
                    'date'    => $date ? intval($date) : time(),
                ];
            }
        }

        // Sort newest first
        usort($items, function($a,$b){ return ($b['date'] <=> $a['date']); });

        // Lightweight keyword extraction
        foreach ($items as &$it) {
            $txt = trim(($it['title'] ?? '') . ' ' . ($it['summary'] ?? ''));
            $it['keywords'] = self::extract_keywords($txt);
        }
        unset($it);

        // Cache for 15 minutes
        set_transient($cache_key, $items, 15 * MINUTE_IN_SECONDS);

        return array_slice($items, 0, $limit);
    }

    private static function extract_keywords(string $text, int $max = 8): array {
        $text  = wp_strip_all_tags($text);
        $textL = function_exists('mb_strtolower') ? mb_strtolower($text) : strtolower($text);
        $textL = preg_replace('/[^a-z0-9\s\-]/', ' ', $textL);

        $parts = preg_split('/\s+/', $textL);
        $parts = array_filter($parts, function($w){
            return (function_exists('mb_strlen') ? mb_strlen($w) : strlen($w)) >= 3;
        });

        $stop = array_flip(self::$STOP);
        $freq = [];
        foreach ($parts as $w) {
            if (isset($stop[$w])) continue;
            if (preg_match('/^\d+$/', $w)) continue;
            $freq[$w] = ($freq[$w] ?? 0) + 1;
        }

        arsort($freq, SORT_NUMERIC);
        $out = array_slice(array_keys($freq), 0, $max);
        return array_values(array_unique($out));
    }
}
