<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

class Miro_Redirections {
    const OPT_SETTINGS   = 'miro_redirect_settings';
    const OPT_RULES      = 'miro_redirect_rules';
    const MENU_PARENT    = 'miro-ai-seo';
    const MENU_SLUG      = 'miro-redirections';

    /** @var array<int, array{hits:int,last_hit:int}> */
    private static $hit_buffer = [];

    public static function defaults(): array {
        return [
            'enabled'                => false,
            'fallback'               => '404',   // 404 | home | custom | 410 | 451
            'fallback_custom'        => '',
            'fallback_451_reason'    => '',
            'default_status'         => 301,     // 301|302|307|308|410|451
            'auto_post_redirect'     => true,
            'auto_term_redirect'     => true,
            'respect_trailing_slash' => true,

            // ✅ Safety: external redirects disabled by default
            'allow_external'         => false,
            'external_hosts'         => '', // comma-separated hosts (example.com, cdn.example.com)
        ];
    }

    public static function init(): void {
        add_action('admin_menu', [__CLASS__, 'menu'], 40);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);

        add_action('admin_post_miro_redir_save',   [__CLASS__, 'handle_save']);
        add_action('admin_post_miro_redir_import', [__CLASS__, 'handle_import']);
        add_action('admin_post_miro_redir_export', [__CLASS__, 'handle_export']);

        add_action('template_redirect', [__CLASS__, 'maybe_redirect'], 0);

        add_action('post_updated', [__CLASS__, 'on_post_updated'], 10, 3);
        add_action('edited_terms', [__CLASS__, 'on_edited_terms'], 10, 2);
        add_action('edited_term',  [__CLASS__, 'on_edited_term'],  10, 3);

        // ✅ Performance: flush hit counts once per request
        add_action('shutdown', [__CLASS__, 'flush_hit_buffer'], 1);
    }

    /* ---------------------------- Capability helpers ---------------------------- */

    private static function cap_view(): string {
        return function_exists('\\miro_ai_cap') ? (string)\miro_ai_cap() : 'manage_options';
    }

    private static function cap_manage(): string {
        return function_exists('\\miro_ai_manage_cap') ? (string)\miro_ai_manage_cap() : 'manage_options';
    }

    private static function is_admin_user(): bool {
        return current_user_can('manage_options');
    }

    /* ---------------------------- Assets ---------------------------- */

    public static function enqueue_assets($hook): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Page detection for asset loading, not form processing.
        $page = isset($_GET['page']) ? sanitize_key((string)wp_unslash($_GET['page'])) : '';
        if ($page !== self::MENU_SLUG) return;

        wp_enqueue_style(
            'miro-redirections-css',
            MIRO_AI_SEO_URL . 'assets/css/miro-redirections.css',
            [],
            (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : null)
        );
    }

    private static function root_plugin_file(): string {
        $candidates = [6,5,4,3,2,1];
        foreach ($candidates as $d) {
            $dir = dirname(__FILE__, $d);
            if (file_exists($dir . '/miro-ai-seo.php')) return $dir . '/miro-ai-seo.php';
            if (file_exists($dir . '/miro-ai-seo-load.php')) return $dir . '/miro-ai-seo-load.php';
        }
        return __FILE__;
    }

    private static function asset_url(string $relative): string {
        $relative = ltrim($relative, '/');
        if (defined('MIRO_AI_SEO_URL')) {
            return trailingslashit(MIRO_AI_SEO_URL) . $relative;
        }
        return plugins_url('/' . $relative, self::root_plugin_file());
    }

    /* ---------------------------- Admin UI ---------------------------- */

    public static function menu(): void {
        $parent = apply_filters('miro_main_menu_slug', self::MENU_PARENT);
        add_submenu_page(
            $parent,
            'Redirections',
            'Redirections',
            self::cap_view(), // ✅ view cap
            self::MENU_SLUG,
            [__CLASS__, 'render_page'],
            31
        );
    }

    public static function render(): void {
        self::render_page();
    }

    public static function render_page(): void {
        if (!current_user_can(self::cap_view())) {
            wp_die(esc_html__('No permission.', 'miro-ai-seo'), '', ['response' => 403]);
        }

        $s     = get_option(self::OPT_SETTINGS, []);
        $s     = is_array($s) ? array_merge(self::defaults(), $s) : self::defaults();

        $rules = get_option(self::OPT_RULES, []);
        if (!is_array($rules)) $rules = [];

        // Normalize older rules (no hits fields yet)
        foreach ($rules as $i => $r) {
            if (!is_array($r)) continue;
            if (!isset($rules[$i]['hits'])) $rules[$i]['hits'] = 0;
            if (!isset($rules[$i]['last_hit'])) $rules[$i]['last_hit'] = 0;
        }

        $total_rules   = count($rules);
        $enabled_rules = 0;
        $hard_410      = 0;
        $hard_451      = 0;
        $total_hits    = 0;

        foreach ($rules as $r) {
            if (!is_array($r)) continue;
            $total_hits += (int)($r['hits'] ?? 0);
            if (!empty($r['enabled'])) {
                $enabled_rules++;
                $st = (int)($r['status'] ?? 301);
                if ($st === 410) $hard_410++;
                if ($st === 451) $hard_451++;
            }
        }

        // Edit mode
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab/edit detection for display, not form processing.
        $edit_idx  = isset($_GET['edit']) ? absint(wp_unslash($_GET['edit'])) : -1;
        $edit_rule = ($edit_idx >= 0 && isset($rules[$edit_idx]) && is_array($rules[$edit_idx])) ? $rules[$edit_idx] : null;

        $hero_icon = esc_url(self::asset_url('assets/img/miro-logo.webp'));
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">Miro Redirections</h1>
        </div>

        <div class="miro-redir-wrap">

            <?php
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Notice display only, not form processing.
            if (!empty($_GET['saved']) || !empty($_GET['imported']) || !empty($_GET['updated'])): ?>
                <div class="mr-notice">✅ Changes saved.</div>
            <?php endif; ?>

            <div class="mr-hero">
                <div class="mr-hero-pill">
                    <?php if (!empty($hero_icon)) : ?>
                        <img src="<?php echo esc_url($hero_icon); ?>" alt="Miro" onerror="this.style.display='none';" />
                    <?php else: ?>
                        <div class="mr-hero-pill-fallback">REDIR</div>
                    <?php endif; ?>
                </div>

                <div class="mr-hero-main">
                    <div class="mr-hero-title-row">
                        <div class="mr-hero-title">Protect your SEO equity with smart redirects.</div>
                        <span class="mr-hero-tag">Core SEO safety</span>
                    </div>

                    <p class="mr-hero-sub">
                        Keep old URLs alive after slug changes, migrations, and legal removals using
                        <strong>SEO-safe 3xx</strong> and hard <strong>410/451</strong> responses.
                    </p>

                    <div class="mr-hero-chips">
                        <div class="mr-chip mr-chip-total"><strong><?php echo intval($total_rules); ?></strong><span>total rules</span></div>
                        <div class="mr-chip mr-chip-active"><strong><?php echo intval($enabled_rules); ?></strong><span>active</span></div>
                        <div class="mr-chip mr-chip-410"><strong><?php echo intval($hard_410); ?></strong><span>410 Gone</span></div>
                        <div class="mr-chip mr-chip-451"><strong><?php echo intval($hard_451); ?></strong><span>451 Legal</span></div>
                        <div class="mr-chip mr-chip-hits"><strong><?php echo intval($total_hits); ?></strong><span>total hits</span></div>
                    </div>
                </div>

                <div class="mr-hero-metrics">
                    <div class="mr-tip-title">
                        <span class="mr-tip-ico">💡</span>
                        <span><strong>Tip:</strong> Don’t spam redirects</span>
                    </div>
                    <div class="mr-tip-sub">
                        Best practice: keep <span class="mr-tip-hl">404 default</span> and use 301 only for true replacements.
                        Avoid redirecting all 404 to homepage.
                    </div>
                </div>
            </div>

            <!-- SETTINGS CARD -->
            <div class="mr-card">
                <h2>Settings</h2>
                <p class="mr-desc">Choose how the redirect engine behaves globally. Green = safe, red = be extra careful.</p>

                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('miro_redir_save'); ?>
                    <input type="hidden" name="action" value="miro_redir_save" />
                    <input type="hidden" name="op" value="save_settings" />

                    <div class="mr-grid mr-grid-one">
                        <div>
                            <h3 class="mr-h3-title">Core engine</h3>

                            <p class="mr-row">
                                <span class="mr-label-row">
                                  <label><input type="checkbox" name="enabled" <?php echo checked(!empty($s['enabled']), true, false); ?>> Enable redirection engine</label>
                                  <span class="mr-tag-mini mr-tag-safe">Recommended</span>
                                </span>
                            </p>

                            <p class="mr-row">
                                <span class="mr-label-row">
                                  <label><input type="checkbox" name="auto_post_redirect" <?php echo checked(!empty($s['auto_post_redirect']), true, false); ?>> Auto post redirects when slugs change</label>
                                  <span class="mr-tag-mini mr-tag-safe">Recommended</span>
                                </span>
                            </p>

                            <p class="mr-row">
                                <span class="mr-label-row">
                                  <label><input type="checkbox" name="auto_term_redirect" <?php echo checked(!empty($s['auto_term_redirect']), true, false); ?>> Auto category/tag redirects when slugs change</label>
                                  <span class="mr-tag-mini mr-tag-safe">Recommended</span>
                                </span>
                            </p>

                            <p class="mr-row">
                                <label><input type="checkbox" name="respect_trailing_slash" <?php echo checked(!empty($s['respect_trailing_slash']), true, false); ?>> Respect site trailing slash setting</label>
                            </p>

                            <hr />

                            <h3 class="mr-h3-title">External redirects (advanced)</h3>
                            <p class="mr-row">
                                <span class="mr-label-row">
                                  <label>
                                    <input type="checkbox" name="allow_external" <?php echo checked(!empty($s['allow_external']), true, false); ?>>
                                    Allow redirects to external domains
                                  </label>
                                  <span class="mr-tag-mini mr-tag-warn">Be careful</span>
                                </span>
                            </p>
                            <p class="mr-row">
                                <label>Allowed external hosts (comma-separated)<br>
                                    <input type="text" name="external_hosts" value="<?php echo esc_attr((string)($s['external_hosts'] ?? '')); ?>" placeholder="example.com, cdn.example.com">
                                </label>
                            </p>
                            <p class="mr-tiny">If enabled, only these hosts will be allowed. Otherwise external targets are blocked.</p>
                        </div>

                        <div>
                            <h3 class="mr-h3-title">404 fallback behavior</h3>

                            <?php $fb = $s['fallback']; ?>
                            <div class="mr-row mr-row-col">
                                <?php foreach ([
                                    '404'   => 'Default 404 page (safe)',
                                    'home'  => 'Redirect to homepage',
                                    'custom'=> 'Redirect to custom URL',
                                    '410'   => '410 Gone (hard removal)',
                                    '451'   => '451 Legal (blocked)'
                                ] as $val => $label): ?>
                                    <label class="mr-label-inline">
                                        <input type="radio" name="fallback" value="<?php echo esc_attr($val); ?>" <?php echo checked($fb, $val, false); ?>>
                                        <?php echo esc_html($label); ?>
                                    </label>
                                <?php endforeach; ?>
                            </div>

                            <p class="mr-row">
                                <label>Custom URL (when fallback = custom)<br>
                                    <input type="text" name="fallback_custom" value="<?php echo esc_attr($s['fallback_custom']); ?>" placeholder="https://example.com/not-found">
                                </label>
                            </p>

                            <p class="mr-row">
                                <label>451 legal reason (optional)<br>
                                    <input type="text" name="fallback_451_reason" value="<?php echo esc_attr($s['fallback_451_reason']); ?>" placeholder="Blocked due to legal demand">
                                </label>
                            </p>

                            <h3 class="mr-h3-title-mt12">Default status</h3>
                            <p class="mr-row">
                                <label>Default redirect status<br>
                                    <select name="default_status">
                                        <?php foreach ([301,302,307,308,410,451] as $st):
                                            $note = '';
                                            if ($st === 301) $note = ' (SEO default)';
                                            if ($st === 302) $note = ' (temporary)';
                                            if ($st === 410 || $st === 451) $note = ' (hard removal)';
                                        ?>
                                            <option value="<?php echo intval($st); ?>" <?php echo selected($s['default_status'], $st, false); ?>>
                                                <?php echo intval($st) . esc_html($note); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </label>
                            </p>
                        </div>
                    </div>

                    <p class="mr-p-mt14">
                        <button class="button button-primary mr-button-primary" type="submit">
                            Save settings
                        </button>
                    </p>
                </form>
            </div>

            <?php if ($edit_rule): ?>
            <!-- EDIT RULE CARD -->
            <div class="mr-card" id="miro-redir-edit">
                <div class="mr-rules-head">
                    <div>
                        <h2 class="mr-h2-title">Edit Rule #<?php echo intval($edit_idx+1); ?></h2>
                        <small>Update this redirect and save.</small>
                    </div>
                    <div>
                        <a class="mr-link" href="<?php echo esc_url(admin_url('admin.php?page='.self::MENU_SLUG)); ?>">✕ Cancel</a>
                    </div>
                </div>

                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="mr-add">
                    <?php wp_nonce_field('miro_redir_save'); ?>
                    <input type="hidden" name="action" value="miro_redir_save" />
                    <input type="hidden" name="op" value="update_rule" />
                    <input type="hidden" name="idx" value="<?php echo intval($edit_idx); ?>" />

                    <label class="mr-w120">Type<br>
                        <select name="rule_type">
                            <?php foreach (['exact'=>'Exact','prefix'=>'Prefix','regex'=>'Regex'] as $v=>$lbl): ?>
                                <option value="<?php echo esc_attr($v); ?>" <?php echo selected(($edit_rule['type'] ?? 'exact'), $v, false); ?>>
                                    <?php echo esc_html($lbl); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>

                    <label class="mr-w240">Source<br>
                        <input type="text" name="rule_source" value="<?php echo esc_attr((string)($edit_rule['source'] ?? '')); ?>" placeholder="/old-url or ^/old/(.*)$">
                    </label>

                    <label class="mr-w240">Target<br>
                        <input type="text" name="rule_target" value="<?php echo esc_attr((string)($edit_rule['target'] ?? '')); ?>" placeholder="/new-url or https://example.com/$1 (ignored for 410/451)">
                    </label>

                    <label class="mr-w120">Status<br>
                        <select name="rule_status">
                            <?php foreach ([301,302,307,308,410,451] as $st): ?>
                                <option value="<?php echo intval($st); ?>" <?php echo selected((int)($edit_rule['status'] ?? 301), $st, false); ?>>
                                    <?php echo intval($st); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>

                    <label class="mr-w200">451 reason<br>
                        <input type="text" name="rule_reason" value="<?php echo esc_attr((string)($edit_rule['reason'] ?? '')); ?>" placeholder="(optional)">
                    </label>

                    <label class="mr-w180">Category<br>
                        <input type="text" name="rule_category" value="<?php echo esc_attr((string)($edit_rule['category'] ?? '')); ?>" placeholder="e.g. Migrations">
                    </label>

                    <label class="mr-w120">Enabled<br>
                        <select name="rule_enabled">
                            <option value="1" <?php echo selected(!empty($edit_rule['enabled']), true, false); ?>>Yes</option>
                            <option value="0" <?php echo selected(empty($edit_rule['enabled']), true, false); ?>>No</option>
                        </select>
                    </label>

                    <button class="button button-primary mr-button-primary-sm" type="submit">
                        Save rule
                    </button>
                </form>
            </div>
            <?php endif; ?>

            <!-- RULES CARD -->
            <div class="mr-card">
                <div class="mr-rules-head">
                    <div>
                        <h2 class="mr-h2-title">Rules</h2>
                        <small>Manual redirects for migrations, campaign URLs, and cleanup.</small>
                    </div>
                    <div><span class="mr-badge">Manual rules</span></div>
                </div>

                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="mr-add">
                    <?php wp_nonce_field('miro_redir_save'); ?>
                    <input type="hidden" name="action" value="miro_redir_save" />
                    <input type="hidden" name="op" value="add_rule" />

                    <label class="mr-w120">Type<br>
                        <select name="rule_type">
                            <option value="exact">Exact</option>
                            <option value="prefix">Prefix</option>
                            <option value="regex">Regex</option>
                        </select>
                    </label>

                    <label class="mr-w240">Source<br>
                        <input type="text" name="rule_source" placeholder="/old-url or ^/old/(.*)$">
                    </label>

                    <label class="mr-w240">Target<br>
                        <input type="text" name="rule_target" placeholder="/new-url or https://example.com/$1 (ignored for 410/451)">
                    </label>

                    <label class="mr-w120">Status<br>
                        <select name="rule_status">
                            <?php foreach ([301,302,307,308,410,451] as $st): ?>
                                <option value="<?php echo intval($st); ?>"><?php echo intval($st); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </label>

                    <label class="mr-w200">451 reason<br>
                        <input type="text" name="rule_reason" placeholder="(optional)">
                    </label>

                    <label class="mr-w180">Category<br>
                        <input type="text" name="rule_category" placeholder="e.g. Migrations">
                    </label>

                    <label class="mr-w120">Enabled<br>
                        <select name="rule_enabled">
                            <option value="1">Yes</option>
                            <option value="0">No</option>
                        </select>
                    </label>

                    <button class="button mr-button-primary-sm" type="submit">
                        Add rule
                    </button>
                </form>

                <?php if (empty($rules)) : ?>
                    <p class="mr-empty">No rules yet. Add your first redirect above.</p>
                <?php else: ?>
                    <table class="mr-table mr-table-mt">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Type</th>
                                <th>Source</th>
                                <th>→ Target</th>
                                <th>Status</th>
                                <th>Hits</th>
                                <th>Reason</th>
                                <th>Category</th>
                                <th>Enabled</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($rules as $i => $r):
                            if (!is_array($r)) continue;

                            $status = intval($r['status'] ?? $s['default_status']);
                            $stClass = 'mr-3xx';
                            if ($status === 410) $stClass = 'mr-410';
                            if ($status === 451) $stClass = 'mr-451';

                            $cat = (string)($r['category'] ?? '');
                            $is_auto_post = ($cat === 'Auto Post Redirect');
                            $is_auto_term = ($cat === 'Auto Term Redirect');

                            $enabled = !empty($r['enabled']);
                            $toggle_class = $enabled ? 'mr-on' : 'mr-off';
                            $toggle_label = $enabled ? 'Pause rule' : 'Enable rule';

                            $edit_url = admin_url('admin.php?page='.self::MENU_SLUG.'&edit='.intval($i).'#miro-redir-edit');

                            $hits = (int)($r['hits'] ?? 0);
                            $last_hit = (int)($r['last_hit'] ?? 0);
                            $last_hit_str = $last_hit ? date_i18n('Y-m-d H:i', $last_hit) : '';
                        ?>
                            <tr>
                                <td><?php echo intval($i+1); ?></td>
                                <td><?php echo esc_html($r['type'] ?? ''); ?></td>
                                <td><code><?php echo esc_html($r['source'] ?? ''); ?></code></td>
                                <td><code><?php echo esc_html($r['target'] ?? ''); ?></code></td>
                                <td><span class="mr-status <?php echo esc_attr($stClass); ?>"><?php echo intval($status); ?></span></td>

                                <td>
                                    <span class="mr-hits"><?php echo intval($hits); ?></span>
                                    <?php if ($last_hit_str): ?>
                                        <span class="mr-last-hit"><?php echo esc_html($last_hit_str); ?></span>
                                    <?php else: ?>
                                        <span class="mr-last-hit">—</span>
                                    <?php endif; ?>
                                </td>

                                <td><?php echo esc_html($r['reason'] ?? ''); ?></td>

                                <td>
                                    <?php if ($is_auto_post): ?>
                                        <span class="mr-auto-inline"><span class="mr-auto-dot"></span>Auto Post</span>
                                    <?php elseif ($is_auto_term): ?>
                                        <span class="mr-auto-inline"><span class="mr-auto-dot"></span>Auto Term</span>
                                    <?php elseif ($cat !== ''): ?>
                                        <span class="mr-badge"><?php echo esc_html($cat); ?></span>
                                    <?php endif; ?>
                                </td>

                                <td><?php echo $enabled ? 'Yes' : 'No'; ?></td>
                                <td>
                                    <div class="mr-actions">
                                        <a class="mr-link" href="<?php echo esc_url($edit_url); ?>">✏️ Edit</a>

                                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                                            <?php wp_nonce_field('miro_redir_save'); ?>
                                            <input type="hidden" name="action" value="miro_redir_save" />
                                            <input type="hidden" name="op" value="toggle" />
                                            <input type="hidden" name="idx" value="<?php echo intval($i); ?>" />
                                            <button class="button button-small mr-btn <?php echo esc_attr($toggle_class); ?>" type="submit">
                                                <span class="mr-dot"></span>
                                                <span><?php echo esc_html($toggle_label); ?></span>
                                            </button>
                                        </form>

                                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                                            <?php wp_nonce_field('miro_redir_save'); ?>
                                            <input type="hidden" name="action" value="miro_redir_save" />
                                            <input type="hidden" name="op" value="delete" />
                                            <input type="hidden" name="idx" value="<?php echo intval($i); ?>" />
                                            <button class="button button-small mr-btn mr-del" type="submit" onclick="return confirm('Delete this rule?')">
                                                <span class="mr-dot"></span>
                                                <span>Delete</span>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

            <!-- IMPORT/EXPORT CARD -->
            <div class="mr-card">
                <h2>Import / Export</h2>
                <p class="mr-desc">Backup your rules or move them between sites using JSON.</p>

                <div class="mr-io">
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                        <?php wp_nonce_field('miro_redir_export'); ?>
                        <input type="hidden" name="action" value="miro_redir_export" />
                        <p class="mr-row">
                            <button class="button mr-button-primary-sm" type="submit">
                                Export JSON
                            </button>
                        </p>
                        <p class="mr-tiny">Downloads all rules as a JSON file.</p>
                    </form>

                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                        <?php wp_nonce_field('miro_redir_import'); ?>
                        <input type="hidden" name="action" value="miro_redir_import" />
                        <p class="mr-row">
                            <textarea class="mr-json" name="json" rows="4" cols="80" placeholder='[{"type":"exact","source":"/old","target":"/new","status":301,"enabled":true}]'></textarea>
                        </p>
                        <p class="mr-row">
                            <button class="button mr-button-primary-sm" type="submit">
                                Import JSON
                            </button>
                        </p>
                        <p class="mr-tiny">Existing rules will be replaced with the imported list.</p>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }

    /* ---------------------------- Save handlers ---------------------------- */

    public static function handle_save(): void {
        // ✅ Admin-only for settings/rules changes
        if (!current_user_can(self::cap_manage())) {
            wp_die(esc_html__('No permission.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        check_admin_referer('miro_redir_save');

        $op = sanitize_text_field(wp_unslash($_POST['op'] ?? ''));

        $s = get_option(self::OPT_SETTINGS, []);
        $s = is_array($s) ? array_merge(self::defaults(), $s) : self::defaults();

        if ($op === 'save_settings') {
            $s['enabled']                = !empty($_POST['enabled']);

            $fb                          = sanitize_text_field(wp_unslash($_POST['fallback'] ?? '404'));
            $s['fallback']               = in_array($fb, ['404','home','custom','410','451'], true) ? $fb : '404';
            $s['fallback_custom']        = sanitize_text_field(wp_unslash($_POST['fallback_custom'] ?? ''));
            $s['fallback_451_reason']    = sanitize_text_field(wp_unslash($_POST['fallback_451_reason'] ?? ''));

            $def                         = absint(wp_unslash($_POST['default_status'] ?? 301));
            $s['default_status']         = in_array($def,[301,302,307,308,410,451],true) ? $def : 301;

            $s['auto_post_redirect']     = !empty($_POST['auto_post_redirect']);
            $s['auto_term_redirect']     = !empty($_POST['auto_term_redirect']);
            $s['respect_trailing_slash'] = !empty($_POST['respect_trailing_slash']);

            $s['allow_external']         = !empty($_POST['allow_external']);
            $s['external_hosts']         = sanitize_text_field(wp_unslash($_POST['external_hosts'] ?? ''));

            update_option(self::OPT_SETTINGS, $s, false);

            wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG.'&saved=1'));
            exit;
        }

        $rules = get_option(self::OPT_RULES, []);
        if (!is_array($rules)) $rules = [];

        if ($op === 'add_rule') {
            $type = in_array(sanitize_text_field(wp_unslash($_POST['rule_type'] ?? 'exact')), ['exact','prefix','regex'], true) ? sanitize_text_field(wp_unslash($_POST['rule_type'])) : 'exact';

            $src_raw = sanitize_text_field(wp_unslash($_POST['rule_source'] ?? ''));
            $tgt     = sanitize_text_field(wp_unslash($_POST['rule_target'] ?? ''));
            $src_raw = trim($src_raw);
            $tgt     = trim($tgt);

            $src = self::normalize_source($src_raw, $type);

            $st   = absint(wp_unslash($_POST['rule_status'] ?? $s['default_status']));
            if (!in_array($st, [301,302,307,308,410,451], true)) $st = (int)$s['default_status'];

            $cat  = sanitize_text_field(wp_unslash($_POST['rule_category'] ?? ''));
            $ena  = !empty($_POST['rule_enabled']);
            $rea  = sanitize_text_field(wp_unslash($_POST['rule_reason'] ?? ''));

            if ($src !== '' && ($st === 410 || $st === 451 || $tgt !== '')) {
                $rules[] = [
                    'id'       => substr(md5($type.'|'.$src.'|'.$tgt.'|'.microtime(true)), 0, 10),
                    'type'     => $type,
                    'source'   => $src,
                    'target'   => $tgt,
                    'status'   => $st,
                    'reason'   => $rea,
                    'category' => $cat,
                    'enabled'  => $ena,
                    'hits'     => 0,
                    'last_hit' => 0,
                ];
            }

            update_option(self::OPT_RULES, $rules, false);
            wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG.'&saved=1'));
            exit;

        } elseif ($op === 'toggle') {
            $idx = absint(wp_unslash($_POST['idx'] ?? -1));
            if (isset($rules[$idx]) && is_array($rules[$idx])) {
                $rules[$idx]['enabled'] = empty($rules[$idx]['enabled']);
                update_option(self::OPT_RULES, $rules, false);
            }
            wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG.'&saved=1'));
            exit;

        } elseif ($op === 'delete') {
            $idx = absint(wp_unslash($_POST['idx'] ?? -1));
            if (isset($rules[$idx])) {
                array_splice($rules, $idx, 1);
                update_option(self::OPT_RULES, $rules, false);
            }
            wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG.'&saved=1'));
            exit;

        } elseif ($op === 'update_rule') {
            $idx = absint(wp_unslash($_POST['idx'] ?? -1));
            if (!isset($rules[$idx]) || !is_array($rules[$idx])) {
                wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG));
                exit;
            }

            $type = in_array(sanitize_text_field(wp_unslash($_POST['rule_type'] ?? 'exact')), ['exact','prefix','regex'], true) ? sanitize_text_field(wp_unslash($_POST['rule_type'])) : 'exact';

            $src_raw = sanitize_text_field(wp_unslash($_POST['rule_source'] ?? ''));
            $tgt     = sanitize_text_field(wp_unslash($_POST['rule_target'] ?? ''));
            $src_raw = trim($src_raw);
            $tgt     = trim($tgt);

            $src = self::normalize_source($src_raw, $type);

            $st   = absint(wp_unslash($_POST['rule_status'] ?? $s['default_status']));
            if (!in_array($st, [301,302,307,308,410,451], true)) $st = (int)$s['default_status'];

            $cat  = sanitize_text_field(wp_unslash($_POST['rule_category'] ?? ''));
            $ena  = !empty($_POST['rule_enabled']);
            $rea  = sanitize_text_field(wp_unslash($_POST['rule_reason'] ?? ''));

            if ($src !== '' && ($st === 410 || $st === 451 || $tgt !== '')) {
                $rules[$idx]['type']     = $type;
                $rules[$idx]['source']   = $src;
                $rules[$idx]['target']   = $tgt;
                $rules[$idx]['status']   = $st;
                $rules[$idx]['reason']   = $rea;
                $rules[$idx]['category'] = $cat;
                $rules[$idx]['enabled']  = $ena;

                if (!isset($rules[$idx]['hits'])) $rules[$idx]['hits'] = 0;
                if (!isset($rules[$idx]['last_hit'])) $rules[$idx]['last_hit'] = 0;

                update_option(self::OPT_RULES, $rules, false);
            }

            wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG.'&updated=1'));
            exit;
        }

        wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG));
        exit;
    }

    public static function handle_import(): void {
        // ✅ Admin-only
        if (!current_user_can(self::cap_manage())) {
            wp_die(esc_html__('No permission.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        check_admin_referer('miro_redir_import');

        $json = sanitize_textarea_field(wp_unslash($_POST['json'] ?? ''));
        $arr  = json_decode($json, true);
        if (!is_array($arr)) $arr = [];
        $rules = [];

        foreach ($arr as $r) {
            if (!is_array($r)) continue;

            $type = in_array(($r['type'] ?? ''), ['exact','prefix','regex'], true) ? (string)$r['type'] : 'exact';

            $src_raw = trim((string)($r['source'] ?? ''));
            $src     = self::normalize_source($src_raw, $type);

            $tgt  = trim((string)($r['target'] ?? ''));
            $st   = (int)($r['status'] ?? 301);
            $rea  = trim((string)($r['reason'] ?? ''));

            if ($src !== '' && in_array($st,[301,302,307,308,410,451],true) && ($st===410 || $st===451 || $tgt !== '')) {
                $rules[] = [
                    'id'       => substr(md5($type.'|'.$src.'|'.$tgt.'|'.microtime(true)), 0, 10),
                    'type'     => $type,
                    'source'   => $src,
                    'target'   => $tgt,
                    'status'   => $st,
                    'reason'   => $rea,
                    'category' => (string)($r['category'] ?? ''),
                    'enabled'  => !empty($r['enabled']),
                    'hits'     => (int)($r['hits'] ?? 0),
                    'last_hit' => (int)($r['last_hit'] ?? 0),
                ];
            }
        }

        update_option(self::OPT_RULES, $rules, false);
        wp_safe_redirect(admin_url('admin.php?page='.self::MENU_SLUG.'&imported=1'));
        exit;
    }

    public static function handle_export(): void {
        // ✅ Admin-only
        if (!current_user_can(self::cap_manage())) {
            wp_die(esc_html__('No permission.', 'miro-ai-seo'), '', ['response' => 403]);
        }
        check_admin_referer('miro_redir_export');

        $rules = get_option(self::OPT_RULES, []);
        if (!is_array($rules)) $rules = [];

        nocache_headers();
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="miro-redirections.json"');
        echo wp_json_encode($rules, JSON_PRETTY_PRINT);
        exit;
    }

    /* ------------------------ Redirect Engine + Hits ------------------------ */

    public static function maybe_redirect(): void {
        $s = get_option(self::OPT_SETTINGS, []);
        $s = is_array($s) ? array_merge(self::defaults(), $s) : self::defaults();
        if (empty($s['enabled'])) return;

        $request_uri = self::get_request_path();
        if ($request_uri === '') return;

        // avoid messing with obvious admin-ish slugs
        if (preg_match('~/(login|admin|dashboard)(/)?$~i', $request_uri)) return;

        $rules = get_option(self::OPT_RULES, []);
        if (!is_array($rules)) $rules = [];

        $cands = self::path_candidates($request_uri, !empty($s['respect_trailing_slash']));

        foreach ($rules as $idx => $r) {
            if (!is_array($r) || empty($r['enabled'])) continue;

            $match = false;
            foreach ($cands as $cand) {
                $m = self::match_rule($r, $cand);
                if ($m !== false) { $match = $m; break; }
            }
            if ($match === false) continue;

            $status = intval($r['status'] ?? $s['default_status']);
            $status = in_array($status,[301,302,307,308,410,451],true) ? $status : 301;

            if ($status === 410 || $status === 451) {
                $reason = trim((string)($r['reason'] ?? ''));
                self::bump_rule_hit((int)$idx);
                self::emit_gone_or_legal($status, $reason ?: ($status===451 ? (string)($s['fallback_451_reason'] ?? '') : ''));
            }

            $target = self::build_target($r, $match);

            // ✅ Validate + allowlist external
            $target = self::validate_target($target, $s);
            if ($target === '') {
                // blocked target -> don't redirect, treat as no-match
                continue;
            }

            self::bump_rule_hit((int)$idx);
            wp_safe_redirect($target, $status);
            exit;
        }

        // Fallback behavior (we are not counting fallback hits here)
        if (is_404()) {
            $fb = (string)($s['fallback'] ?? '404');

            if ($fb === 'home') {
                wp_safe_redirect(home_url('/'), (int)$s['default_status']);
                exit;
            }

            if ($fb === 'custom' && !empty($s['fallback_custom'])) {
                $custom = self::validate_target((string)$s['fallback_custom'], $s);
                if ($custom !== '') {
                    wp_safe_redirect($custom, (int)$s['default_status']);
                    exit;
                }
            }

            if ($fb === '410' || $fb === '451') {
                $status = ($fb === '410') ? 410 : 451;
                $reason = ($status === 451) ? (string)($s['fallback_451_reason'] ?? '') : '';
                self::emit_gone_or_legal($status, $reason);
            }
        }
    }

    protected static function bump_rule_hit(int $idx): void {
        if ($idx < 0) return;

        if (!isset(self::$hit_buffer[$idx])) {
            self::$hit_buffer[$idx] = ['hits' => 0, 'last_hit' => 0];
        }
        self::$hit_buffer[$idx]['hits'] += 1;
        self::$hit_buffer[$idx]['last_hit'] = time();
    }

    public static function flush_hit_buffer(): void {
        if (empty(self::$hit_buffer)) return;

        $rules = get_option(self::OPT_RULES, []);
        if (!is_array($rules) || empty($rules)) return;

        $changed = false;

        foreach (self::$hit_buffer as $idx => $data) {
            if (!isset($rules[$idx]) || !is_array($rules[$idx])) continue;

            if (!isset($rules[$idx]['hits'])) $rules[$idx]['hits'] = 0;
            if (!isset($rules[$idx]['last_hit'])) $rules[$idx]['last_hit'] = 0;

            $rules[$idx]['hits'] = (int)$rules[$idx]['hits'] + (int)$data['hits'];
            $rules[$idx]['last_hit'] = max((int)$rules[$idx]['last_hit'], (int)$data['last_hit']);
            $changed = true;
        }

        self::$hit_buffer = [];

        if ($changed) {
            update_option(self::OPT_RULES, $rules, false);
        }
    }

    /* ------------------------ Target validation (safety) ------------------------ */

    protected static function validate_target(string $target, array $settings): string {
        $target = trim((string)$target);
        if ($target === '') return '';

        // allow relative
        if (isset($target[0]) && $target[0] === '/') {
            return home_url($target);
        }

        // must be http(s) if absolute
        $p = wp_parse_url($target);
        if (empty($p['host']) || empty($p['scheme'])) {
            return '';
        }
        $scheme = strtolower((string)$p['scheme']);
        if (!in_array($scheme, ['http','https'], true)) return '';

        $home_host = (string)wp_parse_url(home_url('/'), PHP_URL_HOST);
        $host      = strtolower((string)$p['host']);

        // same host always OK
        if ($home_host !== '' && $host === strtolower($home_host)) {
            return esc_url_raw($target);
        }

        // external host: require allow_external + host allowlist
        if (empty($settings['allow_external'])) return '';

        $allowed = self::parse_hosts((string)($settings['external_hosts'] ?? ''));
        if (empty($allowed)) return '';

        if (!in_array($host, $allowed, true)) return '';

        // allow wp_safe_redirect to accept this host by adding it to allowed hosts
        add_filter('allowed_redirect_hosts', function($hosts) use ($allowed) {
            foreach ($allowed as $h) {
                if (!in_array($h, $hosts, true)) $hosts[] = $h;
            }
            return $hosts;
        });

        return esc_url_raw($target);
    }

    protected static function parse_hosts(string $csv): array {
        $csv = strtolower(trim($csv));
        if ($csv === '') return [];
        $parts = array_filter(array_map('trim', explode(',', $csv)));
        $out = [];
        foreach ($parts as $h) {
            // strip scheme/path if user pasted a URL
            $p = wp_parse_url($h);
            if (!empty($p['host'])) $h = (string)$p['host'];
            $h = preg_replace('~[^a-z0-9\.\-]~', '', (string)$h);
            if ($h !== '' && !in_array($h, $out, true)) $out[] = $h;
        }
        return $out;
    }

    /* ------------------------ Path + Matching ------------------------ */

    protected static function get_request_path(): string {
        $uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($uri === '') return '';
        $parsed = wp_parse_url($uri);
        $path = (string)($parsed['path'] ?? '');
        $path = $path === '' ? '/' : $path;
        return self::normalize_path($path);
    }

    protected static function normalize_path(string $path): string {
        $path = trim($path);
        if ($path === '') return '/';
        $path = '/'.ltrim($path, '/');
        $path = preg_replace('#/+#','/',$path);
        return $path;
    }

    protected static function strip_to_path(string $maybe_url_or_path): string {
        $s = trim((string)$maybe_url_or_path);
        if ($s === '') return '';
        if (preg_match('#^https?://#i', $s)) {
            $p = wp_parse_url($s);
            $s = (string)($p['path'] ?? '');
        } else {
            $p = wp_parse_url($s);
            if (!empty($p['path'])) $s = (string)$p['path'];
        }
        return self::normalize_path($s);
    }

    protected static function normalize_source(string $src, string $type): string {
        $src = trim((string)$src);
        if ($src === '') return '';
        if ($type === 'regex') return $src;
        return self::strip_to_path($src);
    }

    protected static function path_candidates(string $path, bool $respect_trailing): array {
        $p = self::normalize_path($path);
        $no  = ($p === '/') ? '/' : rtrim($p, '/');
        $yes = ($p === '/') ? '/' : (rtrim($p,'/') . '/');
        $primary = $respect_trailing ? $p : $no;

        $out = [];
        foreach ([$primary, $no, $yes] as $x) {
            if (!in_array($x, $out, true)) $out[] = $x;
        }
        return $out;
    }

    protected static function match_rule(array $r, string $request_path) {
        $type = $r['type'] ?? 'exact';
        $src  = (string)($r['source'] ?? '');
        if ($src === '') return false;

        $src = self::normalize_source($src, $type);
        if ($type !== 'regex' && $src === '') return false;

        if ($type === 'exact') {
            return ($request_path === $src) ? [] : false;
        }
        if ($type === 'prefix') {
            if (strpos($request_path, $src) === 0) {
                $tail = substr($request_path, strlen($src));
                return ['tail'=>$tail];
            }
            return false;
        }
        if ($type === 'regex') {
            $pattern = (string)($r['source'] ?? '');
            if ($pattern === '') return false;
            if ($pattern[0] !== '~') $pattern = '~'.$pattern.'~';
            $result = @preg_match($pattern, $request_path, $m);
            if ($result === 1) return $m;
            return false;
        }
        return false;
    }

    protected static function build_target(array $r, $match): string {
        $status = (int)($r['status'] ?? 301);
        if ($status === 410 || $status === 451) return '';

        $target = trim((string)($r['target'] ?? '/'));

        if (isset($match['tail'])) {
            $target = str_replace('$tail', ltrim((string)$match['tail'],'/'), $target);
            if (isset($target[0]) && $target[0] === '/') {
                return home_url($target);
            }
            return $target;
        }

        if (is_array($match) && !empty($match)) {
            for ($i=1; $i<=9; $i++) {
                if (isset($match[$i])) $target = str_replace('$'.$i, (string)$match[$i], $target);
            }
        }

        if (isset($target[0]) && $target[0] === '/') return home_url($target);
        return $target;
    }

    protected static function emit_gone_or_legal(int $status, string $reason = ''): void {
        header_remove('Location');
        status_header($status);
        header('Content-Type: text/html; charset=utf-8');
        header('X-Robots-Tag: noindex, noarchive', true);

        if ($status === 451 && $reason !== '') {
            header('Link: <'.esc_url(home_url('/')).'>; rel="blocked-by"', false);
        }

        $title = ($status === 410) ? '410 Gone' : '451 Unavailable For Legal Reasons';
        $msg   = ($status === 410)
            ? 'The requested resource is no longer available on this server.'
            : 'The requested resource is unavailable for legal reasons.';

        echo '<!doctype html><html><head><meta charset="utf-8"><title>'.esc_html($title).'</title>';
        echo '<style>
          body{font:16px/1.6 -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Arial,system-ui;padding:48px;color:#111}
          h1{margin:0 0 10px;font-size:28px}
          p{margin:0 0 8px}
        </style></head><body>';
        echo '<h1>'.esc_html($title).'</h1><p>'.esc_html($msg);
        if ($status === 451 && $reason !== '') {
            echo ' Reason: ', esc_html($reason);
        }
        echo '</p>';
        echo '</body></html>';
        exit;
    }

    /* ----------------------- Auto Redirect Hooks ---------------------- */

    public static function on_post_updated($post_ID, $post_after, $post_before): void {
        $s = get_option(self::OPT_SETTINGS, []);
        $s = is_array($s) ? array_merge(self::defaults(), $s) : self::defaults();
        if (empty($s['auto_post_redirect'])) return;

        if ($post_before instanceof \WP_Post && $post_after instanceof \WP_Post) {
            if ($post_before->post_name !== $post_after->post_name && $post_after->post_status === 'publish') {
                $old = get_permalink($post_before);
                $new = get_permalink($post_after);
                self::add_quick_rule_from_urls($old, $new, (int)$s['default_status'], 'Auto Post Redirect');
            }
        }
    }

    public static function on_edited_terms($term_id, $tt_id): void { /* compat noop */ }

    public static function on_edited_term($term_id, $tt_id, $taxonomy): void {
        $s = get_option(self::OPT_SETTINGS, []);
        $s = is_array($s) ? array_merge(self::defaults(), $s) : self::defaults();
        if (empty($s['auto_term_redirect'])) return;

        $term = get_term($term_id, $taxonomy);
        if ($term && !is_wp_error($term)) {
            $meta_key = '_miro_last_term_link';
            $prev = get_term_meta($term_id, $meta_key, true);
            $new  = get_term_link($term, $taxonomy);
            if (!is_wp_error($new) && is_string($new)) {
                if ($prev && $prev !== $new) {
                    self::add_quick_rule_from_urls($prev, $new, (int)$s['default_status'], 'Auto Term Redirect');
                }
                update_term_meta($term_id, $meta_key, $new);
            }
        }
    }

    protected static function add_quick_rule_from_urls(string $old, string $new, int $status, string $category): void {
        $old_path = (string)(wp_parse_url($old, PHP_URL_PATH) ?: '');
        $new_url  = $new;
        if (!$old_path || !$new_url) return;

        $old_path = self::normalize_path($old_path);

        $rules = get_option(self::OPT_RULES, []);
        if (!is_array($rules)) $rules = [];

        $rules[] = [
            'id'       => substr(md5('exact|'.$old_path.'|'.$new_url.'|'.microtime(true)), 0, 10),
            'type'     => 'exact',
            'source'   => $old_path,
            'target'   => $new_url,
            'status'   => in_array($status,[301,302,307,308,410,451],true) ? $status : 301,
            'category' => $category,
            'enabled'  => true,
            'hits'     => 0,
            'last_hit' => 0,
        ];
        update_option(self::OPT_RULES, $rules, false);
    }
}
