<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

class Rest {

    private static $booted = false;

    public static function init() {
        if (self::$booted) return;
        self::$booted = true;
        add_action('rest_api_init', [__CLASS__, 'routes']);
    }

    public static function routes() {

        /* ---------------- Health Check ---------------- */
        register_rest_route('miro/v1', '/health', [
            'methods'  => 'GET',
            'callback' => function () {
                return [
                    'ok'      => true,
                    'version' => defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : 'dev',
                ];
            },
            'permission_callback' => function () {
                return current_user_can('edit_posts');
            }
        ]);

        /* ---------------- AI Stubs ---------------- */
        register_rest_route('miro/v1', '/ai/rewrite', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'ai_rewrite_stub'],
            'permission_callback' => function () { return current_user_can('edit_posts'); },
            'args' => [
                'text' => [
                    'type' => 'string',
                    'required' => true,
                    'sanitize_callback' => 'sanitize_textarea_field',
                    'validate_callback' => function($param) {
                        $len = function_exists('mb_strlen') ? mb_strlen($param) : strlen($param);
                        return $len > 0 && $len <= 50000;
                    }
                ]
            ]
        ]);

        register_rest_route('miro/v1', '/post/generate', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'generate_post_stub'],
            'permission_callback' => function () { return current_user_can('edit_posts'); },
            'args' => [
                'topic' => [
                    'type' => 'string',
                    'required' => true,
                    'sanitize_callback' => 'sanitize_text_field',
                    'validate_callback' => function($param) {
                        $len = function_exists('mb_strlen') ? mb_strlen($param) : strlen($param);
                        return $len > 0 && $len <= 500;
                    }
                ]
            ]
        ]);

        /* ---------------- Internal Links ---------------- */
        register_rest_route('miro/v1', '/links/suggest', [
            'methods'  => 'GET',
            'callback' => [__CLASS__, 'links_suggest'],
            'permission_callback' => function () { return current_user_can('edit_posts'); },
            'args' => [
                'post_id' => [
                    'type' => 'integer',
                    'required' => true,
                    'sanitize_callback' => 'absint',
                    'validate_callback' => function($param) {
                        return $param > 0 && $param <= 2147483647;
                    }
                ],
                'limit' => [
                    'type' => 'integer',
                    'required' => false,
                    'sanitize_callback' => 'absint',
                    'validate_callback' => function($param) {
                        return $param >= 1 && $param <= 100;
                    },
                    'default' => 10
                ]
            ]
        ]);

        register_rest_route('miro/v1', '/links/rebuild', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'links_rebuild'],
            'permission_callback' => function () { return current_user_can('manage_options'); },
            'args' => [
                'per_page' => [
                    'type' => 'integer',
                    'required' => false,
                    'sanitize_callback' => 'absint',
                    'validate_callback' => function($param) {
                        return $param >= 1 && $param <= 1000;
                    },
                    'default' => 50
                ],
                'offset' => [
                    'type' => 'integer',
                    'required' => false,
                    'sanitize_callback' => 'absint',
                    'validate_callback' => function($param) {
                        return $param >= 0 && $param <= 2147483647;
                    },
                    'default' => 0
                ]
            ]
        ]);

        /* ---------------- Trends Generator ---------------- */
        register_rest_route('miro/v1', '/trends/generate_draft', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'trends_generate_draft'],
            'permission_callback' => function () { return current_user_can('edit_posts'); },
            'args' => [
                'topic' => [
                    'type' => 'string',
                    'required' => true,
                    'sanitize_callback' => 'sanitize_text_field',
                    'validate_callback' => function($param) {
                        $len = function_exists('mb_strlen') ? mb_strlen($param) : strlen($param);
                        return $len > 0 && $len <= 500;
                    }
                ]
            ]
        ]);

        /* ---------------- Report: All Posts (NO SCORE HERE) ---------------- */
        register_rest_route('miro/v1', '/analytics/ui/allposts', [
            'methods'  => 'GET',
            'callback' => function () {

                $q = new \WP_Query([
                    'post_type'      => ['post', 'page'],
                    'post_status'    => ['publish','draft','pending','private'],
                    'posts_per_page' => -1,
                    'orderby'        => 'date',
                    'order'          => 'DESC',
                ]);

                $out = [];
                foreach ($q->posts as $p) {
                    $out[] = [
                        'post_id'  => $p->ID,
                        'title'    => $p->post_title,
                        'type'     => $p->post_type,
                        'date'     => $p->post_date,
                        'edit_url' => get_edit_post_link($p->ID,'raw'),
                        'view_url' => get_permalink($p->ID),
                    ];
                }

                return $out;
            },
            'permission_callback' => fn() => current_user_can('edit_posts')
        ]);

        /* ---------------- Report Summary & Report Rows removed ---------------- */
        /* Score system removed for CodeCanyon release */

    }

    /* ---------------- AI Stub Functions ---------------- */

    public static function ai_rewrite_stub($req) {
        $p = $req->get_json_params();
        $t = sanitize_text_field($p['text'] ?? '');
        return [ 'text' => 'Improved: ' . ucfirst($t) ];
    }

    public static function generate_post_stub($req) {
        $p = $req->get_json_params();
        $topic = sanitize_text_field($p['topic'] ?? '');
        return [
            'title'     => "AI Draft: $topic",
            'slug'      => sanitize_title($topic),
            'excerpt'   => "Draft about $topic",
            'body_html' => "<p>Draft content about <strong>$topic</strong>.</p>"
        ];
    }

    /* ---------------- Links Suggestions ---------------- */

    public static function links_suggest($req) {
        try {
            $pid = absint($req->get_param('post_id'));
            $limit = absint($req->get_param('limit') ?? 10);
            if ($pid <= 0) {
                return new \WP_Error('bad_request', 'Invalid post_id', ['status' => 400]);
            }
            return [ 'items' => Embeddings::suggest($pid, $limit) ];
        } catch (\Throwable $e) {
            return new \WP_Error('links_suggest_error', $e->getMessage(), ['status'=>500]);
        }
    }

    public static function links_rebuild($req) {
        try {
            $per = absint($req->get_param('per_page') ?? 50);
            $off = absint($req->get_param('offset') ?? 0);

            $q = new \WP_Query([
                'post_type'      => ['post','page'],
                'post_status'    => 'publish',
                'posts_per_page' => $per,
                'offset'         => $off,
                'fields'         => 'ids'
            ]);

            foreach ($q->posts as $pid) {
                Embeddings::on_save_post($pid, get_post($pid), true);
            }

            return [
                'processed' => count($q->posts),
                'offset'    => $off,
                'done'      => count($q->posts) < $per,
            ];
        } catch (\Throwable $e) {
            return new \WP_Error('links_rebuild_error', $e->getMessage(), ['status'=>500]);
        }
    }

    /* ---------------- Trends Generator ---------------- */

    public static function trends_generate_draft($req) {
        $p = $req->get_json_params();
        $topic = sanitize_text_field($p['topic'] ?? '');
        $title = "AI Draft: $topic";
        $body  = "<p>Quick draft about <strong>$topic</strong>.</p>";
        return [
            'ok'        => true,
            'title'     => $title,
            'meta_desc' => 'Draft about '.$topic,
            'body_html' => $body
        ];
    }

    /**
     * Generate a full article from a topic using AI.
     * Called by Trend Finder to create draft posts.
     * Optimized for SEO best practices and rich content.
     *
     * @param string $topic The topic/title to write about
     * @param int    $word_count Target word count (approximate)
     * @return array|WP_Error Array with title, meta_desc, body_html or error
     */
    public static function generate_article_from_topic(string $topic, int $word_count = 1200) {
        $topic = sanitize_text_field($topic);
        if (empty($topic)) {
            return new \WP_Error('invalid_topic', 'Topic is required.', ['status' => 400]);
        }

        $prompt = sprintf(
            'You are an expert SEO content writer. Create a comprehensive, high-ranking blog article about: "%1$s"' . "\n\n" .
            '## SEO REQUIREMENTS (CRITICAL):' . "\n" .
            '1. **Title**: Create a compelling, click-worthy title (50-60 characters) that includes the primary keyword near the beginning' . "\n" .
            '2. **Meta Description**: Write an enticing meta description (150-155 characters) with primary keyword, a benefit, and implicit CTA' . "\n" .
            '3. **Content Structure**:' . "\n" .
            '   - Start with a hook that addresses the reader\'s pain point or curiosity' . "\n" .
            '   - Include the primary keyword in the first 100 words' . "\n" .
            '   - Use exactly ONE H1 (the title - don\'t include in body)' . "\n" .
            '   - Use 4-6 H2 headings for main sections' . "\n" .
            '   - Use H3 subheadings under H2s where needed' . "\n" .
            '   - Keyword density: 1-2%% naturally distributed' . "\n\n" .
            '## CONTENT ELEMENTS (Include ALL):' . "\n" .
            '1. **Introduction** (100-150 words): Hook + problem statement + what they\'ll learn + brief answer' . "\n" .
            '2. **Table of Contents**: List of main sections (helps with jump links)' . "\n" .
            '3. **Main Content Sections**:' . "\n" .
            '   - Each H2 section should be 150-250 words' . "\n" .
            '   - Include bullet points or numbered lists in at least 2 sections' . "\n" .
            '   - Add a comparison table if relevant to the topic' . "\n" .
            '4. **Pro Tips/Expert Insights**: A highlighted box with 3-5 actionable tips' . "\n" .
            '5. **FAQ Section**: 4-5 common questions with concise answers (schema-ready format)' . "\n" .
            '6. **Conclusion**: Summary + key takeaway + soft CTA' . "\n\n" .
            '## FORMATTING RULES:' . "\n" .
            '- Use <strong> for important terms (2-3 per section)' . "\n" .
            '- Use <em> for emphasis sparingly' . "\n" .
            '- Use <ul> and <ol> for lists' . "\n" .
            '- Use <blockquote> for key quotes or highlights' . "\n" .
            '- Use <table> with <thead> and <tbody> for comparisons' . "\n" .
            '- Add descriptive alt text placeholders: [IMAGE: description of ideal image]' . "\n\n" .
            '## QUALITY STANDARDS:' . "\n" .
            '- Write at 8th-grade reading level (clear, accessible)' . "\n" .
            '- Use active voice (80%%+ of sentences)' . "\n" .
            '- Vary sentence length (mix short punchy + longer explanatory)' . "\n" .
            '- Include statistics or data points where possible (cite as "studies show" or "according to experts")' . "\n" .
            '- No fluff - every sentence should add value' . "\n" .
            '- Target word count: %2$d words (aim for comprehensive but concise)' . "\n\n" .
            '## OUTPUT FORMAT (STRICT JSON):' . "\n" .
            'Return ONLY valid JSON with no markdown wrapping:' . "\n" .
            '{' . "\n" .
            '  "title": "Your SEO-Optimized Title Here (50-60 chars)",' . "\n" .
            '  "meta_desc": "Compelling meta description with keyword and benefit (150-155 chars)",' . "\n" .
            '  "body_html": "<p>Full HTML content starting with intro paragraph...</p>"' . "\n" .
            '}' . "\n\n" .
            'IMPORTANT: The body_html must be properly escaped JSON string with actual HTML tags.',
            $topic,
            $word_count
        );

        // Try miro/ai/complete filter first
        // Use gpt-4o-mini for better quality long-form content
        $result = apply_filters('miro/ai/complete', null, [
            'prompt'  => $prompt,
            'context' => 'trend_article',
            'opts'    => [
                'model'       => 'gpt-4o-mini',
                'temperature' => 0.7,
                'max_tokens'  => min(4096, max(2500, $word_count * 2)),
            ],
        ]);

        // Fallback to miro_ai_complete_inline if available
        if (empty($result) && function_exists('\\Miro_AI_SEO\\miro_ai_complete_inline')) {
            $result = \Miro_AI_SEO\miro_ai_complete_inline($prompt, 'trend_article');
        }

        // If still no result, return a comprehensive SEO stub
        if (empty($result) || is_wp_error($result)) {
            $topic_clean = esc_html($topic);
            $topic_title = ucwords($topic);
            
            return [
                'title'     => sprintf('%s: Complete Guide [%d]', $topic_title, (int)gmdate('Y')),
                'meta_desc' => sprintf('Discover everything about %s. Expert tips, best practices, and actionable insights to help you succeed.', $topic),
                'body_html' => sprintf(
                    '<!-- AI not configured - Edit this SEO-optimized template -->' .
                    '<p><strong>Looking to master %1$s?</strong> You\'re in the right place. This comprehensive guide covers everything you need to know about %1$s, from basics to advanced strategies.</p>' .
                    
                    '<p><em>In this article, you\'ll learn:</em></p>' .
                    '<ul><li>What %1$s is and why it matters</li><li>Key benefits and best practices</li><li>Step-by-step implementation guide</li><li>Common mistakes to avoid</li><li>Expert tips for success</li></ul>' .
                    
                    '<h2>What is %1$s?</h2>' .
                    '<p>%1$s is [add your definition here]. Understanding this concept is essential for anyone looking to [add benefit].</p>' .
                    '<p>[IMAGE: Infographic explaining %1$s basics]</p>' .
                    
                    '<h2>Why %1$s Matters</h2>' .
                    '<p>According to industry experts, %1$s has become increasingly important because:</p>' .
                    '<ol><li><strong>Benefit 1</strong>: [Explain the first key benefit]</li><li><strong>Benefit 2</strong>: [Explain the second key benefit]</li><li><strong>Benefit 3</strong>: [Explain the third key benefit]</li></ol>' .
                    
                    '<h2>How to Get Started with %1$s</h2>' .
                    '<p>Follow these steps to implement %1$s effectively:</p>' .
                    '<h3>Step 1: [First Action]</h3>' .
                    '<p>[Detailed explanation of the first step]</p>' .
                    '<h3>Step 2: [Second Action]</h3>' .
                    '<p>[Detailed explanation of the second step]</p>' .
                    '<h3>Step 3: [Third Action]</h3>' .
                    '<p>[Detailed explanation of the third step]</p>' .
                    
                    '<h2>Best Practices for %1$s</h2>' .
                    '<blockquote><strong>Pro Tip:</strong> The most successful implementations of %1$s focus on [add your insight here].</blockquote>' .
                    '<ul><li>✅ <strong>Do:</strong> [Best practice 1]</li><li>✅ <strong>Do:</strong> [Best practice 2]</li><li>❌ <strong>Avoid:</strong> [Common mistake 1]</li><li>❌ <strong>Avoid:</strong> [Common mistake 2]</li></ul>' .
                    
                    '<h2>%1$s Comparison Table</h2>' .
                    '<table><thead><tr><th>Feature</th><th>Option A</th><th>Option B</th></tr></thead><tbody><tr><td>Cost</td><td>[Value]</td><td>[Value]</td></tr><tr><td>Ease of Use</td><td>[Value]</td><td>[Value]</td></tr><tr><td>Results</td><td>[Value]</td><td>[Value]</td></tr></tbody></table>' .
                    
                    '<h2>Frequently Asked Questions About %1$s</h2>' .
                    '<h3>What is the best way to start with %1$s?</h3>' .
                    '<p>[Your answer here - keep it concise and actionable]</p>' .
                    '<h3>How long does it take to see results from %1$s?</h3>' .
                    '<p>[Your answer here - be specific with timeframes if possible]</p>' .
                    '<h3>Is %1$s suitable for beginners?</h3>' .
                    '<p>[Your answer here - address the reader\'s concerns]</p>' .
                    '<h3>What are the costs associated with %1$s?</h3>' .
                    '<p>[Your answer here - be transparent about investment required]</p>' .
                    
                    '<h2>Conclusion</h2>' .
                    '<p><strong>Key Takeaway:</strong> %1$s is a valuable [skill/tool/strategy] that can help you [main benefit]. Start with the basics, follow the best practices outlined above, and you\'ll be well on your way to success.</p>' .
                    '<p><em>Ready to take the next step? [Add your call-to-action here]</em></p>',
                    $topic_clean
                ),
            ];
        }

        // Try to parse JSON from the AI response
        $parsed = null;
        if (is_string($result)) {
            // Try to extract JSON from the response
            $result = trim($result);
            
            // Remove markdown code blocks if present
            if (preg_match('/```(?:json)?\s*([\s\S]*?)\s*```/', $result, $matches)) {
                $result = trim($matches[1]);
            }
            
            $parsed = json_decode($result, true);
        } elseif (is_array($result)) {
            $parsed = $result;
        }

        // Validate parsed result
        if (is_array($parsed) && !empty($parsed['title']) && !empty($parsed['body_html'])) {
            return [
                'title'     => sanitize_text_field($parsed['title']),
                'meta_desc' => sanitize_text_field($parsed['meta_desc'] ?? ''),
                'body_html' => wp_kses_post($parsed['body_html']),
            ];
        }

        // If parsing failed but we have string content, use it as body
        if (is_string($result) && strlen($result) > 50) {
            return [
                'title'     => ucwords($topic),
                'meta_desc' => sprintf('Learn about %s in this comprehensive guide.', $topic),
                'body_html' => '<div>' . wp_kses_post($result) . '</div>',
            ];
        }

        // Ultimate fallback - still provide SEO structure
        return [
            'title'     => sprintf('%s: Essential Guide [%d]', ucwords($topic), (int)gmdate('Y')),
            'meta_desc' => sprintf('Learn about %s with our expert guide. Actionable tips and insights inside.', $topic),
            'body_html' => sprintf(
                '<p>This article explores <strong>%1$s</strong> in detail. Edit and expand this content with your expertise.</p>' .
                '<h2>Understanding %1$s</h2><p>[Add your introduction here]</p>' .
                '<h2>Key Benefits</h2><ul><li>[Benefit 1]</li><li>[Benefit 2]</li><li>[Benefit 3]</li></ul>' .
                '<h2>How to Get Started</h2><p>[Add your guide here]</p>' .
                '<h2>FAQ</h2><h3>What is %1$s?</h3><p>[Your answer]</p>' .
                '<h2>Conclusion</h2><p>[Summarize key points]</p>',
                esc_html($topic)
            ),
        ];
    }

}
