<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * ── Admin HTTPS + REST URL hardening (prevents "Failed to fetch" on HTTPS) ──
 */
add_action('admin_init', function () {
    // Force REST URLs to https inside admin to avoid mixed-content fetch errors
    add_filter('rest_url', function($url){
        if (is_admin() && strpos($url, 'http://') === 0) {
            $url = preg_replace('#^http://#','https://',$url);
        }
        return $url;
    }, 99);

    // Respect proxy headers (Cloudflare / Nginx reverse proxy)
    if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') {
        $_SERVER['HTTPS'] = 'on';
    }
}, 1);

/**
 * Capability resolver (do NOT redeclare any global!)
 * - If global \miro_ai_cap() exists, use it. Else fallback to manage_options.
 */
if (!function_exists(__NAMESPACE__.'\\_miro_cap')) {
    function _miro_cap(): string {
        if (function_exists('\\miro_ai_cap')) {
            try {
                $cap = \miro_ai_cap();
                if (is_string($cap) && $cap !== '') return $cap;
            } catch (\Throwable $e) {}
        }
        return 'manage_options';
    }
}

/**
 * Minimal AI helper (site filter → OpenAI → Gemini)
 */
if (!function_exists(__NAMESPACE__ . '\\miro_ai_complete_inline')) {
    function miro_ai_complete_inline(string $prompt, string $context = 'analytics') : string {
        $filtered = apply_filters('miro/ai/complete', null, [
            'prompt'  => $prompt,
            'context' => $context,
            'opts'    => ['temperature'=>0.3, 'max_tokens'=>500],
        ]);
        if (is_string($filtered) && $filtered !== '') return $filtered;

        $openai_key = get_option('miro_ai_openai_key');
        if ($openai_key) {
            $payload = [
                'model' => 'gpt-4o-mini',
                'messages' => [
                    ['role'=>'system','content'=>'You are Miro Analytics AI. Be concise and actionable.'],
                    ['role'=>'user','content'=>$prompt],
                ],
                'temperature' => 0.2,
                'max_tokens'  => 500,
            ];
            $res = wp_remote_post('https://api.openai.com/v1/chat/completions', [
                'timeout'=>30,
                'headers'=>[
                    'Authorization'=>'Bearer '.$openai_key,
                    'Content-Type'=>'application/json',
                ],
                'body'=>wp_json_encode($payload),
            ]);
            if (!is_wp_error($res)) {
                $code = wp_remote_retrieve_response_code($res);
                $body = json_decode(wp_remote_retrieve_body($res), true);
                if ($code>=200 && $code<300 && !empty($body['choices'][0]['message']['content'])) {
                    return trim($body['choices'][0]['message']['content']);
                }
            }
        }

        $gemini_key = get_option('miro_ai_gemini_key');
        if ($gemini_key) {
            $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash:generateContent?key=' . rawurlencode($gemini_key);
            $payload = [
                'contents' => [[ 'parts' => [[ 'text' => "You are Miro Analytics AI. Be concise and actionable.\n\n".$prompt ]] ]],
                'generationConfig' => [ 'temperature' => 0.2 ],
            ];
            $res = wp_remote_post($endpoint, [
                'timeout' => 30,
                'headers' => [ 'Content-Type' => 'application/json' ],
                'body'    => wp_json_encode($payload),
            ]);
            if (!is_wp_error($res)) {
                $code = wp_remote_retrieve_response_code($res);
                $body = json_decode(wp_remote_retrieve_body($res), true);
                if ($code>=200 && $code<300 && !empty($body['candidates'][0]['content']['parts'][0]['text'])) {
                    return trim($body['candidates'][0]['content']['parts'][0]['text']);
                }
            }
        }

        return __('AI not connected. Add a key or hook miro/ai/complete.', 'miro-ai-seo');
    }
}

/**
 * Core GSC Analytics (shell + shared helpers; tabs hook into nav/content actions)
 */
class GSC_Analytics_Core
{
    const MENU_PARENT        = 'miro-ai-seo';
    const MENU_SLUG          = 'miro-gsc-analytics';

    // OAuth + property
    const OPT_SETTINGS       = 'pp_miro_gsc_settings';
    const OPT_PROPERTY       = 'pp_miro_gsc_property';

    // Google endpoints
    const GOOGLE_TOKEN_URL   = 'https://oauth2.googleapis.com/token';
    // Official Search Console Search Analytics API (webmasters v3)
    const GSC_QUERY_URL_TMPL = 'https://www.googleapis.com/webmasters/v3/sites/%s/searchAnalytics/query';

    // Cache option
    const OPT_CACHE          = 'miro_gsc_last_queries';

    // First-run sync: state option, cron hook, dedupe lock
    const OPT_SYNC_STATE     = 'miro_gsc_sync_state';
    const OPT_CACHE_READY    = 'miro_gsc_cache_ready';
    const CRON_FIRST_SYNC    = 'miro_gsc_first_sync';
    const FIRST_SYNC_LOCK    = 'miro_gsc_first_sync_scheduled';
    const FIRST_SYNC_LOCK_TTL = 900; // 15 min — avoid scheduling repeatedly

    public static function init(): void
    {
        add_action('admin_menu', [__CLASS__, 'menu'], 25);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
        add_action(self::CRON_FIRST_SYNC, [__CLASS__, 'run_first_sync'], 10, 1);
        add_action('wp_ajax_miro_gsc_clear_cache', [__CLASS__, 'ajax_clear_cache']);
        add_action('wp_ajax_miro_gsc_snapshot', [__CLASS__, 'ajax_snapshot']);
        add_action('wp_ajax_miro_gsc_sync_state', [__CLASS__, 'ajax_sync_state']);
        add_action('wp_ajax_miro_gsc_run_sync_now', [__CLASS__, 'ajax_run_sync_now']);
        add_action('wp_ajax_miro_gsc_run_sync_blocking', [__CLASS__, 'ajax_run_sync_blocking']);
        self::register_rest();
    }

    /** Admin-ajax: run sync in this request so data is fetched even when REST is blocked. Long-running (2–5 min). */
    public static function ajax_run_sync_blocking(): void
    {
        if (!current_user_can(_miro_cap())) {
            wp_send_json(['ok' => false, 'message' => __('Unauthorized.', 'miro-ai-seo')], 403);
        }
        if (isset($_REQUEST['nonce']) && !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['nonce'])), 'wp_rest')) {
            wp_send_json(['ok' => false, 'message' => __('Invalid nonce.', 'miro-ai-seo')], 403);
        }
        if (get_transient('miro_gsc_first_sync_running')) {
            wp_send_json(['ok' => true, 'status' => 'running', 'message' => __('Sync already in progress.', 'miro-ai-seo')]);
            return;
        }
        @set_time_limit(600);
        self::run_first_sync(['reason' => 'run_sync_now']);
        $state = self::get_sync_state();
        wp_send_json([
            'ok'     => true,
            'status' => $state['status'],
            'message' => $state['status'] === 'done' ? __('Sync complete. Data is ready.', 'miro-ai-seo') : ($state['last_error'] ?: __('Sync finished.', 'miro-ai-seo')),
        ]);
    }

    /** Admin-ajax: run sync now (schedules sync and returns; Keywords tab polls sync state until done). */
    public static function ajax_run_sync_now(): void
    {
        if (!current_user_can(_miro_cap())) {
            wp_send_json(['ok' => false, 'message' => __('Unauthorized.', 'miro-ai-seo')], 403);
        }
        if (get_transient('miro_gsc_first_sync_running')) {
            wp_send_json(['ok' => true, 'status' => 'running', 'message' => __('Sync already in progress.', 'miro-ai-seo')]);
            return;
        }
        $scheduled = self::schedule_first_sync('manual');
        $state = self::get_sync_state();
        wp_send_json([
            'ok'        => true,
            'status'    => $state['status'],
            'scheduled'  => $scheduled,
            'message'   => $scheduled ? __('Sync started. Data will appear when finished.', 'miro-ai-seo') : __('Sync already queued or running.', 'miro-ai-seo'),
        ]);
    }

    /** Admin-ajax: snapshot (same response as REST, so page load works when REST route is unavailable). */
    public static function ajax_snapshot(): void
    {
        if (!current_user_can(_miro_cap())) {
            wp_send_json(['code' => 'forbidden', 'message' => __('Unauthorized.', 'miro-ai-seo')], 403);
        }
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        header('Expires: 0');
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing -- Capability checked; days is sanitized via intval.
        $days = max(7, min(90, intval($_GET['days'] ?? $_POST['days'] ?? 30)));
        $req = new \WP_REST_Request('GET', '/miro/v1/analytics/ui/snapshot');
        $req->set_param('days', $days);
        $res = rest_do_request($req);
        if ($res->is_error()) {
            $err = $res->get_error_data();
            wp_send_json($err && is_array($err) ? $err : ['code' => 'error', 'message' => $res->get_error_message()], $res->get_status() ?: 500);
            return;
        }
        wp_send_json($res->get_data());
    }

    /** Admin-ajax: sync state (same response as REST). */
    public static function ajax_sync_state(): void
    {
        if (!current_user_can(_miro_cap())) {
            wp_send_json(['code' => 'forbidden', 'message' => __('Unauthorized.', 'miro-ai-seo')], 403);
        }
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        header('Expires: 0');
        wp_send_json(self::get_sync_state());
    }

    /** Admin-ajax handler for clear cache (works when REST route is unavailable). */
    public static function ajax_clear_cache(): void
    {
        if (!current_user_can(_miro_cap())) {
            wp_send_json_error(['message' => __('Unauthorized.', 'miro-ai-seo')], 403);
        }
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['nonce'])), 'wp_rest')) {
            wp_send_json_error(['message' => __('Invalid nonce.', 'miro-ai-seo')], 403);
        }
        self::clear_analytics_cache();
        wp_send_json_success([
            'ok' => true,
            'message' => __('All analytics cache cleared. Click Sync now to fetch fresh data.', 'miro-ai-seo'),
        ]);
    }

    public static function menu(): void
    {
        if (function_exists('miro_ai_add_submenu_once')) {
            \miro_ai_add_submenu_once(self::MENU_PARENT, 'GSC Analytics', 'GSC Analytics', _miro_cap(), self::MENU_SLUG, [__CLASS__, 'render'], 11);
        } else {
            add_submenu_page(self::MENU_PARENT, 'GSC Analytics', 'GSC Analytics', _miro_cap(), self::MENU_SLUG, [__CLASS__, 'render'], 11);
        }
    }

    /** Enqueue external CSS file (assets/css/miro-analytics.css + miro-momentum.css) */
    public static function enqueue_assets($hook): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin page check for enqueue; capability implied by admin context.
        if (!isset($_GET['page']) || $_GET['page'] !== self::MENU_SLUG) return;

        if (defined('MIRO_AI_SEO_FILE')) {
            // Original analytics CSS (for tabs and other sections)
            $rel  = 'assets/css/miro-analytics.css';
            $url  = plugins_url($rel, MIRO_AI_SEO_FILE);
            $path = plugin_dir_path(MIRO_AI_SEO_FILE) . $rel;
            $ver  = (file_exists($path) && is_readable($path)) ? (string) filemtime($path) : (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');
            wp_enqueue_style('miro-analytics', $url, [], $ver);
            
            // New momentum design CSS
            $rel2  = 'assets/css/miro-momentum.css';
            $url2  = plugins_url($rel2, MIRO_AI_SEO_FILE);
            $path2 = plugin_dir_path(MIRO_AI_SEO_FILE) . $rel2;
            $ver2  = (file_exists($path2) && is_readable($path2)) ? (string) filemtime($path2) : (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');
            wp_enqueue_style('miro-momentum', $url2, [], $ver2);
        }
    }

    /** ===== REST: KPIs + series + AI + overall + report + sync ===== */
    public static function register_rest(): void
    {
        add_action('rest_api_init', function () {

            // Snapshot (reads cache), supports ?days=7|15|30|90 to filter KPIs + series
            // ✅ FIX: Chart + KPIs come from DAILY TOTALS (dimension=date only) to match GSC UI.
            register_rest_route('miro/v1', '/analytics/ui/snapshot', [
                'methods'  => 'GET',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(\WP_REST_Request $req){
                    try {
                    $days = max(7, min(90, intval($req->get_param('days') ?? 30)));

                    $o = get_option(self::OPT_CACHE);
                    
                    // If no cache exists, return empty structure and sync_state for "Syncing… (first run)" UX
                    if (!is_array($o) || empty($o)) {
                        $sync_state = self::get_sync_state();
                        return [
                            'kpis'=>[
                                'search_traffic'     => 0,
                                'search_impressions' => 0,
                                'total_keywords'     => 0,
                                'clicks'             => 0,
                                'impressions'        => 0,
                                'ctr'                => 0.0,
                                'position'           => 0.0,
                                'last_synced'        => '',
                                'traffic_source'     => 'proxy_clicks',
                            ],
                            'series'=>[
                                'labels'=>[],
                                'clicks'=>[],
                                'impressions'=>[],
                                'ctr'=>[],
                                'position'=>[]
                            ],
                            'meta'=>[
                                'cached_range_days' => 0,
                                'ui_days'           => $days,
                                'from'              => '',
                                'to'                => '',
                            ],
                            'sync_state' => $sync_state,
                        ];
                    }
                    
                    $meta = is_array($o) ? ($o['meta'] ?? []) : [];
                    $dailyAll = is_array($o['daily'] ?? null) ? $o['daily'] : [];
                    $rowsAll  = is_array($o['rows']  ?? null) ? $o['rows']  : [];

                    // Fallback for very old cache: if daily missing, use detailed (not ideal but keeps UI alive)
                    if (!$dailyAll && $rowsAll) {
                        // Build daily from detailed (best effort)
                        $tmp = [];
                        foreach ($rowsAll as $r) {
                            $d = (string)($r['date'] ?? '');
                            if ($d === '') continue;
                            if (!isset($tmp[$d])) $tmp[$d] = ['date'=>$d,'clicks'=>0,'impressions'=>0,'posW'=>0];
                            $c = (float)($r['clicks'] ?? 0);
                            $i = (float)($r['impressions'] ?? 0);
                            $p = (float)($r['position'] ?? 0);
                            $tmp[$d]['clicks'] += $c;
                            $tmp[$d]['impressions'] += $i;
                            if ($i > 0 && $p > 0) $tmp[$d]['posW'] += ($p * $i);
                        }
                        ksort($tmp);
                        $dailyAll = [];
                        foreach ($tmp as $d=>$t) {
                            $i = (float)$t['impressions'];
                            $c = (float)$t['clicks'];
                            $pos = ($i > 0 && $t['posW'] > 0) ? ($t['posW'] / $i) : 0.0;
                            $dailyAll[] = [
                                'date'=>$d,
                                'clicks'=>$c,
                                'impressions'=>$i,
                                'ctr'=>($i>0?($c/$i):0.0),
                                'position'=>$pos,
                            ];
                        }
                    }

                    // Compute window bounds from DAILY (source of truth)
                    [$daily, $from, $to] = self::filter_daily_by_days($dailyAll, $days);

                    // Filter detailed rows to same date bounds (for keywords count)
                    $rows = self::filter_rows_by_bounds($rowsAll, $from, $to);

                    // ✅ Detect and exclude incomplete days from the end
                    // Google Search Console data for recent days may be incomplete
                    // We'll show all complete data, excluding only days that are clearly incomplete
                    $tz = wp_timezone();
                    $today = (new \DateTime('today', $tz))->format('Y-m-d');

                    // First, collect all data
                    $tempData = [];
                    foreach ($daily as $r) {
                        $d = (string)($r['date'] ?? '');
                        if ($d === '') continue;
                        
                        // Always exclude today (current day is never complete)
                        if ($d === $today) continue;
                        
                        $tempData[] = [
                            'date' => $d,
                            'clicks' => (float)($r['clicks'] ?? 0),
                            'impressions' => (float)($r['impressions'] ?? 0),
                            'position' => (float)($r['position'] ?? 0),
                        ];
                    }

                    // Sort by date to work from oldest to newest
                    usort($tempData, function($a, $b) {
                        return strcmp($a['date'], $b['date']);
                    });

                    // Calculate average impressions from complete days (exclude last 2 days for calculation)
                    $avgImpressions = 0.0;
                    $daysForAvg = max(1, count($tempData) - 2);
                    if ($daysForAvg > 0) {
                        $impSum = 0.0;
                        for ($i = 0; $i < $daysForAvg; $i++) {
                            $impSum += $tempData[$i]['impressions'];
                        }
                        $avgImpressions = $impSum / $daysForAvg;
                    }

                    // Remove incomplete days from the end
                    // A day is considered incomplete if:
                    // 1. It has 0 impressions (definitely incomplete)
                    // 2. It has impressions < 10% of average (likely incomplete)
                    // 3. It has impressions < 30% of the previous day (likely incomplete compared to yesterday)
                    while (!empty($tempData)) {
                        $last = end($tempData);
                        $isIncomplete = false;
                        
                        if ($last['impressions'] == 0) {
                            // Zero impressions = incomplete day
                            $isIncomplete = true;
                        } elseif ($avgImpressions > 0 && $last['impressions'] < ($avgImpressions * 0.1)) {
                            // Very low impressions compared to average = likely incomplete
                            $isIncomplete = true;
                        } elseif (count($tempData) > 1) {
                            // Compare with previous day - if last day is much lower, it's likely incomplete
                            $prevDay = $tempData[count($tempData) - 2];
                            $prevImpr = $prevDay['impressions'];
                            if ($prevImpr > 0 && $last['impressions'] < ($prevImpr * 0.3)) {
                                // Last day has less than 30% of previous day's impressions = likely incomplete
                                $isIncomplete = true;
                            }
                        }
                        
                        if ($isIncomplete) {
                            array_pop($tempData);
                            // Recalculate average after removing incomplete day
                            $daysForAvg = max(1, count($tempData) - 2);
                            if ($daysForAvg > 0) {
                                $impSum = 0.0;
                                for ($i = 0; $i < $daysForAvg; $i++) {
                                    $impSum += $tempData[$i]['impressions'];
                                }
                                $avgImpressions = $impSum / $daysForAvg;
                            }
                        } else {
                            // Found complete data, stop removing
                            break;
                        }
                    }

                    // Build series + KPIs from complete daily totals
                    $labels=[]; $sC=[]; $sI=[]; $sCTR=[]; $sPOS=[];
                    $C=0.0; $I=0.0; $posW=0.0;

                    foreach ($tempData as $r) {
                        $d = $r['date'];
                        $c = $r['clicks'];
                        $i = $r['impressions'];
                        $p = $r['position'];

                        $labels[] = $d;
                        $sC[] = $c;
                        $sI[] = $i;

                        // CTR as ratio 0..1
                        $sCTR[] = ($i > 0) ? ($c / $i) : 0.0;

                        // Position as reported by GSC for that day
                        $sPOS[] = $p;

                        $C += $c;
                        $I += $i;

                        // Weighted avg position across window
                        if ($i > 0 && $p > 0) $posW += ($p * $i);
                    }

                    // ✅ Correct KPI CTR + Position
                    $avgCtrPct = ($I > 0) ? round(($C / $I) * 100, 2) : 0.0;
                    $avgPos    = ($I > 0 && $posW > 0) ? round($posW / $I, 2) : 0.0;

                    // Keywords count still from detailed rows
                    $kw = [];
                    foreach ($rows as $r) {
                        $q = (string)($r['query'] ?? '');
                        $pos = (float)($r['position'] ?? 0);
                        if ($q !== '' && $pos > 0 && $pos <= 100) $kw[$q] = true;
                    }

                    $search_traffic = apply_filters('miro/analytics/search_traffic', (int)$C, [
                        'source' => 'proxy_clicks',
                        'note'   => 'Using GSC clicks as a proxy for Google pageviews',
                    ]);

                    return [
                        'kpis'=>[
                            'search_traffic'     => (int)$search_traffic,
                            'search_impressions' => (int)$I,
                            'total_keywords'     => (int)count($kw),
                            'clicks'             => (int)$C,
                            'impressions'        => (int)$I,
                            'ctr'                => $avgCtrPct,
                            'position'           => $avgPos,
                            'last_synced'        => sanitize_text_field($meta['last_synced'] ?? ''),
                            'traffic_source'     => 'proxy_clicks',
                        ],
                        'series'=>[
                            'labels'=>$labels,
                            'clicks'=>$sC,
                            'impressions'=>$sI,
                            'ctr'=>$sCTR,        // 0..1 (front multiplies by 100)
                            'position'=>$sPOS
                        ],
                        'meta'=>[
                            'cached_range_days' => (int)($meta['range_days'] ?? 0),
                            'ui_days'           => $days,
                            'from'              => sanitize_text_field($from ?? ($meta['from'] ?? '')),
                            'to'                => sanitize_text_field($to ?? ($meta['to'] ?? '')),
                        ],
                        'sync_state' => self::get_sync_state(),
                    ];
                    } catch (\Throwable $e) {
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            error_log('Miro GSC snapshot error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine()); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- WP_DEBUG only.
                        }
                        return [
                            'code'    => 'snapshot_error',
                            'message' => 'Failed to load analytics data. Please check your connection and try syncing again.',
                        ];
                    }
                }
            ]);

            // AI Summary (traffic + content health)
            register_rest_route('miro/v1', '/analytics/ui/summary', [
                'methods'=>'GET',
                'permission_callback'=>function(){ return current_user_can(_miro_cap()); },
                'callback'=>function(\WP_REST_Request $req){
                    $days = max(7, min(90, intval($req->get_param('days') ?? 30)));

                    $snapReq = new \WP_REST_Request('GET', '/miro/v1/analytics/ui/snapshot');
                    $snapReq->set_param('days', $days);
                    $snapRes = rest_do_request($snapReq);
                    $d = $snapRes->get_data();
                    $k = $d['kpis'] ?? [];

                    $prompt  = "Write a short SEO traffic summary for the last {$days} days using these stats:\n";
                    $prompt .= "Search Traffic (proxy clicks): ".($k['search_traffic']     ?? 0)."\n";
                    $prompt .= "Search Impressions: "         .($k['search_impressions'] ?? 0)."\n";
                    $prompt .= "Total Keywords (<=100): "     .($k['total_keywords']    ?? 0)."\n";
                    $prompt .= "Avg CTR %: "                  .($k['ctr']                ?? 0)."\n";
                    $prompt .= "Avg Position: "               .($k['position']           ?? 0)."\n";


                    $prompt .= "Be concise (3–5 sentences). Call out wins and risks.";
                    $prompt .= " End with one clear next action starting with a verb (e.g. \"Improve\", \"Update\", \"Add\").";

                    return ['text'=> miro_ai_complete_inline($prompt, 'analytics_summary')];
                }
            ]);

            // Ask AI
            register_rest_route('miro/v1', '/analytics/ui/ask', [
                'methods'=>'POST',
                'permission_callback'=>function(){ return current_user_can(_miro_cap()); },
                'callback'=>function(\WP_REST_Request $req){
                    $q    = trim((string)$req->get_param('q'));
                    $days = max(7, min(90, intval($req->get_param('days') ?? 30)));
                    if ($q==='') return new \WP_Error('bad_request','Missing question',['status'=>400]);

                    $snapReq = new \WP_REST_Request('GET', '/miro/v1/analytics/ui/snapshot');
                    $snapReq->set_param('days', $days);
                    $snap = rest_do_request($snapReq)->get_data();
                    $k = $snap['kpis'] ?? [];
                    $prompt = "You are an SEO assistant. Answer briefly using site KPIs when helpful (window={$days}d).\n".
                              "KPIs: traffic={$k['search_traffic']}, impressions={$k['search_impressions']}, total_keywords={$k['total_keywords']}, ctr={$k['ctr']}%, position={$k['position']}.\n".
                              "Question: {$q}\nGive 1–2 concrete actions if relevant.";
                    return ['answer'=> miro_ai_complete_inline($prompt, 'analytics_ask')];
                }
            ]);


            // ===== Primary Sync endpoint (always pulls 90d to cache; UI filters window) =====
            register_rest_route('miro/v1', '/analytics/ui/sync', [
                'methods'  => 'POST',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(\WP_REST_Request $req){

                    $prop = get_option(self::OPT_PROPERTY, []);
                    $uri  = (string)($prop['uri'] ?? '');
                    if ($uri === '') {
                        return new \WP_Error('no_property', 'No GSC property selected. Go to “Miro AI SEO → Google Search Console” and pick one.', ['status'=>400]);
                    }

                    $s = get_option(self::OPT_SETTINGS, []);
                    if (empty($s['refresh_token']) || empty($s['client_id']) || empty($s['client_secret'])) {
                        return new \WP_Error('not_connected', 'GSC not connected. Complete the Google sign-in first.', ['status'=>403]);
                    }

                    $DAYS = 90;

                    $cache = get_option(self::OPT_CACHE);
                    $meta  = is_array($cache) ? ($cache['meta'] ?? []) : [];
                    $last  = isset($meta['last_synced']) ? strtotime($meta['last_synced']) : 0;
                    $now   = time();

                    if ($last && ($now - $last) < 6 * HOUR_IN_SECONDS && intval($meta['range_days'] ?? 0) === $DAYS) {
                        return [
                            'ok'          => true,
                            'skipped'     => true,
                            'message'     => "Cache is fresh (within 6h) for last {$DAYS} days.",
                            'last_synced' => $meta['last_synced'],
                            'rows'        => isset($cache['rows']) ? count($cache['rows']) : 0,
                            'daily'       => isset($cache['daily']) ? count($cache['daily']) : 0,
                        ];
                    }

                    $res = self::sync_last_n_days($DAYS);
                    if (is_wp_error($res)) return $res;

                    return [
                        'ok'          => true,
                        'message'     => "Synced successfully ({$DAYS} days).",
                        'rows'        => intval($res['rows'] ?? 0),
                        'daily'       => intval($res['daily'] ?? 0),
                        'last_synced' => sanitize_text_field($res['last_synced'] ?? ''),
                    ];
                }
            ]);

            // ===== Legacy aliases for old UI (keep working) =====
            register_rest_route('miro/v1', '/gsc/sync/start', [
                'methods'  => 'POST',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(\WP_REST_Request $req){
                    $prop = get_option(self::OPT_PROPERTY, []);
                    $uri  = (string)($prop['uri'] ?? '');
                    if ($uri === '') return new \WP_Error('no_property', 'No GSC property selected.', ['status'=>400]);

                    $s = get_option(self::OPT_SETTINGS, []);
                    if (empty($s['refresh_token']) || empty($s['client_id']) || empty($s['client_secret'])) {
                        return new \WP_Error('not_connected', 'GSC not connected.', ['status'=>403]);
                    }

                    $res  = self::sync_last_n_days(90);
                    if (is_wp_error($res)) return $res;

                    set_transient('miro_sync_last', ['done'=>true,'step'=>'Done','progress'=>100], 5*MINUTE_IN_SECONDS);
                    return ['ok'=>true,'job_id'=>'miro-sync-now'];
                }
            ]);
            register_rest_route('miro/v1', '/gsc/sync/status', [
                'methods'  => 'GET',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(){
                    $s = get_transient('miro_sync_last');
                    return $s ?: ['done'=>true,'step'=>'Done','progress'=>100];
                }
            ]);

            // ===== First-run sync: state (for UX "Syncing…") and manual trigger =====
            register_rest_route('miro/v1', '/gsc/sync-state', [
                'methods'  => 'GET',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(){
                    return self::get_sync_state();
                }
            ]);

            register_rest_route('miro/v1', '/gsc/first-sync-now', [
                'methods'  => 'POST',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(){
                    $scheduled = self::schedule_first_sync('manual');
                    return [
                        'ok'        => true,
                        'message'   => $scheduled ? __('First sync queued. Data will appear in 1–3 minutes.', 'miro-ai-seo') : __('Sync already queued or running.', 'miro-ai-seo'),
                        'scheduled' => $scheduled,
                    ];
                }
            ]);

            // Run sync in this request (for connect flow so data is ready without relying on cron spawn)
            register_rest_route('miro/v1', '/gsc/run-sync-now', [
                'methods'  => 'POST',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(){
                    if (get_transient('miro_gsc_first_sync_running')) {
                        return ['ok' => true, 'status' => 'running', 'message' => __('Sync already in progress.', 'miro-ai-seo')];
                    }
                    // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Long-running sync requires extended execution time.
                    @set_time_limit(600);
                    self::run_first_sync(['reason' => 'run_sync_now']);
                    $state = self::get_sync_state();
                    return [
                        'ok'     => true,
                        'status' => $state['status'],
                        'message' => $state['status'] === 'done' ? __('Sync complete. Data is ready.', 'miro-ai-seo') : ($state['last_error'] ?: __('Sync finished.', 'miro-ai-seo')),
                    ];
                }
            ]);

            register_rest_route('miro/v1', '/gsc/clear-cache', [
                'methods'  => 'POST',
                'permission_callback' => function(){ return current_user_can(_miro_cap()); },
                'callback' => function(){
                    self::clear_analytics_cache();
                    return ['ok' => true, 'message' => __('All analytics cache cleared. Click Sync now to fetch fresh data.', 'miro-ai-seo')];
                }
            ]);

        });
    }

    /**
     * Get sync state for UX (idle|queued|running|done|error).
     */
    public static function get_sync_state(): array
    {
        $state = get_option(self::OPT_SYNC_STATE, []);
        $default = [
            'status'               => 'idle',
            'started_at'           => '',
            'finished_at'          => '',
            'last_error'           => '',
            'last_synced_range_days' => 0,
            'step'                 => '',
            'progress'             => 0,
        ];
        $out = array_merge($default, is_array($state) ? $state : []);
        $prop = get_option(self::OPT_PROPERTY, []);
        $s    = get_option(self::OPT_SETTINGS, []);
        $out['gsc_connected'] = !empty($prop['uri']) && !empty($s['refresh_token']) && !empty($s['client_id']) && !empty($s['client_secret']);
        return $out;
    }

    /**
     * Update sync state (safe merge).
     */
    public static function update_sync_state(array $data): void
    {
        $current = self::get_sync_state();
        update_option(self::OPT_SYNC_STATE, array_merge($current, $data), false);
    }

    /**
     * Clear all GSC analytics cache and tab caches so the next sync fetches fresh data.
     * Does not remove connection (property, tokens) or tracked keywords.
     */
    public static function clear_analytics_cache(): void
    {
        delete_option(self::OPT_CACHE);
        delete_option(self::OPT_CACHE_READY);
        delete_option(self::OPT_SYNC_STATE);
        delete_transient('miro_gsc_first_sync_running');
        delete_transient(self::FIRST_SYNC_LOCK);
        delete_transient('miro_sync_last');
        delete_option('miro_gsc_site_series');
        delete_option('miro_gsc_country_series');
        delete_option('miro_rank_today_site');
        self::update_sync_state(['status' => 'idle', 'step' => '', 'progress' => 0, 'last_error' => '', 'last_synced_range_days' => 0]);
    }

    /** No-op for removed debug log (kept for backward compatibility of call sites). */
    public static function debug_log(string $message, array $context = []): void
    {
    }

    /**
     * Schedule first-run sync (WP-Cron). Dedupe: do not schedule if already scheduled or running.
     * Call after OAuth callback or property save.
     *
     * @param string $reason 'first_connect'|'property_saved'|'manual'
     * @return bool True if event was scheduled, false if skipped (dedupe or no property)
     */
    public static function schedule_first_sync(string $reason = 'first_connect'): bool
    {
        $lock = get_transient(self::FIRST_SYNC_LOCK);
        if ($lock) {
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: skipped (lock present), reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return false;
        }

        $state = self::get_sync_state();
        if (($state['status'] === 'running' || $state['status'] === 'queued')) {
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: skipped (already ' . $state['status'] . '), reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return false;
        }

        $prop = get_option(self::OPT_PROPERTY, []);
        $uri  = (string)($prop['uri'] ?? '');
        if ($uri === '' && defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC first sync: scheduling (no property yet), reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }

        // Schedule for now so it runs as soon as cron is triggered
        $timestamp = time();
        $scheduled = wp_schedule_single_event($timestamp, self::CRON_FIRST_SYNC, [['reason' => $reason]]);
        if (is_wp_error($scheduled)) {
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: wp_schedule_single_event failed: ' . $scheduled->get_error_message()); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return false;
        }

        set_transient(self::FIRST_SYNC_LOCK, 1, self::FIRST_SYNC_LOCK_TTL);
        self::update_sync_state(['status' => 'queued', 'started_at' => '', 'finished_at' => '', 'last_error' => '']);
        if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC first sync: scheduled now, reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }

        // Trigger cron immediately so sync starts in a background request (don't wait for random page load)
        self::spawn_cron_now();
        return true;
    }

    /**
     * Spawn a non-blocking request to wp-cron.php so due events (e.g. first sync) run immediately.
     */
    public static function spawn_cron_now(): void
    {
        $url = add_query_arg('doing_wp_cron', (string) time(), site_url('wp-cron.php'));
        wp_remote_post($url, [
            'timeout'   => 0.01,
            'blocking'  => false,
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core WordPress filter for local SSL verification.
            'sslverify' => apply_filters('https_local_ssl_verify', true),
        ]);
    }

    /**
     * Cron handler: run 90-day sync. No-op if no property or auth. Updates sync state and cache_ready.
     */
    public static function run_first_sync($args = []): void
    {
        // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Long-running sync requires extended execution time.
        @set_time_limit(600);
        $reason = is_array($args) ? (string)($args['reason'] ?? 'cron') : 'cron';
        self::debug_log('run_first_sync started', ['reason' => $reason]);

        $prop = get_option(self::OPT_PROPERTY, []);
        $uri  = (string)($prop['uri'] ?? '');
        $s    = get_option(self::OPT_SETTINGS, []);

        if ($uri === '') {
            self::debug_log('Sync skipped: no property');
            delete_transient(self::FIRST_SYNC_LOCK);
            self::update_sync_state(['status' => 'error', 'last_error' => __('Select a GSC property first (Miro AI SEO → Google Search Console).', 'miro-ai-seo'), 'finished_at' => current_time('mysql')]);
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: no property, skipping. reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return;
        }
        if (empty($s['refresh_token']) || empty($s['client_id']) || empty($s['client_secret'])) {
            self::debug_log('Sync skipped: not connected (missing refresh_token or client credentials)');
            delete_transient(self::FIRST_SYNC_LOCK);
            self::update_sync_state(['status' => 'error', 'last_error' => 'Not connected', 'finished_at' => current_time('mysql')]);
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: not connected, skipping. reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return;
        }
        self::debug_log('Property and auth OK', ['property' => $uri]);

        // Prevent concurrent runs
        if (get_transient('miro_gsc_first_sync_running')) {
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: already running, skipping.'); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return;
        }
        set_transient('miro_gsc_first_sync_running', 1, 600); // 10 min max

        self::update_sync_state([
            'status'     => 'running',
            'started_at' => current_time('mysql'),
            'finished_at' => '',
            'last_error' => '',
            'step'       => __('Preparing…', 'miro-ai-seo'),
            'progress'   => 0,
        ]);

        self::debug_log('Calling sync_last_n_days(90)');
        if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC first sync: starting, property=' . $uri . ', reason=' . $reason); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }

        $res = self::sync_last_n_days(90);

        delete_transient('miro_gsc_first_sync_running');
        delete_transient(self::FIRST_SYNC_LOCK);

        if (is_wp_error($res)) {
            $msg = $res->get_error_code() . ': ' . $res->get_error_message();
            self::debug_log('Sync failed', ['error' => $msg]);
            self::update_sync_state([
                'status'                 => 'error',
                'finished_at'            => current_time('mysql'),
                'last_error'             => $msg,
                'last_synced_range_days' => 0,
                'step'                   => '',
                'progress'               => 0,
            ]);
            if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
                error_log('MIRO GSC first sync: error ' . $msg); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
            }
            return;
        }

        $rows  = (int)($res['rows'] ?? 0);
        $daily = (int)($res['daily'] ?? 0);
        self::debug_log('Sync done', ['rows' => $rows, 'daily' => $daily]);
        self::update_sync_state([
            'status'                 => 'done',
            'finished_at'            => current_time('mysql'),
            'last_error'             => '',
            'last_synced_range_days' => 90,
            'step'                   => '',
            'progress'               => 100,
        ]);
        update_option(self::OPT_CACHE_READY, 1, false);

        if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC first sync: done, rows=' . $rows . ', daily=' . $daily); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }
    }

    /** ===== helpers: date windowing on cached rows ===== */
    private static function compute_window(array $rows, int $days): array {
        $days = max(7, min(90, $days));
        $dates = [];
        foreach ($rows as $r) {
            $d = isset($r['date']) ? (string)$r['date'] : '';
            if ($d !== '') $dates[$d] = true;
        }
        if (!$dates) return [null, null];
        $list = array_keys($dates);
        sort($list);
        $last = end($list);
        $lastTs = strtotime($last);
        $cutTs  = $lastTs - 86400 * ($days - 1);
        return [$last, gmdate('Y-m-d', $cutTs)];
    }

    public static function filter_rows_by_days(array $rows, int $days): array {
        [$last, $cut] = self::compute_window($rows, $days);
        if (!$last || !$cut) return $rows;
        $out = [];
        foreach ($rows as $r) {
            $d = isset($r['date']) ? (string)$r['date'] : '';
            if ($d !== '' && $d >= $cut && $d <= $last) $out[] = $r;
        }
        return $out;
    }

    // ✅ NEW: filter DAILY totals by days (source of truth for chart)
    private static function filter_daily_by_days(array $daily, int $days): array {
        $days = max(7, min(90, $days));
        if (!$daily) return [[], null, null];

        $map = [];
        foreach ($daily as $r) {
            $d = (string)($r['date'] ?? '');
            if ($d !== '') $map[$d] = $r;
        }
        $dates = array_keys($map);
        sort($dates);
        if (!$dates) return [[], null, null];

        $to   = end($dates);
        $toTs = strtotime($to);
        $fromTs = $toTs - 86400 * ($days - 1);
        $from = gmdate('Y-m-d', $fromTs);

        $out = [];
        foreach ($dates as $d) {
            if ($d >= $from && $d <= $to) $out[] = $map[$d];
        }

        return [$out, $from, $to];
    }

    // ✅ NEW: filter detailed rows by exact bounds
    private static function filter_rows_by_bounds(array $rows, ?string $from, ?string $to): array {
        if (!$from || !$to) return $rows;
        $out = [];
        foreach ($rows as $r) {
            $d = isset($r['date']) ? (string)$r['date'] : '';
            if ($d !== '' && $d >= $from && $d <= $to) $out[] = $r;
        }
        return $out;
    }


    /**
     * Date range for GSC queries: always ends at yesterday, never today.
     * Today's data is incomplete and changes throughout the day — including it would confuse users.
     * So we only fetch up to yesterday (last stable data from Search Console).
     */
    public static function date_range(int $days): array
    {
        $tz    = wp_timezone();
        $today = new \DateTime('today', $tz);
        $end   = (clone $today)->modify('-1 day'); // Never include today (incomplete/changing)
        $start = (clone $end)->modify('-'.($days-1).' days');
        return [$start, $end];
    }

    public static function prev_range(\DateTime $start, int $days): array
    {
        $end   = (clone $start)->modify('-1 day');
        $begin = (clone $end)->modify('-'.($days-1).' days');
        return [$begin,$end];
    }

    /** Normalize GSC country to uppercase 2-letter (ISO 3166-1 alpha-2). API may return 2-letter (us) or 3-letter (usa). */
    public static function normalize_country_code(string $raw): string
    {
        $raw = strtoupper(trim($raw));
        if ($raw === '') return '';
        if (strlen($raw) === 2) return $raw;
        $alpha3_to_2 = [
            'USA'=>'US','GBR'=>'GB','DEU'=>'DE','FRA'=>'FR','ITA'=>'IT','ESP'=>'ES','CAN'=>'CA','AUS'=>'AU',
            'IND'=>'IN','JPN'=>'JP','KOR'=>'KR','BRA'=>'BR','MEX'=>'MX','NLD'=>'NL','RUS'=>'RU','POL'=>'PL',
            'TUR'=>'TR','IDN'=>'ID','ZAF'=>'ZA','SWE'=>'SE','NOR'=>'NO','DNK'=>'DK','FIN'=>'FI','IRL'=>'IE',
            'BEL'=>'BE','AUT'=>'AT','CHE'=>'CH','PRT'=>'PT','GRC'=>'GR','ROU'=>'RO','CZE'=>'CZ','HUN'=>'HU',
            'ISR'=>'IL','EGY'=>'EG','SAU'=>'SA','ARE'=>'AE','SGP'=>'SG','MYS'=>'MY','THA'=>'TH','PHL'=>'PH',
            'VNM'=>'VN','ARG'=>'AR','CHL'=>'CL','COL'=>'CO','NZL'=>'NZ','PAK'=>'PK','BGD'=>'BD','NGA'=>'NG',
        ];
        return $alpha3_to_2[$raw] ?? $raw;
    }

    public static function avg_pos_weighted(array $rows)
    {
        $w=0;$s=0;
        foreach($rows as $r){
            $imp=floatval($r['impressions']??0);
            $pos=floatval($r['position']??0);
            if($imp<=0) continue;
            $w+=$imp; $s+=$pos*$imp;
        }
        return $w>0 ? $s/$w : 0.0;
    }

    /** ===== Pull fresh data and write to cache (90d, in chunks to avoid 25k row cap) =====
     * ✅ FIX: We now also fetch DAILY TOTALS using only ['date'] and store them under 'daily'
     */
    public static function sync_last_n_days(int $days = 90) {
        $uri = (string)((get_option(self::OPT_PROPERTY, [])['uri'] ?? ''));
        self::debug_log('sync_last_n_days', ['days' => $days, 'property' => $uri]);
        if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC sync: property=' . $uri . ', days=' . $days); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }

        // Use the helper so we stay aligned with the UI (yesterday back N days)
        [$start, $end] = self::date_range($days);
        self::debug_log('Date range', ['from' => $start->format('Y-m-d'), 'to' => $end->format('Y-m-d')]);
        if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC sync: date_range from=' . $start->format('Y-m-d') . ' to=' . $end->format('Y-m-d')); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }

        // ----------------------------
        // (A) DAILY TOTALS (date only) → MATCHES GSC UI CHART
        // ----------------------------
        $dailyMap = [];

        $chunkStart = clone $start;
        while ($chunkStart <= $end) {
            $chunkEnd = (clone $chunkStart)->modify('+13 days'); // 14-day chunks
            if ($chunkEnd > $end) $chunkEnd = clone $end;

            $payloadDaily = self::payload_dimensions(['date'], [], 1000);
            $payloadDaily['dataState'] = 'all';

            $dataDaily = self::gsc_query($chunkStart, $chunkEnd, $payloadDaily);
            if (is_wp_error($dataDaily)) return $dataDaily;
            self::debug_log('Daily chunk', ['from' => $chunkStart->format('Y-m-d'), 'to' => $chunkEnd->format('Y-m-d'), 'rows' => count($dataDaily['rows'] ?? [])]);

            foreach (($dataDaily['rows'] ?? []) as $r) {
                $keys = $r['keys'] ?? [];
                $rawDate = (string)($keys[0] ?? '');
                $d = $rawDate ? gmdate('Y-m-d', strtotime($rawDate)) : '';
                if ($d === '') continue;

                $c = (float)($r['clicks'] ?? 0);
                $i = (float)($r['impressions'] ?? 0);
                $p = (float)($r['position'] ?? 0);

                $dailyMap[$d] = [
                    'date'        => $d,
                    'clicks'      => $c,
                    'impressions' => $i,
                    'ctr'         => ($i > 0) ? ($c / $i) : 0.0, // ratio 0..1
                    'position'    => $p,
                ];
            }

            $chunkStart->modify('+14 days');
        }

        ksort($dailyMap);
        $daily = array_values($dailyMap);

        self::update_sync_state([
            'step'     => __('Fetching keywords…', 'miro-ai-seo'),
            'progress' => 15,
        ]);

        // ----------------------------
        // (B) DETAILED ROWS (date,query,page,device) → TABLES / KEYWORDS / etc
        // Paginate with startRow so we fetch ALL rows (GSC returns max 25k per request).
        // ----------------------------
        $allRows = [];
        $maxRowsPerRequest = 25000;

        $chunkStart = clone $start;
        $totalChunks = 0;
        $temp = clone $start;
        while ($temp <= $end) {
            $totalChunks++;
            $temp->modify('+7 days');
        }
        if ($totalChunks < 1) $totalChunks = 1;
        $chunkIndex = 0;

        while ($chunkStart <= $end) {
            $chunkEnd = (clone $chunkStart)->modify('+6 days');
            if ($chunkEnd > $end) $chunkEnd = clone $end;

            $chunkIndex++;
            $pct = 15 + (int) round(85 * ($chunkIndex - 1) / $totalChunks);
            $pct = min(99, max(15, $pct));
            self::update_sync_state([
                /* translators: 1: current week number, 2: total number of weeks */
                'step'     => sprintf(__('Fetching week %1$d of %2$d…', 'miro-ai-seo'), $chunkIndex, $totalChunks),
                'progress' => $pct,
            ]);

            $startRow = 0;
            $batchNum = 0;
            do {
                $batchNum++;
                // ✅ REQUIRED DIMENSIONS: date, query, page, device, country (order = keys[] order in response)
                $payload = self::payload_dimensions(
                    ['date', 'query', 'page', 'device', 'country'],
                    [],
                    $maxRowsPerRequest
                );
                $payload['dataState'] = 'all';
                $payload['startRow']   = $startRow;
                // Ensure dimensions are never stripped (explicit for Keywords tab country filter)
                $payload['dimensions'] = ['date', 'query', 'page', 'device', 'country'];

                $data = self::gsc_query($chunkStart, $chunkEnd, $payload);
                if (is_wp_error($data)) return $data;
                $chunkRows = $data['rows'] ?? [];
                $numRows   = count($chunkRows);
                self::debug_log('Keywords chunk', ['week' => $chunkIndex, 'batch' => $batchNum, 'rows' => $numRows, 'startRow' => $startRow]);

                // Update progress so user sees activity (each batch can take 10–30s)
                $rowsSoFar = count($allRows) + $numRows;
                $chunkProgress = ($batchNum === 1 && $numRows < $maxRowsPerRequest) ? 1.0 : min(1.0, 0.25 * $batchNum);
                $pctNow = 15 + (int) round(85 * (($chunkIndex - 1) + $chunkProgress) / $totalChunks);
                $pctNow = min(99, max(15, $pctNow));
                self::update_sync_state([
                    'step'     => sprintf(
                        /* translators: 1: current week number, 2: total weeks, 3: batch number, 4: formatted row count */
                        __('Fetching week %1$d of %2$d… batch %3$d, %4$s rows', 'miro-ai-seo'),
                        $chunkIndex,
                        $totalChunks,
                        $batchNum,
                        number_format_i18n($rowsSoFar)
                    ),
                    'progress' => $pctNow,
                ]);

                foreach ($chunkRows as $r) {
                $keys     = $r['keys'] ?? [];
                
                // Validate that we have the expected number of keys (5: date, query, page, device, country)
                if (count($keys) < 5) {
                    // Log warning but continue - might be old API response format
                    error_log('GSC API: Expected 5 keys but got ' . count($keys) . '. Dimensions: ' . implode(', ', ['date', 'query', 'page', 'device', 'country'])); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- API format diagnostic.
                }
                
                $rawDate  = (string)($keys[0] ?? '');
                $normDate = $rawDate ? gmdate('Y-m-d', strtotime($rawDate)) : '';

                // Normalize country: GSC may return 2-letter (us, gb) or 3-letter (usa, gbr). Store as uppercase 2-letter for Keywords tab.
                $countryRaw = trim((string)($keys[4] ?? ''));
                $countryCode = self::normalize_country_code($countryRaw);
                
                // ✅ Each row MUST include: date, query, page, device, country, clicks, impressions, ctr, position
                $allRows[] = [
                    'date'        => $normDate,
                    'query'       => (string)($keys[1] ?? ''),
                    'page'        => (string)($keys[2] ?? ''),
                    'device'      => (string)($keys[3] ?? ''),
                    'country'     => $countryCode,
                    'clicks'      => (float)($r['clicks'] ?? 0),
                    'impressions' => (float)($r['impressions'] ?? 0),
                    'ctr'         => (float)($r['ctr'] ?? 0),
                    'position'    => (float)($r['position'] ?? 0),
                ];
                }

                $startRow += $maxRowsPerRequest;
            } while ($numRows >= $maxRowsPerRequest);

            // Move forward 7 days (no overlap)
            $chunkStart->modify('+7 days');
        }

        self::update_sync_state([
            'step'     => __('Saving cache…', 'miro-ai-seo'),
            'progress' => 99,
        ]);

        // De-duplicate rows in case of any overlap (safety)
        $map = [];
        foreach ($allRows as $row) {
            $key = $row['date'].'|'.$row['query'].'|'.$row['page'].'|'.$row['device'].'|'.$row['country'];

            if (!isset($map[$key])) {
                $map[$key] = $row;
                continue;
            }

            // Merge metrics
            $a = $map[$key];

            $clicksA = (float)$a['clicks'];
            $imprA   = (float)$a['impressions'];
            $posA    = (float)$a['position'];

            $clicksB = (float)$row['clicks'];
            $imprB   = (float)$row['impressions'];
            $posB    = (float)$row['position'];

            $clicks = $clicksA + $clicksB;
            $impr   = $imprA   + $imprB;

            $pos = 0.0;
            if ($impr > 0) {
                $pos = (($posA * $imprA) + ($posB * $imprB)) / max(1.0, $impr);
            }

            $ctr = ($impr > 0) ? ($clicks / $impr) : 0.0;

            $map[$key] = [
                'date'        => $row['date'],
                'query'       => $row['query'],
                'page'        => $row['page'],
                'device'      => $row['device'],
                'country'     => $row['country'],
                'clicks'      => $clicks,
                'impressions' => $impr,
                'ctr'         => $ctr,
                'position'    => $pos,
            ];
        }

        $rows = array_values($map);

        // Work out real from/to dates from DAILY (truth) first
        $fromDate = $daily ? ($daily[0]['date'] ?? $start->format('Y-m-d')) : $start->format('Y-m-d');
        $toDate   = $daily ? ($daily[count($daily)-1]['date'] ?? $end->format('Y-m-d')) : $end->format('Y-m-d');

        $save = [
            'meta' => [
                'last_synced' => current_time('mysql'),
                'range_days'  => $days,
                'from'        => $fromDate,
                'to'          => $toDate,
            ],
            'daily' => $daily, // ✅ NEW
            'rows'  => $rows,
        ];

        update_option(self::OPT_CACHE, $save, false);
        self::debug_log('Cache written', ['rows' => count($rows), 'daily' => count($daily)]);

        if (defined('MIRO_DEBUG') && MIRO_DEBUG) {
            error_log('MIRO GSC sync: cache written rows=' . count($rows) . ', daily=' . count($daily)); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- MIRO_DEBUG only.
        }
        // Temporary: targeted log for Keywords cache investigation
        error_log('MIRO GSC sync finished: option=' . self::OPT_CACHE . ', property=' . $uri . ', from=' . $fromDate . ', to=' . $toDate . ', rows=' . count($rows) . ', daily=' . count($daily)); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Targeted diagnostic.

        return [
            'rows'        => count($rows),
            'daily'       => count($daily),
            'last_synced' => $save['meta']['last_synced'],
        ];
    }

    public static function payload_dimensions(array $dims, array $args = [], int $rowLimit=250, array $queryFilter = []): array
    {
        $payload = [
            'dimensions'      => $dims,
            'rowLimit'        => $rowLimit,
            'aggregationType' => 'auto',
            'type'            => 'web',  // search filter; do NOT send searchType (API rejects both)
            'dataState'       => 'all',
        ];
        $dimFilters = [];
        if (!empty($args['device']))  $dimFilters[] = ['dimension' => 'device',  'operator'=>'equals', 'expression'=>$args['device']];
        if (!empty($args['country'])) $dimFilters[] = ['dimension' => 'country', 'operator'=>'equals', 'expression'=>$args['country']];
        if (!empty($queryFilter)) foreach ($queryFilter as $q) $dimFilters[] = ['dimension' => 'query', 'operator'=>'equals', 'expression'=>$q];
        if ($dimFilters) $payload['dimensionFilterGroups'] = [[ 'filters' => $dimFilters ]];
        return $payload;
    }

    /**
     * Remove searchType from payload (and any nested arrays). API returns 400 if both type and searchType are sent.
     */
    private static function gsc_strip_search_type(array $payload): array
    {
        unset($payload['searchType']);
        foreach ($payload as $k => $v) {
            if (is_array($v)) {
                $payload[$k] = self::gsc_strip_search_type($v);
            }
        }
        return $payload;
    }

    public static function gsc_query(\DateTime $start, \DateTime $end, array $payload)
    {
        $uri = (string)((get_option(self::OPT_PROPERTY, [])['uri'] ?? ''));
        if ($uri === '') return new \WP_Error('no_property', 'Select a property first.', ['status'=>400]);

        $auth = self::ensure_token();
        if (is_wp_error($auth)) {
            self::debug_log('ensure_token failed', ['code' => $auth->get_error_code(), 'message' => $auth->get_error_message()]);
            return $auth;
        }
        $token = $auth['access_token'];

        $payload['startDate'] = $start->format('Y-m-d');
        $endYmd = $end->format('Y-m-d');
        $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
        $todayYmd = (new \DateTime('today', $tz))->format('Y-m-d');
        if ($endYmd === $todayYmd) {
            $endYmd = (new \DateTime('today', $tz))->modify('-1 day')->format('Y-m-d');
        }
        $payload['endDate'] = $endYmd;
        if (!isset($payload['rowLimit'])) $payload['rowLimit'] = 250;
        if (!isset($payload['type'])) $payload['type'] = 'web';
        // API returns 400 if both type and searchType are present — strip searchType everywhere
        $payload = self::gsc_strip_search_type($payload);

        $url = sprintf(self::GSC_QUERY_URL_TMPL, rawurlencode($uri));
        self::debug_log('GSC API request', ['startDate' => $payload['startDate'], 'endDate' => $payload['endDate'], 'dimensions' => $payload['dimensions'] ?? [], 'rowLimit' => $payload['rowLimit'] ?? 0]);
        $res = wp_remote_post($url, [
            'timeout' => 60,
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'   => 'application/json',
                'Accept'        => 'application/json',
            ],
            'body' => wp_json_encode($payload),
        ]);
        if (is_wp_error($res)) {
            self::debug_log('GSC request network error', ['message' => $res->get_error_message()]);
            return new \WP_Error('gsc_network', 'Could not reach Google: ' . $res->get_error_message(), ['status' => 502]);
        }
        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        $data = json_decode($body, true);
        if ($code !== 200 || !is_array($data)) {
            self::debug_log('GSC API error', ['code' => $code, 'body_preview' => substr($body, 0, 300)]);
            $msg = 'GSC query failed (HTTP ' . $code . ').';
            $err = is_array($data) ? ($data['error'] ?? null) : null;
            if (is_array($err) && !empty($err['message'])) {
                $msg .= ' ' . trim((string) $err['message']);
            } elseif (is_array($err) && !empty($err['errors'][0]['message'])) {
                $msg .= ' ' . trim((string) $err['errors'][0]['message']);
            } elseif ($body !== '' && strlen($body) < 500) {
                $msg .= ' ' . trim(preg_replace('/\s+/', ' ', wp_strip_all_tags($body)));
            }
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                error_log('MIRO GSC query failed: ' . $msg . ' Body: ' . substr($body, 0, 1000)); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- WP_DEBUG_LOG only.
            }
            return new \WP_Error('gsc_error', $msg, ['status' => (int) $code]);
        }
        $rowCount = isset($data['rows']) ? count($data['rows']) : 0;
        self::debug_log('GSC API OK', ['rows' => $rowCount]);
        return $data;
    }

    public static function ensure_token()
    {
        $s   = get_option(self::OPT_SETTINGS, []);
        $acc = (string)($s['access_token'] ?? '');
        $ref = (string)($s['refresh_token'] ?? '');
        $cid = (string)($s['client_id'] ?? '');
        $csc = (string)($s['client_secret'] ?? '');
        $exp = intval($s['access_exp'] ?? 0);

        if ($ref==='' || $cid==='' || $csc==='') {
            self::debug_log('ensure_token: not connected (missing ref/cid/csc)');
            return new \WP_Error('not_connected','Not connected.',['status'=>403]);
        }
        if ($acc && $exp > time()+90) {
            self::debug_log('ensure_token: using cached access_token');
            return ['access_token'=>$acc];
        }
        self::debug_log('ensure_token: refreshing token');
        $res = wp_remote_post(self::GOOGLE_TOKEN_URL, [
            'timeout'=>15,
            'headers'=>['Content-Type'=>'application/x-www-form-urlencoded'],
            'body'=>http_build_query([
                'client_id'=>$cid,'client_secret'=>$csc,'grant_type'=>'refresh_token','refresh_token'=>$ref
            ],'', '&', PHP_QUERY_RFC3986)
        ]);
        if (is_wp_error($res)) {
            return new \WP_Error('auth_network', 'Could not reach Google: ' . $res->get_error_message(), ['status'=>502]);
        }
        $code = wp_remote_retrieve_response_code($res);
        $bodyRaw = wp_remote_retrieve_body($res);
        $body = json_decode($bodyRaw, true);
        if ($code !== 200 || empty($body['access_token'])) {
            $msg = 'Token refresh failed (HTTP ' . $code . ').';
            if (is_array($body) && !empty($body['error_description'])) {
                $msg .= ' ' . trim((string) $body['error_description']);
            } elseif (is_array($body) && !empty($body['error'])) {
                $msg .= ' ' . trim((string) $body['error']);
            }
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                error_log('MIRO GSC token refresh: ' . $msg); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- WP_DEBUG_LOG only.
            }
            return new \WP_Error('auth_error', $msg, ['status'=>401]);
        }
        $s['access_token'] = (string)$body['access_token'];
        $s['access_exp']   = time()+max(60,intval($body['expires_in']??3600))-60;
        update_option(self::OPT_SETTINGS, $s, false);
        self::debug_log('ensure_token: token refreshed OK');
        return ['access_token'=>$s['access_token']];
    }

    /** ===== Page Render ===== */
    public static function render(): void
    {
        if (!current_user_can(_miro_cap())) wp_die(esc_html__('You do not have permission.', 'miro-ai-seo'));

        $propOpt = get_option(self::OPT_PROPERTY, []);
        $propUri = (string)($propOpt['uri'] ?? '');
        $set     = get_option(self::OPT_SETTINGS, []);
        $hasRef  = !empty($set['refresh_token']);

        echo '<div class="wrap"><h1 style="display:none;">Miro GSC Analytics</h1>';
        echo '<div class="miro-ana-wrap">';

        // Connection status (hidden by default, shown if not connected)
        $hasClientCreds = !empty($set['client_id']) && !empty($set['client_secret']);
        $fullyConnected = $propUri && $hasRef && $hasClientCreds;
        if (!$fullyConnected) {
            $gscUrl = esc_url(admin_url('admin.php?page=miro-gsc'));
            echo '<div style="background:#fef3c7;border:1px solid #f59e0b;border-radius:12px;padding:20px 24px;margin-bottom:24px;">';
            echo '<div style="font-weight:600;margin-bottom:8px;">Connect Google Search Console to see data</div>';
            echo '<div style="color:#92400e;margin-bottom:12px;">';
            if (!$propUri) {
                echo 'Select a property in Google Search Console.';
            } elseif (!$hasRef) {
                echo 'Complete the OAuth connection to connect your account.';
            } else {
                echo 'Connection credentials are incomplete; reconnect if you still see zeros after syncing.';
            }
            echo '</div>';
            echo '<a href="'.esc_attr($gscUrl).'" class="button button-primary">Open Google Search Console</a>';
            echo '</div>';
        }
        if (false) { // replaced by $fullyConnected notice above
            echo '<div style="background:#fff;border:1px solid rgba(0,0,0,0.06);border-radius:16px;padding:20px;margin-bottom:24px;box-shadow:0 8px 24px rgba(0,0,0,0.06);">';
            echo '<div><strong>Property:</strong> '.($propUri ? esc_html($propUri) : '<span style="color:#d63638;">None</span>').'</div>';
            echo '<div><strong>Auth:</strong> '.($hasRef ? '<span style="color:#1a7f37;">Connected</span>' : '<span style="color:#d63638;">Not connected</span>').'</div>';
        echo '<div class="muted">Use “Miro AI SEO → Google Search Console” to connect & pick a property.</div>';
            echo '</div>';
        }

        if (!$propUri) {
            echo '</div></div>';
            return;
        }

        // ===== MOMENTUM HEADER =====
        echo '<div class="miro-momentum-header">';
        echo '<div class="miro-momentum-brand">';
        echo '<div class="miro-momentum-logo">M</div>';
        echo '<div class="miro-momentum-brand-text">MIRO SEO</div>';
        echo '</div>';
        echo '<div class="miro-momentum-header-controls">';
        echo '<div class="miro-momentum-sync-info" id="miro-ana-last">'.esc_html__('Last synced: —','miro-ai-seo').'</div>';
        echo '<select id="miro-range" class="miro-momentum-range-select">';
        echo '<option value="7">7 days</option>';
        echo '<option value="15">15 days</option>';
        echo '<option value="30" selected>30 days</option>';
        echo '<option value="90">90 days</option>';
        echo '</select>';
        echo '<button id="miro-sync-now" class="miro-momentum-sync-btn">'.esc_html__('Sync now','miro-ai-seo').'</button>';
        echo '<button type="button" id="miro-clear-cache" class="button button-secondary" style="margin-left:8px;">'.esc_html__('Clear all cache','miro-ai-seo').'</button>';
        echo '</div>';
        echo '</div>';

        // ===== MAIN CHART CARD =====
        echo '<div class="miro-momentum-chart-card">';
        echo '<div class="miro-momentum-chart-title" id="miro-momentum-chart-title">'.esc_html__('Your SEO momentum (last 7 days)','miro-ai-seo').'</div>';
        echo '<div class="miro-momentum-chart-container" style="position:relative;">';
        echo '<canvas id="miro-ana-chart"></canvas>';
        echo '</div>';
        
        // Legend with checkboxes
        echo '<div class="miro-momentum-chart-legend">';
        echo '<div class="miro-momentum-legend-item active" data-series="clicks">';
        echo '<div class="miro-momentum-legend-checkbox"></div>';
        echo '<div class="miro-momentum-legend-dot" style="background:#D8B15A;"></div>';
        echo '<span class="miro-momentum-legend-label">Clicks</span>';
        echo '</div>';
        echo '<div class="miro-momentum-legend-item active" data-series="impressions">';
        echo '<div class="miro-momentum-legend-checkbox"></div>';
        echo '<div class="miro-momentum-legend-dot" style="background:#F4D03F;"></div>';
        echo '<span class="miro-momentum-legend-label">Impressions</span>';
        echo '</div>';
        echo '<div class="miro-momentum-legend-item active" data-series="ctr">';
        echo '<div class="miro-momentum-legend-checkbox"></div>';
        echo '<div class="miro-momentum-legend-dot" style="background:#9ca3af;"></div>';
        echo '<span class="miro-momentum-legend-label">Click Appeal</span>';
        echo '</div>';
        echo '<div class="miro-momentum-legend-item active" data-series="position">';
        echo '<div class="miro-momentum-legend-checkbox"></div>';
        echo '<div class="miro-momentum-legend-dot" style="border:2px dashed #6366f1;background:transparent;width:14px;height:14px;"></div>';
        echo '<span class="miro-momentum-legend-label">Ranking Strength</span>';
        echo '</div>';
        echo '</div>';
        
        // Insight line
        echo '<div class="miro-momentum-insight">';
        echo '<svg class="miro-momentum-insight-icon" width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M10 2C5.58 2 2 5.58 2 10C2 14.42 5.58 18 10 18C14.42 18 18 14.42 18 10C18 5.58 14.42 2 10 2ZM11 14H9V9H11V14ZM10 7.5C9.17 7.5 8.5 6.83 8.5 6C8.5 5.17 9.17 4.5 10 4.5C10.83 4.5 11.5 5.17 11.5 6C11.5 6.83 10.83 7.5 10 7.5Z" fill="#D8B15A"/>';
        echo '</svg>';
        echo '<div class="miro-momentum-insight-text" id="miro-momentum-insight">'.esc_html__('Loading insights...','miro-ai-seo').'</div>';
        echo '</div>';
        echo '</div>';

        // ===== KPI CARDS ROW =====
        echo '<div class="miro-momentum-kpi-row">';

        
        // Search Demand (using search_traffic)
        echo '<div class="miro-momentum-kpi-card">';
        echo '<svg class="miro-momentum-kpi-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M12 2L2 7L12 12L22 7L12 2Z" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '<path d="M2 17L12 22L22 17" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '<path d="M2 12L12 17L22 12" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '</svg>';
        echo '<div class="miro-momentum-kpi-label">Search Demand</div>';
        echo '<div class="miro-momentum-kpi-value" data-kpi="search_traffic">—</div>';
        echo '<div class="miro-momentum-kpi-delta" id="kpi-delta-traffic" style="display:none;"></div>';
        echo '<div class="miro-momentum-kpi-hint">Estimated pageviews from Google</div>';
        echo '</div>';
        
        // Avg Position
        echo '<div class="miro-momentum-kpi-card">';
        echo '<svg class="miro-momentum-kpi-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M12 2L15.09 8.26L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L8.91 8.26L12 2Z" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '</svg>';
        echo '<div class="miro-momentum-kpi-label">Avg Position</div>';
        echo '<div class="miro-momentum-kpi-value" data-kpi="position">—</div>';
        echo '<div class="miro-momentum-kpi-delta" id="kpi-delta-position" style="display:none;"></div>';
        echo '<div class="miro-momentum-kpi-hint">Lower number = better rankings</div>';
        echo '</div>';
        
        // Clicks
        echo '<div class="miro-momentum-kpi-card">';
        echo '<svg class="miro-momentum-kpi-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M18 13V19A2 2 0 0 1 16 21H5A2 2 0 0 1 3 19V8A2 2 0 0 1 5 6H11" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '<path d="M15 3H21V9" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '<path d="M10 14L21 3" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '</svg>';
        echo '<div class="miro-momentum-kpi-label">Clicks</div>';
        echo '<div class="miro-momentum-kpi-value" data-kpi="clicks">—</div>';
        echo '<div class="miro-momentum-kpi-delta" id="kpi-delta-clicks" style="display:none;"></div>';
        echo '<div class="miro-momentum-kpi-hint">Total clicks from search</div>';
        echo '</div>';
        
        // Click Appeal (CTR)
        echo '<div class="miro-momentum-kpi-card">';
        echo '<svg class="miro-momentum-kpi-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M22 11.08V12A10 10 0 1 1 5.93 7.5" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '<path d="M22 4L12 14.01L9 11.01" stroke="#D8B15A" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
        echo '</svg>';
        echo '<div class="miro-momentum-kpi-label">Click Appeal</div>';
        echo '<div class="miro-momentum-kpi-value" data-kpi="ctr">—%</div>';
        echo '<div class="miro-momentum-kpi-delta" id="kpi-delta-ctr" style="display:none;"></div>';
        echo '<div class="miro-momentum-kpi-hint">Click-through rate percentage</div>';
        echo '</div>';
        
        echo '</div>'; // End KPI row

        // ===== INSIGHT PANEL =====
        echo '<div class="miro-momentum-insight-panel">';
        echo '<svg class="miro-momentum-insight-panel-icon" width="24" height="24" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M10 2C5.58 2 2 5.58 2 10C2 14.42 5.58 18 10 18C14.42 18 18 14.42 18 10C18 5.58 14.42 2 10 2ZM11 14H9V9H11V14ZM10 7.5C9.17 7.5 8.5 6.83 8.5 6C8.5 5.17 9.17 4.5 10 4.5C10.83 4.5 11.5 5.17 11.5 6C11.5 6.83 10.83 7.5 10 7.5Z" fill="#D8B15A"/>';
        echo '</svg>';
        echo '<div class="miro-momentum-insight-panel-text" id="miro-ana-summary">'.esc_html__('Loading AI insights...','miro-ai-seo').'</div>';
        echo '</div>';

        // Tabs nav + content
        echo '<div class="tabs" id="miroTabs">';

        $tabs_order = [
            '\\Miro_AI_SEO\\GSC_Tab_Overview',              // Overview
            '\\Miro_AI_SEO\\GSC_Tab_Presence',              // Presence on Google
            '\\Miro_AI_SEO\\GSC_Tab_Pages',                 // Pages
            '\\Miro_AI_SEO\\GSC_Tab_RankFlow',              // RankFlow
                '\\Miro_AI_SEO\\GSC_Tab_Easy_Keywords', // ✅ ADD THIS RIGHT HERE
            '\\Miro_AI_SEO\\GSC_Tab_Rank_Today',   // ✅ NEW
            '\\Miro_AI_SEO\\GSC_Tab_Keywords',              // Keywords
            '\\Miro_AI_SEO\\GSC_Tab_Rank',                  // Rank Tracker
            '\\Miro_AI_SEO\\GSC_Tab_AI',                    // AI Suggestions
            '\\Miro_AI_SEO\\Miro_Analytics_Content_Gap',    // Content Gap
            '\\Miro_AI_SEO\\GSC_Tab_QuickWins',             // Quick-Wins
        ];

        foreach ($tabs_order as $cls) {
            if (class_exists($cls) && method_exists($cls, 'nav')) {
                $cls::nav();
            }
        }
        do_action('miro_gsc_tabs_nav_after');

        echo '</div>';

        // Content panels stay hooked to the old action – no breakage
        do_action('miro_gsc_tabs_content');

        echo '</div></div>';

        // ===== JS (tabs + days filter wiring + charts + endpoints) =====
        $rest_base       = site_url('index.php');
        $rest_snapshot   = add_query_arg('rest_route', '/miro/v1/analytics/ui/snapshot', $rest_base);
        $rest_sync_state = add_query_arg('rest_route', '/miro/v1/gsc/sync-state', $rest_base);
        $rest_summary    = add_query_arg('rest_route', '/miro/v1/analytics/ui/summary', $rest_base);
        $rest_ask        = add_query_arg('rest_route', '/miro/v1/analytics/ui/ask', $rest_base);
        $rest_overall    = add_query_arg('rest_route', '/miro/v1/analytics/ui/overall', $rest_base);
        $rest_sync       = add_query_arg('rest_route', '/miro/v1/analytics/ui/sync', $rest_base);
        $rest_first_sync = add_query_arg('rest_route', '/miro/v1/gsc/first-sync-now', $rest_base);
        $rest_run_sync   = add_query_arg('rest_route', '/miro/v1/gsc/run-sync-now', $rest_base);
        $rest_clear_cache = add_query_arg('rest_route', '/miro/v1/gsc/clear-cache', $rest_base);
        $clear_cache_ajax = add_query_arg('action', 'miro_gsc_clear_cache', admin_url('admin-ajax.php'));
        $snapshot_ajax   = add_query_arg('action', 'miro_gsc_snapshot', admin_url('admin-ajax.php'));
        $sync_state_ajax = add_query_arg('action', 'miro_gsc_sync_state', admin_url('admin-ajax.php'));
        $run_sync_ajax   = add_query_arg('action', 'miro_gsc_run_sync_blocking', admin_url('admin-ajax.php'));
        $rest_nonce      = wp_create_nonce('wp_rest');

        echo '<script>
(function(){

  // Tabs
  const tabs = document.querySelectorAll(".tab-link");
  const pans = document.querySelectorAll(".tab-pan");
  function activate(id){
    tabs.forEach(t=>t.classList.remove("active"));
    pans.forEach(p=>p.style.display="none");
    const a=[...tabs].find(x=>x.getAttribute("href")==="#"+id);
    const s=document.getElementById(id);
    if(a) a.classList.add("active");
    if(s) s.style.display="block";
    window.dispatchEvent(new CustomEvent("miro-tab-activated",{detail:{tab:id}}));
  }
  tabs.forEach(t=>t.addEventListener("click", e=>{
    e.preventDefault();
    const id=t.getAttribute("href").slice(1);
    activate(id);
    history.replaceState(null,"","#"+id);
  }));
  const hash=(location.hash||"").replace(/^#/,"");
  if(hash && document.getElementById(hash)) activate(hash);
  else if(tabs.length) activate(tabs[0].getAttribute("href").slice(1));

  // REST (snapshot + syncState + runSync use admin-ajax so data loads when REST is blocked)
  const REST = {
    snapshot:   "'.esc_js($snapshot_ajax).'",
    syncState:  "'.esc_js($sync_state_ajax).'",
    firstSync:   "'.esc_js($rest_first_sync).'",
    runSyncNow:  "'.esc_js($rest_run_sync).'",
    runSyncNowAjax: "'.esc_js($run_sync_ajax).'",
    clearCache: "'.esc_js($rest_clear_cache).'",
    clearCacheAjax: "'.esc_js($clear_cache_ajax).'",
    summary:    "'.esc_js($rest_summary).'",
    ask:        "'.esc_js($rest_ask).'",
    overall:    "'.esc_js($rest_overall).'",
    sync:       "'.esc_js($rest_sync).'",
    nonce:      "'.esc_js($rest_nonce).'"
  };

  function GET(u){
    return fetch(u,{headers:{ "X-WP-Nonce":REST.nonce }, credentials:"same-origin"})
      .then(function(r){ return r.text().then(function(t){
        if (!r.ok) {
          var msg = "Request failed (" + r.status + "). ";
          if (r.status === 502 || r.status === 504) msg += "Server or proxy timeout—try again in a moment.";
          else if (r.status === 403) msg += "Access denied. Check you are logged in.";
          throw new Error(msg);
        }
        try { return JSON.parse(t); } catch(e) {
          throw new Error("Server returned an invalid response. Try refreshing the page or run Sync now.");
        }
      }); })
      .then(function(data){
        if (data && data.code && data.message) {
          throw new Error(data.message || "API Error: " + data.code);
        }
        return data;
      });
  }

  function POST(u,b){
    return fetch(u,{
      method:"POST",
      headers:{
        "Content-Type":"application/json",
        "X-WP-Nonce":REST.nonce
      },
      body:JSON.stringify(b||{}),
      credentials:"same-origin"
    }).then(function(r){
      return r.text().then(function(t){
        if (!r.ok) {
          var msg = "Request failed (" + r.status + "). ";
          if (r.status === 502 || r.status === 504) msg += "Sync may have timed out—it can run in the background. Check back in 1–2 minutes or try again.";
          else if (t && t.length < 200 && !t.trim().startsWith("<")) msg += t;
          throw new Error(msg);
        }
        try { return JSON.parse(t); } catch(e) {
          throw new Error("Server returned an invalid response. If you clicked Sync, the sync may still be running in the background—check back in 1–2 minutes.");
        }
      });
    });
  }

  function setK(k,v){
    // Support both old and new KPI selectors
    let el=document.querySelector(".kval[data-kpi=\'"+k+"\']") || 
           document.querySelector(".miro-momentum-kpi-value[data-kpi=\'"+k+"\']");
    if(el) el.textContent=v;
  }
  function fmtN(n){ return (n||0).toLocaleString(); }
  function fmtP(n){ return (n||0).toFixed(2)+"%"; }

  // Charts
  var chart;

  function loadChart(cb){
    if (window.Chart) return cb();
    var s=document.createElement("script");
    s.src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js";
    s.onload=cb;
    document.head.appendChild(s);
  }

  // ==== Bars + CTR + Position, all STOP on last Impressions day ====
  function paintChart(labels, clicks, impr, ctr, pos) {
    var ctx = document.getElementById("miro-ana-chart");
    if (!ctx) return;
    if (chart) { try { chart.destroy(); } catch (e) {} }

    // --- helpers ---
    function toArray(a) {
      return Array.isArray(a) ? a.slice() : (a ? [a] : []);
    }
    function normNumeric(arr) {
      var out = [];
      for (var i = 0; i < arr.length; i++) {
        var v = arr[i];
        if (v == null) v = 0;
        v = Number(v);
        if (isNaN(v)) v = 0;
        out.push(v);
      }
      return out;
    }

    labels = toArray(labels);
    clicks = toArray(clicks);
    impr   = toArray(impr);
    ctr    = toArray(ctr);
    pos    = toArray(pos);

    var len = labels.length;
    if (!len) return;

    // ---- 1) find last day where impressions > 0 ----
    var lastIdx = -1;
    for (var i = 0; i < len; i++) {
      var v = Number(impr[i] || 0);
      if (v > 0) lastIdx = i;
    }
    // if no impressions at all, just show everything
    if (lastIdx === -1) lastIdx = len - 1;

    // ---- 2) slice EVERYTHING to that last impressions day ----
    function sliceToLast(arr) {
      return arr.slice(0, lastIdx + 1);
    }

    labels = sliceToLast(labels);
    clicks = sliceToLast(clicks);
    impr   = sliceToLast(impr);
    ctr    = sliceToLast(ctr);
    pos    = sliceToLast(pos);

    // normalize to numbers
    clicks = normNumeric(clicks);
    impr   = normNumeric(impr);
    var ctrPercent   = normNumeric(ctr).map(function (v) { return v * 100; });
    var positionVals = normNumeric(pos);

    // ---- 3) build chart (4 line series) ----
    chart = new Chart(ctx, {
      type: "line",
      data: {
        labels: labels,
        datasets: [
          // Clicks (gold line)
          {
            label: "Clicks",
            data: clicks,
            yAxisID: "y",
            borderWidth: 2,
            pointRadius: 0,
            pointHoverRadius: 6,
            pointHoverBackgroundColor: "#D8B15A",
            pointHoverBorderColor: "#D8B15A",
            pointHoverBorderWidth: 2,
            tension: 0.4,
            borderColor: "#D8B15A",
            backgroundColor: "transparent",
            hidden: false
          },
          // Impressions (light gold line)
          {
            label: "Impressions",
            data: impr,
            yAxisID: "y",
            borderWidth: 2,
            pointRadius: 0,
            pointHoverRadius: 6,
            pointHoverBackgroundColor: "#F4D03F",
            pointHoverBorderColor: "#F4D03F",
            pointHoverBorderWidth: 2,
            tension: 0.4,
            borderColor: "#F4D03F",
            backgroundColor: "transparent",
            hidden: false
          },
          // Click Appeal / CTR (gray line, right axis)
          {
            label: "Click Appeal",
            data: ctrPercent,
            yAxisID: "y1",
            borderWidth: 2,
            pointRadius: 0,
            pointHoverRadius: 6,
            pointHoverBackgroundColor: "#9ca3af",
            pointHoverBorderColor: "#9ca3af",
            pointHoverBorderWidth: 2,
            tension: 0.4,
            borderColor: "#9ca3af",
            backgroundColor: "transparent",
            hidden: false
          },
          // Ranking Strength / Position (dashed line, right axis, reversed)
          {
            label: "Ranking Strength",
            data: positionVals,
            yAxisID: "y2",
            borderWidth: 2,
            borderDash: [5, 5],
            pointRadius: 0,
            pointHoverRadius: 6,
            pointHoverBackgroundColor: "#6366f1",
            pointHoverBorderColor: "#6366f1",
            pointHoverBorderWidth: 2,
            tension: 0.4,
            borderColor: "#6366f1",
            backgroundColor: "transparent",
            hidden: false
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          mode: "index",
          intersect: false
        },
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            enabled: false,
            external: function(ctx) {
              var chart = ctx.chart;
              var tooltip = ctx.tooltip;
              var parent = chart.canvas && chart.canvas.parentNode;
              if (!parent) return;
              var el = parent.querySelector(".miroFloatTip");
              if (!el) {
                el = document.createElement("div");
                el.className = "miroFloatTip";
                el.setAttribute("aria-hidden", "true");
                parent.style.position = "relative";
                parent.appendChild(el);
              }
              if (!tooltip || tooltip.opacity === 0 || !tooltip.dataPoints || tooltip.dataPoints.length === 0) {
                el.style.opacity = "0";
                el.style.pointerEvents = "none";
                return;
              }
              function formatMetric(label, v) {
                if (v === null || v === undefined) return "–";
                if (v === 0) return "0";
                var num = Number(v);
                if (isNaN(num)) return String(v);
                if (label === "Click Appeal") return num.toFixed(2) + "%";
                if (label === "Ranking Strength") return num.toFixed(2);
                if (label === "Clicks" || label === "Impressions") return num.toLocaleString();
                return String(v);
              }
              var title = (tooltip.title && tooltip.title[0] !== undefined) ? String(tooltip.title[0]) : "";
              if (title && title.match(/^\d{4}-\d{2}-\d{2}$/)) {
                var d = new Date(title + "T00:00:00");
                title = d.toLocaleDateString("en-US", { month: "short", day: "numeric", year: "numeric" });
              }
              var rowsHtml = tooltip.dataPoints.map(function(dp) {
                var label = (dp.dataset && dp.dataset.label) ? String(dp.dataset.label) : "";
                var v = (dp.parsed && dp.parsed.y !== undefined && dp.parsed.y !== null) ? dp.parsed.y : dp.raw;
                var val = formatMetric(label, v);
                var color = (dp.dataset && (dp.dataset.borderColor || dp.dataset.backgroundColor)) || "#999";
                return "<div class=\"tRow\"><span class=\"tDot\" style=\"background:" + color + "\"></span><span class=\"tLab\">" + (label || "") + "</span><span class=\"tVal\">" + val + "</span></div>";
              }).join("");
              el.innerHTML = "<div class=\"tTitle\">" + (title || "—") + "</div>" + rowsHtml;
              var rect = chart.canvas.getBoundingClientRect();
              var left = rect.left + (tooltip.caretX || 0) + 12;
              var top = rect.top + (tooltip.caretY || 0) - 12;
              var tipRect = el.getBoundingClientRect();
              var containerRect = parent.getBoundingClientRect();
              left = Math.max(containerRect.left, Math.min(left, containerRect.right - tipRect.width));
              top = Math.max(containerRect.top, Math.min(top, containerRect.bottom - tipRect.height));
              el.style.left = left + "px";
              el.style.top = top + "px";
              el.style.opacity = "1";
              el.style.pointerEvents = "none";
            }
          }
        },
        scales: {
          y: {
            type: "linear",
            beginAtZero: true,
            title: { display: false },
            grid: { drawBorder: false }
          },
          y1: {
            type: "linear",
            position: "right",
            beginAtZero: true,
            title: { display: false },
            grid: { drawBorder: false, display: false },
            ticks: {
              callback: function(val) {
                return Number(val).toFixed(1) + "%";
              }
            }
          },
          y2: {
            type: "linear",
            position: "right",
            reverse: true,
            beginAtZero: false,
            suggestedMin: 1,
            title: { display: false },
            grid: { display: false, drawBorder: false }
          },
          x: {
            grid: {
              display: true,
              color: "rgba(0,0,0,0.05)",
              drawBorder: false
            },
            ticks: {
              maxRotation: 0,
              autoSkip: true,
              maxTicksLimit: 8,
              font: { size: 11 }
            }
          },
          y: {
            grid: {
              display: true,
              color: "rgba(0,0,0,0.05)",
              drawBorder: false
            },
            ticks: {
              font: { size: 11 }
            }
          }
        }
      }
    });
    
    // Add legend checkbox functionality
    var legendItems = document.querySelectorAll(".miro-momentum-legend-item");
    legendItems.forEach(function(item) {
      item.addEventListener("click", function() {
        var series = this.getAttribute("data-series");
        var isActive = this.classList.contains("active");
        
        if (isActive) {
          this.classList.remove("active");
        } else {
          this.classList.add("active");
        }
        
        // Toggle chart dataset visibility
        if (chart && chart.data && chart.data.datasets) {
          var datasetIndex = -1;
          if (series === "clicks") datasetIndex = 0;
          else if (series === "impressions") datasetIndex = 1;
          else if (series === "ctr") datasetIndex = 2;
          else if (series === "position") datasetIndex = 3;
          
          if (datasetIndex >= 0 && chart.data.datasets[datasetIndex]) {
            var meta = chart.getDatasetMeta(datasetIndex);
            meta.hidden = isActive;
            chart.update();
          }
        }
      });
    });
  }

  function buildTrendMeta(series){
    // Old function - no longer used in new design
    // Insight text is now updated in refreshAll()
    return;
    var el = document.getElementById("miro-ana-trend-meta");
    if (!el) return;

    var labels = series.labels || [];
    if (!labels.length) {
      el.innerHTML = "<span class=\\"muted\\">Not enough data to calculate trends yet.</span>";
      return;
    }

    function calcTrend(arr, inverse){
      if (!arr || arr.length < 2) return null;
      var first = Number(arr[0] || 0);
      var last  = Number(arr[arr.length - 1] || 0);

      if (!inverse) {
        var abs = last - first;
        var pct = first !== 0 ? (abs / Math.abs(first)) * 100 : 0;
        return { first:first, last:last, abs:abs, pct:pct };
      } else {
        // inverse metric (position): lower is better
        var absImp = first - last;
        var pctImp = first !== 0 ? (absImp / Math.abs(first)) * 100 : 0;
        return { first:first, last:last, abs:absImp, pct:pctImp };
      }
    }

    var tClicks = calcTrend(series.clicks);
    var tImpr   = calcTrend(series.impressions);
    var tCtr    = calcTrend((series.ctr || []).map(function(x){ return x * 100; }));
    var tPos    = calcTrend(series.position, true); // inverse: improvement when going DOWN

    function card(metricKey, label, t, suffix, opts){
      if (!t) return "";
      suffix = suffix || "";
      opts = opts || {};
      var improvedLabel = opts.improvedLabel || "up";
      var worseLabel    = opts.worseLabel    || "down";

      var dirText;
      var deltaClass;
      if (t.pct > 0.1) {
        dirText    = "↑ " + improvedLabel + " (+" + t.pct.toFixed(1) + "%)";
        deltaClass = "miro-trend-up";
      } else if (t.pct < -0.1) {
        dirText    = "↓ " + worseLabel + " (" + t.pct.toFixed(1) + "%)";
        deltaClass = "miro-trend-down";
      } else {
        dirText    = "≈ Flat";
        deltaClass = "miro-trend-flat";
      }

      var first = t.first.toLocaleString(undefined,{maximumFractionDigits:1});
      var last  = t.last.toLocaleString(undefined,{maximumFractionDigits:1});

      return ""
        + "<div class=\\"card kpi kpi-trend kpi-trend-" + metricKey + "\\">"
        +   "<div class=\\"title\\">" + label + "</div>"
        +   "<div class=\\"kval\\">" + first + " → " + last + (suffix ? " " + suffix : "") + "</div>"
        +   "<div class=\\"kpi-sub " + deltaClass + "\\">" + dirText + "</div>"
        + "</div>";
    }

    var html = "";

    if (tClicks) {
      html += card("clicks", "Clicks", tClicks, "", {
        improvedLabel: "higher",
        worseLabel: "lower"
      });
    }
    if (tImpr) {
      html += card("impr", "Impressions", tImpr, "", {
        improvedLabel: "higher",
        worseLabel: "lower"
      });
    }
    if (tCtr) {
      html += card("ctr", "CTR", tCtr, "%", {
        improvedLabel: "higher",
        worseLabel: "lower"
      });
    }
    if (tPos) {
      html += card("position", "Position", tPos, "", {
        improvedLabel: "better (lower)",
        worseLabel: "worse"
      });
    }

    if (!html) {
      el.innerHTML = "<span class=\\"muted\\">Not enough data to calculate trends yet.</span>";
    } else {
      el.innerHTML = "<div class=\\"miro-ana-trend-row\\">" + html + "</div>";
    }
  }

  // Range + Sync + Refresh
  const btnSync = document.getElementById("miro-sync-now");
  const selRange = document.getElementById("miro-range");

  const btnClearCache = document.getElementById("miro-clear-cache");
  if (btnClearCache && REST.clearCacheAjax) {
    btnClearCache.addEventListener("click", function(){
      var btn = this;
      var origText = btn.textContent;
      btn.disabled = true;
      btn.textContent = "Clearing…";
      var fd = new FormData();
      fd.append("action", "miro_gsc_clear_cache");
      fd.append("nonce", REST.nonce);
      fetch(REST.clearCacheAjax, { method: "POST", credentials: "same-origin", body: fd })
        .then(function(r){ return r.json().then(function(d){ return { ok: r.ok, data: d }; }); })
        .then(function(o){
          var ok = o.ok && o.data && (o.data.ok || o.data.success);
          if (ok) { location.reload(); return; }
          btn.disabled = false;
          btn.textContent = origText;
          var d = o.data && o.data.data ? o.data.data : o.data;
          var msg = (d && d.message) || "Failed to clear cache.";
          alert(msg);
        })
        .catch(function(){
          btn.disabled = false;
          btn.textContent = origText;
          alert("Network error clearing cache.");
        });
    });
  }

  function getSelectedRange(){
    let v = parseInt(selRange.value,10);
    if (isNaN(v)) v=30;
    return Math.max(7, Math.min(90, v));
  }

  function saveRange(v){
    try{ localStorage.setItem("miro_ana_range_days", String(v)); }catch(e){}
  }
  function loadRange(){
    try{
      const s = localStorage.getItem("miro_ana_range_days");
      const v = s ? parseInt(s,10) : 0;
      if (v && selRange) selRange.value = String(v);
    }catch(e){}
  }

  var _gscSyncElapsedInterval = null;
  function startSyncElapsedTimer(){
    if (_gscSyncElapsedInterval) clearInterval(_gscSyncElapsedInterval);
    var start = Date.now();
    function tick(){
      var el = document.getElementById("gsc-sync-elapsed");
      if (!el) return;
      var sec = Math.floor((Date.now() - start) / 1000);
      var m = Math.floor(sec / 60);
      var s = sec % 60;
      el.textContent = "Elapsed: " + m + ":" + (s < 10 ? "0" : "") + s;
    }
    tick();
    _gscSyncElapsedInterval = setInterval(tick, 1000);
  }
  function stopSyncElapsedTimer(){
    if (_gscSyncElapsedInterval) { clearInterval(_gscSyncElapsedInterval); _gscSyncElapsedInterval = null; }
  }

  function refreshAll(){
    const days = getSelectedRange();
    var snapshotUrl = REST.snapshot + "&days=" + days + "&_=" + (Date.now ? Date.now() : 0);

    return GET(snapshotUrl).then(function(s){
      if (!s || !s.kpis) {
        console.error("No data received from snapshot endpoint");
        return;
      }
      
      setK("search_traffic", fmtN(s.kpis.search_traffic));
      setK("search_impressions", fmtN(s.kpis.search_impressions));
      setK("total_keywords", fmtN(s.kpis.total_keywords));
      setK("ctr", fmtP(s.kpis.ctr));
      setK("position", (s.kpis.position||0).toFixed(2));
      setK("clicks", fmtN(s.kpis.clicks || s.kpis.search_traffic));
      
      // Update chart title with selected range
      const titleEl = document.getElementById("miro-momentum-chart-title");
      if (titleEl) {
        var daysText = (s.meta && s.meta.ui_days ? s.meta.ui_days : days) + " days";
        titleEl.textContent = "Your SEO momentum (last " + daysText + ")";
      }

      let last=document.getElementById("miro-ana-last");
      if(last){
        var syncText = s.kpis.last_synced || "—";
        var rangeText = (s.meta && s.meta.ui_days ? s.meta.ui_days : days) + "d";
        last.textContent = "Last synced: " + syncText + " • Range: " + rangeText;
      }

      // Update insight text
      const insightEl = document.getElementById("miro-momentum-insight");
      if (insightEl && s.series && s.series.labels && s.series.labels.length >= 2) {
        var labels = s.series.labels || [];
        var clicksFirst = s.series.clicks[0] || 0;
        var clicksLast = s.series.clicks[labels.length - 1] || 0;
        var posFirst = s.series.position[0] || 0;
        var posLast = s.series.position[labels.length - 1] || 0;
        var ctrFirst = (s.series.ctr[0] || 0) * 100;
        var ctrLast = (s.series.ctr[labels.length - 1] || 0) * 100;
        
        var insight = "Your rankings ";
        if (posLast < posFirst && posFirst > 0) {
          insight += "improved this week";
        } else if (posLast > posFirst && posFirst > 0) {
          insight += "declined this week";
        } else {
          insight += "stayed stable this week";
        }
        
        if (ctrLast < ctrFirst * 0.95 && ctrFirst > 0) {
          insight += ", but CTR dipped slightly.";
        } else if (ctrLast > ctrFirst * 1.05 && ctrFirst > 0) {
          insight += ", and CTR improved.";
        } else {
          insight += ".";
        }
        
        insightEl.textContent = insight;
      } else if (insightEl) {
        insightEl.textContent = "Not enough data to generate insights yet. Sync your data first.";
      }

      if (s.series && s.series.labels && s.series.labels.length > 0) {
        loadChart(function(){
          paintChart(
            s.series.labels,
            s.series.clicks,
            s.series.impressions,
            s.series.ctr,
            s.series.position
          );
        });
      } else {
        console.warn("No series data available for chart");
        const chartContainer = document.querySelector(".miro-momentum-chart-container");
        if (chartContainer && (!s.series || !s.series.labels || s.series.labels.length === 0)) {
          var syncState = (s && s.sync_state) ? s.sync_state : {};
          var status = syncState.status || "idle";
          var isFirstSync = (status === "queued" || status === "running");
          if (isFirstSync) {
            chartContainer.innerHTML =
              "<div style=\"display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;color:#374151;font-size:14px;text-align:center;padding:40px;\">" +
              "<div style=\"margin-bottom:8px;font-weight:600;\">⏳ Syncing… (first run)</div>" +
              "<div id=\"gsc-sync-step\" style=\"font-size:13px;color:#6b7280;margin-bottom:4px;\">" + (syncState.step || "Preparing…") + "</div>" +
              "<div style=\"margin-top:12px;width:100%;max-width:320px;height:10px;background:#e5e7eb;border-radius:6px;overflow:hidden;\">" +
              "<div id=\"gsc-sync-progress\" style=\"height:100%;width:" + (syncState.progress || 0) + "%;background:#2563eb;border-radius:6px;transition:width 0.4s ease;\"></div></div>" +
              "<div id=\"gsc-sync-pct\" style=\"font-size:12px;margin-top:8px;color:#6b7280;\">" + (syncState.progress || 0) + "%</div>" +
              "<div id=\"gsc-sync-elapsed\" style=\"font-size:12px;color:#9ca3af;margin-top:6px;\">Elapsed: 0:00</div>" +
              "<div style=\"font-size:11px;color:#9ca3af;margin-top:8px;\">This can take 2–5 min. Don’t close this page.</div>" +
              "</div>";
            startSyncElapsedTimer();
            if (window._gscSyncPoll) clearInterval(window._gscSyncPoll);
            window._gscSyncPoll = setInterval(function(){
              fetch(REST.syncState, { headers: { "X-WP-Nonce": REST.nonce }, credentials: "same-origin" })
                .then(function(r){ return r.ok ? r.json() : r.json().catch(function(){ return {}; }); })
                .then(function(st){
                  var elStep = document.getElementById("gsc-sync-step");
                  var elProg = document.getElementById("gsc-sync-progress");
                  var elPct = document.getElementById("gsc-sync-pct");
                  if (elStep) elStep.textContent = st.step || "Syncing…";
                  var pct = Math.max(0, Math.min(100, parseInt(st.progress, 10) || 0));
                  if (elProg) elProg.style.width = pct + "%";
                  if (elPct) elPct.textContent = pct + "%";
          if (st.status === "done") {
            clearInterval(window._gscSyncPoll);
            window._gscSyncPoll = null;
            stopSyncElapsedTimer();
            setTimeout(function(){ location.reload(); }, 500);
          } else if (st.status === "error") {
            clearInterval(window._gscSyncPoll);
            window._gscSyncPoll = null;
            stopSyncElapsedTimer();
            chartContainer.innerHTML = "<div style=\"display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;color:#6b7280;font-size:14px;text-align:center;padding:40px;\"><div style=\"margin-bottom:8px;color:#dc2626;\">❌ Sync failed</div><div style=\"font-size:13px;\">" + (st.last_error || "Unknown error") + "</div><div style=\"margin-top:12px;font-size:12px;\">Click Sync now to try again.</div></div>";
                  }
                })
                .catch(function(){});
            }, 2000);
          } else {
            var connected = syncState.gsc_connected !== false;
            var gscUrl = (typeof window !== "undefined" && window.location && window.location.origin) ? (window.location.origin + "/wp-admin/admin.php?page=miro-gsc") : "";
            var msg = connected
              ? "<div style=\"margin-bottom:12px;\">📊 No data yet</div><div style=\"font-size:13px;\">Click <strong>Sync now</strong> above to fetch your Google Search Console data. This can take 2–5 minutes.</div>"
              : "<div style=\"margin-bottom:12px;\">🔗 Connect Google Search Console first</div><div style=\"font-size:13px;margin-bottom:12px;\">Connect your account and select a property, then return here and click <strong>Sync now</strong>.</div>" + (gscUrl ? "<div><a href=\"" + gscUrl + "\" class=\"button button-primary\">Open Google Search Console →</a></div>" : "");
            chartContainer.innerHTML = "<div style=\"display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;color:#6b7280;font-size:14px;text-align:center;padding:40px;\">" + msg + "</div>";
          }
        }
      }

      return GET(REST.summary + "&days=" + days).then(function(r){
        let el=document.getElementById("miro-ana-summary");
        if(el) el.textContent=r.text||"—";
      }).catch(function(err){
        console.error("Failed to load AI summary:", err);
      });

    }).catch(function(err){
      console.error("Failed to fetch snapshot data:", err);
      var chartContainer = document.querySelector(".miro-momentum-chart-container");
      var friendlyMsg = "Run Sync now above to fetch data, or refresh the page.";
      if (chartContainer) {
        chartContainer.innerHTML =
          "<div style=\"display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;color:#374151;font-size:14px;text-align:center;padding:40px;\">" +
          "<div style=\"margin-bottom:8px;color:#dc2626;font-weight:600;\">Failed to load analytics data</div>" +
          "<div style=\"font-size:13px;color:#6b7280;margin-bottom:12px;\">" + friendlyMsg + "</div>" +
          "<div style=\"font-size:12px;color:#9ca3af;\">Click <strong>Sync now</strong> above to fetch data, or refresh the page and try again.</div>" +
          "</div>";
      }
    });
  }

  function showSyncProgressAndPoll(){
    var chartContainer = document.querySelector(".miro-momentum-chart-container");
    if (!chartContainer) return;
    chartContainer.innerHTML =
      "<div style=\"display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;color:#374151;font-size:14px;text-align:center;padding:40px;\">" +
      "<div style=\"margin-bottom:8px;font-weight:600;\">⏳ Syncing…</div>" +
      "<div id=\"gsc-sync-step\" style=\"font-size:13px;color:#6b7280;margin-bottom:4px;\">Preparing…</div>" +
      "<div style=\"margin-top:12px;width:100%;max-width:320px;height:10px;background:#e5e7eb;border-radius:6px;overflow:hidden;\">" +
      "<div id=\"gsc-sync-progress\" style=\"height:100%;width:0%;background:#2563eb;border-radius:6px;transition:width 0.4s ease;\"></div></div>" +
      "<div id=\"gsc-sync-pct\" style=\"font-size:12px;margin-top:8px;color:#6b7280;\">0%</div>" +
      "<div id=\"gsc-sync-elapsed\" style=\"font-size:12px;color:#9ca3af;margin-top:6px;\">Elapsed: 0:00</div>" +
      "<div style=\"font-size:11px;color:#9ca3af;margin-top:8px;\">This can take 2–5 min. Don’t close this page.</div>" +
      "</div>";
    startSyncElapsedTimer();
    if (window._gscSyncPoll) clearInterval(window._gscSyncPoll);
    window._gscSyncPoll = setInterval(function(){
      fetch(REST.syncState, { headers: { "X-WP-Nonce": REST.nonce }, credentials: "same-origin" })
        .then(function(r){ return r.ok ? r.json() : r.text().then(function(){ return {}; }); })
        .then(function(st){
          var elStep = document.getElementById("gsc-sync-step");
          var elProg = document.getElementById("gsc-sync-progress");
          var elPct = document.getElementById("gsc-sync-pct");
          if (elStep) elStep.textContent = st.step || "Syncing…";
          var pct = Math.max(0, Math.min(100, parseInt(st.progress, 10) || 0));
          if (elProg) elProg.style.width = pct + "%";
          if (elPct) elPct.textContent = pct + "%";
          if (st.status === "done") {
            clearInterval(window._gscSyncPoll);
            window._gscSyncPoll = null;
            stopSyncElapsedTimer();
            var sb = document.getElementById("miro-sync-now");
            if (sb) { sb.disabled = false; sb.textContent = "Sync now"; }
            setTimeout(function(){ location.reload(); }, 500);
          } else if (st.status === "error") {
            clearInterval(window._gscSyncPoll);
            window._gscSyncPoll = null;
            stopSyncElapsedTimer();
            var sb = document.getElementById("miro-sync-now");
            if (sb) { sb.disabled = false; sb.textContent = "Sync now"; }
            chartContainer.innerHTML = "<div style=\"display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;color:#6b7280;font-size:14px;text-align:center;padding:40px;\"><div style=\"margin-bottom:8px;color:#dc2626;\">❌ Sync failed</div><div style=\"font-size:13px;\">" + (st.last_error || "Unknown error") + "</div><div style=\"margin-top:12px;font-size:12px;\">Click Sync now to try again.</div></div>";
          }
        })
        .catch(function(){});
    }, 2000);
  }

  if(btnSync){
    btnSync.addEventListener("click", function(){
      var original = btnSync.textContent;
      btnSync.disabled = true;
      btnSync.textContent = "Syncing…";
      showSyncProgressAndPoll();
      var runUrl = (typeof REST !== "undefined" && REST.runSyncNowAjax) ? REST.runSyncNowAjax : (REST.runSyncNow || REST.firstSync);
      var useAjax = runUrl.indexOf("admin-ajax") !== -1;
      if (useAjax) {
        var fd = new FormData();
        fd.append("action", "miro_gsc_run_sync_blocking");
        fd.append("nonce", REST.nonce);
        fetch(runUrl, { method: "POST", credentials: "same-origin", body: fd }).catch(function(){
          if (REST.firstSync) fetch(REST.firstSync, { method: "POST", headers: { "Content-Type": "application/json", "X-WP-Nonce": REST.nonce }, credentials: "same-origin", body: "{}" }).catch(function(){});
        });
      } else {
        fetch(runUrl, { method: "POST", headers: { "Content-Type": "application/json", "X-WP-Nonce": REST.nonce }, credentials: "same-origin", body: "{}" }).catch(function(){
          if (REST.firstSync) POST(REST.firstSync, {}).catch(function(){});
        });
      }
    });
  }

  if (selRange){
    selRange.addEventListener("change", function(){
      saveRange(getSelectedRange());
      refreshAll();
    });
  }

  function boot(){
    loadRange();

    var startSync = (typeof window !== "undefined" && window.location && window.location.search && window.location.search.indexOf("start_sync=1") !== -1);
    if (startSync) {
      try {
        var params = new URLSearchParams(window.location.search);
        params.delete("start_sync");
        var qs = params.toString();
        history.replaceState(null, "", window.location.pathname + (qs ? "?" + qs : "") + (window.location.hash || ""));
      } catch (e) {}
      if (btnSync) btnSync.disabled = true;
      btnSync && (btnSync.textContent = "Syncing…");
      showSyncProgressAndPoll();
      var runUrl = (typeof REST !== "undefined" && REST.runSyncNowAjax) ? REST.runSyncNowAjax : (REST.runSyncNow || REST.firstSync);
      if (runUrl && runUrl.indexOf("admin-ajax") !== -1) {
        var fd = new FormData();
        fd.append("action", "miro_gsc_run_sync_blocking");
        fd.append("nonce", REST.nonce);
        fetch(runUrl, { method: "POST", credentials: "same-origin", body: fd }).catch(function(){});
      } else if (runUrl) {
        fetch(runUrl, { method: "POST", headers: { "Content-Type": "application/json", "X-WP-Nonce": REST.nonce }, credentials: "same-origin", body: "{}" }).catch(function(){});
      } else {
        POST(REST.firstSync, {}).catch(function(){});
      }
    }

    // Check if we have any cached data first (errors shown by refreshAll() in chart area)
    if (!startSync) {
      refreshAll().catch(function(err){
        console.error("Initial data load failed:", err);
      });
    }

    const btnReport = document.getElementById("miro-ana-open-report");
    if (btnReport) {
      btnReport.addEventListener("click", function () {
        activate("tab-report");
        history.replaceState(null, "", "#tab-report");
        const tabsWrap = document.getElementById("miroTabs");
        if (tabsWrap) {
          tabsWrap.scrollIntoView({ behavior: "smooth" });
        }
      });
    }

    const btn=document.getElementById("miro-ana-ask-btn");
    const input=document.getElementById("miro-ana-ask");
    const ans=document.getElementById("miro-ana-ans");
    if(btn && input && ans){
      btn.addEventListener("click", function(){
        var q=(input.value||"").trim();
        if(!q) return;
        btn.disabled=true;
        const original=btn.textContent;
        btn.textContent="Thinking…";
        ans.textContent="…";

        POST(REST.ask+"&days="+getSelectedRange(), {q:q})
          .then(function(r){ ans.textContent=r.answer||"—"; })
          .catch(function(){ ans.textContent="AI error."; })
          .finally(function(){ btn.disabled=false; btn.textContent=original; });
      });
      input.addEventListener("keydown", function(e){
        if(e.key==="Enter") btn.click();
      });
    }
  }

  document.addEventListener("DOMContentLoaded", boot);

})();

</script>';
    }
}

/** Boot */
add_action('plugins_loaded', ['\\Miro_AI_SEO\\GSC_Analytics_Core','init'], 20);
