<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

class GSC_Tab_Keywords
{
    // mbstring fallbacks
    private static function lstr($s){ return function_exists('mb_strtolower') ? mb_strtolower($s) : strtolower($s); }
    private static function mbpos($hay,$needle){ return function_exists('mb_strpos') ? mb_strpos($hay,$needle) : strpos($hay,$needle); }

    const REST_NS    = 'miro/v1';
    const R_KEYWORDS = '/gsc/analytics/keywords';
    const R_AI       = '/gsc/analytics/keywords/ai';

    public static function init(): void
    {
        add_action('rest_api_init', [__CLASS__, 'routes']);

        if (is_admin()) {
            add_action('miro_gsc_tabs_nav',     [__CLASS__, 'nav']);
            add_action('miro_gsc_tabs_content', [__CLASS__, 'content']);
        }
    }

    public static function nav(): void
    {
        echo '<a href="#tab-keywords" class="tab-link">Keywords</a>';
    }

    public static function content(): void
    {
        $nonce      = wp_create_nonce('wp_rest');
        $kwURL      = esc_js(rest_url(self::REST_NS . self::R_KEYWORDS));
        $aiURL      = esc_js(rest_url(self::REST_NS . self::R_AI));
        $trackURL   = esc_js(rest_url(self::REST_NS . '/gsc/rank/tracked'));
        $syncStart  = esc_js(rest_url(self::REST_NS . '/gsc/sync/start'));
        $syncStatus = esc_js(rest_url(self::REST_NS . '/gsc/sync/status'));
        $site       = esc_js(wp_strip_all_tags(get_bloginfo('name') ?: 'Your site'));
        $gscPage    = esc_url(admin_url('admin.php?page=miro-gsc'));

        echo '<section id="tab-keywords" class="tab-pan" style="display:none">';

        echo '<style>
        #tab-keywords .miro-page-cell{display:flex;align-items:center;gap:8px;flex-wrap:nowrap;}
        #tab-keywords .miro-page-cell a{font-weight:500;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;min-width:0;flex:1;}
        #tab-keywords td:nth-child(3) .miro-page-cell{flex-wrap:wrap;}
        #tab-keywords td:nth-child(3) .miro-page-cell a{white-space:normal !important;overflow:visible !important;text-overflow:clip !important;word-break:break-word;}
        #tab-keywords .miro-page-badges{display:flex;flex-wrap:nowrap;gap:6px;font-size:11px;margin-top:0;flex-shrink:0;}
        #tab-keywords .miro-badge-chip{display:inline-flex;align-items:center;border-radius:999px;padding:2px 9px;border:1px solid rgba(148,163,184,.55);background:#fff;white-space:nowrap;}
        #tab-keywords .miro-badge-score{border-color:rgba(16,185,129,.55);color:#047857;}
        #tab-keywords .miro-badge-index{border-color:rgba(99,102,241,.55);color:#3730a3;}
        #tab-keywords .miro-badge-words{border-color:rgba(156,163,175,.65);color:#374151;}
        #tab-keywords .miro-kw-drawer{position:fixed;top:32px;right:0;bottom:0;width:390px;max-width:100%;background:#fff;box-shadow:-2px 0 28px rgba(15,23,42,.15);border-left:1px solid rgba(226,232,240,.9);z-index:9999;transform:translateX(100%);transition:transform .22s ease-out;}
        #tab-keywords .miro-kw-drawer.open{transform:translateX(0);}
        #tab-keywords .miro-kw-drawer-inner{height:100%;padding:16px 18px 18px;display:flex;flex-direction:column;gap:10px;}
        #tab-keywords .miro-kw-close{border:none;background:transparent;font-size:20px;line-height:1;cursor:pointer;margin-left:auto;}
        #tab-keywords .miro-kw-meta small{display:block;color:#6b7280;margin-bottom:6px;}
        #tab-keywords .miro-kw-pages table{width:100%;border-collapse:collapse;font-size:12px;}
        #tab-keywords .miro-kw-pages th,#tab-keywords .miro-kw-pages td{border-bottom:1px solid #e5e7eb;padding:6px 4px;text-align:left;}
        #tab-keywords .miro-kw-pages a{word-break:break-word;}
        @media (max-width:900px){#tab-keywords .miro-kw-drawer{top:0;width:100%;}}
        </style>';

        echo '<div class="kw-page-header">';
        echo '<div><h2 class="kw-page-header-title">Keywords</h2><p class="kw-page-header-sub">Cached GSC queries with trend, intent, and best page. Click a row to open the keyword drawer.</p></div>';
        echo '</div>';

        echo '<div class="grid">';

        echo '  <div id="kwAlert" class="card full" style="display:none;background:#fef3c7;border:1px solid #fde68a;border-radius:18px;">
                  <div class="kw-card-header"><span class="kw-card-title">Cache not found</span></div>
                  <p id="kwAlertMessage" class="muted" style="margin:0 0 12px 0;">This tab uses cached GSC data. Sync your GSC data first.</p>
                  <div class="controls">
                    <a class="btn" href="'.esc_url($gscPage).'">Open GSC Page</a>
                    <button class="btn primary" id="kwQuickSync">Quick Sync (Background)</button>
                    <span class="muted" id="kwQuickSyncStatus" style="display:none">Starting…</span>
                  </div>
                </div>';

        echo '  <div class="card full">
                  <div class="kw-card-header">
                    <span class="kw-card-title">AI Keyword Playbook</span>
                    <span class="badge muted">site-wide</span>
                  </div>
                  <div class="kw-filters">
                    <span class="kw-filters-label">Device</span><select id="aiDevice"><option value="">All</option><option>DESKTOP</option><option>MOBILE</option><option>TABLET</option></select>
                    <span class="kw-filters-label">Country</span><select id="aiCountry" class="input kw-input-sm"><?php echo self::render_country_options(); ?></select>
                    <span class="kw-filters-label">Days</span><select id="aiDays"><option value="7">7</option><option value="15">15</option><option value="30" selected>30</option><option value="90">90</option></select>
                    <button class="btn primary" id="aiBuild">Generate</button>
                    <span class="muted" id="aiStatus" style="display:none">Working…</span>
                  </div>
                  <div id="kwAI" style="white-space:pre-wrap;font-size:13px;line-height:1.6;"></div>
                </div>';

        echo '  <div class="card full">
                  <div class="kw-card-header">
                    <span class="kw-card-title">Keywords</span>
                    <span id="kwTotal" class="badge">Total: —</span>
                  </div>
                  <p class="muted" style="margin:0 0 14px 0;font-size:12px;">Trend = Top URL position. Hover the line for exact day.</p>

                  <div class="kw-filters">
                    <input id="kwSearch" class="input" placeholder="Search query…"/>
                    <span class="kw-filters-label">Device</span><select id="kwDevice"><option value="">All</option><option>DESKTOP</option><option>MOBILE</option><option>TABLET</option></select>
                    <span class="kw-filters-label">Country</span><select id="kwCountry" class="input kw-input-sm"><?php echo self::render_country_options(); ?></select>
                    <span class="kw-filters-label">Days</span><select id="kwDays"><option value="7">7</option><option value="15">15</option><option value="30" selected>30</option><option value="90">90</option></select>
                    <span class="kw-filters-label">Segment</span><select id="kwSegment"><option value="">All</option><option value="striking">Striking Distance</option><option value="winners">Big Winners</option><option value="losers">Big Losers</option><option value="brand">Brand</option><option value="nonbrand">Non-brand</option></select>
                    <span class="kw-filters-label">Sort</span><select id="kwSort"><option value="clicks_desc">Clicks ↓</option><option value="opportunity_desc">Opportunity ↓</option><option value="pos_asc">Best Pos ↑</option><option value="impressions_desc">Impr ↓</option><option value="pages_desc">Cannibal ↓</option><option value="winners_desc">Winners ↑</option><option value="losers_desc">Losers ↓</option></select>
                    <span class="kw-filters-label">Trend</span><select id="kwTrendMetric"><option value="clicks" selected>Clicks</option><option value="impressions">Impr</option><option value="position">Position</option><option value="ctr">CTR</option></select>
                    <span class="kw-filters-label">Per page</span><select id="kwPer"><option>25</option><option>50</option><option>100</option></select>
                    <div class="divider"></div>
                    <button class="btn" id="kwExport">Export CSV</button>
                    <button class="btn primary" id="kwReload">Apply</button>
                  </div>

                  <div class="miro-table-wrap">
                    <table class="miro-table">
                        <thead>
                            <tr>
                                <th>Query</th>
                                <th>Intent</th>
                                <th>Best Page</th>
                                <th>Clicks<br><span class="muted">Δ</span></th>
                                <th>Impr<br><span class="muted">Δ</span></th>
                                <th>CTR<br><span class="muted">Δ</span></th>
                                <th>Pos<br><span class="muted">Δ</span></th>
                                <th>Opp</th>
                                <th>Pages</th>
                                <th>Trend</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="kwBody"></tbody>
                    </table>
                  </div>

                  <div class="controls">
                    <button class="btn" id="kwPrev">Prev</button>
                    <span class="muted" id="kwPage">Page 1</span>
                    <button class="btn" id="kwNext">Next</button>
                  </div>
                </div>';

        echo '</div>';

        echo '<div id="kwDrawer" class="miro-kw-drawer">
                <div class="miro-kw-drawer-inner">
                    <button type="button" class="miro-kw-close" aria-label="Close">&times;</button>
                    <div class="miro-kw-meta">
                        <small id="kwDrawerMeta"></small>
                        <h3 id="kwDrawerTitle" style="margin:0 0 4px;font-size:16px;font-weight:500;"></h3>
                    </div>
                    <div id="kwDrawerMain" class="miro-kw-main"></div>
                    <div id="kwDrawerPages" class="miro-kw-pages"></div>
                </div>
              </div>';

        if (defined('MIRO_AI_SEO_FILE') && defined('MIRO_AI_SEO_URL') && defined('MIRO_AI_SEO_VERSION')) {
            wp_enqueue_style('miro-gsc-keywords-css', MIRO_AI_SEO_URL . 'assets/css/keywords.css', [], MIRO_AI_SEO_VERSION);
            
            $inline_css_rel = 'assets/css/miro-gsc-keywords-inline.css';
            $inline_css_path = plugin_dir_path(MIRO_AI_SEO_FILE) . $inline_css_rel;
            $inline_css_url = MIRO_AI_SEO_URL . $inline_css_rel;
            $inline_css_ver = (file_exists($inline_css_path) && is_readable($inline_css_path)) ? (string) filemtime($inline_css_path) : MIRO_AI_SEO_VERSION;
            wp_enqueue_style('miro-gsc-keywords-inline', $inline_css_url, ['miro-gsc-keywords-css'], $inline_css_ver);
            
            wp_enqueue_script('miro-gsc-keywords', MIRO_AI_SEO_URL . 'assets/js/keywords.js', ['jquery'], MIRO_AI_SEO_VERSION, true);
        } else {
            $js_path = plugin_dir_path(__FILE__) . '../../../../assets/js/keywords.js';
            $js_ver  = (file_exists($js_path) && is_readable($js_path)) ? (string) filemtime($js_path) : (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');
            wp_enqueue_script('miro-gsc-keywords', plugins_url('assets/js/keywords.js', __FILE__), ['jquery'], $js_ver, true);
        }

        $country_list = array_merge(
            [['code' => '', 'name' => 'All countries']],
            self::get_countries_from_cache()
        );
        wp_localize_script('miro-gsc-keywords', 'miroGSCKeywords', [
            'nonce'      => $nonce,
            'kwURL'      => $kwURL,
            'aiURL'      => $aiURL,
            'trackURL'   => $trackURL,
            'syncStart'  => $syncStart,
            'syncStatus' => $syncStatus,
            'site'       => wp_strip_all_tags(get_bloginfo('name') ?: 'Your site'),
            'countries'  => $country_list,
        ]);

        echo '<div id="rfKwTooltip" class="rf-rank-tooltip"></div>';

        echo '</section>';

        echo '<style>
        .rf-rank-tooltip {
            position: fixed;
            min-width: 180px;
            max-width: 280px;
            background: #ffffff;
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            padding: 12px 16px;
            font-size: 12px;
            color: #374151;
            box-shadow: 0 12px 40px rgba(0,0,0,0.15), 0 4px 12px rgba(0,0,0,0.1);
            pointer-events: none;
            display: none;
            z-index: 30;
            font-weight: 500;
            backdrop-filter: blur(10px);
            line-height: 1.6;
        }
        #tab-keywords td.kwTrendCell {
            vertical-align: middle;
        }
        #tab-keywords td.kwTrendCell .rf-trend {
            margin-top: 0;
        }
        </style>';
    }

    public static function routes(): void
    {
        $perm = [__CLASS__, 'perm_cb'];

        register_rest_route(self::REST_NS, self::R_KEYWORDS, [
            'methods'             => 'GET',
            'permission_callback' => $perm,
            'callback'            => [__CLASS__, 'rest_keywords_cached'],
        ]);

        register_rest_route(self::REST_NS, self::R_AI, [
            'methods'             => 'GET',
            'permission_callback' => $perm,
            'callback'            => [__CLASS__, 'rest_ai_summary_cached'],
        ]);

        add_filter('rest_post_dispatch', [__CLASS__, 'rest_no_cache_headers'], 10, 3);
    }

    /**
     * Prevent host/page cache from serving stale Keywords/AI data after sync.
     * Without these headers, Hostinger (or other caches) can return old empty response.
     */
    public static function rest_no_cache_headers($response, $server, $request)
    {
        $route = $request->get_route();
        if ($route === null) {
            return $response;
        }
        $prefix = '/' . self::REST_NS . self::R_KEYWORDS;
        $is_keywords = (strpos($route, $prefix) === 0);
        $is_ai = (strpos($route, '/' . self::REST_NS . self::R_AI) === 0);
        if (!$is_keywords && !$is_ai) {
            return $response;
        }
        if ($response instanceof \WP_REST_Response) {
            $response->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
            $response->header('Pragma', 'no-cache');
            $response->header('Expires', '0');
        }
        return $response;
    }

    public static function perm_cb(): bool
    {
        if (function_exists(__NAMESPACE__.'\\_miro_cap')) {
            return current_user_can(\Miro_AI_SEO\_miro_cap());
        }
        if (function_exists('\\miro_ai_cap')) {
            return current_user_can(\miro_ai_cap());
        }
        return current_user_can('edit_posts');
    }

    public static function rest_keywords_cached(\WP_REST_Request $r)
    {
        if (function_exists('ob_get_level')) { while (ob_get_level()) { @ob_end_clean(); } }

        $opt  = get_option('miro_gsc_last_queries');
        $opt  = is_array($opt) ? $opt : [];
        $rows = (isset($opt['rows']) && is_array($opt['rows'])) ? $opt['rows'] : [];

        // Only treat as "cache not found" when sync has never run (no meta.last_synced).
        // A completed sync can write rows=[] (e.g. new property, no data); that is valid cache.
        $has_synced = isset($opt['meta']['last_synced']) && (string)$opt['meta']['last_synced'] !== '';

        if (!$rows && !$has_synced) {
            $prop = get_option('pp_miro_gsc_property', []);
            $settings = get_option('pp_miro_gsc_settings', []);
            $uri = isset($prop['uri']) ? (string)$prop['uri'] : '';
            $has_token = !empty($settings['refresh_token']);
            if ($uri === '' || !$has_token) {
                $message = 'Connect your GSC account and select a property first (Miro AI SEO → Google Search Console), then run Sync.';
                $hint = 'connect_first';
            } else {
                $message = 'No cached data yet. Open the GSC page and run Sync, or use Quick Sync below.';
                $hint = 'sync_first';
            }
            return new \WP_REST_Response([
                'items'       => [],
                'total'       => 0,
                'page'        => 1,
                'spark_dates' => [],
                'code'        => 'no_cache',
                'message'     => $message,
                'hint'        => $hint,
            ], 200);
        }

        $args    = self::args($r);
        $days    = $args['days'];
        $device  = $args['device'];
        $country = $args['country'];
        $search  = $args['q'];
        $segment = strtolower((string)($r->get_param('segment') ?: ''));

        $page = max(1, intval($r->get_param('page') ?: 1));
        $per  = max(5, min(100, intval($r->get_param('per_page') ?: 25)));
        $sort = (string)($r->get_param('sort') ?: 'clicks_desc');
        $wantCSV = ((string)$r->get_param('csv') === '1');

        $anchorTo = self::cache_anchor_to($opt, $rows);

        [$dates_cur, $datesListCur]   = self::date_set_for_window($anchorTo, $days);
        [$dates_prev, $datesListPrev] = self::prev_date_set_for_window($anchorTo, $days);

        $filter = function(array $rr, array $datesSet) use ($device, $country, $search){
            $d = (string)($rr['date'] ?? '');
            if (!$d || !isset($datesSet[$d])) return false;

            if ($device !== '' && (string)($rr['device'] ?? '') !== $device) return false;
            if ($country !== '' && (string)($rr['country'] ?? '') !== $country) return false;

            if ($search){
                $q = self::lstr((string)($rr['query'] ?? ''));
                $p = self::lstr((string)($rr['page']  ?? ''));
                $needle = self::lstr($search);
                if (self::mbpos($q, $needle) === false && self::mbpos($p, $needle) === false) return false;
            }
            return true;
        };

        $cur = [];
        $prev = [];
        foreach ($rows as $rr) {
            if ($filter($rr, $dates_cur))  $cur[]  = $rr;
            if ($filter($rr, $dates_prev)) $prev[] = $rr;
        }

        // If query+page exists, ignore query-only rows
        if (self::has_page_rows($cur)) {
            $cur = array_values(array_filter($cur, fn($x)=> (string)($x['page'] ?? '') !== ''));
        }
        if (self::has_page_rows($prev)) {
            $prev = array_values(array_filter($prev, fn($x)=> (string)($x['page'] ?? '') !== ''));
        }

        $brand   = self::brand_tokens();
        $aggCurr = self::aggregate_by_query($cur);
        $aggPrev = self::aggregate_by_query($prev);

        $itemsAll = [];
        foreach ($aggCurr as $q => $s) {

            $clicks = (float)($s['clicks'] ?? 0);
            $impr   = (float)($s['impressions'] ?? 0);
            $ctr    = ($impr > 0) ? ($clicks / $impr) : 0.0;

            // keyword avg position (all URLs) — keep for calculations + drawer
            $pos_keyword = (($s['posw_w'] ?? 0) > 0)
                ? (($s['posw_sum'] ?? 0) / max(1.0, (float)$s['posw_w']))
                : 0.0;

            // Build daily series (keyword totals)
            $seriesClicks = [];
            $seriesImpr   = [];
            $seriesCtr    = [];
            foreach ($datesListCur as $d) {
                $cDay = (float)($s['clicks_by_date'][$d] ?? 0.0);
                $iDay = (float)($s['impressions_by_date'][$d] ?? 0.0);
                $seriesClicks[] = $cDay;
                $seriesImpr[]   = $iDay;
                $seriesCtr[]    = ($iDay > 0) ? ($cDay / $iDay) : null;
            }

            // back-compat spark = clicks
            $spark = $seriesClicks;

            // clicks trend for winner/loser segment
            $trendClicks = 0.0;
            $n = count($spark);
            if ($n >= 4) {
                $slice = min(5, max(2, (int)floor($n / 3)));
                $startAvg = array_sum(array_slice($spark, 0, $slice)) / max(1, $slice);
                $endAvg   = array_sum(array_slice($spark, -$slice)) / max(1, $slice);
                $trendClicks = $endAvg - $startAvg;
            }

            // prev window totals (keyword)
            $pv = $aggPrev[$q] ?? null;

            $clicks_prev = $pv ? (float)($pv['clicks'] ?? 0) : 0.0;
            $impr_prev   = $pv ? (float)($pv['impressions'] ?? 0) : 0.0;
            $ctr_prev    = ($impr_prev > 0) ? ($clicks_prev / $impr_prev) : 0.0;

            $pos_keyword_prev = ($pv && ($pv['posw_w'] ?? 0) > 0)
                ? (($pv['posw_sum'] ?? 0) / max(1.0, (float)$pv['posw_w']))
                : 0.0;

            // per-page breakdown + primary page
            $pages_list    = [];
            $primary_url   = '';
            $primary_click = -1;

            if (!empty($s['per_page']) && is_array($s['per_page'])) {
                foreach ($s['per_page'] as $url => $ps) {
                    $pc = (float)($ps['clicks'] ?? 0);
                    $pi = (float)($ps['impressions'] ?? 0);

                    $pp = 0.0;
                    if (($ps['posw_w'] ?? 0) > 0) {
                        $pp = ((float)$ps['posw_sum']) / max(1.0, (float)$ps['posw_w']);
                    }

                    $pages_list[] = [
                        'url'         => (string)$url,
                        'clicks'      => (int)round($pc),
                        'impressions' => (int)round($pi),
                        'position'    => (float)$pp,
                        'ctr'         => ($pi > 0 ? ($pc / $pi) : 0.0),
                    ];

                    // primary by clicks
                    if ($pc > $primary_click) {
                        $primary_click = $pc;
                        $primary_url   = (string)$url;
                    }
                }

                usort($pages_list, function($a,$b){
                    $c = ($b['clicks'] <=> $a['clicks']);
                    if ($c !== 0) return $c;
                    return ($b['impressions'] <=> $a['impressions']);
                });
            }

            // ✅ Top URL position (primary URL) — THIS becomes "position" in the table
            $pos_top = $pos_keyword;
            $pos_top_prev = $pos_keyword_prev;

            // ✅ Position SERIES for sparkline when metric=position = primary URL series
            $seriesPos = [];
            if ($primary_url && isset($s['per_page'][$primary_url])) {
                $pps = $s['per_page'][$primary_url];

                // overall top url pos
                if (($pps['posw_w'] ?? 0) > 0) {
                    $pos_top = ((float)$pps['posw_sum']) / max(1.0, (float)$pps['posw_w']);
                }

                // per-day top url pos
                foreach ($datesListCur as $d) {
                    $pw = (float)($pps['pos_w_by_date'][$d] ?? 0.0);
                    $ps = (float)($pps['pos_sum_by_date'][$d] ?? 0.0);
                    $seriesPos[] = ($pw > 0) ? ($ps / $pw) : null;
                }

                // prev top url pos (same url)
                if ($pv && isset($pv['per_page'][$primary_url])) {
                    $ppv = $pv['per_page'][$primary_url];
                    if (($ppv['posw_w'] ?? 0) > 0) {
                        $pos_top_prev = ((float)$ppv['posw_sum']) / max(1.0, (float)$ppv['posw_w']);
                    }
                }
            } else {
                // fallback: keyword avg series
                foreach ($datesListCur as $d) {
                    $pw = (float)($s['pos_w_by_date'][$d] ?? 0.0);
                    $ps = (float)($s['pos_sum_by_date'][$d] ?? 0.0);
                    $seriesPos[] = ($pw > 0) ? ($ps / $pw) : null;
                }
            }

            // Opportunity should use keyword avg position (not just one URL)
            $opp = $impr * max(0.0, self::expected_ctr($pos_keyword, $device) - $ctr);

            // Page-level meta for primary URL
            $page_post_id = 0;
            $page_title   = '';
            $page_score   = null;
            $page_index   = '';
            $page_links   = null;
            $page_words   = null;

            if ($primary_url) {
                $page_post_id = url_to_postid($primary_url);
                if ($page_post_id) {
                    $page_title = get_the_title($page_post_id);
                    $page_score = self::read_page_score($page_post_id);

                    $page_index = get_post_meta($page_post_id, '_miro_index_verdict', true);
                    if (!is_string($page_index)) $page_index = '';

                    $page_links = (int)get_post_meta($page_post_id, '_miro_internal_links_count', true);

                    $content    = (string)get_post_field('post_content', $page_post_id);
                    $page_words = $content !== '' ? str_word_count(wp_strip_all_tags($content)) : 0;
                }
            }

            $itemsAll[] = [
                'query'            => $q,
                'clicks'           => (int)round($clicks),
                'impressions'      => (int)round($impr),
                'ctr'              => (float)$ctr,

                // ✅ TABLE POSITION = TOP URL position
                'position'         => (float)$pos_top,
                'position_prev'    => (float)$pos_top_prev,

                // ✅ Keep keyword avg position for drawer / debug
                'position_keyword'      => (float)$pos_keyword,
                'position_keyword_prev' => (float)$pos_keyword_prev,

                'opportunity'      => (float)$opp,
                'pages'            => max(1, count($s['pages_set'] ?? [])),
                'intent'           => self::intent_of($q),
                'is_brand'         => self::is_brand($q, $brand),

                'spark'            => array_map('floatval', $spark),
                'trend'            => (float)$trendClicks,

                // ✅ SERIES: position is top URL series now
                'series' => [
                    'clicks'      => array_map('floatval', $seriesClicks),
                    'impressions' => array_map('floatval', $seriesImpr),
                    'position'    => $seriesPos,
                    'ctr'         => $seriesCtr,
                ],

                'clicks_prev'      => (int)round($clicks_prev),
                'impressions_prev' => (int)round($impr_prev),
                'ctr_prev'         => (float)$ctr_prev,

                'page_url'     => $primary_url,
                'page_post_id' => $page_post_id,
                'page_title'   => $page_title,
                'page_score'   => $page_score,
                'page_index'   => $page_index,
                'page_links'   => $page_links,
                'page_words'   => $page_words,

                'pages_list'   => $pages_list,
            ];
        }

        // Segment filter
        if ($segment && $segment !== 'all') {
            $itemsAll = array_values(array_filter($itemsAll, function($it) use ($segment){
                switch ($segment) {
                    case 'striking':
                        // use keyword avg position for "striking"
                        return (($it['position_keyword'] ?? 0) >= 5 && ($it['position_keyword'] ?? 0) <= 20 && ($it['impressions'] ?? 0) >= 100);
                    case 'winners':
                        return (($it['trend'] ?? 0) > 0.5 && ($it['clicks'] ?? 0) > ($it['clicks_prev'] ?? 0));
                    case 'losers':
                        return (($it['trend'] ?? 0) < -0.5 && ($it['clicks_prev'] ?? 0) > 0 && ($it['clicks'] ?? 0) < ($it['clicks_prev'] ?? 0));
                    case 'brand':
                        return !empty($it['is_brand']);
                    case 'nonbrand':
                        return empty($it['is_brand']);
                    default:
                        return true;
                }
            }));
        }

        // Sort
        usort($itemsAll, function($a,$b) use ($sort){
            switch ($sort) {
                case 'opportunity_desc':   return ($b['opportunity'] <=> $a['opportunity']);
                case 'pos_asc':            return ($a['position']    <=> $b['position']); // top-url pos
                case 'impressions_desc':   return ($b['impressions'] <=> $a['impressions']);
                case 'pages_desc':         return ($b['pages']       <=> $a['pages']);
                case 'winners_desc':       return ($b['trend']       <=> $a['trend']);
                case 'losers_desc':        return ($a['trend']       <=> $b['trend']);
                default:                   return ($b['clicks']      <=> $a['clicks']);
            }
        });

        $total    = count($itemsAll);
        $startIdx = ($page-1) * $per;
        $paged    = array_slice($itemsAll, $startIdx, $per);

        if ($wantCSV) {
            $csv = self::to_csv($paged);
            $resp = new \WP_REST_Response($csv);
            $resp->header('Content-Type','text/csv; charset=utf-8');
            $resp->header('Content-Disposition','attachment; filename=keywords.csv');
            return $resp;
        }

        return rest_ensure_response([
            'ok'          => true,
            'items'       => $paged,
            'page'        => $page,
            'total'       => $total,
            'sampled'     => ($total > count($paged)),
            'spark_dates' => $datesListCur,
        ]);
    }

    public static function rest_ai_summary_cached(\WP_REST_Request $r)
    {
        if (function_exists('ob_get_level')) { while (ob_get_level()) { @ob_end_clean(); } }

        $opt  = get_option('miro_gsc_last_queries');
        $opt  = is_array($opt) ? $opt : [];
        $rows = (isset($opt['rows']) && is_array($opt['rows'])) ? $opt['rows'] : [];

        $has_synced = isset($opt['meta']['last_synced']) && (string)$opt['meta']['last_synced'] !== '';
        if (!$rows && !$has_synced) {
            return new \WP_Error('no_cache','No cached data for AI summary. Sync first.', ['status'=>400]);
        }

        $args   = self::args($r);
        $days   = $args['days'];
        $device = $args['device'];
        $country = $args['country'];

        $anchorTo = self::cache_anchor_to($opt, $rows);
        [$dates_cur] = self::date_set_for_window($anchorTo, $days);

        $filtered = [];
        foreach ($rows as $rr) {
            $d = (string)($rr['date'] ?? '');
            if (!$d || !isset($dates_cur[$d])) continue;
            if ($device !== '' && (string)($rr['device'] ?? '') !== $device) continue;
            if ($country !== '' && (string)($rr['country'] ?? '') !== $country) continue;
            $filtered[] = $rr;
        }

        if (self::has_page_rows($filtered)) {
            $filtered = array_values(array_filter($filtered, fn($x)=> (string)($x['page'] ?? '') !== ''));
        }

        $byQP = [];
        foreach ($filtered as $row) {
            $q = (string)($row['query'] ?? '');
            if ($q === '') continue;
            $p = (string)($row['page'] ?? '');
            $k = $q.'|'.$p;

            if (!isset($byQP[$k])) $byQP[$k] = [
                'query'=>$q,'page'=>$p,
                'clicks'=>0.0,'impressions'=>0.0,
                'posw_sum'=>0.0,'posw_w'=>0.0
            ];

            $c   = (float)($row['clicks'] ?? 0);
            $i   = (float)($row['impressions'] ?? 0);
            $pos = (float)($row['position'] ?? 0);

            $byQP[$k]['clicks']      += $c;
            $byQP[$k]['impressions'] += $i;

            if ($i > 0 && $pos > 0 && $pos <= 1000) {
                $byQP[$k]['posw_sum'] += $pos*$i;
                $byQP[$k]['posw_w']   += $i;
            }
        }

        $sample = [];
        foreach ($byQP as $row) {
            $impr = (float)$row['impressions'];
            $clk  = (float)$row['clicks'];
            $ctr  = ($impr > 0) ? ($clk / $impr) : 0.0;
            $pos  = ($row['posw_w'] > 0) ? ($row['posw_sum'] / $row['posw_w']) : 0.0;
            $opp  = $impr * max(0.0, self::expected_ctr($pos, $device) - $ctr);

            $sample[] = [
                'query' => $row['query'],
                'page'  => $row['page'],
                'clicks'=> (int)round($clk),
                'impressions' => (int)round($impr),
                'ctr'   => (float)$ctr,
                'position' => (float)$pos,
                'opp'   => (float)$opp,
            ];
        }

        $topByOpp  = $sample;
        usort($topByOpp,  fn($a,$b)=> ($b['opp'] <=> $a['opp']));
        $topByImpr = $sample;
        usort($topByImpr, fn($a,$b)=> ($b['impressions'] <=> $a['impressions']));

        $topOpp  = array_slice($topByOpp,  0, 10);
        $topImpr = array_slice($topByImpr, 0, 10);

        $site   = (string)($r->get_param('site') ?: (get_bloginfo('name') ?: 'this site'));
        $prompt = self::ai_prompt($site, $args, $topOpp, $topImpr);

        $aiOut = null;
        if (has_filter('miro/ai/complete')) {
            $aiOut = apply_filters('miro/ai/complete', null, [
                'prompt'  => $prompt,
                'context' => 'keywords',
                'opts'    => ['temperature'=>0.4,'max_tokens'=>600],
            ]);
        } elseif (function_exists(__NAMESPACE__.'\\miro_ai_complete_inline')) {
            $aiOut = \Miro_AI_SEO\miro_ai_complete_inline($prompt, 'keywords');
        }
        if (!is_string($aiOut) || $aiOut === '') {
            $aiOut = self::fallback_ai_summary($topOpp, $topImpr);
        }

        return rest_ensure_response(['ok'=>true,'summary'=>$aiOut]);
    }

    private static function args(\WP_REST_Request $r): array
    {
        $days = intval($r->get_param('days') ?: 30);
        if (!in_array($days, [7,15,30,90], true)) $days = 30;

        $device = (string)($r->get_param('device') ?: '');
        if ($device && !in_array($device, ['DESKTOP','MOBILE','TABLET'], true)) $device = '';

        return [
            'days'    => max(7, $days),
            'device'  => $device,
            'country' => strtoupper(trim((string)($r->get_param('country') ?: ''))),
            'q'       => (string)($r->get_param('q') ?: ''),
        ];
    }

    private static function has_page_rows(array $rows): bool
    {
        foreach ($rows as $rr) {
            if ((string)($rr['page'] ?? '') !== '') return true;
        }
        return false;
    }

    /** ISO 3166-1 alpha-2 country codes => names (common set for Keywords country filter) */
    private static function country_names(): array
    {
        return [
            'US' => 'United States', 'GB' => 'United Kingdom', 'CA' => 'Canada', 'AU' => 'Australia',
            'DE' => 'Germany', 'FR' => 'France', 'IN' => 'India', 'ES' => 'Spain', 'IT' => 'Italy',
            'NL' => 'Netherlands', 'BR' => 'Brazil', 'MX' => 'Mexico', 'JP' => 'Japan', 'KR' => 'South Korea',
            'RU' => 'Russia', 'PL' => 'Poland', 'TR' => 'Turkey', 'ID' => 'Indonesia', 'ZA' => 'South Africa',
            'SE' => 'Sweden', 'NO' => 'Norway', 'DK' => 'Denmark', 'FI' => 'Finland', 'IE' => 'Ireland',
            'BE' => 'Belgium', 'AT' => 'Austria', 'CH' => 'Switzerland', 'PT' => 'Portugal', 'GR' => 'Greece',
            'RO' => 'Romania', 'CZ' => 'Czech Republic', 'HU' => 'Hungary', 'IL' => 'Israel', 'EG' => 'Egypt',
            'SA' => 'Saudi Arabia', 'AE' => 'United Arab Emirates', 'SG' => 'Singapore', 'MY' => 'Malaysia',
            'TH' => 'Thailand', 'PH' => 'Philippines', 'VN' => 'Vietnam', 'AR' => 'Argentina', 'CL' => 'Chile',
            'CO' => 'Colombia', 'NZ' => 'New Zealand', 'PK' => 'Pakistan', 'BD' => 'Bangladesh', 'NG' => 'Nigeria',
        ];
    }

    /** Country list for dropdown: from cache when available, otherwise full static list so the dropdown is never empty. */
    public static function get_countries_from_cache(): array
    {
        $names = self::country_names();
        $opt   = get_option('miro_gsc_last_queries');
        $opt   = is_array($opt) ? $opt : [];
        $rows  = (isset($opt['rows']) && is_array($opt['rows'])) ? $opt['rows'] : [];
        $codes = [];
        foreach ($rows as $rr) {
            $c = (string)($rr['country'] ?? '');
            if ($c !== '') $codes[$c] = true;
        }
        // If we have countries in cache, use those (with names from map or code as fallback)
        if (!empty($codes)) {
            $out = [];
            foreach (array_keys($codes) as $code) {
                $out[] = ['code' => $code, 'name' => $names[$code] ?? $code];
            }
            usort($out, function ($a, $b) { return strcasecmp($a['name'], $b['name']); });
            return $out;
        }
        // No country data in cache: show full list so user can still pick (filter will apply after next sync with country data)
        $out = [];
        foreach ($names as $code => $name) {
            $out[] = ['code' => $code, 'name' => $name];
        }
        usort($out, function ($a, $b) { return strcasecmp($a['name'], $b['name']); });
        return $out;
    }

    /** HTML options for country select: "All countries" + country list (from cache or full static list). */
    public static function render_country_options(): string
    {
        $countries = self::get_countries_from_cache();
        $html = '<option value="">All countries</option>';
        foreach ($countries as $c) {
            $html .= '<option value="' . esc_attr($c['code']) . '">' . esc_html($c['name']) . '</option>';
        }
        return $html;
    }

    private static function cache_anchor_to(array $opt, array $rows): string
    {
        $meta = is_array($opt['meta'] ?? null) ? $opt['meta'] : [];
        $to = (string)($meta['to'] ?? '');
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $to)) return $to;

        $max = '';
        foreach ($rows as $r) {
            $d = (string)($r['date'] ?? '');
            if ($d !== '' && $d > $max) $max = $d;
        }
        if ($max) return $max;

        $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
        $today = new \DateTime('today', $tz);
        return $today->modify('-1 day')->format('Y-m-d');
    }

    private static function date_set_for_window(string $toYmd, int $days): array
    {
        $days = max(7, min(90, $days));
        $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');

        $to = new \DateTime($toYmd, $tz);
        $from = (clone $to)->modify('-'.($days-1).' days');

        $set = [];
        $list = [];
        $d = clone $from;
        while ($d <= $to) {
            $k = $d->format('Y-m-d');
            $set[$k] = true;
            $list[] = $k;
            $d->modify('+1 day');
        }
        return [$set, $list];
    }

    private static function prev_date_set_for_window(string $toYmd, int $days): array
    {
        $days = max(7, min(90, $days));
        $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');

        $toCur = new \DateTime($toYmd, $tz);
        $fromCur = (clone $toCur)->modify('-'.($days-1).' days');

        $toPrev = (clone $fromCur)->modify('-1 day');
        $fromPrev = (clone $toPrev)->modify('-'.($days-1).' days');

        $set = [];
        $list = [];
        $d = clone $fromPrev;
        while ($d <= $toPrev) {
            $k = $d->format('Y-m-d');
            $set[$k] = true;
            $list[] = $k;
            $d->modify('+1 day');
        }
        return [$set, $list];
    }

    private static function brand_tokens(): array
    {
        $tokens = [];
        $name = wp_strip_all_tags(get_bloginfo('name') ?: '');
        if ($name) $tokens[] = self::lstr($name);
        $home = home_url('/');
        $host = wp_parse_url($home, PHP_URL_HOST);
        if ($host) {
            $host = preg_replace('/^www\./i','',$host);
            $host = preg_replace('/\..*$/','',$host);
            if ($host) $tokens[] = self::lstr($host);
        }
        return array_values(array_unique(array_filter($tokens)));
    }

    private static function is_brand($q, $brandTokens): bool
    {
        $lq = self::lstr($q);
        foreach ($brandTokens as $t) {
            if ($t && self::mbpos($lq, $t) !== false) return true;
        }
        return false;
    }

    private static function intent_of($q): string
    {
        $l = ' '.self::lstr($q).' ';
        if (preg_match('/ buy | price | coupon | deal | discount | plan | subscribe | order /', $l)) return 'transactional';
        if (preg_match('/ best | compare | vs | review | top | alternatives? /', $l)) return 'commercial';
        if (preg_match('/ how | what | why | guide | tutorial | meaning | examples? /', $l)) return 'informational';
        if (preg_match('/ login | dashboard | app | download | official /', $l)) return 'navigational';
        return 'informational';
    }

    private static function expected_ctr($pos, $device): float
    {
        $p = max(1.0, min(100.0, (float)$pos));
        $d = ($device === 'MOBILE') ? 0.9 : 1.0;
        if ($p <= 1)  return 0.30 * $d;
        if ($p <= 2)  return 0.16 * $d;
        if ($p <= 3)  return 0.10 * $d;
        if ($p <= 4)  return 0.075 * $d;
        if ($p <= 5)  return 0.060 * $d;
        if ($p <= 6)  return 0.050 * $d;
        if ($p <= 8)  return 0.035 * $d;
        if ($p <= 10) return 0.025 * $d;
        if ($p <= 15) return (0.020 - (($p-10)*0.0012)) * $d;
        if ($p <= 20) return (0.014 - (($p-15)*0.0010)) * $d;
        if ($p <= 40) return (0.009 -  ($p-20)*0.00018) * $d;
        return 0.005 * $d;
    }

    // ✅ UPDATED: now stores per-page-per-day position weights so top-URL series works
    private static function aggregate_by_query($rows): array
    {
        $agg = [];
        foreach ($rows as $row) {
            $q   = (string)($row['query'] ?? '');
            if ($q === '') continue;

            $p    = (string)($row['page'] ?? '');
            $c    = (float)($row['clicks'] ?? 0);
            $i    = (float)($row['impressions'] ?? 0);
            $pos  = (float)($row['position'] ?? 0);
            $date = (string)($row['date'] ?? '');

            if ($c < 0) $c = 0;
            if ($i < 0) $i = 0;
            if ($pos < 0 || $pos > 1000) $pos = 0;

            if (!isset($agg[$q])) {
                $agg[$q] = [
                    'clicks'              => 0.0,
                    'impressions'         => 0.0,
                    'posw_sum'            => 0.0,
                    'posw_w'              => 0.0,
                    'pages_set'           => [],
                    'per_page'            => [],
                    'clicks_by_date'      => [],
                    'impressions_by_date' => [],
                    'pos_sum_by_date'     => [],
                    'pos_w_by_date'       => [],
                ];
            }

            $agg[$q]['clicks']      += $c;
            $agg[$q]['impressions'] += $i;

            if ($i > 0 && $pos > 0) {
                $agg[$q]['posw_sum'] += $pos * $i;
                $agg[$q]['posw_w']   += $i;
            }

            if ($date !== '') {
                $agg[$q]['clicks_by_date'][$date]      = ($agg[$q]['clicks_by_date'][$date] ?? 0.0) + $c;
                $agg[$q]['impressions_by_date'][$date] = ($agg[$q]['impressions_by_date'][$date] ?? 0.0) + $i;

                if ($i > 0 && $pos > 0) {
                    $agg[$q]['pos_sum_by_date'][$date] = ($agg[$q]['pos_sum_by_date'][$date] ?? 0.0) + ($pos * $i);
                    $agg[$q]['pos_w_by_date'][$date]   = ($agg[$q]['pos_w_by_date'][$date]   ?? 0.0) + $i;
                } else {
                    $agg[$q]['pos_sum_by_date'][$date] = ($agg[$q]['pos_sum_by_date'][$date] ?? 0.0);
                    $agg[$q]['pos_w_by_date'][$date]   = ($agg[$q]['pos_w_by_date'][$date]   ?? 0.0);
                }
            }

            if ($p !== '') {
                $agg[$q]['pages_set'][$p] = true;

                if (!isset($agg[$q]['per_page'][$p])) {
                    $agg[$q]['per_page'][$p] = [
                        'clicks'      => 0.0,
                        'impressions' => 0.0,
                        'posw_sum'    => 0.0,
                        'posw_w'      => 0.0,
                        // NEW per-day
                        'pos_sum_by_date' => [],
                        'pos_w_by_date'   => [],
                    ];
                }

                $agg[$q]['per_page'][$p]['clicks']      += $c;
                $agg[$q]['per_page'][$p]['impressions'] += $i;

                if ($i > 0 && $pos > 0) {
                    $agg[$q]['per_page'][$p]['posw_sum'] += $pos * $i;
                    $agg[$q]['per_page'][$p]['posw_w']   += $i;

                    if ($date !== '') {
                        $agg[$q]['per_page'][$p]['pos_sum_by_date'][$date] = ($agg[$q]['per_page'][$p]['pos_sum_by_date'][$date] ?? 0.0) + ($pos * $i);
                        $agg[$q]['per_page'][$p]['pos_w_by_date'][$date]   = ($agg[$q]['per_page'][$p]['pos_w_by_date'][$date]   ?? 0.0) + $i;
                    }
                } else {
                    if ($date !== '') {
                        $agg[$q]['per_page'][$p]['pos_sum_by_date'][$date] = ($agg[$q]['per_page'][$p]['pos_sum_by_date'][$date] ?? 0.0);
                        $agg[$q]['per_page'][$p]['pos_w_by_date'][$date]   = ($agg[$q]['per_page'][$p]['pos_w_by_date'][$date]   ?? 0.0);
                    }
                }
            }
        }
        return $agg;
    }

    private static function read_page_score(int $post_id): ?float
    {
        // Score system removed - always return null
        return null;
    }

    private static function to_csv($rows): string
    {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- In-memory stream (php://temp) for CSV, not filesystem.
        $out = fopen('php://temp','r+');
        fputcsv($out, ['query','intent','clicks','impressions','ctr','position_top_url','position_keyword','opportunity','pages']);
        foreach ($rows as $r) {
            fputcsv($out, [
                (string)($r['query'] ?? ''),
                (string)($r['intent'] ?? ''),
                intval($r['clicks'] ?? 0),
                intval($r['impressions'] ?? 0),
                round(100*floatval($r['ctr'] ?? 0), 2).'%',
                round(floatval($r['position'] ?? 0), 1),
                round(floatval($r['position_keyword'] ?? 0), 1),
                intval(round($r['opportunity'] ?? 0)),
                intval($r['pages'] ?? 1),
            ]);
        }
        rewind($out);
        return (string)stream_get_contents($out);
    }

    private static function ai_prompt($site, $args, $topOpp, $topImpr): string
    {
        $country = (!empty($args['country'])) ? $args['country'] : 'ALL';
        $device  = (!empty($args['device']))  ? $args['device']  : 'ALL';
        $days    = $args['days'] ?? 30;

        $listOpp = [];
        foreach ($topOpp as $x) {
            $opp = $x['opp'] ?? 0;
            $listOpp[] = sprintf("- %s (pos %.1f, CTR %.1f%%, impr %d, est. opp +%d)",
                $x['query'], $x['position'], $x['ctr']*100, $x['impressions'], round($opp)
            );
        }
        $listImpr = [];
        foreach ($topImpr as $x) {
            $listImpr[] = sprintf("- %s (pos %.1f, CTR %.1f%%, impr %d)",
                $x['query'], $x['position'], $x['ctr']*100, $x['impressions']
            );
        }

        return "You are an SEO strategist. Create a punchy, buy-worthy one-pager called 'AI Keyword Playbook' for {$site}.
Country: {$country}. Device: {$device}. Window: last {$days} days.

Sections (short bullets only):
1) Quick Wins (CTR-gap upgrades) — Title patterns + Meta ideas.
2) Big Rocks (pos 11–20 high impr) — content angles.
3) Internal Links — where to link from/topics.
4) FAQ Schema — 4–6 real Q&As.
5) Action Checklist — 6 steps for non-experts.

Top Opportunities:
".implode("\n", array_slice($listOpp,0,8))."

High-Impression Terms:
".implode("\n", array_slice($listImpr,0,8))."
";
    }

    private static function fallback_ai_summary($topOpp, $topImpr): string
    {
        $b = [];
        $b[] = "✅ Quick Wins";
        $i=0; foreach ($topOpp as $x) { if ($i++>=5) break; $b[] = "• ".$x['query']." — Sharper title + stronger meta."; }
        $b[] = "🪨 Big Rocks";
        $i=0; foreach ($topImpr as $x) { if ($i++>=3) break; $b[] = "• ".$x['query']." — Add comparison + FAQs."; }
        $b[] = "🔗 Internal Links — link 3 related posts with natural anchors.";
        $b[] = "❓ FAQ Schema — add 4–6 real questions.";
        $b[] = "🚀 Checklist — titles, links, comparison block, FAQs, re-index.";
        return implode("\n", $b);
    }
}
