<?php

namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

class GSC_Tab_Overview {
    const NS    = 'miro/v1';
    const R_SUM = '/gsc/overview_summary';
    const R_SER = '/gsc/overview_series';
    const R_WIN = '/gsc/overview_winners';
    const R_LOS = '/gsc/overview_losers';
    const R_AI  = '/gsc/overview_ai';    // NEW: site-level series cache (date-only) => matches GSC console totals
    const R_COUNTRIES = '/gsc/overview_countries';    // Top countries by traffic
    const OPT_SITE = 'miro_gsc_site_series';    // Existing query cache (good for keyword tables, NOT for totals)
    const OPT_QUERIES = 'miro_gsc_last_queries';
    const OPT_COUNTRY_SERIES = 'miro_gsc_country_series';    // Country+date cache for country breakdowns

    public static function init(): void {
        add_action('miro_gsc_tabs_nav',     [__CLASS__, 'nav']);
        add_action('miro_gsc_tabs_content', [__CLASS__, 'content']);
        add_action('rest_api_init',         [__CLASS__, 'routes']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
    }

    public static function enqueue_assets($hook): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Page detection for asset loading, not form processing.
        if (!isset($_GET['page']) || $_GET['page'] !== 'miro-gsc-analytics') return;
        if (defined('MIRO_AI_SEO_FILE') && defined('MIRO_AI_SEO_URL') && defined('MIRO_AI_SEO_VERSION')) {
            // Enqueue CSS
            $css_rel = 'assets/css/miro-gsc-overview.css';
            $css_path = plugin_dir_path(MIRO_AI_SEO_FILE) . $css_rel;
            $css_url = MIRO_AI_SEO_URL . $css_rel;
            $css_ver = (file_exists($css_path) && is_readable($css_path)) ? (string) filemtime($css_path) : MIRO_AI_SEO_VERSION;
            wp_enqueue_style('miro-gsc-overview', $css_url, ['miro-analytics'], $css_ver);
            // Enqueue JS
            $js_rel = 'assets/js/miro-gsc-overview.js';
            $js_path = plugin_dir_path(MIRO_AI_SEO_FILE) . $js_rel;
            $js_url = MIRO_AI_SEO_URL . $js_rel;
            $js_ver = (file_exists($js_path) && is_readable($js_path)) ? (string) filemtime($js_path) : MIRO_AI_SEO_VERSION;
            wp_enqueue_script('miro-gsc-overview', $js_url, [], $js_ver, true);
            // Pass config object to JS
            $nonce  = wp_create_nonce('wp_rest');
            $sumURL = esc_js(rest_url(self::NS . self::R_SUM));
            $serURL = esc_js(rest_url(self::NS . self::R_SER));
            $winURL = esc_js(rest_url(self::NS . self::R_WIN));
            $losURL = esc_js(rest_url(self::NS . self::R_LOS));
            $aiURL  = esc_js(rest_url(self::NS . self::R_AI));
            $countriesURL = esc_js(rest_url(self::NS . self::R_COUNTRIES));
            wp_localize_script('miro-gsc-overview', 'MIRO_GSC_OVERVIEW', [
                'nonce'  => $nonce,
                'sumURL' => $sumURL,
                'serURL' => $serURL,
                'winURL' => $winURL,
                'losURL' => $losURL,
                'aiURL'  => $aiURL,
                'countriesURL' => $countriesURL,
            ]);
        }
    }

    public static function nav() {
        echo '<a href="#tab-overview" class="tab-link active">Overview</a>';
    }

    public static function content() {
        ?>
<section id="tab-overview" class="tab-pan" style="display:none;">
  <div class="miro-page-header">
    <div>
      <h2 class="miro-page-header-title">Overview</h2>
      <p class="miro-page-header-sub">Search performance summary, key metrics, and keyword movers.</p>
    </div>
    <div class="miro-page-header-meta" id="ovPageMeta">
      <span id="ovLastSynced">Last synced: —</span>
      <span id="ovRange">—</span>
    </div>
  </div>
  <div class="grid">
    <!-- KPI row -->
    <div class="card kpi miro-card">
      <div class="kpi-title">Search Traffic</div>
      <div class="kpi-help miro-muted">Google Search clicks (property totals).</div>
      <div class="kpi-val" id="kpiClicks">—</div>
      <div class="kpi-delta" id="kpiClicksDelta">? —</div>
    </div>
    <div class="card kpi miro-card">
      <div class="kpi-title">Search Impressions</div>
      <div class="kpi-help miro-muted">Property totals (matches Search Console).</div>
      <div class="kpi-val" id="kpiImpr">—</div>
      <div class="kpi-delta" id="kpiImprDelta">? —</div>
    </div>
    <div class="card kpi miro-card">
      <div class="kpi-title">Total Keywords <span class="badge-est">Estimate</span></div>
      <div class="kpi-help miro-muted">Unique queries (from query cache; can differ slightly).</div>
      <div class="kpi-val" id="kpiKw">—</div>
      <div class="kpi-delta" id="kpiKwDelta">? —</div>
    </div>
    <div class="card kpi miro-card">
      <div class="kpi-title">SEO Visibility</div>
      <div class="kpi-help miro-muted">Internal score based on clicks, impressions &amp; position.</div>
      <div class="kpi-val" id="kpiVis">—</div>
      <div class="kpi-delta" id="kpiVisDelta">? —</div>
    </div>
    <!-- Data Trust -->
    <div class="card wide miro-card">
      <div class="trust-box miro-card-body" id="dataTrust">
        <div class="miro-card-header" style="margin-bottom:12px;">
          <span class="miro-card-title">Data Trust</span>
        </div>
        <div class="trust-text">Totals (Clicks/Impressions/CTR/Avg position) come from <strong>property-level "date" data</strong> and should match Search Console.</div>
        <div class="trust-text">Some widgets marked <span class="badge-est">Estimate</span> use the <strong>query cache</strong> (limited rows). They are directional and can differ slightly.</div>
        <div class="trust-row" id="dataTrustPills"></div>
      </div>
    </div>
    <!-- Key metrics (formerly Donut) -->
    <div class="card wide miro-card ov-metrics-card">
      <div class="miro-card-header">
        <span class="miro-card-title">Key metrics at a glance</span>
      </div>
      <div class="miro-card-body">
        <div class="ov-metrics-grid">
          <div class="ov-metric-card">
            <div class="ov-metric-ring-wrap"><svg id="donutVis" class="ov-metric-ring" viewBox="0 0 100 100"></svg><span class="ov-metric-value" id="donutVisVal">—</span></div>
            <span class="ov-metric-trend" id="donutVisTrend"></span>
            <div class="ov-metric-label">Visibility vs previous period</div>
          </div>
          <div class="ov-metric-card">
            <div class="ov-metric-ring-wrap"><svg id="donutCTR" class="ov-metric-ring" viewBox="0 0 100 100"></svg><span class="ov-metric-value" id="donutCTRVal">—</span></div>
            <span class="ov-metric-trend" id="donutCTRTrend"></span>
            <div class="ov-metric-label">CTR vs previous period</div>
          </div>
          <div class="ov-metric-card">
            <div class="ov-metric-ring-wrap"><svg id="donutRank" class="ov-metric-ring" viewBox="0 0 100 100"></svg><span class="ov-metric-value" id="donutRankVal">—</span></div>
            <span class="ov-metric-trend" id="donutRankTrend"></span>
            <div class="ov-metric-label">Impressions on page 1 (pos 1–10) <span class="badge-est">Est.</span></div>
          </div>
          <div class="ov-metric-card">
            <div class="ov-metric-ring-wrap"><svg id="donutBrand" class="ov-metric-ring" viewBox="0 0 100 100"></svg><span class="ov-metric-value" id="donutBrandVal">—</span></div>
            <span class="ov-metric-trend" id="donutBrandTrend"></span>
            <div class="ov-metric-label">Brand clicks share <span class="badge-est">Est.</span></div>
          </div>
          <div class="ov-metric-card">
            <div class="ov-metric-ring-wrap"><svg id="donutNonBrand" class="ov-metric-ring" viewBox="0 0 100 100"></svg><span class="ov-metric-value" id="donutNonBrandVal">—</span></div>
            <span class="ov-metric-trend" id="donutNonBrandTrend"></span>
            <div class="ov-metric-label">Non-brand clicks share <span class="badge-est">Est.</span></div>
          </div>
        </div>
        <div class="ov-metrics-explain">
          <div class="ov-metrics-explain-inner">
            <div class="ov-metrics-explain-title">How to read</div>
            <ul class="ov-metrics-explain-list">
              <li><strong>Visibility</strong> — trend score vs last period</li>
              <li><strong>CTR</strong> — how well impressions turn into clicks</li>
              <li><strong>Page 1</strong> — share of impressions in top 10 (estimate)</li>
              <li><strong>Brand / Non-brand</strong> — share of branded vs organic SEO clicks</li>
            </ul>
            <div class="ov-metrics-explain-legend">Gold = improvement, red = decline, grey = flat</div>
          </div>
        </div>
        <div id="donutNote" class="donut-note"></div>
      </div>
    </div>
    <!-- AI Insights + Top Countries: side-by-side -->
    <div class="miro-grid-2 ov-grid-span">
      <div class="card miro-card">
        <div class="miro-card-header">
          <span class="miro-card-title">AI Insights — Top 5 Things to Fix Today</span>
        </div>
        <div class="miro-card-body" id="aiInsights">Loading insights…</div>
      </div>
      <div class="card miro-card">
        <div class="miro-card-header">
          <span class="miro-card-title">Top Countries by Traffic</span>
        </div>
        <div class="miro-card-body" id="topCountries">Loading countries…</div>
      </div>
    </div>
    <!-- Trend metric + GSC days selector (centered) -->
    <div class="card wide miro-card ov-trend-metric-card">
      <div class="miro-card-header">
        <span class="miro-card-title">Trend Metric for Winners &amp; Losers</span>
      </div>
      <div class="miro-card-body ov-trend-controls-body">
        <p class="miro-muted ov-trend-controls-desc">Choose metric for trend lines and period for GSC data.</p>
        <div class="ov-controls ov-trend-controls-center">
          <span class="ov-controls-label">Period</span>
          <button type="button" class="ov-btn ov-days-btn" data-ov-days="7">7 days</button>
          <button type="button" class="ov-btn ov-days-btn active" data-ov-days="28">28 days</button>
          <button type="button" class="ov-btn ov-days-btn" data-ov-days="90">90 days</button>
          <span class="ov-controls-divider"></span>
          <span class="ov-controls-label">Trend metric</span>
          <button type="button" class="ov-btn ov-trend-metric" data-trend="clicks">Clicks / day</button>
          <button type="button" class="ov-btn ov-trend-metric active" data-trend="impressions">Impressions / day</button>
          <button type="button" class="ov-btn ov-trend-metric" data-trend="ctr">CTR</button>
          <button type="button" class="ov-btn ov-trend-metric" data-trend="position">Avg position</button>
        </div>
      </div>
    </div>
    <!-- Winning + Losing: side-by-side -->
    <div class="miro-grid-2 ov-grid-span">
      <div class="card miro-card ov-winlose-card">
        <div class="miro-card-header">
          <span class="miro-card-title" id="winnersTitle">Winning Keywords</span>
          <span class="miro-card-meta ov-badge-count" id="winnersCount">0</span>
        </div>
        <div class="miro-card-body ov-winlose-body">
          <p class="miro-small" style="margin:0 0 12px 0;">Trend = selected metric over time for this keyword.</p>
          <div class="ov-table-wrap">
            <table class="ov-winlose-table">
              <thead>
                <tr><th>Query</th><th>Clicks</th><th>CTR</th><th>Pos</th><th id="wTrendHead">Trend (clicks/day)</th></tr>
              </thead>
              <tbody id="winnersBody"></tbody>
            </table>
          </div>
        </div>
      </div>
      <div class="card miro-card ov-winlose-card">
        <div class="miro-card-header">
          <span class="miro-card-title" id="losersTitle">Losing Keywords</span>
          <span class="miro-card-meta ov-badge-count" id="losersCount">0</span>
        </div>
        <div class="miro-card-body ov-winlose-body">
          <p class="miro-small" style="margin:0 0 12px 0;">Trend = selected metric over time for this keyword.</p>
          <div class="ov-table-wrap">
            <table class="ov-winlose-table">
              <thead>
                <tr><th>Query</th><th>Clicks</th><th>CTR</th><th>Pos</th><th id="lTrendHead">Trend (clicks/day)</th></tr>
              </thead>
              <tbody id="losersBody"></tbody>
            </table>
          </div>
        </div>
      </div>
    </div>
  </div>
  <div id="ovSparkTooltip" class="ov-tooltip"></div>
</section>
        <?php
    }

    public static function routes() {
        $perm = fn() => current_user_can(\Miro_AI_SEO\_miro_cap());
        register_rest_route(self::NS, self::R_SUM, ['methods'=>'GET','permission_callback'=>$perm,'callback'=>[__CLASS__,'rest_summary']]);
        register_rest_route(self::NS, self::R_SER, ['methods'=>'GET','permission_callback'=>$perm,'callback'=>[__CLASS__,'rest_series']]);
        register_rest_route(self::NS, self::R_WIN, ['methods'=>'GET','permission_callback'=>$perm,'callback'=>[__CLASS__,'rest_winners']]);
        register_rest_route(self::NS, self::R_LOS, ['methods'=>'GET','permission_callback'=>$perm,'callback'=>[__CLASS__,'rest_losers']]);
        register_rest_route(self::NS, self::R_AI,  ['methods'=>'POST','permission_callback'=>$perm,'callback'=>[__CLASS__,'rest_ai']]);
        register_rest_route(self::NS, self::R_COUNTRIES, ['methods'=>'GET','permission_callback'=>$perm,'callback'=>[__CLASS__,'rest_countries']]);
    }

    /* =======================
       SITE SERIES CACHE (date)
       ======================= */
    private static function ensure_site_cache(int $days = 120) {
        $cache = get_option(self::OPT_SITE);
        $ok    = is_array($cache) && !empty($cache['by_date']) && !empty($cache['range']['end']);
        // refresh if missing or older than 6h
        $stale = true;
        if ($ok && !empty($cache['updated_at'])) {
            $stale = (time() - intval($cache['updated_at'])) > (6 * HOUR_IN_SECONDS);
        }
        if ($ok && !$stale) {
            return $cache;
        }
        $rebuilt = self::rebuild_site_cache($days);
        if (is_wp_error($rebuilt)) {
            // if we have old cache, return it anyway
            return $ok ? $cache : $rebuilt;
        }
        return $rebuilt;
    }

    private static function rebuild_site_cache(int $days = 120) {
        // Use site timezone and avoid "today" partial data
        $tz    = wp_timezone();
        $today = new \DateTime('today', $tz);
        $end   = (clone $today)->modify('-1 day');
        $start = (clone $end)->modify('-' . ($days - 1) . ' days');
        // IMPORTANT: date-only report => matches GSC console totals
        $resp = \Miro_AI_SEO\GSC_Analytics_Core::gsc_query(
            $start,
            $end,
            \Miro_AI_SEO\GSC_Analytics_Core::payload_dimensions(['date'], [], 1000)
        );
        if (is_wp_error($resp)) return $resp;
        $by = [];
        foreach (($resp['rows'] ?? []) as $row) {
            $d = (string)($row['keys'][0] ?? '');
            if ($d === '') continue;
            $clicks = floatval($row['clicks'] ?? 0);
            $impr   = floatval($row['impressions'] ?? 0);
            // GSC gives ctr/position already aggregated at property level for date dimension
            $ctr    = floatval($row['ctr'] ?? 0);
            $pos    = floatval($row['position'] ?? 0);
            // If no impressions, treat as "no signal": CTR=0, Position=100 (worst) to avoid fake boosts in score
            if ($impr <= 0) {
                $ctr = 0.0;
                $pos = 100.0;
            }
            $by[$d] = [
                'clicks'      => $clicks,
                'impressions' => $impr,
                'ctr'         => $ctr,
                'position'    => $pos,
            ];
        }
        if (!$by) {
            $empty = [
                'updated_at' => time(),
                'range'      => ['start'=>null,'end'=>null],
                'by_date'    => [],
            ];
            update_option(self::OPT_SITE, $empty, false);
            return $empty;
        }
        ksort($by);
        $startDate = array_key_first($by);
        $endDate   = array_key_last($by);
        $cache = [
            'updated_at' => time(),
            'range'      => ['start'=>$startDate,'end'=>$endDate],
            'by_date'    => $by,
        ];
        update_option(self::OPT_SITE, $cache, false);
        return $cache;
    }

    /* =======================
       COUNTRY SERIES CACHE (country, date)
       ======================= */
    private static function ensure_country_cache(int $days = 120) {
        $cache = get_option(self::OPT_COUNTRY_SERIES);
        $ok    = is_array($cache) && !empty($cache['by_date_country']) && !empty($cache['range']['end']);
        // refresh if missing or older than 6h
        $stale = true;
        if ($ok && !empty($cache['updated_at'])) {
            $stale = (time() - intval($cache['updated_at'])) > (6 * HOUR_IN_SECONDS);
        }
        if ($ok && !$stale) {
            return $cache;
        }
        $rebuilt = self::rebuild_country_cache($days);
        if (is_wp_error($rebuilt)) {
            // if we have old cache, return it anyway
            return $ok ? $cache : $rebuilt;
        }
        return $rebuilt;
    }

    private static function rebuild_country_cache(int $days = 120) {
        // Use site timezone and avoid "today" partial data
        $tz    = wp_timezone();
        $today = new \DateTime('today', $tz);
        $end   = (clone $today)->modify('-1 day');
        $start = (clone $end)->modify('-' . ($days - 1) . ' days');
        
        // Fetch country+date dimensions from GSC
        $resp = \Miro_AI_SEO\GSC_Analytics_Core::gsc_query(
            $start,
            $end,
            \Miro_AI_SEO\GSC_Analytics_Core::payload_dimensions(['country', 'date'], [], 25000)
        );
        if (is_wp_error($resp)) return $resp;
        
        $by_date_country = [];
        foreach (($resp['rows'] ?? []) as $row) {
            $keys = $row['keys'] ?? [];
            $country = strtoupper(trim((string)($keys[0] ?? '')));
            $d = (string)($keys[1] ?? '');
            
            // Skip empty country or date
            if ($country === '' || $d === '') continue;
            
            $clicks = floatval($row['clicks'] ?? 0);
            $impr   = floatval($row['impressions'] ?? 0);
            $ctr    = floatval($row['ctr'] ?? 0);
            $pos    = floatval($row['position'] ?? 0);
            
            // If no impressions, treat as "no signal": CTR=0, Position=100 (worst)
            if ($impr <= 0) {
                $ctr = 0.0;
                $pos = 100.0;
            }
            
            if (!isset($by_date_country[$d])) {
                $by_date_country[$d] = [];
            }
            
            $by_date_country[$d][$country] = [
                'clicks'      => $clicks,
                'impressions' => $impr,
                'ctr'         => $ctr,
                'position'    => $pos,
            ];
        }
        
        if (!$by_date_country) {
            $empty = [
                'updated_at' => time(),
                'range'      => ['start'=>null,'end'=>null],
                'by_date_country' => [],
            ];
            update_option(self::OPT_COUNTRY_SERIES, $empty, false);
            return $empty;
        }
        
        ksort($by_date_country);
        $startDate = array_key_first($by_date_country);
        $endDate   = array_key_last($by_date_country);
        
        $cache = [
            'updated_at' => time(),
            'range'      => ['start'=>$startDate,'end'=>$endDate],
            'by_date_country' => $by_date_country,
        ];
        update_option(self::OPT_COUNTRY_SERIES, $cache, false);
        return $cache;
    }

    /* SUMMARY: totals based on date-only cache (matches Console) */
    public static function rest_summary(\WP_REST_Request $r) {
        $days = max(7, min(90, intval($r->get_param('days') ?: 28)));
        $site = self::ensure_site_cache(120);
        if (is_wp_error($site)) {
            return [
                'ok'=>true,
                'search_traffic'=>0,'impressions'=>0,'total_keywords'=>0,'delta_keywords'=>0,
                'visibility'=>0,'visibility_norm'=>0,'ctr_norm'=>0,'clicks'=>0,'ctr'=>0,'position'=>0,
                'page1_share'=>0,'page1_impressions'=>0,'brand_clicks'=>0,'nonbrand_clicks'=>0,
                'brand_share'=>0,'nonbrand_share'=>0,
                'delta'=>['visibility'=>0,'clicks'=>0,'impressions'=>0,'ctr'=>0,'position'=>0,'page1_share'=>0,'brand_share'=>0,'nonbrand_share'=>0],
                'trust'=>['site_ok'=>false,'query_ok'=>false],
            ];
        }
        $by = $site['by_date'] ?? [];
        if (!$by) {
            return [
                'ok'=>true,
                'search_traffic'=>0,'impressions'=>0,'total_keywords'=>0,'delta_keywords'=>0,
                'visibility'=>0,'visibility_norm'=>0,'ctr_norm'=>0,'clicks'=>0,'ctr'=>0,'position'=>0,
                'page1_share'=>0,'page1_impressions'=>0,'brand_clicks'=>0,'nonbrand_clicks'=>0,
                'brand_share'=>0,'nonbrand_share'=>0,
                'delta'=>['visibility'=>0,'clicks'=>0,'impressions'=>0,'ctr'=>0,'position'=>0,'page1_share'=>0,'brand_share'=>0,'nonbrand_share'=>0],
                'trust'=>['site_ok'=>false,'query_ok'=>false],
            ];
        }
        $dates = array_keys($by);
        sort($dates);
        $last   = end($dates);
        $lastTs = strtotime($last);
        $currStartTs = $lastTs - 86400 * ($days - 1);
        $prevEndTs   = $currStartTs - 86400;
        $prevStartTs = $prevEndTs - 86400 * ($days - 1);
        $sumWindow = function(int $startTs, int $endTs) use ($by) {
            $clicks = 0.0; $impr = 0.0;
            $posW   = 0.0; $posImpr = 0.0;
            foreach ($by as $d => $m) {
                $ts = strtotime($d);
                if ($ts === false || $ts < $startTs || $ts > $endTs) continue;
                $c = floatval($m['clicks'] ?? 0);
                $i = floatval($m['impressions'] ?? 0);
                $p = floatval($m['position'] ?? 100);
                $clicks += $c;
                $impr   += $i;
                if ($i > 0) {
                    // impression-weighted position
                    $posW += ($p * $i);
                    $posImpr += $i;
                }
            }
            $ctr = ($impr > 0) ? ($clicks / $impr) : 0.0;
            // if no impression-weighted position signal, treat as worst
            $pos = ($posImpr > 0) ? ($posW / $posImpr) : 100.0;
            // internal visibility score (safe: pos=100 when no data)
            $vis = ($clicks*2) + ($impr*0.03) + ((100 - $pos) * 5);
            return compact('clicks','impr','ctr','pos','vis');
        };
        $curr = $sumWindow($currStartTs, $lastTs);
        $prev = $sumWindow($prevStartTs, $prevEndTs);
        // Keywords count uses query cache (estimate)
        $kwCurrVal = 0; $kwPrevVal = 0;
        $qopt  = get_option(self::OPT_QUERIES);
        $qrows = is_array($qopt['rows'] ?? null) ? $qopt['rows'] : [];
        if ($qrows) {
            $seenC = []; $seenP = [];
            foreach ($qrows as $row) {
                $d = (string)($row['date'] ?? '');
                $q = (string)($row['query'] ?? '');
                if ($d==='' || $q==='') continue;
                $ts = strtotime($d);
                if ($ts === false) continue;
                $im  = floatval($row['impressions'] ?? 0);
                $pos = floatval($row['position'] ?? 0);
                if ($im <= 0 || $pos <= 0 || $pos > 100) continue;
                if ($ts >= $currStartTs && $ts <= $lastTs) $seenC[$q] = true;
                if ($ts >= $prevStartTs && $ts <= $prevEndTs) $seenP[$q] = true;
            }
            $kwCurrVal = count($seenC);
            $kwPrevVal = count($seenP);
        }
        // brand shares & page1 share use query cache (estimate)
        $brandCurr=0.0; $nonCurr=0.0; $brandPrev=0.0; $nonPrev=0.0;
        $page1_c=0.0; $off_c=0.0; $page1_p=0.0; $off_p=0.0;
        $blog   = strtolower(get_bloginfo('name') ?: '');
        $domain = strtolower(wp_parse_url(home_url(), PHP_URL_HOST) ?: '');
        $tokensRaw = preg_split('/[\s\.\-]+/', $blog . ' ' . $domain);
        $tokens = array_values(array_filter($tokensRaw, function($t){
            $t = trim($t);
            if ($t === '') return false;
            $ban = ['www','com','net','org','app','apps'];
            if (in_array($t,$ban,true)) return false;
            return strlen($t) >= 3;
        }));
        foreach ($qrows as $row) {
            $d  = (string)($row['date'] ?? '');
            $q  = strtolower((string)($row['query'] ?? ''));
            if ($d==='' || $q==='') continue;
            $ts = strtotime($d);
            if ($ts === false) continue;
            $cl  = floatval($row['clicks'] ?? 0);
            $im  = floatval($row['impressions'] ?? 0);
            $pos = floatval($row['position'] ?? 0);
            $isBrand = false;
            foreach ($tokens as $tok){
                if ($tok !== '' && strpos($q, $tok) !== false){ $isBrand = true; break; }
            }
            if ($ts >= $currStartTs && $ts <= $lastTs) {
                if ($isBrand) $brandCurr += $cl; else $nonCurr += $cl;
                if ($im > 0 && $pos > 0) { if ($pos <= 10) $page1_c += $im; else $off_c += $im; }
            } elseif ($ts >= $prevStartTs && $ts <= $prevEndTs) {
                if ($isBrand) $brandPrev += $cl; else $nonPrev += $cl;
                if ($im > 0 && $pos > 0) { if ($pos <= 10) $page1_p += $im; else $off_p += $im; }
            }
        }
        $totCurrClicks = $brandCurr + $nonCurr;
        $totPrevClicks = $brandPrev + $nonPrev;
        $brandShareC    = $totCurrClicks > 0 ? $brandCurr / $totCurrClicks : 0;
        $nonBrandShareC = $totCurrClicks > 0 ? $nonCurr   / $totCurrClicks : 0;
        $brandShareP    = $totPrevClicks > 0 ? $brandPrev / $totPrevClicks : 0;
        $nonBrandShareP = $totPrevClicks > 0 ? $nonPrev   / $totPrevClicks : 0;
        $denC = $page1_c + $off_c;
        $denP = $page1_p + $off_p;
        $page1ShareC = $denC > 0 ? $page1_c / $denC : 0;
        $page1ShareP = $denP > 0 ? $page1_p / $denP : 0;
        // trust meta for UI
        $siteUpdated = intval($site['updated_at'] ?? 0);
        $siteRange   = is_array($site['range'] ?? null) ? $site['range'] : ['start'=>null,'end'=>null];
        $qUpdated = 0;
        if (is_array($qopt) && isset($qopt['updated_at'])) $qUpdated = intval($qopt['updated_at']);
        $qRowsCount = is_array($qrows) ? count($qrows) : 0;
        return [
            'ok'               => true,
            'search_traffic'   => $curr['clicks'],
            'impressions'      => $curr['impr'],
            'total_keywords'   => $kwCurrVal,
            'delta_keywords'   => $kwCurrVal - $kwPrevVal,
            'visibility'       => $curr['vis'],
            'visibility_norm'  => max(0,min(1,$curr['vis']/10000)),
            'ctr_norm'         => max(0,min(1,$curr['ctr']/0.3)),
            'clicks'           => $curr['clicks'],
            'ctr'              => $curr['ctr'],
            'position'         => $curr['pos'],
            'page1_share'      => $page1ShareC,
            'page1_impressions'=> $page1_c,
            'brand_clicks'     => $brandCurr,
            'nonbrand_clicks'  => $nonCurr,
            'brand_share'      => $brandShareC,
            'nonbrand_share'   => $nonBrandShareC,
            'delta' => [
                'visibility'      => $curr['vis']   - $prev['vis'],
                'clicks'          => $curr['clicks']- $prev['clicks'],
                'impressions'     => $curr['impr']  - $prev['impr'],
                'ctr'             => $curr['ctr']   - $prev['ctr'],
                'position'        => $curr['pos']   - $prev['pos'],
                'page1_share'     => $page1ShareC   - $page1ShareP,
                'brand_share'     => $brandShareC   - $brandShareP,
                'nonbrand_share'  => $nonBrandShareC- $nonBrandShareP,
            ],
            'trust' => [
                'site_ok'         => true,
                'site_updated_at' => $siteUpdated,
                'site_range'      => $siteRange,
                'query_ok'        => ($qRowsCount > 0),
                'query_updated_at'=> $qUpdated,
                'query_rows'      => $qRowsCount,
            ],
        ];
    }

    /* SITE-LEVEL SERIES: date-only cache => matches Console */
    public static function rest_series(\WP_REST_Request $r) {
        $days = max(7, min(90, intval($r->get_param('days') ?: 28)));
        $site = self::ensure_site_cache(120);
        if (is_wp_error($site)) {
            return [
                'ok'=>true,'range'=>['start'=>null,'end'=>null],'dates'=>[],
                'clicks'=>[],'impressions'=>[],'position'=>[],'ctr'=>[],'score'=>[],
            ];
        }
        $by = $site['by_date'] ?? [];
        if (!$by) {
            return [
                'ok'=>true,'range'=>['start'=>null,'end'=>null],'dates'=>[],
                'clicks'=>[],'impressions'=>[],'position'=>[],'ctr'=>[],'score'=>[],
            ];
        }
        $dates = array_keys($by);
        sort($dates);
        $last   = end($dates);
        $tz = wp_timezone();
        $startD = (new \DateTime($last, $tz))->modify('-' . ($days - 1) . ' days');
        $clicks=[]; $impr=[]; $posArr=[]; $ctrArr=[]; $score=[]; $dateArr=[];
        $cursor = clone $startD;
        $endD   = new \DateTime($last, $tz);
        while ($cursor <= $endD) {
            $d = $cursor->format('Y-m-d');
            $m = $by[$d] ?? ['clicks'=>0,'impressions'=>0,'ctr'=>0,'position'=>100];
            $c  = floatval($m['clicks'] ?? 0);
            $i  = floatval($m['impressions'] ?? 0);
            $ct = floatval($m['ctr'] ?? 0);
            $p  = floatval($m['position'] ?? 100);
            // Safety: on 0 impression days, no "free score"
            if ($i <= 0) {
                $ct = 0.0;
                $p  = 100.0;
            }
            $clicks[] = $c;
            $impr[]   = $i;
            $posArr[] = $p;
            $ctrArr[] = $ct;
            $score[]  = ($c*2) + ($i*0.03) + ((100-$p)*5);
            $dateArr[]= $d;
            $cursor->modify('+1 day');
        }
        $startDate = $dateArr ? $dateArr[0] : null;
        $endDate   = $dateArr ? $dateArr[count($dateArr)-1] : null;
        return [
            'ok'          => true,
            'range'       => ['start'=>$startDate,'end'=>$endDate],
            'dates'       => $dateArr,
            'clicks'      => $clicks,
            'impressions' => $impr,
            'position'    => $posArr,
            'ctr'         => $ctrArr,
            'score'       => $score,
        ];
    }

    public static function rest_ai(\WP_REST_Request $r){
        $body  = json_decode($r->get_body(), true) ?: [];
        $delta = $body['delta'] ?? [];
        $fixes = [];
        if (!empty($delta['position']) && $delta['position'] < 0) $fixes[] = 'Rankings improved this period — update top pages with fresh content, FAQs and internal links to lock in gains.';
        elseif (!empty($delta['position']) && $delta['position'] > 0) $fixes[] = 'Some pages lost rankings — compare them with top competitors and improve content depth and relevance.';
        if (!empty($delta['ctr']) && $delta['ctr'] < 0) $fixes[] = 'CTR dropped — rewrite titles and meta descriptions for key pages to better match search intent and add a clear benefit.';
        elseif (!empty($delta['ctr']) && $delta['ctr'] > 0) $fixes[] = 'CTR improved — replicate the winning title/meta patterns across similar pages.';
        if (!empty($delta['impressions']) && $delta['impressions'] > 0 && (!empty($delta['ctr']) && $delta['ctr'] <= 0)) {
            $fixes[] = 'Impressions are up but CTR is flat/down — strengthen titles with numbers, power words, and solve-the-problem angles.';
        }
        if (!empty($delta['clicks']) && $delta['clicks'] < 0 && (!empty($delta['position']) && $delta['position'] < 0)) {
            $fixes[] = 'Clicks dropped even though rankings improved — search demand may be shifting, target more long-tail variations of your main keywords.';
        } elseif (!empty($delta['clicks']) && $delta['clicks'] < 0) {
            $fixes[] = 'Clicks dropped — check for technical issues (noindex, canonicals, redirects) and make sure important pages are still in the sitemap.';
        }
        if (count($fixes) < 5) {
            $fixes[] = 'Add FAQ sections and FAQ schema to important posts to capture more rich results.';
            $fixes[] = 'Add internal links from top-traffic posts to money pages to push more authority.';
            $fixes[] = 'Refresh older content with updated examples, screenshots, and current year references.';
            $fixes[] = 'Check mobile performance and page speed (Core Web Vitals) for your top landing pages.';
            $fixes[] = 'Merge overlapping/duplicate content and redirect weaker pages into stronger ones.';
        }
        $fixes = array_slice($fixes, 0, 5);
        $html = '<ul style="margin:0;padding-left:18px;">';
        foreach ($fixes as $li) $html .= '<li>' . esc_html($li) . '</li>';
        $html .= '</ul>';
        return ['ok' => true, 'html' => $html];
    }

    public static function rest_winners(\WP_REST_Request $r){ return self::movers($r,false); }
    public static function rest_losers(\WP_REST_Request $r){ return self::movers($r,true); }

    /**
     * REST: Top countries by traffic
     * Uses dedicated country+date cache (OPT_COUNTRY_SERIES), not query cache
     */
    public static function rest_countries(\WP_REST_Request $r) {
        $days = max(7, min(90, intval($r->get_param('days') ?: 28)));
        
        $cache = self::ensure_country_cache(120);
        if (is_wp_error($cache)) {
            return ['ok' => false, 'msg' => 'Failed to load country cache'];
        }
        
        if (!is_array($cache) || empty($cache['by_date_country']) || empty($cache['range']['end'])) {
            return ['ok' => false, 'msg' => 'No country data available'];
        }
        
        $by_date_country = $cache['by_date_country'] ?? [];
        $lastDate = $cache['range']['end'] ?? '';
        
        if (!$lastDate) {
            return ['ok' => false, 'msg' => 'No date data'];
        }
        
        // Determine window: end = last date in cache, start = end - (days-1)
        $endTs = strtotime($lastDate);
        if ($endTs === false) {
            return ['ok' => false, 'msg' => 'Invalid end date'];
        }
        $startTs = $endTs - 86400 * ($days - 1);
        
        // Aggregate across window by country
        $byCountry = [];
        foreach ($by_date_country as $date => $countries) {
            $ts = strtotime($date);
            if ($ts === false || $ts < $startTs || $ts > $endTs) continue;
            
            foreach ($countries as $country => $metrics) {
                // Exclude empty and UNKNOWN countries
                $country = strtoupper(trim($country));
                if ($country === '' || $country === 'UNKNOWN') continue;
                
                if (!isset($byCountry[$country])) {
                    $byCountry[$country] = [
                        'clicks' => 0.0,
                        'impressions' => 0.0,
                        'pos_w' => 0.0,
                        'pos_impr' => 0.0,
                    ];
                }
                
                $c = floatval($metrics['clicks'] ?? 0);
                $i = floatval($metrics['impressions'] ?? 0);
                $p = floatval($metrics['position'] ?? 0);
                
                $byCountry[$country]['clicks'] += $c;
                $byCountry[$country]['impressions'] += $i;
                
                // Impression-weighted position
                if ($i > 0 && $p > 0) {
                    $byCountry[$country]['pos_w'] += ($p * $i);
                    $byCountry[$country]['pos_impr'] += $i;
                }
            }
        }
        
        // Calculate CTR and position for each country
        $out = [];
        foreach ($byCountry as $country => $data) {
            $ctr = ($data['impressions'] > 0) ? ($data['clicks'] / $data['impressions']) : 0.0;
            $pos = ($data['pos_impr'] > 0) ? ($data['pos_w'] / $data['pos_impr']) : 0.0;
            
            $out[] = [
                'country' => $country,
                'clicks' => round($data['clicks']),
                'impressions' => round($data['impressions']),
                'ctr' => round($ctr * 100, 2),  // Return as percent 0..100
                'position' => round($pos, 2),
            ];
        }
        
        // Sort by clicks descending
        usort($out, function($a, $b) {
            return $b['clicks'] <=> $a['clicks'];
        });
        
        // Determine actual range used
        $actualStart = '';
        $actualEnd = $lastDate;
        foreach (array_keys($by_date_country) as $d) {
            $ts = strtotime($d);
            if ($ts !== false && $ts >= $startTs && $ts <= $endTs) {
                if ($actualStart === '' || $d < $actualStart) {
                    $actualStart = $d;
                }
            }
        }
        if ($actualStart === '') {
            $actualStart = $actualEnd;
        }
        
        return [
            'ok' => true,
            'range' => ['start' => $actualStart, 'end' => $actualEnd],
            'countries' => array_slice($out, 0, 10), // Top 10
        ];
    }

    /**
     * Movers is live GSC. Fixed: impression-weighted position aggregation (matches GSC better).
     */
    private static function movers(\WP_REST_Request $r, bool $losing){
        $window = max(3, intval( $r->get_param('window') ?: 7 ));
        $tz     = wp_timezone();
        $today  = new \DateTime('today', $tz);
        $end2   = (clone $today)->modify('-1 day');
        $start2 = (clone $end2)->modify('-' . ($window - 1) . ' days');
        $end1   = (clone $start2)->modify('-1 day');
        $start1 = (clone $end1)->modify('-' . ($window - 1) . ' days');
        $fetch = function(\DateTime $s, \DateTime $e){
            $resp = \Miro_AI_SEO\GSC_Analytics_Core::gsc_query(
                $s,
                $e,
                \Miro_AI_SEO\GSC_Analytics_Core::payload_dimensions(['query','date'], [], 500)
            );
            if (is_wp_error($resp)) return $resp;
            $agg = [];
            foreach (($resp['rows'] ?? []) as $row) {
                $q = (string)($row['keys'][0] ?? '');
                $d = (string)($row['keys'][1] ?? '');
                if ($q === '' || $d === '') continue;
                if (!isset($agg[$q])) {
                    $agg[$q] = [
                        'clicks'    => 0.0,
                        'impr'      => 0.0,
                        'pos_w'     => 0.0, // position * impressions
                        'pos_impr'  => 0.0, // impressions used for position weighting
                        'd_clicks'  => [],
                        'd_impr'    => [],
                        'd_ctr'     => [],
                        'd_pos'     => [],
                    ];
                }
                $cl  = floatval($row['clicks'] ?? 0);
                $im  = floatval($row['impressions'] ?? 0);
                $pos = floatval($row['position'] ?? 0);
                $ctr = $im > 0 ? $cl / $im : 0.0;
                $agg[$q]['clicks']  += $cl;
                $agg[$q]['impr']    += $im;
                if ($im > 0 && $pos > 0) {
                    $agg[$q]['pos_w']    += ($pos * $im);
                    $agg[$q]['pos_impr'] += $im;
                }
                $agg[$q]['d_clicks'][$d] = (int)round($cl);
                $agg[$q]['d_impr'][$d]   = (int)round($im);
                $agg[$q]['d_ctr'][$d]    = $ctr;
                $agg[$q]['d_pos'][$d]    = $pos;
            }
            foreach ($agg as $k => $v) {
                $agg[$k]['position'] = ($v['pos_impr'] > 0) ? ($v['pos_w'] / $v['pos_impr']) : 0.0;
            }
            return $agg;
        };
        $prevWin = $fetch($start1, $end1);
        if (is_wp_error($prevWin)) return $prevWin;
        $currWin = $fetch($start2, $end2);
        if (is_wp_error($currWin)) return $currWin;
        $out = [];
        foreach ($currWin as $q => $cur) {
            $prev = $prevWin[$q] ?? [
                'clicks'=>0,'impr'=>0,'position'=>0,'d_clicks'=>[],'d_impr'=>[],'d_ctr'=>[],'d_pos'=>[],
            ];
            $prevClicks  = floatval($prev['clicks'] ?? 0);
            $currClicks  = floatval($cur['clicks']  ?? 0);
            $deltaClicks = $currClicks - $prevClicks;
            $sparkClicks=[]; $sparkImpr=[]; $sparkCtr=[]; $sparkPos=[];
            $cursor = clone $start1;
            while ($cursor <= $end2) {
                $k = $cursor->format('Y-m-d');
                $cl = ($cursor <= $end1) ? (int)($prev['d_clicks'][$k] ?? 0) : (int)($cur['d_clicks'][$k] ?? 0);
                $im = ($cursor <= $end1) ? (int)($prev['d_impr'][$k]   ?? 0) : (int)($cur['d_impr'][$k]   ?? 0);
                $ct = ($cursor <= $end1) ? (float)($prev['d_ctr'][$k]  ?? 0) : (float)($cur['d_ctr'][$k]  ?? 0);
                $po = ($cursor <= $end1) ? (float)($prev['d_pos'][$k]  ?? 0) : (float)($cur['d_pos'][$k]  ?? 0);
                $sparkClicks[]=$cl; $sparkImpr[]=$im; $sparkCtr[]=$ct; $sparkPos[]=$po;
                $cursor->modify('+1 day');
            }
            $out[] = [
                'query'        => $q,
                'clicks'       => intval(round($currClicks)),
                'delta_clicks' => intval(round($deltaClicks)),
                'ctr'          => ($cur['impr'] > 0 ? $cur['clicks'] / $cur['impr'] : 0),
                'position'     => floatval($cur['position'] ?? 0),
                'spark_clicks' => $sparkClicks,
                'spark_impr'   => $sparkImpr,
                'spark_ctr'    => $sparkCtr,
                'spark_pos'    => $sparkPos,
                'spark'        => $sparkClicks,
            ];
        }
        usort($out, function($A,$B) use ($losing){
            return $losing ? ($A['delta_clicks'] <=> $B['delta_clicks']) : ($B['delta_clicks'] <=> $A['delta_clicks']);
        });
        return [
            'ok'    => true,
            'items' => array_slice($out, 0, 20),
            'range' => ['start' => $start1->format('Y-m-d'), 'end' => $end2->format('Y-m-d')],
        ];
    }
}
