<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

class Miro_GSC_Connect
{
    const MENU_PARENT          = 'miro-ai-seo';
    const MENU_SLUG            = 'miro-gsc';

    const OPT_SETTINGS         = 'pp_miro_gsc_settings';
    const OPT_STATUS           = 'pp_miro_gsc_status';
    const OPT_PROPERTY         = 'pp_miro_gsc_property';
    const OPT_PROPERTIES_CACHE = 'pp_miro_gsc_properties';

    const DEFAULT_BROKER_BASE  = 'https://connect.miroseo.com';

    const REST_NS              = 'miro/v1';
    const REST_CB_ROUTE        = '/gsc/oauth/callback';  // Match other routes in codebase
    const REST_OAUTH_SECRET    = '/gsc/oauth/secret';     // Broker exchanges one-time token for secret (no secret in URL)
    const REST_LIST_PROPS      = '/gsc/properties';       // Match other routes in codebase
    const OAUTH_TOKEN_PREFIX   = 'miro_gsc_oauth_token_';

    const GOOGLE_TOKEN_URL     = 'https://oauth2.googleapis.com/token';
    const GSC_API_BASE         = 'https://searchconsole.googleapis.com/webmasters/v3';

    const STATE_PREFIX         = 'miro_gsc_state_';

    public static function init(): void
    {
        add_action('admin_menu', [__CLASS__, 'menu'], 12);
        add_action('admin_post_miro_gsc_disconnect', [__CLASS__, 'disconnect']);
        add_action('admin_post_miro_gsc_save_secret', [__CLASS__, 'save_hmac_secret']);
        add_action('admin_post_miro_gsc_save_property', [__CLASS__, 'save_property']);

        // Ensure broker secret exists on init
        self::ensure_broker_secret();

        // Register REST routes - use direct method call instead of closure
        add_action('rest_api_init', [__CLASS__, 'register_rest_routes'], 10);

        // Ensure our full-width CSS is loaded here as well
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
    }

    /**
     * Register REST API routes
     */
    public static function register_rest_routes(): void {
        // Public endpoint for external OAuth broker — security enforced via state + HMAC inside callback. POST only.
        register_rest_route(self::REST_NS, self::REST_CB_ROUTE, [
            'methods'             => 'POST',
            'permission_callback' => [__CLASS__, 'oauth_callback_permission'],
            'callback'             => [__CLASS__, 'oauth_callback'],
        ]);

        // Broker exchanges one-time token for secret (secret never sent in URL).
        register_rest_route(self::REST_NS, self::REST_OAUTH_SECRET, [
            'methods'             => 'POST',
            'permission_callback' => [__CLASS__, 'oauth_secret_permission'],
            'callback'             => [__CLASS__, 'oauth_secret_exchange'],
            'args'                 => [
                'token' => [
                    'required'          => true,
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ]);

        register_rest_route(self::REST_NS, self::REST_LIST_PROPS, [
            'methods'             => 'GET',
            'permission_callback' => function(){ return current_user_can(\miro_ai_cap()); },
            'callback'            => [__CLASS__, 'rest_list_properties'],
        ]);
    }

    /**
     * Ensure broker secret exists in options. Generate if missing.
     * Never log or print the secret.
     */
    private static function ensure_broker_secret(): void
    {
        $secret = get_option('miro_broker_secret', '');
        if (empty($secret) || !is_string($secret) || strlen($secret) < 32) {
            // Generate secure random secret (32-64 chars, alphanumeric + special)
            $length = random_int(32, 64);
            $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*()_+-=[]{}|;:,.<>?';
            $secret = '';
            $max = strlen($chars) - 1;
            for ($i = 0; $i < $length; $i++) {
                $secret .= $chars[random_int(0, $max)];
            }
            update_option('miro_broker_secret', $secret, false); // autoload = false
        }
    }

    /**
     * Get site ID (unique identifier for this WordPress installation)
     */
    private static function get_site_id(): string
    {
        $site_id = get_option('miro_site_id', '');
        if (empty($site_id)) {
            $site_id = wp_generate_password(32, false);
            update_option('miro_site_id', $site_id, false);
        }
        return $site_id;
    }

    /**
     * Public endpoint for external OAuth broker — security enforced via state + HMAC inside callback.
     * Rejects non-POST; invalid/missing body or headers are rejected inside oauth_callback().
     */
    public static function oauth_callback_permission(\WP_REST_Request $r): bool
    {
        return $r->get_method() === 'POST';
    }

    /**
     * Permission callback for oauth/secret: POST only, valid one-time token required in JSON body.
     * Keeps endpoint usable by broker without WordPress auth; rejects missing/empty/invalid token.
     */
    public static function oauth_secret_permission(\WP_REST_Request $r): bool
    {
        if ($r->get_method() !== 'POST') {
            return false;
        }
        $raw = $r->get_body();
        if (!is_string($raw) || $raw === '') {
            return false;
        }
        $data = json_decode($raw, true);
        if (!is_array($data) || !isset($data['token'])) {
            return false;
        }
        $token = is_string($data['token']) ? trim($data['token']) : '';
        if ($token === '' || strlen($token) < 16 || strlen($token) > 64) {
            return false;
        }
        if (!ctype_alnum($token)) {
            return false;
        }
        return true;
    }

    /**
     * Exchange one-time token for broker secret. Broker calls this (server-side) so the secret
     * is never sent in the OAuth start URL. Secret remains in options; transient is used only for this exchange.
     */
    public static function oauth_secret_exchange(\WP_REST_Request $r)
    {
        $token = $r->get_param('token');
        if (empty($token) || !is_string($token)) {
            return new \WP_Error('bad_request', 'Missing or invalid token.', ['status' => 400]);
        }
        $token = sanitize_text_field($token);
        $transient_key = self::OAUTH_TOKEN_PREFIX . $token;
        $secret = get_transient($transient_key);
        if ($secret === false || !is_string($secret)) {
            return new \WP_Error('invalid_token', 'Invalid or expired token.', ['status' => 403]);
        }
        delete_transient($transient_key);
        return ['secret' => $secret];
    }

    public static function enqueue_assets($hook): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin page check for enqueue only; value sanitized.
        $page = isset($_GET['page']) ? sanitize_key((string) wp_unslash($_GET['page'])) : '';
        if ($page !== self::MENU_SLUG) return;
        if (!defined('MIRO_AI_SEO_FILE')) return;

        $rel  = 'assets/css/miro-analytics.css';
        $url  = plugins_url($rel, MIRO_AI_SEO_FILE);
        $path = plugin_dir_path(MIRO_AI_SEO_FILE) . $rel;
        $ver  = (file_exists($path) && is_readable($path)) ? (string) filemtime($path) : (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');

        wp_enqueue_style('miro-analytics', $url, [], $ver);
        wp_enqueue_style('miro-gsc-connect-css', plugins_url('assets/css/miro-gsc-connect.css', MIRO_AI_SEO_FILE), [], $ver);
        
        // Enqueue Fix Center style hero CSS
        $rel_alt  = 'assets/css/miro-alt.css';
        $url_alt  = plugins_url($rel_alt, MIRO_AI_SEO_FILE);
        $path_alt = plugin_dir_path(MIRO_AI_SEO_FILE) . $rel_alt;
        $ver_alt  = (file_exists($path_alt) && is_readable($path_alt)) ? (string) filemtime($path_alt) : (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');
        
        wp_enqueue_style('miro-alt', $url_alt, ['ppmiro-admin'], $ver_alt);
    }

    public static function menu(): void
    {
        if (function_exists('miro_ai_add_submenu_once')) {
            \miro_ai_add_submenu_once(self::MENU_PARENT, 'Google Search Console', 'Google Search Console', \miro_ai_cap(), self::MENU_SLUG, [__CLASS__, 'render'], 10);
        } else {
            add_submenu_page(self::MENU_PARENT, 'Google Search Console', 'Google Search Console', \miro_ai_cap(), self::MENU_SLUG, [__CLASS__, 'render'], 10);
        }
    }

    public static function render(): void
    {
        if (!current_user_can(\miro_ai_cap())) wp_die(esc_html__('You do not have permission.', 'miro-ai-seo-free'));

        $s           = get_option(self::OPT_SETTINGS, []);
        $connected   = !empty($s['refresh_token']);
        $access_exp  = !empty($s['access_exp']) ? intval($s['access_exp']) : 0;
        // Auto-generated broker settings (hidden from users)
        $broker      = self::DEFAULT_BROKER_BASE; // Always https://connect.miroseo.com
        // Ensure hardcoded values are saved in options
        if (empty($s['broker_base']) || $s['broker_base'] !== self::DEFAULT_BROKER_BASE) {
            $s['broker_base'] = self::DEFAULT_BROKER_BASE;
        }
        // Ensure broker secret exists (generated on init if missing)
        self::ensure_broker_secret();

        $uid     = get_current_user_id();
        $cbUrl   = rest_url(self::REST_NS . self::REST_CB_ROUTE);
        $site_id = self::get_site_id();
        $site_url = home_url('/');

        // Fresh state (valid 10m)
        $state = wp_generate_password(24, false);
        set_transient(self::STATE_PREFIX . $state, ['uid'=>$uid,'ts'=>time()], 10 * MINUTE_IN_SECONDS);

        // One-time token for broker to fetch secret (secret never in URL — broker calls REST_OAUTH_SECRET with token)
        $broker_secret = get_option('miro_broker_secret', '');
        if (empty($broker_secret)) {
            self::ensure_broker_secret();
            $broker_secret = get_option('miro_broker_secret', '');
        }
        $oauth_token = wp_generate_password(32, false);
        set_transient(self::OAUTH_TOKEN_PREFIX . $oauth_token, $broker_secret, 10 * MINUTE_IN_SECONDS);

        // Start URL: no secret in query string; broker uses oauth_token to fetch secret via POST /gsc/oauth/secret
        $qs = http_build_query([
            'site_url'     => $site_url,
            'site_id'      => $site_id,
            'callback'     => $cbUrl,
            'uid'          => $uid,
            'state'        => $state,
            'oauth_token'  => $oauth_token,
        ], '', '&', PHP_QUERY_RFC3986);
        $startUrl = trailingslashit($broker) . 'oauth/gsc/start.php?' . $qs;

        // Property + status
        $prop       = get_option(self::OPT_PROPERTY, []);
        $prop_uri   = isset($prop['uri']) ? (string)$prop['uri'] : '';
        $status     = get_option(self::OPT_STATUS, []);
        $last_err   = isset($status['last_error']) ? (string)$status['last_error'] : '';

        // REST nonce for properties fetch
        $restUrl   = rest_url(self::REST_NS . self::REST_LIST_PROPS);
        $restNonce = wp_create_nonce('wp_rest');

        $af_logo = defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/img/miro-logo.webp' : '';
        echo '<div class="wrap">';
        echo '<h1 class="wp-heading-inline">Google Search Console</h1>';
        echo '</div>';

        echo '<div class="miro-alt-wrap">';
        
        // Hero Banner (Fix Center style)
        echo '<div class="af-hero">';
        echo '<div class="af-hero-pill">';
        echo '<div class="af-hero-pill-inner">';
        if (!empty($af_logo)) {
            echo '<img src="' . esc_url($af_logo) . '" alt="' . esc_attr__('Miro AI SEO logo', 'miro-ai-seo-free') . '">';
        } else {
            echo '<div class="af-hero-pill-fallback">GSC</div>';
        }
        echo '</div>';
        echo '</div>';
        echo '<div class="af-hero-main">';
        echo '<div class="af-hero-title-row">';
        echo '<div class="af-hero-title">Connect your Google Search Console account</div>';
        echo '<span class="af-hero-tag">GSC Integration</span>';
        echo '</div>';
        echo '<p class="af-hero-sub">';
        echo 'Sync your GSC data to track keywords, clicks, impressions, and rankings. OAuth connection via secure broker.';
        echo '</p>';
        echo '<div class="af-hero-chips">';
        echo '<div class="af-chip af-chip-pro ' . ($connected ? 'af-chip-safe' : 'af-chip-alt') . '">';
        echo '<span class="af-dot"></span>';
        echo ($connected ? 'Connected' : 'Not Connected');
        echo '<span class="af-chip-sub">OAuth Status</span>';
        echo '</div>';
        if ($connected && $prop_uri) {
            echo '<div class="af-chip af-chip-pro af-chip-scan">';
            echo '<span class="af-dot"></span>';
            echo 'Property';
            echo '<span class="af-chip-sub">' . esc_html($prop_uri) . '</span>';
            echo '</div>';
        }
        echo '</div>';
        echo '</div>';
        echo '</div>';

        // Card: Connection
        echo '<div class="af-card" style="margin-bottom: 16px;">';
        echo '<div class="af-card-header">';
        echo '<div class="af-card-title">Connection</div>';
        echo '</div>';
        echo '<div class="af-toolbar" style="margin-bottom: 12px;">';
        echo '<span>Status: '.($connected ? '<span style="color: #16a34a; font-weight: 600;">✓ Connected</span>' : '<span style="color: #b91c1c; font-weight: 600;">Not connected</span>').'</span>';
        echo '</div>';
        echo '<div class="af-toolbar">';
        if ($connected) {
            echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'" style="display:inline;">';
            wp_nonce_field('miro_gsc_disconnect');
            echo '<input type="hidden" name="action" value="miro_gsc_disconnect" />';
            echo '<button class="button" type="submit">Disconnect</button>';
            echo '</form>';

            echo '<button class="button" type="button" onclick="location.reload()">Refresh Page</button>';
        } else {
            echo '<a class="button button-primary" href="'.esc_url($startUrl).'" target="_blank" rel="noopener">Connect to Google</a>';
        }
        echo '</div>';
        if ($access_exp) echo '<p class="af-muted" style="margin-top: 8px;">Access token expires at <code>'.esc_html(date_i18n('Y-m-d H:i:s', $access_exp)).'</code> — will auto-refresh when needed.</p>';
        if ($last_err) echo '<p style="margin-top: 8px; color: #b91c1c;"><strong>Last error:</strong> <span class="mono">'.esc_html($last_err).'</span></p>';
        
        // Show success message if just connected
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only display flag; value sanitized.
        $connected_param = isset($_GET['connected']) ? sanitize_key((string) wp_unslash($_GET['connected'])) : '';
        if ($connected_param === '1') {
            echo '<div class="notice notice-success is-dismissible" style="margin-top: 12px; margin-bottom: 0;"><p><strong>✓ Success!</strong> Google Search Console is now connected. You can now fetch properties and sync data.</p></div>';
        }

        // Show "Syncing… (first run)" when connected but cache empty and sync queued/running
        $gsc_cache   = get_option('miro_gsc_last_queries', []);
        $cache_empty = !is_array($gsc_cache) || empty($gsc_cache['rows']) || empty($gsc_cache['daily']);
        $sync_state  = get_option('miro_gsc_sync_state', []);
        $sync_status = isset($sync_state['status']) ? (string) $sync_state['status'] : 'idle';
        $is_syncing  = $connected && $cache_empty && ($sync_status === 'queued' || $sync_status === 'running');
        if ($is_syncing) {
            echo '<div class="notice notice-info is-dismissible" style="margin-top: 12px; margin-bottom: 0;"><p><strong>Syncing… (first run)</strong> We are fetching your last 90 days of data in the background. The dashboard will show data in 1–3 minutes. No need to press Sync.</p></div>';
        }

        echo '</div>';

        // Card: Broker & HMAC - HIDDEN (auto-generated per-site secret)
        // Broker Settings are auto-generated and hidden from users:
        // - Broker Base: https://connect.miroseo.com
        // - HMAC Secret: Auto-generated per-site secret stored in options
        // - OAuth Redirect URI: https://connect.miroseo.com/oauth/gsc/callback.php

        // Card: Property picker
        echo '<div class="af-card" style="margin-bottom: 16px;">';
        echo '<div class="af-card-header">';
        echo '<div class="af-card-title">Property</div>';
        echo '</div>';
        if (!$connected) {
            echo '<p class="af-muted">Connect first, then pick a property.</p>';
        } else {
            echo '<div class="af-toolbar"><button class="button" id="miroFetchProps">Fetch Properties</button><span id="miroPropsStatus" class="af-muted"></span></div>';
            echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'" style="margin-top: 12px;">';
            wp_nonce_field('miro_gsc_save_property');
            echo '<input type="hidden" name="action" value="miro_gsc_save_property" />';
            echo '<div class="af-field" style="margin-bottom: 12px;">';
            echo '<label class="af-label"><strong>Select Property</strong></label>';
            echo '<select name="property_uri" id="miroPropSelect" class="regular-text" style="margin-right: 8px; min-width: 340px;">';
            if ($prop_uri) {
                echo '<option value="'.esc_attr($prop_uri).'" selected>'.esc_html($prop_uri).'</option>';
            } else {
                echo '<option value="">— Select —</option>';
            }
            echo '</select>';
            echo '<button class="button button-primary" type="submit">Save</button>';
            echo '</div>';
            echo '</form>';
            echo '<p class="af-muted" style="margin-top: 8px;">Domain properties look like <code class="mono">sc-domain:example.com</code>; URL-prefix like <code class="mono">https://example.com/</code>.</p>';
        }
        echo '</div>';

        echo '</div>'; // /miro-alt-wrap

        // JS to fetch properties with REST Nonce
        if ($connected) {
            echo '<script>
            (function(){
                const btn = document.getElementById("miroFetchProps");
                const status = document.getElementById("miroPropsStatus");
                const sel = document.getElementById("miroPropSelect");
                if(!btn) return;
                btn.addEventListener("click", async ()=>{
                    status.textContent = " Fetching...";
                    btn.disabled = true;
                    try{
                        const res = await fetch("'.esc_js($restUrl).'", {
                            method: "GET",
                            credentials: "same-origin",
                            headers: { "X-WP-Nonce": "'.esc_js($restNonce).'" }
                        });
                        const data = await res.json();
                        if(!res.ok){ throw new Error((data && data.message) || "Error"); }
                        sel.innerHTML = "";
                        (data.items||[]).forEach(p=>{
                            const o = document.createElement("option");
                            o.value = p.uri; o.textContent = p.uri + (p.permissionLevel?(" — "+p.permissionLevel):"");
                            sel.appendChild(o);
                        });
                        status.textContent = " Done.";
                        if(sel.options.length === 0){
                            const o = document.createElement("option");
                            o.value = ""; o.textContent = "No verified properties found";
                            sel.appendChild(o);
                        }
                    }catch(e){
                        console.error(e);
                        status.textContent = " Failed.";
                        alert("Failed to fetch properties: "+e.message);
                    }finally{
                        btn.disabled = false;
                    }
                });
            })();
            </script>';
        }
    }

    /* ===== Actions ===== */

    public static function disconnect(): void
    {
        if (!current_user_can(\miro_ai_cap())) wp_die(esc_html__('You do not have permission to perform this action.', 'miro-ai-seo-free'), '', ['response' => 403]);
        check_admin_referer('miro_gsc_disconnect');
        delete_option(self::OPT_SETTINGS);
        delete_option(self::OPT_PROPERTY);
        update_option(self::OPT_STATUS, ['state'=>'idle','last_error'=>'','last_ok_at'=>current_time('mysql')], false);
        delete_option('miro_gsc_sync_state');
        delete_option('miro_gsc_cache_ready');
        delete_transient('miro_gsc_first_sync_scheduled');
        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG . '&disconnected=1'));
        exit;
    }

    public static function save_hmac_secret(): void
    {
        // This function is kept for backward compatibility but values are now auto-generated
        // Broker base is always DEFAULT_BROKER_BASE, HMAC secret is auto-generated
        if (!current_user_can(\miro_ai_cap())) wp_die(esc_html__('You do not have permission to perform this action.', 'miro-ai-seo-free'), '', ['response' => 403]);
        check_admin_referer('miro_gsc_save_secret');

        $s = get_option(self::OPT_SETTINGS, []);
        // Auto-generated values (hidden from users)
        $s['broker_base'] = self::DEFAULT_BROKER_BASE; // Always https://connect.miroseo.com
        // Ensure broker secret exists (generated if missing)
        self::ensure_broker_secret();
        $s = get_option(self::OPT_SETTINGS, []);

        update_option(self::OPT_SETTINGS, $s, false);
        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG . '&saved=1'));
        exit;
    }

    public static function save_property(): void
    {
        if (!current_user_can(\miro_ai_cap())) wp_die(esc_html__('You do not have permission to perform this action.', 'miro-ai-seo-free'), '', ['response' => 403]);
        check_admin_referer('miro_gsc_save_property');

        $uri = isset($_POST['property_uri']) ? sanitize_text_field(wp_unslash($_POST['property_uri'])) : '';
        $uri = trim($uri);
        if ($uri === '') {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG . '&prop=empty'));
            exit;
        }

        // Validate allowed formats: sc-domain:example.com or https:// URL prefix
        $is_domain = (strpos($uri, 'sc-domain:') === 0);
        $is_url    = (strpos($uri, 'https://') === 0);
        if (!$is_domain && !$is_url) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG . '&prop=empty'));
            exit;
        }
        if ($is_url && !wp_http_validate_url($uri) && !wp_http_validate_url(rtrim($uri, '/') . '/')) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG . '&prop=empty'));
            exit;
        }

        update_option(self::OPT_PROPERTY, [
            'uri'      => $uri,
            'type'     => ($is_domain ? 'domain' : 'url-prefix'),
            'saved_at' => current_time('mysql'),
        ], false);

        // Trigger first-run sync in background so dashboard shows data without user pressing Sync.
        if (class_exists('\Miro_AI_SEO\GSC_Analytics_Core') && method_exists('\Miro_AI_SEO\GSC_Analytics_Core', 'schedule_first_sync')) {
            \Miro_AI_SEO\GSC_Analytics_Core::schedule_first_sync('property_saved');
        }

        wp_safe_redirect(admin_url('admin.php?page=miro-gsc-analytics&start_sync=1'));
        exit;
    }

    /* ===== Broker → WP OAuth callback ===== */

    /**
     * Public endpoint for external OAuth broker — security enforced via state + HMAC inside callback.
     * POST only; invalid/missing body or headers cause early rejection.
     */
    public static function oauth_callback(\WP_REST_Request $r)
    {
        // Early rejection: non-POST (permission_callback already restricts; double-check)
        if ($r->get_method() !== 'POST') {
            return new \WP_Error('method_not_allowed', 'Method not allowed.', ['status' => 405]);
        }

        // Get raw body from php://input (exactly as received, no modifications)
        $raw = file_get_contents('php://input');
        if ($raw === false) {
            $raw = (string) $r->get_body();
        }

        // Early rejection: invalid or missing body
        if ($raw === '' || strlen($raw) > 20000) {
            return new \WP_Error('bad_request', 'Invalid payload size.', ['status'=>400]);
        }

        $body = json_decode($raw, true);
        if (!is_array($body)) {
            return new \WP_Error('bad_request', 'Invalid JSON payload.', ['status'=>400]);
        }

        $uid   = intval($body['uid'] ?? 0);
        $state = (string)($body['state'] ?? '');

        if (!$uid || !$state) return new \WP_Error('bad_state', 'Missing uid or state.', ['status'=>403]);

        // CSRF protection: Validate state transient (created during OAuth initiation)
        $st = get_transient(self::STATE_PREFIX . $state);
        // Validate state transient exists and is an array
        if (!$st || !is_array($st)) {
            return new \WP_Error('bad_state', 'Invalid or expired state.', ['status'=>403]);
        }

        // Validate state contains uid and timestamp
        if (!isset($st['uid']) || !isset($st['ts'])) {
            return new \WP_Error('bad_state', 'Invalid state structure.', ['status'=>403]);
        }

        $st_uid = intval($st['uid'] ?? 0);
        $st_ts  = intval($st['ts'] ?? 0);

        // Validate user ID matches
        if ($st_uid !== $uid) {
            return new \WP_Error('bad_state', 'User ID mismatch.', ['status'=>403]);
        }

        // Validate TTL: state age must be <= 10 minutes
        if ($st_ts <= 0 || (time() - $st_ts) > (10 * MINUTE_IN_SECONDS)) {
            return new \WP_Error('bad_state', 'State expired.', ['status'=>403]);
        }

        // Step 1: Replay protection - check timestamp
        $callback_timestamp = '';
        
        // Check $_SERVER first (raw headers)
        if (isset($_SERVER['HTTP_X_MIRO_TIMESTAMP'])) {
            $callback_timestamp = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_MIRO_TIMESTAMP']));
        }
        
        // Check REST request headers (case-insensitive)
        if ($callback_timestamp === '') {
            $callback_timestamp = (string) $r->get_header('x-miro-timestamp');
            if ($callback_timestamp === '') {
                $callback_timestamp = (string) $r->get_header('X-MIRO-TIMESTAMP');
            }
        }
        
        // Also check all headers for debugging
        if ($callback_timestamp === '') {
            $all_headers = $r->get_headers();
            foreach ($all_headers as $header_name => $header_value) {
                if (stripos($header_name, 'x-miro-timestamp') !== false || stripos($header_name, 'miro-timestamp') !== false) {
                    $callback_timestamp = is_array($header_value) ? (string) $header_value[0] : (string) $header_value;
                    break;
                }
            }
        }
        
        // Check in body as fallback (broker should include timestamp in JSON body)
        if ($callback_timestamp === '' && isset($body['timestamp'])) {
            $callback_timestamp = (string) $body['timestamp'];
        }
        
        // Fallback: Use state transient timestamp if broker doesn't send header (shouldn't happen in production)
        if ($callback_timestamp === '') {
            $callback_timestamp = (string) $st_ts;
        }
        
        $callback_timestamp = trim($callback_timestamp);
        if ($callback_timestamp === '') {
            return new \WP_Error('no_timestamp', 'Missing timestamp. Broker must send X-MIRO-TIMESTAMP header or timestamp in body.', ['status'=>403]);
        }
        
        $ts = intval($callback_timestamp);
        $now = time();
        $age = abs($now - $ts);
        
        // Reject if older than 10 minutes (using state TTL as max age)
        // State transient is valid for 10 minutes, so allow up to that age
        if ($age > (10 * MINUTE_IN_SECONDS)) {
            return new \WP_Error('stale_request', 'Request too old (replay attack protection).', ['status'=>403]);
        }

        // Step 2: Validate HMAC signature
        // Get HMAC header (X-MIRO-HMAC or x-miro-hmac)
        $provided_hmac = isset($_SERVER['HTTP_X_MIRO_HMAC']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_X_MIRO_HMAC'])) : '';
        if ($provided_hmac === '') {
            $provided_hmac = (string) $r->get_header('x-miro-hmac');
            if ($provided_hmac === '') {
                $provided_hmac = (string) $r->get_header('X-MIRO-HMAC');
            }
        }
        $provided_hmac = trim($provided_hmac);

        if ($provided_hmac === '') {
            return new \WP_Error('no_hmac', 'Missing HMAC header.', ['status'=>403]);
        }

        // Get broker secret from options (per-site secret)
        $broker_secret = get_option('miro_broker_secret', '');
        if ($broker_secret === '') {
            // Ensure it exists (should have been created on init)
            self::ensure_broker_secret();
            $broker_secret = get_option('miro_broker_secret', '');
            if ($broker_secret === '') {
                return new \WP_Error('no_secret', 'Broker secret not configured.', ['status'=>500]);
            }
        }

        // Calculate HMAC of raw body using broker secret
        // Use raw body exactly as received (no modifications)
        $calc_hmac = hash_hmac('sha256', $raw, $broker_secret);
        
        if (!hash_equals($calc_hmac, $provided_hmac)) {
            return new \WP_Error('bad_hmac', 'Invalid HMAC. Check broker secret matches.', ['status'=>403]);
        }

        // HMAC verified - now save tokens
        $acc   = sanitize_text_field((string)($body['access_token'] ?? ''));
        $ref   = sanitize_text_field((string)($body['refresh_token'] ?? ''));
        $expIn = intval($body['expires_in'] ?? 0);
        $cid   = sanitize_text_field((string)($body['client_id'] ?? ''));
        $csec  = sanitize_text_field((string)($body['client_secret'] ?? ''));

        if ($cid)  $s['client_id']     = $cid;
        if ($csec) $s['client_secret'] = $csec;
        if ($ref)  $s['refresh_token'] = $ref;
        if ($acc)  $s['access_token']  = $acc;
        if ($expIn) $s['access_exp']   = time() + max(60, $expIn) - 60;

        $s['broker_base']  = self::DEFAULT_BROKER_BASE;
        // Ensure broker secret exists (should already be set, but double-check)
        self::ensure_broker_secret();
        $s['connected_at'] = current_time('mysql');

        update_option(self::OPT_SETTINGS, $s, false);

        $st2 = get_option(self::OPT_STATUS, []);
        $st2['state']      = 'connected';
        $st2['last_error'] = '';
        $st2['last_ok_at'] = current_time('mysql');
        update_option(self::OPT_STATUS, $st2, false);

        // Delete state transient only after HMAC verification succeeds and tokens are saved successfully
        delete_transient(self::STATE_PREFIX . $state);

        // Trigger first-run sync in background (WP-Cron). Sync runs only when property is set; dedupe inside.
        if (class_exists('\Miro_AI_SEO\GSC_Analytics_Core') && method_exists('\Miro_AI_SEO\GSC_Analytics_Core', 'schedule_first_sync')) {
            \Miro_AI_SEO\GSC_Analytics_Core::schedule_first_sync('first_connect');
        }

        // Redirect to Connect page so user can select a property; after they save property they go to Analytics with sync
        $redirect_url = admin_url('admin.php?page=' . self::MENU_SLUG . '&connected=1');
        return [
            'ok' => true,
            'state' => 'connected',
            'access_exp' => ($s['access_exp'] ?? 0),
            'message' => 'Successfully connected to Google Search Console',
            'redirect_url' => $redirect_url
        ];
    }

    /* ===== REST: properties list ===== */

    public static function rest_list_properties(\WP_REST_Request $r)
    {
        $auth = self::ensure_access_token();
        if (is_wp_error($auth)) return $auth;

        $token = $auth['access_token'];
        $url   = self::GSC_API_BASE . '/sites';
        $res   = self::http_json($url, 'GET', null, ['Authorization: Bearer '.$token]);

        if (is_wp_error($res)) return $res;
        [$code, $json] = $res;

        if ($code !== 200 || !is_array($json)) {
            self::log_error('sites.list failed: HTTP '.$code);
            return new \WP_Error('gsc_error', 'Failed to fetch properties (HTTP '.$code.').', ['status'=>502]);
        }

        $items = [];
        foreach (($json['siteEntry'] ?? []) as $e) {
            $items[] = [
                'uri'             => (string)($e['siteUrl'] ?? ''),
                'permissionLevel' => (string)($e['permissionLevel'] ?? ''),
            ];
        }

        update_option(self::OPT_PROPERTIES_CACHE, ['items'=>$items,'updated_at'=>current_time('mysql')], false);

        return ['ok'=>true, 'items'=>$items];
    }

    /* ===== Token & HTTP helpers ===== */

    private static function ensure_access_token()
    {
        $s = get_option(self::OPT_SETTINGS, []);
        $acc = (string)($s['access_token'] ?? '');
        $exp = intval($s['access_exp'] ?? 0);
        $ref = (string)($s['refresh_token'] ?? '');
        $cid = (string)($s['client_id'] ?? '');
        $csc = (string)($s['client_secret'] ?? '');

        if ($ref === '' || $cid === '' || $csc === '') {
            return new \WP_Error('not_connected', 'Not connected to Google.', ['status'=>403]);
        }

        if ($acc && ($exp > (time() + 90))) return ['access_token'=>$acc];

        $res = self::http_form(self::GOOGLE_TOKEN_URL, [
            'client_id'     => $cid,
            'client_secret' => $csc,
            'grant_type'    => 'refresh_token',
            'refresh_token' => $ref,
        ]);
        if (is_wp_error($res)) return $res;
        [$code, $json] = $res;

        if ($code !== 200 || empty($json['access_token'])) {
            self::log_error('token refresh failed: HTTP '.$code);
            return new \WP_Error('auth_error', 'Token refresh failed. Please reconnect.', ['status'=>401]);
        }

        $s['access_token'] = (string)$json['access_token'];
        $s['access_exp']   = time() + max(60, intval($json['expires_in'] ?? 3600)) - 60;
        update_option(self::OPT_SETTINGS, $s, false);

        return ['access_token'=>$s['access_token']];
    }

    private static function http_json($url, $method = 'GET', $json = null, array $headers = [])
    {
        $args = [
            'method'  => strtoupper($method),
            'timeout' => 15,
            'headers' => array_merge(['Accept'=>'application/json'], self::kvs($headers)),
        ];
        if ($json !== null) {
            $args['headers']['Content-Type'] = 'application/json';
            $args['body'] = is_string($json) ? $json : wp_json_encode($json);
        }
        $res = wp_remote_request($url, $args);
        if (is_wp_error($res)) return $res;
        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        $data = json_decode($body, true);
        return [$code, $data];
    }

    private static function http_form($url, array $fields)
    {
        $res = wp_remote_post($url, [
            'timeout' => 15,
            'headers' => ['Content-Type'=>'application/x-www-form-urlencoded'],
            'body'    => http_build_query($fields, '', '&', PHP_QUERY_RFC3986),
        ]);
        if (is_wp_error($res)) return $res;
        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        $data = json_decode($body, true);
        return [$code, $data];
    }

    private static function kvs(array $h)
    {
        $out = [];
        foreach ($h as $k=>$v) {
            if (is_int($k)) {
                $p = explode(':', $v, 2);
                if (count($p) === 2) $out[trim($p[0])] = trim($p[1]);
            } else {
                $out[$k] = $v;
            }
        }
        return $out;
    }

    private static function log_error($msg)
    {
        $st = get_option(self::OPT_STATUS, []);
        $st['last_error'] = (string)$msg;
        $st['last_err_at']= current_time('mysql');
        update_option(self::OPT_STATUS, $st, false);
    }
}
