<?php
/**
 * Index Monitor module for Miro AI SEO Suite
 */
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) { exit; }

class Miro_Index_Monitor {
    const OPT_KEY                 = 'miro_index_settings';
    const CRON_HOOK               = 'miro_index_monitor_daily';
    const CRON_BACKFILL           = 'miro_index_backfill_tick';   // 5m backfill tick
    const OPT_BACKFILL            = 'miro_index_backfill_state';  // backfill state

    const META_STATUS             = '_miro_index_status';
    const META_DETAILS            = '_miro_index_details';
    const META_LAST_CHECK         = '_miro_index_last_check';
    const META_FIRST_SEEN_NOTIDX  = '_miro_index_first_seen_notidx';
    const META_RETRY_COUNT        = '_miro_index_retry_count';

    public static function init(array $args = []) {
        // 5m cron schedule
        add_filter('cron_schedules', function ($s) {
            if (!isset($s['five_minutes'])) {
                $s['five_minutes'] = ['interval' => 300, 'display' => 'Every 5 Minutes'];
            }
            return $s;
        });

        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_menu', [__CLASS__, 'add_settings_submenu']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);

        add_filter('manage_post_posts_columns', [__CLASS__, 'add_index_column']);
        add_action('manage_post_posts_custom_column', [__CLASS__, 'render_index_column'], 10, 2);
        add_filter('manage_edit-post_sortable_columns', [__CLASS__, 'make_index_sortable']);

        add_action('add_meta_boxes', [__CLASS__, 'add_metabox']);

        add_filter('bulk_actions-edit-post', [__CLASS__, 'register_bulk_actions']);
        add_filter('handle_bulk_actions-edit-post', [__CLASS__, 'handle_bulk_actions'], 10, 3);

        add_action('transition_post_status', [__CLASS__, 'on_transition_status'], 10, 3);

        // Daily check + single post ad-hoc
        add_action(self::CRON_HOOK, [__CLASS__, 'cron_daily_batch']);
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_event(time() + 3600, 'daily', self::CRON_HOOK);
        }

        // Quick check REST for metabox button
        add_action('rest_api_init', function () {
            register_rest_route('miro/v1', '/index/quick_check', [
                'methods'  => 'POST',
                'permission_callback' => function() { return current_user_can('edit_posts'); },
                'callback' => function(\WP_REST_Request $req) {
                    $post_id = intval($req->get_param('post_id'));
                    if (!$post_id) return new \WP_Error('bad_request','post_id required',['status'=>400]);
                    self::quick_check_for_post($post_id);
                    return rest_ensure_response(['ok'=>true]);
                }
            ]);
        });

        // Backfill (5-minute tick)
        add_action(self::CRON_BACKFILL, [__CLASS__, 'cron_backfill_tick']);

        // Kick backfill automatically on first-run
        self::maybe_start_backfill();

        // Ensure history table exists
        self::ensure_log_table();

        // Load Index REST
        $rest_file = __DIR__ . '/rest/class-miro-rest-index.php';
        if (file_exists($rest_file)) {
            require_once $rest_file;
            if (class_exists('\\Miro_AI_SEO\\Miro_REST_Index')) {
                \Miro_AI_SEO\Miro_REST_Index::init();
            }
        }
    }

    /** Ensure index history table exists (very lightweight) */
    protected static function ensure_log_table(): void {
        global $wpdb;
        $tbl = $wpdb->prefix . 'ppmiro_index_log';
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS `$tbl` (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            post_id BIGINT UNSIGNED NOT NULL,
            checked_at DATETIME NOT NULL,
            status VARCHAR(40) NOT NULL,
            verdict VARCHAR(40) DEFAULT '' NOT NULL,
            coverage_state VARCHAR(191) DEFAULT '' NOT NULL,
            indexing_allowed TINYINT(1) DEFAULT 0 NOT NULL,
            page_fetch_state VARCHAR(60) DEFAULT '' NOT NULL,
            http_code INT DEFAULT 0 NOT NULL,
            device VARCHAR(20) DEFAULT 'MOBILE' NOT NULL,
            rich_types TEXT NULL,
            notes TEXT NULL,
            PRIMARY KEY (id),
            KEY post_time (post_id, checked_at),
            KEY status_idx (status)
        ) $charset;";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    /** SETTINGS */
    public static function defaults() : array {
        return [
            'gsc_connected'         => false,
            'gsc_property'          => '',
            'indexnow_key'          => '',
            'auto_check_on_publish' => true,
            'daily_recheck'         => true,
            'alert_days'            => 7,
            'backoff'               => true,
            'ping_sitemaps'         => true,
            'show_posts_column'     => true,
            'show_metabox_badges'   => true,
            'email_alerts'          => false,
            'email_to'              => get_option('admin_email'),
        ];
    }
    public static function get_settings() : array {
        $opt = get_option(self::OPT_KEY, []);
        if (!is_array($opt)) $opt = [];
        return array_merge(self::defaults(), $opt);
    }
    public static function register_settings() {
        register_setting('miro_index_settings_group', self::OPT_KEY, [
            'type' => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize_settings'],
            'default' => self::defaults(),
        ]);
    }
    public static function sanitize_settings($val) {
        $d = self::defaults();
        if (!is_array($val)) $val = [];
        return [
            'gsc_connected'         => !empty($val['gsc_connected']),
            'gsc_property'          => sanitize_text_field($val['gsc_property'] ?? ''),
            'indexnow_key'          => sanitize_text_field($val['indexnow_key'] ?? ''),
            'auto_check_on_publish' => !empty($val['auto_check_on_publish']),
            'daily_recheck'         => !empty($val['daily_recheck']),
            'alert_days'            => max(1, intval($val['alert_days'] ?? $d['alert_days'])),
            'backoff'               => !empty($val['backoff']),
            'ping_sitemaps'         => !empty($val['ping_sitemaps']),
            'show_posts_column'     => !empty($val['show_posts_column']),
            'show_metabox_badges'   => !empty($val['show_metabox_badges']),
            'email_alerts'          => !empty($val['email_alerts']),
            'email_to'              => sanitize_email($val['email_to'] ?? $d['email_to']),
        ];
    }

    /** Get property saved by GSC Connect (preferred) */
    public static function gsc_property_from_connect() : string {
        if (class_exists('\\Miro_AI_SEO\\Miro_GSC_Connect')) {
            $prop = get_option(\Miro_AI_SEO\Miro_GSC_Connect::OPT_PROPERTY, []);
            if (is_array($prop) && !empty($prop['uri'])) {
                return (string)$prop['uri'];
            }
        }
        return '';
    }

    /** SETTINGS UI */
    public static function add_settings_submenu() {
        $parent = apply_filters('miro_main_menu_slug', 'miro-ai-seo');

        global $menu;
        $parent_exists = false;
        if (is_array($menu)) {
            foreach ($menu as $m) {
                if (!empty($m[2]) && $m[2] === $parent) { $parent_exists = true; break; }
            }
        }
        if (!$parent_exists) {
            add_options_page(
                __('Index Monitor', 'miro-ai-seo'),
                __('Index Monitor', 'miro-ai-seo'),
                'manage_options',
                'miro-index',
                [__CLASS__, 'render_settings_page']
            );
            return;
        }

        add_submenu_page(
            $parent,
            __('Index Monitor', 'miro-ai-seo'),
            __('Index Monitor', 'miro-ai-seo'),
            'manage_options',
            'miro-index',
            [__CLASS__, 'render_settings_page'],
            32
        );
    }

public static function render_settings_page() {
    $opt = self::get_settings();
    $gsc_page_url = admin_url('admin.php?page=miro-gsc');
    $prop_from_connect = self::gsc_property_from_connect();
    $effective_prop = $prop_from_connect ?: $opt['gsc_property']; // prefer Connect's saved property

    $rest_base = trailingslashit( rest_url('miro/v1') );
    $nonce     = wp_create_nonce('wp_rest');
    $af_logo = defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/img/miro-logo.webp' : '';
    ?>
    <div class="wrap">
      <h1 class="wp-heading-inline"><?php esc_html_e('Index Monitor', 'miro-ai-seo'); ?></h1>
    </div>
    
    <div class="miro-alt-wrap">
      <!-- Hero Banner (Fix Center style) -->
      <div class="af-hero">
          <div class="af-hero-pill">
              <div class="af-hero-pill-inner">
                  <?php if (!empty($af_logo)): ?>
                      <img src="<?php echo esc_url($af_logo); ?>" alt="<?php echo esc_attr__('Miro AI SEO logo', 'miro-ai-seo'); ?>">
                  <?php else: ?>
                      <div class="af-hero-pill-fallback">📊</div>
                  <?php endif; ?>
              </div>
          </div>
          <div class="af-hero-main">
              <div class="af-hero-title-row">
                  <div class="af-hero-title">Monitor your content's indexing status</div>
                  <span class="af-hero-tag">Index Monitor</span>
              </div>
              <p class="af-hero-sub">
                  Track which posts are indexed by Google, identify issues, and automatically check new content.
              </p>
              <div class="af-hero-chips">
                  <div class="af-chip af-chip-pro af-chip-alt"><span class="af-dot"></span>GSC<span class="af-chip-sub">Google Search Console</span></div>
                  <div class="af-chip af-chip-pro af-chip-scan"><span class="af-dot"></span>Auto Check<span class="af-chip-sub">On publish</span></div>
                  <div class="af-chip af-chip-pro af-chip-safe"><span class="af-dot"></span>Daily Scan<span class="af-chip-sub">Re-check not indexed</span></div>
              </div>
          </div>
      </div>
      
      <div class="af-card" style="margin-bottom: 16px;">
      <form method="post" action="options.php" class="miro-index-form">
        <?php settings_fields('miro_index_settings_group'); ?>
        <?php $field = self::OPT_KEY; ?>

        <h2><?php esc_html_e('Connection', 'miro-ai-seo'); ?></h2>
        <table class="form-table">
          <tr>
            <th><?php esc_html_e('Google Search Console', 'miro-ai-seo'); ?></th>
            <td>
              <p class="description" style="margin-bottom:6px;">
                <?php
                echo wp_kses_post(sprintf(
                    /* translators: %s: URL to the GSC Connect admin page. */
                    __('Manage connection in <a href="%s">GSC Connect</a>. Index Monitor will use its selected property by default.', 'miro-ai-seo'),
                    esc_url($gsc_page_url)
                )); ?>
              </p>

              <div style="display:flex;align-items:center;gap:8px;margin:6px 0;">
                <select id="miro-gsc-prop" class="regular-text" style="min-width:340px;">
                    <?php if ($effective_prop): ?>
                        <option value="<?php echo esc_attr($effective_prop); ?>" selected><?php echo esc_html($effective_prop); ?></option>
                    <?php else: ?>
                        <option value=""><?php esc_html_e('— Select —', 'miro-ai-seo'); ?></option>
                    <?php endif; ?>
                </select>
                <button type="button" class="button" id="miro-gsc-refresh"><?php esc_html_e('Fetch from GSC', 'miro-ai-seo'); ?></button>
              </div>

              <input type="hidden" id="miro-gsc-prop-hidden" name="<?php echo esc_attr($field); ?>[gsc_property]" value="<?php echo esc_attr($opt['gsc_property']); ?>" />

              <p class="description" id="miro-gsc-prop-note">
                <?php
                if ($prop_from_connect) {
                    echo wp_kses_post(sprintf(
                        /* translators: %s: GSC property URL or identifier. */
                        __('Using <strong>GSC Connect</strong> property: <code>%s</code>. You can set an Index Monitor override by saving the selector above.', 'miro-ai-seo'),
                        esc_html($prop_from_connect)
                    ));
                } else {
                    esc_html_e('No property set in GSC Connect yet. Pick one above to store a local override until you save it in GSC Connect.', 'miro-ai-seo');
                }
                ?>
              </p>
            </td>
          </tr>

          <tr>
            <th><?php esc_html_e('IndexNow Key (Bing)', 'miro-ai-seo'); ?></th>
            <td>
              <input type="text" class="regular-text" name="<?php echo esc_attr($field); ?>[indexnow_key]" value="<?php echo esc_attr($opt['indexnow_key']); ?>" placeholder="your-indexnow-key" />
              <p class="description"><?php esc_html_e('If set, “Re-index” will submit the URL to IndexNow.', 'miro-ai-seo'); ?></p>
            </td>
          </tr>
        </table>

        <h2><?php esc_html_e('Automation', 'miro-ai-seo'); ?></h2>
        <table class="form-table">
          <tr>
            <th><?php esc_html_e('On Publish', 'miro-ai-seo'); ?></th>
            <td><label><input type="checkbox" name="<?php echo esc_attr($field); ?>[auto_check_on_publish]" <?php checked(!empty($opt['auto_check_on_publish'])); ?> /> <?php esc_html_e('Auto-check newly published posts', 'miro-ai-seo'); ?></label></td>
          </tr>
          <tr>
            <th><?php esc_html_e('Daily Re-check', 'miro-ai-seo'); ?></th>
            <td><label><input type="checkbox" name="<?php echo esc_attr($field); ?>[daily_recheck]" <?php checked(!empty($opt['daily_recheck'])); ?> /> <?php esc_html_e('Daily re-check not-indexed posts', 'miro-ai-seo'); ?></label></td>
          </tr>
          <tr>
            <th><?php esc_html_e('Alert Days', 'miro-ai-seo'); ?></th>
            <td>
              <input type="number" min="1" style="width:80px" name="<?php echo esc_attr($field); ?>[alert_days]" value="<?php echo intval($opt['alert_days']); ?>" />
              <label><input type="checkbox" name="<?php echo esc_attr($field); ?>[backoff]" <?php checked(!empty($opt['backoff'])); ?> /> <?php esc_html_e('Backoff after 2 failed attempts', 'miro-ai-seo'); ?></label>
            </td>
          </tr>
          <tr>
            <th><?php esc_html_e('Sitemaps', 'miro-ai-seo'); ?></th>
            <td><label><input type="checkbox" name="<?php echo esc_attr($field); ?>[ping_sitemaps]" <?php checked(!empty($opt['ping_sitemaps'])); ?> /> <?php esc_html_e('Ping Google/Bing when post updates', 'miro-ai-seo'); ?></label></td>
          </tr>
        </table>

        <h2><?php esc_html_e('UI', 'miro-ai-seo'); ?></h2>
        <table class="form-table">
          <tr>
            <th><?php esc_html_e('Posts Table', 'miro-ai-seo'); ?></th>
            <td><label><input type="checkbox" name="<?php echo esc_attr($field); ?>[show_posts_column]" <?php checked(!empty($opt['show_posts_column'])); ?> /> <?php esc_html_e('Show Index column', 'miro-ai-seo'); ?></label></td>
          </tr>
          <tr>
            <th><?php esc_html_e('Metabox', 'miro-ai-seo'); ?></th>
            <td><label><input type="checkbox" name="<?php echo esc_attr($field); ?>[show_metabox_badges]" <?php checked(!empty($opt['show_metabox_badges'])); ?> /> <?php esc_html_e('Show colored badges', 'miro-ai-seo'); ?></label></td>
          </tr>
          <tr>
            <th><?php esc_html_e('Email Alerts', 'miro-ai-seo'); ?></th>
            <td>
              <label><input type="checkbox" name="<?php echo esc_attr($field); ?>[email_alerts]" <?php checked(!empty($opt['email_alerts'])); ?> /> <?php esc_html_e('Enable email alerts for not-indexed ≥ X days', 'miro-ai-seo'); ?></label><br/>
              <input type="email" class="regular-text" name="<?php echo esc_attr($field); ?>[email_to]" value="<?php echo esc_attr($opt['email_to']); ?>" />
            </td>
          </tr>
        </table>

        <?php submit_button(); ?>
      </form>
      </div> <!-- /af-card -->
      
      <div class="af-card">
        <div class="af-card-header">
            <div class="af-card-title"><?php esc_html_e('Scan All Posts', 'miro-ai-seo'); ?></div>
        </div>
        <p class="description" style="margin-top:-6px;">
          <?php esc_html_e('Run URL Inspection across your site. Use Background for safe server-side backfill, or Sync Scan to process in chunks right now.', 'miro-ai-seo'); ?>
        </p>

        <div style="display:flex;flex-wrap:wrap;gap:12px;align-items:center;margin:10px 0;">
          <label><?php esc_html_e('Post type', 'miro-ai-seo'); ?>
            <select id="miro-scan-pt">
              <option value="any"><?php esc_html_e('Any public type', 'miro-ai-seo'); ?></option>
              <option value="post">post</option>
              <option value="page">page</option>
            </select>
          </label>

          <label><input type="checkbox" id="miro-scan-only-missing" checked> <?php esc_html_e('Only missing (never checked)', 'miro-ai-seo'); ?></label>

          <label><?php esc_html_e('Status filter', 'miro-ai-seo'); ?>
            <select id="miro-scan-status">
              <option value=""><?php esc_html_e('— Any —', 'miro-ai-seo'); ?></option>
              <option value="unknown">unknown</option>
              <option value="discovered_not_crawled">discovered_not_crawled</option>
              <option value="crawled_not_indexed">crawled_not_indexed</option>
              <option value="blocked_or_error">blocked_or_error</option>
            </select>
          </label>

          <label><?php esc_html_e('Per page', 'miro-ai-seo'); ?>
            <input type="number" id="miro-scan-per" value="50" min="1" max="200" style="width:90px;">
          </label>

          <label><?php esc_html_e('Property override (optional)', 'miro-ai-seo'); ?>
            <input type="text" id="miro-scan-prop" value="" placeholder="sc-domain:example.com or https://example.com/">
          </label>
        </div>

        <div style="display:flex;gap:8px;align-items:center;margin:10px 0;">
          <button type="button" class="button button-primary" id="miro-scan-sync"><?php esc_html_e('Scan All (Sync, paged)', 'miro-ai-seo'); ?></button>
          <button type="button" class="button" id="miro-scan-stop" disabled><?php esc_html_e('Stop', 'miro-ai-seo'); ?></button>
          <button type="button" class="button" id="miro-scan-bg"><?php esc_html_e('Background (Queue)', 'miro-ai-seo'); ?></button>
          <span id="miro-scan-msg" style="margin-left:8px;color:#334155;"></span>
        </div>

        <div id="miro-scan-progress" style="display:none;margin-top:8px;">
          <div style="height:10px;background:#e5e7eb;border-radius:8px;overflow:hidden;">
            <div id="miro-scan-bar" style="height:10px;width:0%;background:#3b82f6;"></div>
          </div>
          <div id="miro-scan-stats" style="margin-top:6px;font-size:12px;color:#64748b;"></div>
        </div>
      </div>
    </div> <!-- /miro-alt-wrap -->

    <script>
    (function(){
      const rest = "<?php echo esc_js($rest_base); ?>";
      const nonce = "<?php echo esc_js($nonce); ?>";

      const $ = (sel) => document.querySelector(sel);
      const btnSync = $("#miro-scan-sync");
      const btnStop = $("#miro-scan-stop");
      const btnBg   = $("#miro-scan-bg");
      const msg     = $("#miro-scan-msg");
      const barWrap = $("#miro-scan-progress");
      const bar     = $("#miro-scan-bar");
      const stats   = $("#miro-scan-stats");

      let stopFlag = false;

      function setMsg(t){ msg.textContent = t || ""; }
      function setBar(pct){ bar.style.width = Math.max(0, Math.min(100, pct)) + "%"; }
      function setStats(txt){ stats.textContent = txt || ""; }

      async function postJSON(url, body){
        const res = await fetch(url, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json', 'X-WP-Nonce': nonce },
          body: JSON.stringify(body || {})
        });
        const data = await res.json().catch(()=> ({}));
        if(!res.ok) throw new Error(data && data.message || ('HTTP '+res.status));
        return data;
      }

      async function scanAllPaged(){
        stopFlag = false;
        btnSync.disabled = true;
        btnStop.disabled = false;
        btnBg.disabled   = true;
        barWrap.style.display = '';
        setBar(0);
        setStats('');
        setMsg('Starting…');

        const per   = Math.max(1, Math.min(200, parseInt($("#miro-scan-per").value || "50", 10)));
        const pt    = $("#miro-scan-pt").value || 'any';
        const onlyM = $("#miro-scan-only-missing").checked;
        const stat  = $("#miro-scan-status").value || '';
        const prop  = ($("#miro-scan-prop").value || '').trim();

        // First page to discover total pages
        let page = 1, processed = 0, total = 0, pages = 0;

        try {
          const first = await postJSON(rest + 'index/scan_all', { per_page: per, page, only_missing: onlyM, status_filter: stat, post_type: pt, property: prop });
          processed += (first.processed || 0);
          total  = first.total || 0;
          pages  = first.pages || 1;
          setMsg('Scanning…');
          setBar(total ? (processed/total)*100 : (page/pages)*100);
          setStats(`Page ${page}/${pages} — processed ${processed}/${total}`);

          while(!stopFlag && page < pages){
            page++;
            const r = await postJSON(rest + 'index/scan_all', { per_page: per, page, only_missing: onlyM, status_filter: stat, post_type: pt, property: prop });
            processed += (r.processed || 0);
            total  = r.total || total;
            pages  = r.pages || pages;
            setBar(total ? (processed/total)*100 : (page/pages)*100);
            setStats(`Page ${page}/${pages} — processed ${processed}/${total}`);
          }

          if (stopFlag) {
            setMsg('Stopped.');
          } else {
            setMsg('Done.');
            setBar(100);
          }
        } catch (e) {
          console.error(e);
          setMsg('Error: ' + e.message);
        } finally {
          btnSync.disabled = false;
          btnStop.disabled = true;
          btnBg.disabled   = false;
        }
      }

      btnSync.addEventListener('click', scanAllPaged);
      btnStop.addEventListener('click', () => { stopFlag = true; });
      btnBg.addEventListener('click', async () => {
        btnBg.disabled = true;
        setMsg('Queuing background backfill…');
        try {
          await postJSON(rest + 'index/rescan', {});
          setMsg('Queued. First batch is being processed.');
        } catch (e) {
          setMsg('Error: ' + e.message);
        } finally {
          btnBg.disabled = false;
        }
      });

      // GSC property dropdown (optional fetch)
      const btnGSC = document.getElementById('miro-gsc-refresh');
      const selGSC = document.getElementById('miro-gsc-prop');
      const hidGSC = document.getElementById('miro-gsc-prop-hidden');
      if (btnGSC && selGSC && hidGSC) {
        btnGSC.addEventListener('click', async () => {
          btnGSC.disabled = true;
          btnGSC.textContent = 'Fetching…';
          try {
            const res = await fetch(rest + 'index/sites', { headers: { 'X-WP-Nonce': nonce }});
            const data = await res.json();
            if (data && data.sites && Array.isArray(data.sites)) {
              selGSC.innerHTML = '';
              selGSC.appendChild(new Option('— Select —',''));
              data.sites.forEach(s => {
                selGSC.appendChild(new Option(s.siteUrl, s.siteUrl));
              });
            }
          } catch (e) {
            alert('Failed: ' + e.message);
          } finally {
            btnGSC.disabled = false;
            btnGSC.textContent = 'Fetch from GSC';
          }
        });
        selGSC.addEventListener('change', () => {
          hidGSC.value = selGSC.value || '';
        });
      }
    })();
    </script>
    <?php
}


    /** ASSETS */
    public static function enqueue_assets($hook) {
        wp_enqueue_style('miro-index-css', plugins_url('css/index.css', __FILE__), [], '0.1.4');
        
        // Enqueue Fix Center style hero CSS
        if (strpos($hook, '_page_miro-index') !== false) {
            if (defined('MIRO_AI_SEO_FILE')) {
                $rel_alt  = 'assets/css/miro-alt.css';
                $url_alt  = plugins_url($rel_alt, MIRO_AI_SEO_FILE);
                $path_alt = plugin_dir_path(MIRO_AI_SEO_FILE) . $rel_alt;
                $ver_alt  = (file_exists($path_alt) && is_readable($path_alt)) ? (string) filemtime($path_alt) : (defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');
                
                wp_enqueue_style('miro-alt', $url_alt, ['ppmiro-admin'], $ver_alt);
            }
        }

        $should_admin = ($hook === 'post.php' || $hook === 'post-new.php' || strpos($hook, '_page_miro-index') !== false);
        $rest_base = trailingslashit( rest_url('miro/v1') );

        if ($should_admin) {
            wp_enqueue_script('miro-index-admin', plugins_url('js/index-admin.js', __FILE__), ['jquery'], '0.1.4', true);
            wp_localize_script('miro-index-admin', 'MiroIndex', [
                'rest'    => esc_url_raw($rest_base),
                'nonce'   => wp_create_nonce('wp_rest'),
                'opt'     => self::get_settings(),
                'gscProp' => self::gsc_property_from_connect(),
                'gscList' => esc_url_raw( $rest_base . 'gsc/properties' ),
            ]);
        }
        if ($hook === 'post.php' || $hook === 'post-new.php') {
            wp_enqueue_script('miro-index-metabox', plugins_url('js/index-metabox.js', __FILE__), ['jquery'], '0.1.3', true);
            wp_localize_script('miro-index-metabox', 'MiroIndex', [
                'rest'    => esc_url_raw($rest_base),
                'nonce'   => wp_create_nonce('wp_rest'),
                'opt'     => self::get_settings(),
                'gscProp' => self::gsc_property_from_connect(),
            ]);
        }
    }

    /** POSTS LIST COLUMN */
    public static function add_index_column($columns) {
        $opt = self::get_settings();
        if (empty($opt['show_posts_column'])) return $columns;
        $columns['miro_index'] = __('Index', 'miro-ai-seo');
        return $columns;
    }
    public static function render_index_column($column, $post_id) {
        if ($column !== 'miro_index') return;
        $status = get_post_meta($post_id, self::META_STATUS, true) ?: 'unknown';
        $last = get_post_meta($post_id, self::META_LAST_CHECK, true);
        $badge = self::status_badge($status);
        $tt = $last ? (string) $last : __('Never checked', 'miro-ai-seo');
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $tt escaped via esc_attr, $badge via wp_kses_post
        echo '<span class="miro-badge" title="' . esc_attr($tt) . '">' . wp_kses_post($badge) . '</span>';
    }
    public static function make_index_sortable($cols) {
        $cols['miro_index'] = 'miro_index';
        return $cols;
    }

    /** METABOX */
    public static function add_metabox() {
        $opt = self::get_settings();
        if (empty($opt['show_metabox_badges'])) return;
        $types = get_post_types(['public'=>true], 'names');
        foreach ($types as $pt) {
            add_meta_box(
                'miro-index-box',
                esc_html__('Miro Index Monitor', 'miro-ai-seo'),
                [__CLASS__, 'render_metabox'],
                $pt,
                'side',
                'high'
            );
        }
    }

    public static function render_metabox($post) {
        $status  = get_post_meta($post->ID, self::META_STATUS, true) ?: 'unknown';
        $details = get_post_meta($post->ID, self::META_DETAILS, true);
        $details = is_string($details) ? json_decode($details, true) : (is_array($details) ? $details : []);
        $last    = get_post_meta($post->ID, self::META_LAST_CHECK, true);
        ?>
        <div class="miro-index-box">
            <div><?php echo wp_kses_post(self::status_badge($status)); ?></div>
            <p><strong><?php esc_html_e('Verdict:', 'miro-ai-seo'); ?></strong> <?php echo esc_html($details['verdict'] ?? '—'); ?></p>
            <p><strong><?php esc_html_e('Last crawl:', 'miro-ai-seo'); ?></strong> <?php echo esc_html($details['lastCrawlTime'] ?? '—'); ?></p>
            <?php if (!empty($details['canonicalUrl'])): ?>
                <p><strong><?php esc_html_e('Canonical:', 'miro-ai-seo'); ?></strong> <code><?php echo esc_html($details['canonicalUrl']); ?></code></p>
            <?php endif; ?>
            <div style="display:flex; gap:6px; margin-top:8px;">
              <button type="button" class="button button-primary miro-index-check" data-post="<?php echo intval($post->ID); ?>"><?php esc_html_e('Check Now', 'miro-ai-seo'); ?></button>
              <button type="button" class="button miro-index-ping" data-post="<?php echo intval($post->ID); ?>"><?php esc_html_e('Re-index (Best Effort)', 'miro-ai-seo'); ?></button>
            </div>
            <p class="description" id="miro-index-msg" style="margin-top:6px;"></p>
        </div>
        <?php
    }

    /** BULK ACTIONS */
    public static function register_bulk_actions($bulk_actions) {
        $bulk_actions['miro_check_index'] = __('Check Index Status', 'miro-ai-seo');
        $bulk_actions['miro_reindex']     = __('Re-index (Best Effort)', 'miro-ai-seo');
        return $bulk_actions;
    }
    public static function handle_bulk_actions($redirect_to, $doaction, $post_ids) {
        $rest_file = __DIR__ . '/rest/class-miro-rest-index.php';
        if (!class_exists('\\Miro_AI_SEO\\Miro_REST_Index') && file_exists($rest_file)) {
            require_once $rest_file;
        }

        if ($doaction === 'miro_check_index') {
            foreach ($post_ids as $pid) {
                $pid = (int)$pid;
                if (class_exists('\\Miro_AI_SEO\\Miro_REST_Index')) {
                    \Miro_AI_SEO\Miro_REST_Index::perform_check_for_post($pid);
                } else {
                    update_post_meta($pid, self::META_LAST_CHECK, gmdate('c'));
                }
            }
            return add_query_arg('miro_bulk_checked', count($post_ids), $redirect_to);
        }

        if ($doaction === 'miro_reindex') {
            $opt = self::get_settings();
            $sitemap = home_url('/sitemap.xml');
            foreach ($post_ids as $pid) {
                $pid = (int)$pid;
                $url = get_permalink($pid);
                if (!$url) continue;

                if (!empty($opt['ping_sitemaps'])) {
                    wp_remote_get('https://www.google.com/ping?sitemap=' . urlencode($sitemap));
                    wp_remote_get('https://www.bing.com/ping?sitemap=' . urlencode($sitemap));
                }

                if (!empty($opt['indexnow_key'])) {
                    $endpoint = 'https://api.indexnow.org/indexnow';
                    $body = [
                        'host'        => wp_parse_url(home_url(), PHP_URL_HOST),
                        'key'         => $opt['indexnow_key'],
                        'urlList'     => [$url],
                        'keyLocation' => home_url('/' . $opt['indexnow_key'] . '.txt'),
                    ];
                    wp_remote_post($endpoint, [
                        'headers' => ['Content-Type' => 'application/json'],
                        'body'    => wp_json_encode($body),
                        'timeout' => 10,
                    ]);
                }
            }
            return add_query_arg('miro_bulk_reindexed', count($post_ids), $redirect_to);
        }

        return $redirect_to;
    }

    /** PUBLISH → queue first check */
    public static function on_transition_status($new, $old, $post) {
        if ($new === $old) return;
        if ($new !== 'publish') return;
        $opt = self::get_settings();
        if (empty($opt['auto_check_on_publish'])) return;
        wp_schedule_single_event(time() + 600, self::CRON_HOOK, [['single_post_id' => $post->ID]]);
    }

    /** CRON (daily batch or single) */
    public static function cron_daily_batch($args = []) {
        $opt = self::get_settings();

        // Single post ad-hoc check
        if (!empty($args['single_post_id'])) {
            self::do_check((int)$args['single_post_id']);
            return;
        }

        if (empty($opt['daily_recheck'])) return;

        // Re-check a limited batch of non-indexed posts
        $q = new \WP_Query([
            'post_type'      => 'post',
            'post_status'    => 'publish',
            'meta_query'     => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for status filter
                [
                    'key'     => self::META_STATUS,
                    'value'   => 'indexed',
                    'compare' => '!='
                ]
            ],
            'posts_per_page' => 25,
            'fields'         => 'ids',
            'orderby'        => 'date',
            'order'          => 'DESC'
        ]);
        if ($q->have_posts()) {
            foreach ($q->posts as $pid) {
                self::do_check((int)$pid);
            }
        }

        if (!empty($opt['email_alerts'])) {
            self::send_alerts($opt);
        }
    }

    /** ===== BACKFILL ===== */
    protected static function maybe_start_backfill(): void {
        $have_any = get_posts([
            'post_type'    => 'any',
            'post_status'  => 'publish',
            'meta_key'     => self::META_LAST_CHECK, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Existence check for backfill
            'meta_compare' => 'EXISTS',
            'numberposts'  => 1,
            'fields'       => 'ids',
            'no_found_rows'=> true,
        ]);
        if (!$have_any) {
            self::queue_full_backfill(20);
        }
    }

    public static function queue_full_backfill(int $batchSize = 20): void {
        $st = get_option(self::OPT_BACKFILL, []);
        if (!is_array($st) || empty($st['running'])) {
            update_option(self::OPT_BACKFILL, [
                'running'    => true,
                'batch_size' => max(1, min(50, $batchSize)),
                'last_run'   => current_time('mysql'),
            ], false);
        }
        if (!wp_next_scheduled(self::CRON_BACKFILL)) {
            wp_schedule_event(time() + 60, 'five_minutes', self::CRON_BACKFILL);
        }
    }

    public static function cron_backfill_tick(): void {
        $st = get_option(self::OPT_BACKFILL, []);
        if (!is_array($st) || empty($st['running'])) return;

        $batch = max(1, min(50, intval($st['batch_size'] ?? 20)));

        $q = new \WP_Query([
            'post_type'      => get_post_types(['public'=>true], 'names'),
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => $batch,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'meta_query'     => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for backfill filter
                [
                    'key'     => self::META_LAST_CHECK,
                    'compare' => 'NOT EXISTS',
                ],
            ],
            'no_found_rows'  => true,
        ]);

        $processed = 0;
        if (!empty($q->posts)) {
            foreach ($q->posts as $pid) {
                $pid = (int)$pid;
                if (!get_post_meta($pid, self::META_LAST_CHECK, true)) {
                    self::do_check($pid);
                    $processed++;
                }
            }
        }

        if ($processed === 0) {
            $ts = wp_next_scheduled(self::CRON_BACKFILL);
            if ($ts) wp_unschedule_event($ts, self::CRON_BACKFILL);
            delete_option(self::OPT_BACKFILL);
            return;
        }

        $st['last_run'] = current_time('mysql');
        update_option(self::OPT_BACKFILL, $st, false);
    }

    /** CORE */
    public static function do_check(int $post_id) {
        // Use REST class to perform + log; fallback to quick if not present
        if (!class_exists('\\Miro_AI_SEO\\Miro_REST_Index')) {
            $rest_file = __DIR__ . '/rest/class-miro-rest-index.php';
            if (file_exists($rest_file)) require_once $rest_file;
        }
        if (class_exists('\\Miro_AI_SEO\\Miro_REST_Index')) {
            try {
                \Miro_AI_SEO\Miro_REST_Index::perform_check_for_post($post_id, '', 'MOBILE');
                return;
            } catch (\Throwable $e) {
                // fall through to quick
            }
        }
        self::quick_check_for_post($post_id);
    }

    /**
     * Quick fallback check without GSC:
     * - GET the URL (follow redirects), capture final status & headers
     * - Detect meta robots & X-Robots-Tag noindex
     * - Detect canonical
     * - Check presence in sitemap
     * - Normalize to statuses
     */
    public static function quick_check_for_post(int $post_id) : void {
        $url = get_permalink($post_id);
        if (!$url) {
            update_post_meta($post_id, self::META_STATUS, 'unknown');
            update_post_meta($post_id, self::META_LAST_CHECK, gmdate('c'));
            return;
        }

        $resp = self::fetch_url($url);
        $code = intval($resp['code'] ?? 0);
        $html = (string)($resp['body'] ?? '');
        $headers = $resp['headers'] ?? [];

        $details = [
            'verdict'       => 'FALLBACK',
            'httpCode'      => $code,
            'finalUrl'      => $resp['final_url'] ?? $url,
            'xRobots'       => $headers['x-robots-tag'] ?? '',
            'lastCrawlTime' => '—',
        ];

        $blocked = false;
        if ($code >= 400) { $blocked = true; }
        if (!empty($headers['x-robots-tag']) && stripos($headers['x-robots-tag'], 'noindex') !== false) {
            $blocked = true;
            $details['verdict'] = 'BLOCKED_X_ROBOTS';
        }

        $meta = self::parse_meta_robots_and_canonical($html);
        $details['metaRobots']  = $meta['robots'] ?? '';
        $details['canonicalUrl'] = $meta['canonical'] ?? '';

        if (!$blocked && !empty($meta['robots']) && stripos($meta['robots'], 'noindex') !== false) {
            $blocked = true;
            $details['verdict'] = 'BLOCKED_META';
        }

        $host = wp_parse_url(home_url(), PHP_URL_HOST);
        $canonHost = $details['canonicalUrl'] ? wp_parse_url($details['canonicalUrl'], PHP_URL_HOST) : '';
        $canon_offsite = ($details['canonicalUrl'] && $canonHost && $host && strcasecmp($canonHost, $host) !== 0);

        $in_sitemap = self::url_in_sitemap($url);

        $status = 'unknown';
        if ($blocked) {
            $status = 'blocked_or_error';
        } else {
            if ($code === 200) {
                $status = $in_sitemap ? 'crawled_not_indexed' : 'discovered_not_crawled';
            }
            if ($canon_offsite) {
                $status = 'crawled_not_indexed';
                $details['verdict'] = 'CANONICAL_OFFSITE';
            }
        }

        update_post_meta($post_id, self::META_STATUS, $status);
        update_post_meta($post_id, self::META_DETAILS, wp_json_encode($details));
        update_post_meta($post_id, self::META_LAST_CHECK, gmdate('c'));

        if ($status !== 'indexed') {
            $first = get_post_meta($post_id, self::META_FIRST_SEEN_NOTIDX, true);
            if (!$first) update_post_meta($post_id, self::META_FIRST_SEEN_NOTIDX, gmdate('c'));
        } else {
            delete_post_meta($post_id, self::META_FIRST_SEEN_NOTIDX);
            delete_post_meta($post_id, self::META_RETRY_COUNT);
        }
    }

    /** Fetch URL with redirect follow (best-effort). */
    public static function fetch_url(string $url) : array {
        $seen = 0;
        $final = $url;
        $body = '';
        $headers = [];
        $code = 0;

        while ($seen < 3) {
            $seen++;
            $res = wp_remote_get($final, ['timeout' => 10, 'redirection' => 0]);
            if (is_wp_error($res)) {
                return ['final_url'=>$final, 'code'=>0, 'headers'=>[], 'body'=>''];
            }
            $code = intval(wp_remote_retrieve_response_code($res));
            $headers = array_change_key_case(wp_remote_retrieve_headers($res)->getAll() ?? [], CASE_LOWER);
            if ($code >= 300 && $code < 400 && !empty($headers['location'])) {
                $loc = is_array($headers['location']) ? end($headers['location']) : $headers['location'];
                $final = wp_http_validate_url($loc) ? $loc : wp_make_link_relative($loc);
                if (!preg_match('#^https?://#i', $final)) {
                    $final = home_url($final);
                }
                continue;
            }
            $body = (string) wp_remote_retrieve_body($res);
            break;
        }
        return ['final_url'=>$final, 'code'=>$code, 'headers'=>$headers, 'body'=>$body];
    }

    /** Parse meta robots + canonical (made public for reuse) */
    public static function parse_meta_robots_and_canonical(string $html) : array {
        $out = ['robots'=>'', 'canonical'=>''];
        if ($html === '') return $out;

        if (preg_match('/<meta[^>]+name=["\']robots["\'][^>]+content=["\']([^"\']+)["\']/i', $html, $m)) {
            $out['robots'] = trim($m[1]);
        }
        if (preg_match('/<link[^>]+rel=["\']canonical["\'][^>]+href=["\']([^"\']+)["\']/i', $html, $m)) {
            $out['canonical'] = trim($m[1]);
        }
        if (preg_match('/x-robots-tag\s*:\s*([^\s<]+)/i', $html, $m)) {
            $out['robots'] = $out['robots'] ?: trim($m[1]);
        }
        return $out;
    }

    /** Check if URL appears in sitemap(s) (made public for reuse) */
    public static function url_in_sitemap(string $url) : bool {
        $host = trailingslashit(home_url());
        $candidates = [
            $host . 'sitemap_index.xml',
            $host . 'sitemap.xml',
        ];
        foreach ($candidates as $sm) {
            $res = wp_remote_get($sm, ['timeout'=>6]);
            if (is_wp_error($res)) continue;
            $code = intval(wp_remote_retrieve_response_code($res));
            if ($code !== 200) continue;
            $xml = (string) wp_remote_retrieve_body($res);
            if ($xml === '') continue;

            if (stripos($xml, '<sitemapindex') !== false && preg_match_all('#<loc>([^<]+)</loc>#i', $xml, $m)) {
                $children = array_slice($m[1] ?? [], 0, 8);
                foreach ($children as $child) {
                    $r = wp_remote_get($child, ['timeout'=>6]);
                    if (!is_wp_error($r) && intval(wp_remote_retrieve_response_code($r)) === 200) {
                        $x = (string) wp_remote_retrieve_body($r);
                        if ($x && stripos($x, $url) !== false) return true;
                    }
                }
            } else {
                if (stripos($xml, $url) !== false) return true;
            }
        }
        return false;
    }

    /** Alerts */
    protected static function send_alerts(array $opt) {
        $days = max(1, intval($opt['alert_days'] ?? 7));
        $cut = strtotime("-{$days} days");
        $q = new \WP_Query([
            'post_type' => 'post',
            'post_status' => 'publish',
            'meta_query' => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for alert filter
                [
                    'key' => self::META_STATUS,
                    'value' => 'indexed',
                    'compare' => '!='
                ],
                [
                    'key' => self::META_FIRST_SEEN_NOTIDX,
                    'value' => gmdate('c', $cut),
                    'compare' => '<=',
                    'type' => 'DATETIME',
                ]
            ],
            'posts_per_page' => 50,
            'fields' => 'ids',
        ]);
        if (!$q->have_posts()) return;

        /* translators: %d: Number of days. */
        $lines = [sprintf(__('The following posts have not been indexed for %d+ days:', 'miro-ai-seo'), $days)];
        foreach ($q->posts as $pid) {
            $lines[] = get_the_title($pid) . ' — ' . get_permalink($pid);
        }
        wp_mail($opt['email_to'], 'Miro Index Monitor Alert', implode("\n", $lines));
    }

    /** BADGE */
    public static function status_badge(string $status) : string {
        $map = [
            'indexed'                 => ['🟢','Indexed','green'],
            'crawled_not_indexed'     => ['🟡','Crawled, not indexed','yellow'],
            'discovered_not_crawled'  => ['🟡','Discovered, not crawled','yellow'],
            'blocked_or_error'        => ['🔴','Blocked / Error','red'],
            'unknown'                 => ['⚪','Unknown','gray'],
        ];
        $m = $map[$status] ?? $map['unknown'];
        return '<span class="miro-badge--' . esc_attr($m[2]) . '"><strong>' . esc_html($m[0] . ' ' . $m[1]) . '</strong></span>';
    }

    /** Normalize GSC response → status */
    public static function normalize_status(array $gsc) : string {
        $verdict   = strtoupper($gsc['verdict'] ?? '');
        $coverage  = strtoupper($gsc['coverageState'] ?? '');
        $indexing  = strtoupper($gsc['indexingState'] ?? '');
        $robots    = strtoupper($gsc['robotsTxtState'] ?? '');
        $fetch     = strtoupper($gsc['pageFetchState'] ?? '');

        if ($verdict === 'PASS' && ($indexing === 'INDEXING_ALLOWED' || $coverage === 'SUBMITTED_AND_INDEXED' || $coverage === 'INDEXED')) {
            return 'indexed';
        }
        if (strpos($coverage, 'DISCOVERED') !== false) return 'discovered_not_crawled';
        if (strpos($coverage, 'CRAWLED') !== false || $verdict === 'NEUTRAL' || $verdict === 'PARTIAL') return 'crawled_not_indexed';
        if ($robots === 'BLOCKED' || $indexing === 'INDEXING_BLOCKED' || in_array($fetch, ['SOFT_404','NOT_FOUND','ACCESS_DENIED','REDIRECT_ERROR','INTERNAL_CRAWL_ERROR'], true))
            return 'blocked_or_error';
        return 'unknown';
    }
}
