<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) { exit; }

/**
 * REST controller for Index Monitor
 * Routes:
 *  - GET  /miro/v1/index/sites
 *  - POST /miro/v1/index/check     { post_id, property? }
 *  - POST /miro/v1/index/ping      { post_id }
 *  - GET  /miro/v1/index/status    ?page=&per_page=&filter=
 *  - POST /miro/v1/index/rescan    (queue backfill + immediate tick)
 *  - POST /miro/v1/index/scan_all  (synchronous chunked scanner)
 */
class Miro_REST_Index {

    public static function init() {
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    public static function register_routes() {

        // List verified GSC properties (settings picker)
        register_rest_route('miro/v1', '/index/sites', [
            'methods'  => 'GET',
            'permission_callback' => function(){ return current_user_can('manage_options'); },
            'callback' => [__CLASS__, 'route_sites'],
        ]);

        // Run URL Inspection for a single post (optionally override property)
        register_rest_route('miro/v1', '/index/check', [
            'methods'  => 'POST',
            'permission_callback' => function(){ return current_user_can('edit_posts'); },
            'callback' => [__CLASS__, 'route_check'],
            'args' => [
                'post_id' => ['required' => true,  'type' => 'integer'],
                'property'=> ['required' => false, 'type' => 'string'],
            ]
        ]);

        // Best-effort reindex (sitemaps + IndexNow)
        register_rest_route('miro/v1', '/index/ping', [
            'methods'  => 'POST',
            'permission_callback' => function(){ return current_user_can('edit_posts'); },
            'callback' => [__CLASS__, 'route_ping'],
            'args' => [
                'post_id' => ['required' => true, 'type' => 'integer'],
            ]
        ]);

        // Paginated index status list (reads our stored postmeta)
        register_rest_route('miro/v1', '/index/status', [
            'methods'  => 'GET',
            'permission_callback' => function(){ return current_user_can('edit_posts'); },
            'callback' => [__CLASS__, 'route_status'],
            'args' => [
                'page'     => ['type' => 'integer', 'default' => 1],
                'per_page' => ['type' => 'integer', 'default' => 20],
                'filter'   => ['type' => 'string',  'default' => 'all'], // '', 'indexed', ...
            ]
        ]);

        // Background backfill: queue all + run first tick immediately
        register_rest_route('miro/v1', '/index/rescan', [
            'methods'  => 'POST',
            'permission_callback' => function(){ return current_user_can('manage_options'); },
            'callback' => [__CLASS__, 'route_rescan'],
        ]);

        // Synchronous, chunked scan you can loop from UI
        register_rest_route('miro/v1', '/index/scan_all', [
            'methods'  => 'POST',
            'permission_callback' => function(){ return current_user_can('manage_options'); },
            'callback' => [__CLASS__, 'route_scan_all'],
            'args' => [
                'per_page'      => ['type'=>'integer','default'=>25],
                'page'          => ['type'=>'integer','default'=>1],
                'only_missing'  => ['type'=>'boolean','default'=>true],
                'status_filter' => ['type'=>'string','default'=>''], // '', 'indexed', 'crawled_not_indexed', ...
                'post_type'     => ['type'=>'string','default'=>'any'], // 'post', 'page', 'any'
                'property'      => ['type'=>'string','default'=>''], // GSC property override
            ],
        ]);
    }

    /* ======================= Private helpers ======================= */

    private static function get_gsc_access_token() {
        // 1) Try Rank Tracker (shared tokens)
        if (class_exists('\\Miro_AI_SEO\\Miro_Rank_Tracker') && method_exists('\\Miro_AI_SEO\\Miro_Rank_Tracker', 'get_access_token')) {
            $tok = \Miro_AI_SEO\Miro_Rank_Tracker::get_access_token();
            if ($tok) return $tok;
        }

        // 2) Fallback to GSC Connect stored token (and refresh if needed)
        if (class_exists('\\Miro_AI_SEO\\Miro_GSC_Connect')) {
            $s   = get_option(\Miro_AI_SEO\Miro_GSC_Connect::OPT_SETTINGS, []);
            $acc = (string)($s['access_token'] ?? '');
            $exp = intval($s['access_exp'] ?? 0);
            $ref = (string)($s['refresh_token'] ?? '');
            $cid = (string)($s['client_id'] ?? '');
            $csc = (string)($s['client_secret'] ?? '');

            if ($acc && $exp > (time() + 90)) return $acc;

            if ($ref && $cid && $csc) {
                $res = wp_remote_post(\Miro_AI_SEO\Miro_GSC_Connect::GOOGLE_TOKEN_URL, [
                    'timeout' => 15,
                    'headers' => ['Content-Type'=>'application/x-www-form-urlencoded'],
                    'body'    => http_build_query([
                        'client_id'     => $cid,
                        'client_secret' => $csc,
                        'grant_type'    => 'refresh_token',
                        'refresh_token' => $ref,
                    ], '', '&', PHP_QUERY_RFC3986),
                ]);
                if (!is_wp_error($res)) {
                    $code = wp_remote_retrieve_response_code($res);
                    $body = wp_remote_retrieve_body($res);
                    $json = json_decode($body, true);
                    if ($code === 200 && !empty($json['access_token'])) {
                        $acc = (string)$json['access_token'];
                        $ttl = intval($json['expires_in'] ?? 3600);
                        $s['access_token'] = $acc;
                        $s['access_exp']   = time() + max(60, $ttl) - 60;
                        update_option(\Miro_AI_SEO\Miro_GSC_Connect::OPT_SETTINGS, $s, false);
                        return $acc;
                    }
                }
            }
        }

        return '';
    }

    private static function pt_from_param($p) {
        if ($p && $p !== 'any') return [$p];
        return get_post_types(['public'=>true], 'names');
    }

    private static function is_local_url(string $url) : bool {
        $h = strtolower(wp_parse_url($url, PHP_URL_HOST) ?: '');
        if (in_array($h, ['localhost','127.0.0.1'], true)) return true;
        if (function_exists('str_ends_with')) {
            if (str_ends_with($h, '.local') || str_ends_with($h, '.test')) return true;
        } else {
            if (preg_match('~(\.local|\.test)$~i', $h)) return true;
        }
        return false;
    }

    protected static function save_unknown_and_return(int $post_id, string $url, array $why) {
        $now = gmdate('c');
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_STATUS, 'unknown');
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_DETAILS, wp_json_encode($why));
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_LAST_CHECK, $now);
        return ['status'=>'unknown','details'=>$why,'last_check'=>$now,'url'=>$url];
    }

    protected static function save_fake_and_return(int $post_id, string $url) {
        $gsc = self::fake_gsc_result($url);
        $status = \Miro_AI_SEO\Miro_Index_Monitor::normalize_status($gsc);
        $now = gmdate('c');

        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_STATUS, $status);
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_DETAILS, wp_json_encode($gsc));
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_LAST_CHECK, $now);

        if ($status !== 'indexed') {
            if (!get_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_FIRST_SEEN_NOTIDX, true)) {
                update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_FIRST_SEEN_NOTIDX, $now);
            }
            $rc = (int) get_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_RETRY_COUNT, true);
            update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_RETRY_COUNT, $rc + 1);
        } else {
            delete_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_FIRST_SEEN_NOTIDX);
            delete_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_RETRY_COUNT);
        }

        return [
            'status'      => $status,
            'details'     => $gsc,
            'last_check'  => $now,
            'url'         => $url,
            'siteUrl'     => '(dev-stub)',
        ];
    }

    protected static function fake_gsc_result(string $url) : array {
        $h = crc32($url);
        $r = $h % 5;
        $map = [
            0 => ['verdict' => 'PASS',    'indexingState' => 'INDEXING_ALLOWED', 'coverageState' => 'Indexed',                              'robotsTxtState' => 'ALLOWED', 'pageFetchState' => 'SUCCESSFUL'],
            1 => ['verdict' => 'PARTIAL', 'indexingState' => 'INDEXING_ALLOWED', 'coverageState' => 'Crawled - currently not indexed',     'robotsTxtState' => 'ALLOWED', 'pageFetchState' => 'SUCCESSFUL'],
            2 => ['verdict' => 'PARTIAL', 'indexingState' => 'INDEXING_ALLOWED', 'coverageState' => 'Discovered - currently not indexed',  'robotsTxtState' => 'ALLOWED', 'pageFetchState' => 'REDIRECT_ERROR'],
            3 => ['verdict' => 'FAIL',    'indexingState' => 'INDEXING_BLOCKED', 'coverageState' => 'Blocked by robots.txt',               'robotsTxtState' => 'BLOCKED', 'pageFetchState' => 'ACCESS_DENIED'],
            4 => ['verdict' => 'FAIL',    'indexingState' => 'INDEXING_ALLOWED', 'coverageState' => 'Crawled - currently not indexed',     'robotsTxtState' => 'ALLOWED', 'pageFetchState' => 'SOFT_404'],
        ];
        $o = $map[$r];
        $o['lastCrawlTime'] = gmdate('c', time() - ($h % 86400));
        $o['canonicalUrl']  = $url;
        return $o;
    }

    /* ======================= Routes ======================= */

    public static function route_sites(\WP_REST_Request $req) {
        // Prefer a filter the GSC module exposes
        $sites = apply_filters('miro/gsc/sites', null);

        // Fallback to option cache if filter not implemented
        if (!is_array($sites)) {
            $sites = get_option('miro_gsc_cached_sites');
        }
        if (!is_array($sites)) $sites = [];

        // Normalize
        $out = [];
        foreach ($sites as $s) {
            if (is_string($s)) {
                $out[] = [
                    'siteUrl'  => $s,
                    'type'     => (strpos($s, 'sc-domain:') === 0 ? 'DOMAIN' : 'URL_PREFIX'),
                    'verified' => true,
                ];
            } elseif (is_array($s)) {
                $out[] = [
                    'siteUrl'  => $s['siteUrl'] ?? ($s['property'] ?? ''),
                    'type'     => $s['type'] ?? ((isset($s['siteUrl']) && strpos($s['siteUrl'], 'sc-domain:') === 0) ? 'DOMAIN' : 'URL_PREFIX'),
                    'verified' => isset($s['verified']) ? (bool)$s['verified'] : true,
                ];
            }
        }

        return rest_ensure_response(['ok'=>true, 'sites'=>$out]);
    }

    /**
     * Core URL Inspection for a single post
     */
    public static function perform_check_for_post(int $post_id, string $property_override = '') {
        $url = get_permalink($post_id);
        if (!$url) return new \WP_Error('bad_post', 'Post not found');

        // Dev/localhost stub for smooth local dev
        if ((defined('MIRO_INDEX_DEV_STUB') && MIRO_INDEX_DEV_STUB) || self::is_local_url($url)) {
            return self::save_fake_and_return($post_id, $url);
        }

        // Resolve property (override → Index Monitor → Rank Tracker → GSC Connect)
        $prop = trim($property_override);
        if (!$prop && class_exists('\\Miro_AI_SEO\\Miro_Index_Monitor')) {
            $im = \Miro_AI_SEO\Miro_Index_Monitor::get_settings();
            if (!empty($im['gsc_property'])) $prop = (string)$im['gsc_property'];
        }
        if (!$prop && class_exists('\\Miro_AI_SEO\\Miro_Rank_Tracker')) {
            $rtOpt = get_option(\Miro_AI_SEO\Miro_Rank_Tracker::OPT, []);
            if (!empty($rtOpt['gsc_property'])) $prop = (string)$rtOpt['gsc_property'];
        }
        if (!$prop && class_exists('\\Miro_AI_SEO\\Miro_GSC_Connect')) {
            $gscProp = get_option(\Miro_AI_SEO\Miro_GSC_Connect::OPT_PROPERTY, []);
            if (!empty($gscProp['uri'])) $prop = (string)$gscProp['uri'];
        }

        // OAuth token (Rank Tracker first; else GSC Connect)
        $token = self::get_gsc_access_token();

        if (!$prop) {
            return self::save_unknown_and_return($post_id, $url, ['reason' => 'no_property']);
        }
        if (!$token) {
            return self::save_unknown_and_return($post_id, $url, ['reason' => 'no_token']);
        }

        // URL Inspection API
        $endpoint = 'https://searchconsole.googleapis.com/v1/urlInspection/index:inspect';
        $body = [
            'inspectionUrl' => $url,
            'siteUrl'       => $prop,
            'languageCode'  => 'en-US',
        ];
        $res = wp_remote_post($endpoint, [
            'timeout' => 30,
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode($body),
        ]);

        if (is_wp_error($res)) {
            return self::save_unknown_and_return($post_id, $url, [
                'reason' => 'api_error',
                'error'  => $res->get_error_message()
            ]);
        }

        $code = (int) wp_remote_retrieve_response_code($res);
        $raw  = wp_remote_retrieve_body($res);
        if ($code !== 200) {
            $err = ['reason'=>'api_error', 'http'=>$code, 'body'=>$raw];
            if ($code === 400 || $code === 403) {
                $err['hint'] = 'Verify siteUrl matches a verified property that owns the inspected URL.';
            }
            return self::save_unknown_and_return($post_id, $url, $err);
        }

        $json = json_decode($raw, true) ?: [];
        $idx  = $json['inspectionResult']['indexStatusResult'] ?? [];
        $details = [
            'verdict'        => $idx['verdict'] ?? '',
            'coverageState'  => $idx['coverageState'] ?? '',
            'indexingState'  => $idx['indexingState'] ?? '',
            'robotsTxtState' => $idx['robotsTxtState'] ?? '',
            'pageFetchState' => $idx['pageFetchState'] ?? '',
            'lastCrawlTime'  => $idx['lastCrawlTime'] ?? '',
            'canonicalUrl'   => $idx['googleCanonical'] ?? ($idx['userCanonical'] ?? ''),
        ];

        $status = \Miro_AI_SEO\Miro_Index_Monitor::normalize_status($details);
        $now = gmdate('c');

        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_STATUS, $status);
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_DETAILS, wp_json_encode($details));
        update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_LAST_CHECK, $now);

        if ($status !== 'indexed') {
            if (!get_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_FIRST_SEEN_NOTIDX, true)) {
                update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_FIRST_SEEN_NOTIDX, $now);
            }
            $rc = (int) get_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_RETRY_COUNT, true);
            update_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_RETRY_COUNT, $rc + 1);
        } else {
            delete_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_FIRST_SEEN_NOTIDX);
            delete_post_meta($post_id, \Miro_AI_SEO\Miro_Index_Monitor::META_RETRY_COUNT);
        }

        return [
            'status'      => $status,
            'details'     => $details,
            'last_check'  => $now,
            'url'         => $url,
            'siteUrl'     => $prop,
        ];
    }

    public static function route_check(\WP_REST_Request $req) {
        $pid  = intval($req->get_param('post_id'));
        if (!$pid) return new \WP_Error('bad_request', 'post_id required');

        $prop = sanitize_text_field($req->get_param('property') ?: '');
        $res  = self::perform_check_for_post($pid, $prop);

        if (is_wp_error($res)) return $res;
        return rest_ensure_response($res);
    }

    public static function route_ping(\WP_REST_Request $req) {
        $pid = intval($req->get_param('post_id'));
        if (!$pid) return new \WP_Error('bad_request', 'post_id required');

        $url = get_permalink($pid);
        if (!$url) return new \WP_Error('bad_post', 'Post not found');

        $opt = \Miro_AI_SEO\Miro_Index_Monitor::get_settings();
        $actions = [];

        if (!empty($opt['ping_sitemaps'])) {
            $sitemap = home_url('/sitemap.xml');
            wp_remote_get('https://www.google.com/ping?sitemap=' . urlencode($sitemap));
            wp_remote_get('https://www.bing.com/ping?sitemap=' . urlencode($sitemap));
            $actions[] = 'sitemap_ping';
        }

        if (!empty($opt['indexnow_key'])) {
            $endpoint = 'https://api.indexnow.org/indexnow';
            $body = [
                'host'        => wp_parse_url(home_url(), PHP_URL_HOST),
                'key'         => $opt['indexnow_key'],
                'urlList'     => [$url],
                'keyLocation' => home_url('/' . $opt['indexnow_key'] . '.txt'),
            ];
            wp_remote_post($endpoint, [
                'headers' => ['Content-Type' => 'application/json'],
                'body'    => wp_json_encode($body),
                'timeout' => 10,
            ]);
            $actions[] = 'indexnow_submit';
        }

        return rest_ensure_response([
            'ok'      => true,
            'url'     => $url,
            'actions' => $actions
        ]);
    }

    public static function route_status(\WP_REST_Request $req) {
        $page   = max(1, intval($req->get_param('page')));
        $per    = max(1, min(100, intval($req->get_param('per_page'))));
        $filter = sanitize_text_field($req->get_param('filter') ?: 'all');

        $meta_query = [];
        if ($filter && $filter !== 'all') {
            $meta_query[] = [
                'key'     => \Miro_AI_SEO\Miro_Index_Monitor::META_STATUS,
                'value'   => $filter,
                'compare' => '=',
            ];
        }

        $q = new \WP_Query([
            'post_type'      => 'post',
            'post_status'    => 'publish',
            'posts_per_page' => $per,
            'paged'          => $page,
            'fields'         => 'ids',
            'meta_query'     => $meta_query, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for status filter
            'orderby'        => 'date',
            'order'          => 'DESC',
        ]);

        $rows = [];
        foreach ($q->posts as $pid) {
            $rows[] = [
                'post_id'    => $pid,
                'title'      => get_the_title($pid),
                'url'        => get_permalink($pid),
                'status'     => get_post_meta($pid, \Miro_AI_SEO\Miro_Index_Monitor::META_STATUS, true) ?: 'unknown',
                'last_check' => get_post_meta($pid, \Miro_AI_SEO\Miro_Index_Monitor::META_LAST_CHECK, true),
                'details'    => json_decode(get_post_meta($pid, \Miro_AI_SEO\Miro_Index_Monitor::META_DETAILS, true) ?: '[]', true),
            ];
        }

        return rest_ensure_response([
            'page'     => $page,
            'per_page' => $per,
            'total'    => intval($q->found_posts),
            'rows'     => $rows,
        ]);
    }

    public static function route_rescan(\WP_REST_Request $req) {
        \Miro_AI_SEO\Miro_Index_Monitor::queue_full_backfill(20);
        // Run one immediate tick so UI shows progress now
        \Miro_AI_SEO\Miro_Index_Monitor::cron_backfill_tick();

        return rest_ensure_response([
            'ok'     => true,
            'queued' => true,
            'note'   => 'Backfill scheduled. First batch processed immediately.'
        ]);
    }

    /**
     * Synchronous, chunked scan (call repeatedly from UI until done=true)
     * Args:
     *  - per_page (1..200)
     *  - page (>=1)
     *  - only_missing (bool)   : scan only posts w/o _miro_index_last_check
     *  - status_filter (string): scan only posts with that stored status
     *  - post_type ('any'|'post'|'page'|custom)
     *  - property (string)     : GSC property override
     */
    public static function route_scan_all(\WP_REST_Request $req) {
        $per        = max(1, min(200, (int)$req->get_param('per_page')));
        $page       = max(1, (int)$req->get_param('page'));
        $onlyMiss   = (bool) $req->get_param('only_missing');
        $statusFilt = trim((string)$req->get_param('status_filter') ?: '');
        $ptParam    = trim((string)$req->get_param('post_type') ?: 'any');
        $prop       = trim((string)$req->get_param('property') ?: '');

        // Build post type array
        $types = ($ptParam && $ptParam !== 'any')
            ? [$ptParam]
            : get_post_types(['public'=>true], 'names');

        // Meta filters
        $meta_query = [];
        if ($onlyMiss) {
            $meta_query[] = [
                'key'     => \Miro_AI_SEO\Miro_Index_Monitor::META_LAST_CHECK,
                'compare' => 'NOT EXISTS',
            ];
        }
        if ($statusFilt !== '') {
            $meta_query[] = [
                'key'     => \Miro_AI_SEO\Miro_Index_Monitor::META_STATUS,
                'value'   => $statusFilt,
                'compare' => '=',
            ];
        }

        $q = new \WP_Query([
            'post_type'      => $types,
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => $per,
            'paged'          => $page,
            'meta_query'     => $meta_query, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for status filter
            'orderby'        => 'date',
            'order'          => 'DESC',
        ]);

        $processed = 0;
        foreach ($q->posts as $pid) {
            $pid = (int)$pid;
            self::perform_check_for_post($pid, $prop);
            $processed++;
        }

        // Count total for same filter (cheap count query)
        $qCount = new \WP_Query([
            'post_type'      => $types,
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => 1,
            'paged'          => 1,
            'meta_query'     => $meta_query, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required for status filter
            'orderby'        => 'date',
            'order'          => 'DESC',
        ]);
        $total = (int)$qCount->found_posts;
        $pages = (int)max(1, ceil($total / $per));

        return rest_ensure_response([
            'ok'        => true,
            'processed' => $processed,
            'page'      => $page,
            'per_page'  => $per,
            'total'     => $total,
            'pages'     => $pages,
            'done'      => ($page >= $pages),
        ]);
    }
}
