<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — One place for optional SEO features (each can be turned on/off).
 * Tabs: Settings, Local SEO, Schema, Broken Links, Breadcrumbs, OG & Twitter, Sitemap, Canonical & Duplicates.
 */
class Miro_SEO_Hub {

    const MENU_PARENT = 'miro-ai-seo';
    const MENU_SLUG   = 'miro-seo-hub';
    const OPT_SETTINGS = 'miro_seo_hub_settings';

    /** Tab slugs and labels (order = tab order). */
    const TABS = [
        'settings'   => 'Settings',
        'local_seo'  => 'Local SEO',
        'schema'      => 'Schema',
        'broken_links'=> 'Broken Links',
        'breadcrumbs' => 'Breadcrumbs',
        'og_twitter'  => 'OG & Twitter',
        'sitemap'     => 'Sitemap',
    ];

    /** Return translated tab label for a slug (uses literals for i18n extraction). */
    private static function get_tab_label(string $slug): string {
        switch ($slug) {
            case 'settings':
                return __('Settings', 'miro-ai-seo-free');
            case 'local_seo':
                return __('Local SEO', 'miro-ai-seo-free');
            case 'schema':
                return __('Schema', 'miro-ai-seo-free');
            case 'broken_links':
                return __('Broken Links', 'miro-ai-seo-free');
            case 'breadcrumbs':
                return __('Breadcrumbs', 'miro-ai-seo-free');
            case 'og_twitter':
                return __('OG & Twitter', 'miro-ai-seo-free');
            case 'sitemap':
                return __('Sitemap', 'miro-ai-seo-free');
            default:
                return $slug;
        }
    }

    public static function init(): void {
        add_action('admin_menu', [__CLASS__, 'menu'], 35);
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'assets']);
    }

    public static function menu(): void {
        if (function_exists('miro_ai_add_submenu_once')) {
            miro_ai_add_submenu_once(
                self::MENU_PARENT,
                __('SEO Hub', 'miro-ai-seo-free'),
                __('SEO Hub', 'miro-ai-seo-free'),
                function_exists('miro_ai_cap') ? \miro_ai_cap() : 'edit_posts',
                self::MENU_SLUG,
                [__CLASS__, 'render_page'],
                35
            );
        } else {
            add_submenu_page(
                self::MENU_PARENT,
                __('SEO Hub', 'miro-ai-seo-free'),
                __('SEO Hub', 'miro-ai-seo-free'),
                function_exists('miro_ai_cap') ? \miro_ai_cap() : 'edit_posts',
                self::MENU_SLUG,
                [__CLASS__, 'render_page'],
                35
            );
        }
    }

    public static function register_settings(): void {
        register_setting('miro_seo_hub_group', self::OPT_SETTINGS, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize_settings'],
        ]);
    }

    public static function sanitize_settings($input): array {
        if (!is_array($input)) return self::default_settings();
        $out = [];
        foreach (array_keys(self::TABS) as $key) {
            if ($key === 'settings') continue;
            $out[$key] = !empty($input[$key]) ? 1 : 0;
        }
        return $out;
    }

    public static function default_settings(): array {
        $out = [];
        foreach (array_keys(self::TABS) as $key) {
            if ($key === 'settings') continue;
            $out[$key] = 0; // Default off — enable only if user wants it; avoids conflict with other SEO plugins.
        }
        return $out;
    }

    public static function get_settings(): array {
        $s = get_option(self::OPT_SETTINGS, []);
        return array_merge(self::default_settings(), is_array($s) ? $s : []);
    }

    public static function is_tab_enabled(string $tab): bool {
        if ($tab === 'settings') return true;
        $s = self::get_settings();
        return !empty($s[$tab]);
    }

    /** Update a tab's active state (used when saving from within a tab). */
    public static function set_tab_enabled(string $tab, bool $enabled): void {
        $s = self::get_settings();
        $s[$tab] = $enabled ? 1 : 0;
        update_option(self::OPT_SETTINGS, $s);
    }

    public static function assets($hook): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Page detection for asset loading, not form processing.
        $page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : '';
        if ($page !== self::MENU_SLUG) return;
        $ver = defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0';
        if (defined('MIRO_AI_SEO_URL')) {
            wp_enqueue_style('miro-alt', MIRO_AI_SEO_URL . 'assets/css/miro-alt.css', [], $ver);
            wp_enqueue_style('miro-seo-hub', MIRO_AI_SEO_URL . 'assets/css/miro-seo-hub.css', ['miro-alt'], $ver);
        }
        wp_enqueue_media();
        wp_enqueue_script('miro-seo-hub', defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/js/miro-seo-hub.js' : '', ['jquery', 'media-editor'], $ver, true);
    }

    public static function render_page(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'edit_posts')) {
            wp_die(esc_html__('You do not have permission.', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        $settings = self::get_settings();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab detection for page render, not form processing.
        $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'settings';
        if (!isset(self::TABS[$current_tab])) $current_tab = 'settings';

        if (defined('MIRO_AI_SEO_URL')) {
            wp_enqueue_style('miro-alt', MIRO_AI_SEO_URL . 'assets/css/miro-alt.css', [], defined('MIRO_AI_SEO_VERSION') ? MIRO_AI_SEO_VERSION : '1.0.0');
        }
        $af_logo = defined('MIRO_AI_SEO_URL') ? MIRO_AI_SEO_URL . 'assets/img/miro-logo.webp' : '';
        ?>
        <div class="wrap"><h1 class="wp-heading-inline"><?php echo esc_html__('SEO Hub', 'miro-ai-seo-free'); ?></h1></div>
        <div class="miro-alt-wrap miro-seo-hub">
            <div class="af-hero">
                <div class="af-hero-pill">
                    <div class="af-hero-pill-inner">
                        <?php if (!empty($af_logo)) : ?>
                            <img src="<?php echo esc_url($af_logo); ?>" alt="<?php echo esc_attr__('Miro AI SEO', 'miro-ai-seo-free'); ?>">
                        <?php else : ?>
                            <div class="af-hero-pill-fallback">SEO</div>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="af-hero-main">
                    <div class="af-hero-title-row">
                        <div class="af-hero-title"><?php echo esc_html__('All-in-one SEO controls', 'miro-ai-seo-free'); ?></div>
                        <span class="af-hero-tag alt"><?php echo esc_html__('SEO Hub', 'miro-ai-seo-free'); ?></span>
                    </div>
                    <p class="af-hero-sub"><?php echo esc_html__('Local SEO, Schema, Broken links, Breadcrumbs, Open Graph, Sitemap, Canonical. Turn each section on or off so it plays nice with other plugins.', 'miro-ai-seo-free'); ?></p>
                    <div class="af-hero-chips">
                        <div class="af-chip-pro af-chip-alt">
                            <strong><?php echo esc_html(count(array_filter($settings))); ?></strong>
                            <span class="af-chip-sub"><?php echo esc_html__('sections on', 'miro-ai-seo-free'); ?></span>
                        </div>
                        <div class="af-chip-pro af-chip-safe">
                            <strong><?php echo esc_html__('Optional', 'miro-ai-seo-free'); ?></strong>
                            <span class="af-chip-sub"><?php echo esc_html__('per-section toggle', 'miro-ai-seo-free'); ?></span>
                        </div>
                    </div>
                </div>
                <div class="af-hero-metrics">
                    <div class="af-tip-title">
                        <span class="af-tip-ico">💡</span>
                        <span><strong><?php echo esc_html__('Tip', 'miro-ai-seo-free'); ?></strong></span>
                    </div>
                    <div class="af-tip-sub">
                        <?php echo esc_html__('Disable any section in the Settings tab if Rank Math, Yoast or another plugin already handles it.', 'miro-ai-seo-free'); ?>
                    </div>
                </div>
            </div>

            <div class="af-card miro-seo-hub-card">
                <nav class="miro-seo-hub-tabs" role="tablist">
                    <?php foreach (self::TABS as $slug => $label) : ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=' . self::MENU_SLUG . '&tab=' . $slug)); ?>"
                           class="miro-seo-hub-tab <?php echo $current_tab === $slug ? 'active' : ''; ?>"><?php echo esc_html(self::get_tab_label($slug)); ?></a>
                    <?php endforeach; ?>
                </nav>

                <div class="miro-seo-hub-content">
                    <?php
                    switch ($current_tab) {
                        case 'settings':
                            self::render_tab_settings($settings);
                            break;
                        case 'local_seo':
                            if (class_exists(__NAMESPACE__ . '\\SEO_Hub_Local_SEO')) {
                                SEO_Hub_Local_SEO::render_form();
                            } else {
                                self::render_tab_placeholder('local_seo', __('Local SEO', 'miro-ai-seo-free'), __('Business info, NAP, LocalBusiness schema, service areas. Turn off if you use another local SEO plugin.', 'miro-ai-seo-free'));
                            }
                            break;
                        case 'schema':
                            if (class_exists(__NAMESPACE__ . '\\SEO_Hub_Schema')) {
                                SEO_Hub_Schema::render_form();
                            } else {
                                self::render_tab_placeholder('schema', __('Schema / Structured Data', 'miro-ai-seo-free'), __('Article, FAQ, HowTo, Organization. Turn off if Rank Math or Yoast already outputs schema.', 'miro-ai-seo-free'));
                            }
                            break;
                        case 'broken_links':
                            if (class_exists(__NAMESPACE__ . '\\SEO_Hub_Broken_Links')) {
                                SEO_Hub_Broken_Links::render_form();
                            } else {
                                self::render_tab_placeholder('broken_links', __('Broken Link Checker', 'miro-ai-seo-free'), __('Scan posts for broken internal and external links. Report and fix suggestions.', 'miro-ai-seo-free'));
                            }
                            break;
                        case 'breadcrumbs':
                            if (class_exists(__NAMESPACE__ . '\\SEO_Hub_Breadcrumbs')) {
                                SEO_Hub_Breadcrumbs::render_form();
                            } else {
                                self::render_tab_placeholder('breadcrumbs', __('Breadcrumbs', 'miro-ai-seo-free'), __('Shortcode and BreadcrumbList schema. Turn off if your theme or another plugin already handles breadcrumbs.', 'miro-ai-seo-free'));
                            }
                            break;
                        case 'og_twitter':
                            if (class_exists(__NAMESPACE__ . '\\SEO_Hub_OG_Twitter')) {
                                SEO_Hub_OG_Twitter::render_form();
                            } else {
                                self::render_tab_placeholder('og_twitter', __('OG & Twitter Cards', 'miro-ai-seo-free'), __('Open Graph and Twitter Card meta for social sharing. Turn off if another plugin handles this.', 'miro-ai-seo-free'));
                            }
                            break;
                        case 'sitemap':
                            if (class_exists(__NAMESPACE__ . '\\SEO_Hub_Sitemap')) {
                                SEO_Hub_Sitemap::render_form();
                            } else {
                                self::render_tab_placeholder('sitemap', __('Sitemap', 'miro-ai-seo-free'), __('Enhanced sitemap (exclusions). Turn off if another plugin handles sitemaps.', 'miro-ai-seo-free'));
                            }
                            break;
                        default:
                            self::render_tab_settings($settings);
                    }
                    ?>
                </div>
            </div>
        </div>
        <?php
    }

    protected static function render_tab_settings(array $settings): void {
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('Which sections are active', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Disable any section you don’t need or that another plugin already handles. When disabled, that section won’t output anything on the site.', 'miro-ai-seo-free'); ?></p>
        <form method="post" action="options.php" class="miro-seo-hub-settings-form">
            <?php settings_fields('miro_seo_hub_group'); ?>
            <table class="form-table">
                <?php foreach (self::TABS as $slug => $label) : if ($slug === 'settings') continue; ?>
                    <tr>
                        <th scope="row"><?php echo esc_html(self::get_tab_label($slug)); ?></th>
                        <td>
                            <label>
                                <input type="hidden" name="<?php echo esc_attr(self::OPT_SETTINGS); ?>[<?php echo esc_attr($slug); ?>]" value="0" />
                                <input type="checkbox" name="<?php echo esc_attr(self::OPT_SETTINGS); ?>[<?php echo esc_attr($slug); ?>]" value="1" <?php checked(!empty($settings[$slug])); ?> />
                                <?php echo esc_html__('Enable', 'miro-ai-seo-free'); ?>
                            </label>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
            <p class="submit"><input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save', 'miro-ai-seo-free'); ?>" /></p>
        </form>
        <?php
    }

    protected static function render_tab_placeholder(string $tab_slug, string $title, string $description): void {
        $enabled = self::is_tab_enabled($tab_slug);
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html($title); ?></h2>
        <?php if (!$enabled) : ?>
            <div class="notice notice-warning inline"><p><?php echo esc_html__('This section is disabled. Enable it in the Settings tab to use it.', 'miro-ai-seo-free'); ?></p></div>
        <?php else : ?>
            <p class="description"><?php echo esc_html($description); ?></p>
            <p class="miro-seo-hub-coming"><?php echo esc_html__('Configuration and tools for this section are coming in the next update.', 'miro-ai-seo-free'); ?></p>
        <?php endif; ?>
        <?php
    }
}
