<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — Breadcrumbs: shortcode [miro_breadcrumbs] and optional BreadcrumbList schema.
 */
class SEO_Hub_Breadcrumbs {

    const OPT = 'miro_seo_hub_breadcrumbs';
    const SAVE_ACTION = 'miro_seo_hub_save_breadcrumbs';
    const NONCE = 'miro_seo_hub_breadcrumbs_nonce';

    public static function init(): void {
        add_action('admin_init', [__CLASS__, 'register_setting']);
        add_action('admin_post_' . self::SAVE_ACTION, [__CLASS__, 'handle_save']);
        add_action('wp_head', [__CLASS__, 'output_breadcrumb_schema'], 9);
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_front_style']);
        add_shortcode('miro_breadcrumbs', [__CLASS__, 'shortcode_breadcrumbs']);
    }

    public static function enqueue_front_style(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('breadcrumbs') || is_admin()) return;
        if (!defined('MIRO_AI_SEO_URL') || !defined('MIRO_AI_SEO_VERSION')) return;
        wp_enqueue_style(
            'miro-breadcrumbs',
            MIRO_AI_SEO_URL . 'assets/css/miro-breadcrumbs.css',
            [],
            MIRO_AI_SEO_VERSION
        );
    }

    public static function register_setting(): void {
        register_setting('miro_seo_hub_breadcrumbs_group', self::OPT, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
        ]);
    }

    public static function default_settings(): array {
        return [
            'home_label'    => 'Home',
            'separator'     => '»',
            'output_schema' => 1,
            'css_class'     => 'miro-breadcrumbs',
        ];
    }

    public static function get_settings(): array {
        $s = get_option(self::OPT, []);
        return array_merge(self::default_settings(), is_array($s) ? $s : []);
    }

    public static function sanitize($input): array {
        if (!is_array($input)) return self::default_settings();
        $out = self::default_settings();
        $out['home_label']    = sanitize_text_field($input['home_label'] ?? 'Home');
        if ($out['home_label'] === '') $out['home_label'] = 'Home';
        $out['separator']     = sanitize_text_field($input['separator'] ?? '»');
        $out['output_schema'] = !empty($input['output_schema']) ? 1 : 0;
        $out['css_class']     = preg_replace('/[^a-zA-Z0-9_\-\s]/', '', (string) ($input['css_class'] ?? 'miro-breadcrumbs'));
        $out['css_class']     = trim($out['css_class']) ?: 'miro-breadcrumbs';
        return $out;
    }

    public static function handle_save(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SAVE_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passed to self::sanitize() which sanitizes all fields.
        $opts = isset($_POST[self::OPT]) && is_array($_POST[self::OPT]) ? wp_unslash($_POST[self::OPT]) : [];
        update_option(self::OPT, self::sanitize($opts));
        Miro_SEO_Hub::set_tab_enabled('breadcrumbs', !empty(sanitize_text_field(wp_unslash($_POST['section_active'] ?? ''))));
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=breadcrumbs&saved=1'));
        exit;
    }

    /**
     * Build list of breadcrumb items: each is [ 'name' => string, 'url' => string|false ].
     * Last item typically has url false (current page).
     */
    public static function get_trail(): array {
        $home_label = self::get_settings()['home_label'];
        $items = [];
        $items[] = ['name' => $home_label, 'url' => home_url('/')];

        if (is_singular()) {
            $post = get_post();
            if (!$post) return $items;

            if (is_singular('post')) {
                $cats = get_the_category($post->ID);
                if (!empty($cats)) {
                    $cat = reset($cats);
                    $items[] = ['name' => $cat->name, 'url' => get_category_link($cat->term_id)];
                }
            }
            if (is_page() && $post->post_parent) {
                $ancestors = get_post_ancestors($post->ID);
                $ancestors = array_reverse($ancestors);
                foreach ($ancestors as $aid) {
                    $parent = get_post($aid);
                    if ($parent && $parent->post_status === 'publish') {
                        $items[] = ['name' => get_the_title($parent), 'url' => get_permalink($parent)];
                    }
                }
            }
            $items[] = ['name' => get_the_title($post), 'url' => false];
            return $items;
        }

        if (is_category()) {
            $cat = get_queried_object();
            if ($cat && isset($cat->term_id)) {
                if ($cat->parent) {
                    $ancestors = get_ancestors($cat->term_id, 'category');
                    $ancestors = array_reverse($ancestors);
                    foreach ($ancestors as $tid) {
                        $t = get_term($tid, 'category');
                        if ($t && !is_wp_error($t)) {
                            $items[] = ['name' => $t->name, 'url' => get_category_link($t->term_id)];
                        }
                    }
                }
                $items[] = ['name' => $cat->name, 'url' => false];
            }
            return $items;
        }

        if (is_tag()) {
            $tag = get_queried_object();
            if ($tag && isset($tag->name)) {
                $items[] = ['name' => $tag->name, 'url' => false];
            }
            return $items;
        }

        if (is_author()) {
            $author = get_queried_object();
            if ($author && isset($author->display_name)) {
                $items[] = ['name' => $author->display_name, 'url' => false];
            }
            return $items;
        }

        if (is_post_type_archive()) {
            $obj = get_queried_object();
            $name = $obj && isset($obj->labels->name) ? $obj->labels->name : (is_array($obj) && isset($obj['label']) ? $obj['label'] : __('Archive', 'miro-ai-seo-free'));
            $items[] = ['name' => $name, 'url' => false];
            return $items;
        }

        if (is_date()) {
            if (is_year()) {
                $items[] = ['name' => get_the_date('Y'), 'url' => false];
            } elseif (is_month()) {
                $items[] = ['name' => get_the_date('F Y'), 'url' => false];
            } else {
                $items[] = ['name' => get_the_date(), 'url' => false];
            }
            return $items;
        }

        if (is_search()) {
            /* translators: %s: Search query string. */
            $items[] = ['name' => sprintf(__('Search for "%s"', 'miro-ai-seo-free'), get_search_query()), 'url' => false];
            return $items;
        }

        if (is_404()) {
            $items[] = ['name' => __('Page not found', 'miro-ai-seo-free'), 'url' => false];
            return $items;
        }

        if (is_home() && !is_front_page()) {
            $items[] = ['name' => get_the_title(get_option('page_for_posts')), 'url' => false];
            return $items;
        }

        return $items;
    }

    /** Output BreadcrumbList JSON-LD in wp_head when enabled. */
    public static function output_breadcrumb_schema(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('breadcrumbs')) return;
        $s = self::get_settings();
        if (empty($s['output_schema'])) return;

        $trail = self::get_trail();
        if (count($trail) < 2) return;

        $list = [];
        $pos = 1;
        $current_url = is_singular() ? get_permalink() : (is_search() ? get_search_link() : '');
        if (!$current_url) $current_url = home_url('/');
        foreach ($trail as $item) {
            $url = $item['url'] ? $item['url'] : $current_url;
            $list[] = [
                '@type'    => 'ListItem',
                'position' => $pos++,
                'name'     => $item['name'],
                'item'     => $url,
            ];
        }
        $data = ['@context' => 'https://schema.org', '@type' => 'BreadcrumbList', 'itemListElement' => $list];
        echo '<script type="application/ld+json">' . "\n" . wp_json_encode($data) . "\n" . '</script>' . "\n";
    }

    public static function shortcode_breadcrumbs($atts): string {
        if (!Miro_SEO_Hub::is_tab_enabled('breadcrumbs')) return '';
        $a = shortcode_atts([
            'class' => '',
            'sep'   => '',
        ], $atts ?? [], 'miro_breadcrumbs');
        $s = self::get_settings();
        $sep = $a['sep'] !== '' ? $a['sep'] : $s['separator'];
        $css = $a['class'] !== '' ? esc_attr($a['class']) : esc_attr($s['css_class']);
        $trail = self::get_trail();
        if (empty($trail)) return '';

        $out = '<nav class="' . $css . '" aria-label="' . esc_attr__('Breadcrumb', 'miro-ai-seo-free') . '"><ol class="' . $css . '__list" itemscope itemtype="https://schema.org/BreadcrumbList">';
        $pos = 1;
        $n = count($trail);
        foreach ($trail as $i => $item) {
            $out .= '<li class="' . $css . '__item" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">';
            if ($item['url']) {
                $out .= '<a itemprop="item" href="' . esc_url($item['url']) . '"><span itemprop="name">' . esc_html($item['name']) . '</span></a>';
            } else {
                $out .= '<span itemprop="name">' . esc_html($item['name']) . '</span>';
            }
            $out .= '<meta itemprop="position" content="' . $pos . '" />';
            $out .= '</li>';
            if ($i < $n - 1) {
                $out .= '<li class="' . $css . '__sep" aria-hidden="true">' . esc_html($sep) . '</li>';
            }
            $pos++;
        }
        $out .= '</ol></nav>';
        return $out;
    }

    public static function render_form(): void {
        $s = self::get_settings();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Redirect param for success message display, not form processing.
        $saved = isset($_GET['saved']) && sanitize_text_field(wp_unslash($_GET['saved'])) === '1';
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('Breadcrumbs', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Display a breadcrumb trail with the shortcode and optionally output BreadcrumbList schema. Turn off if your theme or another plugin already handles breadcrumbs.', 'miro-ai-seo-free'); ?></p>
        <?php if ($saved) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="miro-seo-hub-breadcrumbs-form">
            <input type="hidden" name="action" value="<?php echo esc_attr(self::SAVE_ACTION); ?>" />
            <?php wp_nonce_field(self::SAVE_ACTION, self::NONCE); ?>

            <table class="form-table">
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Status', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Active', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="checkbox" name="section_active" value="1" <?php checked(Miro_SEO_Hub::is_tab_enabled('breadcrumbs')); ?> /> <?php echo esc_html__('Enable this section (shortcode + schema)', 'miro-ai-seo-free'); ?></label>
                        <p class="description"><?php echo esc_html__('Leave unchecked if another plugin handles breadcrumbs.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="breadcrumbs_home_label"><?php echo esc_html__('Home label', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="breadcrumbs_home_label" name="<?php echo esc_attr(self::OPT); ?>[home_label]" value="<?php echo esc_attr($s['home_label']); ?>" class="regular-text" />
                        <p class="description"><?php echo esc_html__('Text for the first (home) link.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="breadcrumbs_separator"><?php echo esc_html__('Separator', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="breadcrumbs_separator" name="<?php echo esc_attr(self::OPT); ?>[separator]" value="<?php echo esc_attr($s['separator']); ?>" class="small-text" maxlength="10" />
                        <p class="description"><?php echo esc_html__('Character between items (e.g. » or >).', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="breadcrumbs_css_class"><?php echo esc_html__('CSS class', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="breadcrumbs_css_class" name="<?php echo esc_attr(self::OPT); ?>[css_class]" value="<?php echo esc_attr($s['css_class']); ?>" class="regular-text" />
                        <p class="description"><?php echo esc_html__('Class for the breadcrumb container (default: miro-breadcrumbs).', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Schema', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label>
                            <input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[output_schema]" value="0" />
                            <input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[output_schema]" value="1" <?php checked(!empty($s['output_schema'])); ?> />
                            <?php echo esc_html__('Output BreadcrumbList JSON-LD in the page head', 'miro-ai-seo-free'); ?>
                        </label>
                    </td>
                </tr>
            </table>
            <p class="submit"><input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save', 'miro-ai-seo-free'); ?>" /></p>
        </form>

        <div class="af-card" style="margin-top:20px;">
            <h3 style="margin:0 0 8px;"><?php echo esc_html__('Shortcode', 'miro-ai-seo-free'); ?></h3>
            <p class="description"><?php echo esc_html__('Insert the breadcrumb trail in your theme or content:', 'miro-ai-seo-free'); ?></p>
            <code style="display:inline-block; margin:8px 0; padding:8px 12px; background:#f1f5f9; border-radius:4px;">[miro_breadcrumbs]</code>
            <p class="description"><?php echo esc_html__('Optional attributes: class (override container class), sep (override separator). Example: [miro_breadcrumbs sep="&gt;"]', 'miro-ai-seo-free'); ?></p>
        </div>
        <?php
    }
}
