<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — Broken Link Checker: scan posts/pages for broken internal and external links.
 */
class SEO_Hub_Broken_Links {

    const OPT = 'miro_seo_hub_broken_links';
    const SCAN_ACTION = 'miro_seo_hub_scan_broken_links';
    const SAVE_ACTION = 'miro_seo_hub_save_broken_links';
    const AJAX_ACTION = 'miro_seo_hub_scan_broken_links';
    const NONCE = 'miro_seo_hub_broken_nonce';
    const SAVE_NONCE = 'miro_seo_hub_broken_save_nonce';
    const DEFAULT_MAX_URLS = 40;
    const REQUEST_TIMEOUT = 8;

    public static function init(): void {
        add_action('admin_post_' . self::SCAN_ACTION, [__CLASS__, 'handle_scan']);
        add_action('admin_post_' . self::SAVE_ACTION, [__CLASS__, 'handle_save_settings']);
        add_action('wp_ajax_' . self::AJAX_ACTION, [__CLASS__, 'ajax_scan']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'localize_script'], 20);
    }

    public static function localize_script($hook): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Page/tab detection for script localization, not form processing.
        if (!isset($_GET['page']) || $_GET['page'] !== Miro_SEO_Hub::MENU_SLUG || (isset($_GET['tab']) && $_GET['tab'] !== 'broken_links')) {
            return;
        }
        wp_localize_script('miro-seo-hub', 'miroBrokenLinks', [
            'ajaxUrl'    => admin_url('admin-ajax.php'),
            'action'     => self::AJAX_ACTION,
            'nonceName'  => self::NONCE,
            'nonce'      => wp_create_nonce(self::SCAN_ACTION),
            'i18n'       => [
                'scanning' => __('Scanning…', 'miro-ai-seo-free'),
                'pleaseWait' => __('Checking links. This may take 1–2 minutes. Please don’t close this page.', 'miro-ai-seo-free'),
                'error'   => __('Scan failed. Please try again.', 'miro-ai-seo-free'),
            ],
        ]);
    }

    public static function get_stored(): array {
        $data = get_option(self::OPT, []);
        return [
            'last_scan'   => isset($data['last_scan']) ? (int) $data['last_scan'] : 0,
            'post_types'  => isset($data['post_types']) && is_array($data['post_types']) ? $data['post_types'] : ['post', 'page'],
            'max_urls'    => isset($data['max_urls']) ? max(10, min(100, (int) $data['max_urls'])) : self::DEFAULT_MAX_URLS,
            'results'     => isset($data['results']) && is_array($data['results']) ? $data['results'] : [],
        ];
    }

    public static function save_results(array $results, array $post_types, int $max_urls): void {
        update_option(self::OPT, [
            'last_scan'  => time(),
            'post_types' => $post_types,
            'max_urls'    => $max_urls,
            'results'    => $results,
        ]);
    }

    /** Extract hrefs from HTML content. */
    public static function extract_links(string $html): array {
        $urls = [];
        if (preg_match_all('/<a\s[^>]*href\s*=\s*["\']([^"\']+)["\']/i', $html, $m)) {
            foreach ($m[1] as $url) {
                $url = trim($url);
                if ($url !== '' && strpos($url, '#') !== 0 && strpos($url, 'mailto:') !== 0 && strpos($url, 'tel:') !== 0) {
                    $urls[] = $url;
                }
            }
        }
        return array_unique($urls);
    }

    /** Make URL absolute. */
    public static function absolute_url(string $url, string $base): string {
        $url = trim($url);
        if (preg_match('#^https?://#i', $url)) return $url;
        if (strpos($url, '//') === 0) return 'https:' . $url;
        $base = rtrim($base, '/');
        if (strpos($url, '/') === 0) return $base . $url;
        return $base . '/' . ltrim($url, '/');
    }

    public static function is_internal(string $url): bool {
        $home = home_url();
        $host_home = wp_parse_url($home, PHP_URL_HOST);
        $host_url  = wp_parse_url($url, PHP_URL_HOST);
        if (!$host_home || !$host_url) return false;
        return strtolower($host_home) === strtolower($host_url);
    }

    /** Check internal URL (same site). */
    public static function check_internal(string $url): array {
        $path = wp_parse_url($url, PHP_URL_PATH);
        if (!$path) return ['ok' => false, 'error' => 'Invalid URL'];
        $path = trim($path, '/');
        if ($path === '') return ['ok' => true];

        $home_path = wp_parse_url(home_url(), PHP_URL_PATH);
        if ($home_path) $path_relative = preg_replace('#^' . preg_quote(trim($home_path, '/'), '#') . '/?#', '', $path);
        else $path_relative = $path;

        $post = get_page_by_path($path_relative, OBJECT, ['post', 'page']);
        if ($post && $post->post_status === 'publish') return ['ok' => true];
        if (get_page_by_path($path_relative, OBJECT, get_post_types(['public' => true]))) return ['ok' => true];
        return ['ok' => false, 'error' => '404 or not found'];
    }

    /** Check external URL via HTTP. */
    public static function check_external(string $url): array {
        $resp = wp_remote_head($url, [
            'timeout'    => self::REQUEST_TIMEOUT,
            'redirection'=> 5,
            'user-agent' => 'Miro-Broken-Link-Checker/1.0',
        ]);
        if (is_wp_error($resp)) return ['ok' => false, 'error' => $resp->get_error_message()];
        $code = wp_remote_retrieve_response_code($resp);
        if ($code >= 200 && $code < 400) return ['ok' => true];
        return ['ok' => false, 'error' => 'HTTP ' . $code];
    }

    /** Run the scan; returns results array. Used by both form POST and AJAX. */
    public static function run_scan(array $post_types, int $max_urls): array {
        $home = home_url();
        $to_check = [];
        $posts = get_posts([
            'post_type'      => $post_types,
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ]);
        foreach ($posts as $post_id) {
            $post = get_post($post_id);
            if (!$post) continue;
            $content = $post->post_content;
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core WordPress filter.
            $content = apply_filters('the_content', $content);
            $links = self::extract_links($content);
            foreach ($links as $href) {
                $abs = self::absolute_url($href, $home);
                if (!isset($to_check[$abs])) $to_check[$abs] = [];
                $to_check[$abs][] = ['post_id' => $post_id, 'post_title' => $post->post_title];
            }
        }

        $urls_order = array_keys($to_check);
        $urls_order = array_slice($urls_order, 0, $max_urls);
        $results = [];
        foreach ($urls_order as $url) {
            $sources = $to_check[$url];
            $first = $sources[0];
            $internal = self::is_internal($url);
            $check = $internal ? self::check_internal($url) : self::check_external($url);
            if (!$check['ok']) {
                $results[] = [
                    'post_id'     => $first['post_id'],
                    'post_title'  => $first['post_title'],
                    'edit_url'    => get_edit_post_link($first['post_id'], 'raw'),
                    'url'         => $url,
                    'error'       => $check['error'] ?? 'Unknown',
                    'is_internal' => $internal,
                ];
            }
        }
        return $results;
    }

    public static function handle_scan(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SCAN_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Long-running scan requires extended timeout.
        set_time_limit(300);
        $stored = self::get_stored();
        $post_types = isset($_POST['post_types']) && is_array($_POST['post_types']) ? array_map('sanitize_key', wp_unslash($_POST['post_types'])) : $stored['post_types'];
        if (empty($post_types)) $post_types = ['post', 'page'];
        $max_urls = isset($_POST['max_urls']) ? max(10, min(100, absint(wp_unslash($_POST['max_urls'])))) : $stored['max_urls']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via absint(wp_unslash())
        $results = self::run_scan($post_types, $max_urls);
        self::save_results($results, $post_types, $max_urls);
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=broken_links&scan_done=1'));
        exit;
    }

    public static function ajax_scan(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_send_json_error(['message' => esc_html__('Unauthorized', 'miro-ai-seo-free')], 403);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SCAN_ACTION)) {
            wp_send_json_error(['message' => esc_html__('Invalid nonce', 'miro-ai-seo-free')], 403);
        }
        // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Long-running scan requires extended timeout.
        set_time_limit(300);
        $stored = self::get_stored();
        $post_types = isset($_POST['post_types']) && is_array($_POST['post_types']) ? array_map('sanitize_key', wp_unslash($_POST['post_types'])) : $stored['post_types'];
        if (empty($post_types)) $post_types = ['post', 'page'];
        $max_urls = isset($_POST['max_urls']) ? max(10, min(100, absint(wp_unslash($_POST['max_urls'])))) : $stored['max_urls']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via absint(wp_unslash())
        $results = self::run_scan($post_types, $max_urls);
        self::save_results($results, $post_types, $max_urls);
        wp_send_json_success([
            'redirect' => admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=broken_links&scan_done=1'),
        ]);
    }

    public static function handle_save_settings(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::SAVE_NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::SAVE_NONCE])), self::SAVE_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        Miro_SEO_Hub::set_tab_enabled('broken_links', !empty(sanitize_text_field(wp_unslash($_POST['section_active'] ?? ''))));
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=broken_links&saved=1'));
        exit;
    }

    public static function render_form(): void {
        $stored = self::get_stored();
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Redirect params for display, not form processing.
        $scan_done = isset($_GET['scan_done']) && sanitize_text_field(wp_unslash($_GET['scan_done'])) === '1';
        $saved = isset($_GET['saved']) && sanitize_text_field(wp_unslash($_GET['saved'])) === '1';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended

        $post_types = get_post_types(['public' => true], 'objects');
        $allowed = ['post', 'page'];
        foreach (array_keys($post_types) as $pt) {
            if (in_array($pt, ['post', 'page'], true)) continue;
            $allowed[] = $pt;
        }
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('Broken Link Checker', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Scan published posts and pages for broken links (internal and external). Fix or remove broken URLs in the editor. Turn off if you use another plugin for this.', 'miro-ai-seo-free'); ?></p>
        <?php if ($scan_done) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Scan complete.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>
        <?php if ($saved) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>

        <div class="af-card" style="margin-bottom:16px;">
            <h3 style="margin:0 0 12px;"><?php echo esc_html__('Status', 'miro-ai-seo-free'); ?></h3>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <input type="hidden" name="action" value="<?php echo esc_attr(self::SAVE_ACTION); ?>" />
                <?php wp_nonce_field(self::SAVE_ACTION, self::SAVE_NONCE); ?>
                <p>
                    <label><input type="checkbox" name="section_active" value="1" <?php checked(Miro_SEO_Hub::is_tab_enabled('broken_links')); ?> /> <?php echo esc_html__('Active — Enable this section', 'miro-ai-seo-free'); ?></label>
                    <span class="description"><?php echo esc_html__('Leave unchecked if another plugin handles broken link checking.', 'miro-ai-seo-free'); ?></span>
                </p>
                <p><button type="submit" class="button"><?php echo esc_html__('Save', 'miro-ai-seo-free'); ?></button></p>
            </form>
        </div>

        <div class="af-card" style="margin-bottom:16px;">
            <h3 style="margin:0 0 12px;"><?php echo esc_html__('Run scan', 'miro-ai-seo-free'); ?></h3>
            <div id="miro-broken-links-scanning" class="miro-broken-links-scanning" aria-hidden="true">
                <div class="miro-broken-links-scanning-inner">
                    <div class="miro-broken-links-progress-bar" role="progressbar" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100" aria-label="<?php echo esc_attr__('Scanning', 'miro-ai-seo-free'); ?>"></div>
                    <p class="miro-broken-links-scanning-title"><?php echo esc_html__('Scanning…', 'miro-ai-seo-free'); ?></p>
                    <p class="miro-broken-links-scanning-desc"><?php echo esc_html__('Checking links. This may take 1–2 minutes. Please don’t close this page.', 'miro-ai-seo-free'); ?></p>
                </div>
            </div>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="miro-broken-links-scan-form">
                <input type="hidden" name="action" value="<?php echo esc_attr(self::SCAN_ACTION); ?>" />
                <?php wp_nonce_field(self::SCAN_ACTION, self::NONCE); ?>
                <p>
                    <label><?php echo esc_html__('Post types:', 'miro-ai-seo-free'); ?></label>
                    <?php foreach ($allowed as $pt) :
                        $obj = get_post_type_object($pt);
                        $label = $obj ? $obj->labels->name : $pt;
                        $checked = in_array($pt, $stored['post_types'], true);
                        ?>
                        <label style="margin-right:12px;"><input type="checkbox" name="post_types[]" value="<?php echo esc_attr($pt); ?>" <?php checked($checked); ?> /> <?php echo esc_html($label); ?></label>
                    <?php endforeach; ?>
                </p>
                <p>
                    <label for="broken_max_urls"><?php echo esc_html__('Max URLs to check per scan:', 'miro-ai-seo-free'); ?></label>
                    <input type="number" id="broken_max_urls" name="max_urls" value="<?php echo esc_attr($stored['max_urls']); ?>" min="10" max="100" step="5" />
                    <span class="description"><?php echo esc_html__('Higher = slower scan.', 'miro-ai-seo-free'); ?></span>
                </p>
                <p>
                    <button type="submit" class="button button-primary" id="miro-broken-links-scan-btn"><?php echo esc_html__('Scan now', 'miro-ai-seo-free'); ?></button>
                </p>
            </form>
            <?php if ($stored['last_scan'] > 0) : ?>
                <p class="description"><?php echo esc_html__('Last scan:', 'miro-ai-seo-free'); ?> <?php echo esc_html(wp_date(get_option('date_format') . ' ' . get_option('time_format'), $stored['last_scan'])); ?></p>
            <?php endif; ?>
        </div>

        <div class="af-card">
            <h3 style="margin:0 0 12px;"><?php echo esc_html__('Broken links', 'miro-ai-seo-free'); ?></h3>
            <?php if (empty($stored['results'])) : ?>
                <p class="description"><?php echo esc_html__('No broken links found. Run a scan to check.', 'miro-ai-seo-free'); ?></p>
            <?php else : ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th scope="col"><?php echo esc_html__('Post', 'miro-ai-seo-free'); ?></th>
                            <th scope="col"><?php echo esc_html__('Broken URL', 'miro-ai-seo-free'); ?></th>
                            <th scope="col"><?php echo esc_html__('Error', 'miro-ai-seo-free'); ?></th>
                            <th scope="col"><?php echo esc_html__('Type', 'miro-ai-seo-free'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stored['results'] as $row) : ?>
                            <tr>
                                <td>
                                    <?php if (!empty($row['edit_url'])) : ?>
                                        <a href="<?php echo esc_url($row['edit_url']); ?>" target="_blank" rel="noopener"><?php echo esc_html($row['post_title']); ?></a>
                                    <?php else : ?>
                                        <?php echo esc_html($row['post_title']); ?>
                                    <?php endif; ?>
                                </td>
                                <td><code style="word-break:break-all;"><?php echo esc_html($row['url']); ?></code></td>
                                <td><?php echo esc_html($row['error']); ?></td>
                                <td><?php echo !empty($row['is_internal']) ? esc_html__('Internal', 'miro-ai-seo-free') : esc_html__('External', 'miro-ai-seo-free'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <p class="description" style="margin-top:12px;"><?php echo esc_html__('Edit each post to remove or fix the broken link.', 'miro-ai-seo-free'); ?></p>
            <?php endif; ?>
        </div>
        <?php
    }
}
