<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — Local SEO tab: NAP, opening hours, LocalBusiness schema, service area, shortcode.
 */
class SEO_Hub_Local_SEO {

    const OPT = 'miro_seo_hub_local_seo';
    const SAVE_ACTION = 'miro_seo_hub_save_local_seo';
    const NONCE = 'miro_seo_hub_local_nonce';

    public static function init(): void {
        add_action('admin_init', [__CLASS__, 'register_setting']);
        add_action('admin_post_' . self::SAVE_ACTION, [__CLASS__, 'handle_save']);
        add_action('wp_head', [__CLASS__, 'output_schema'], 5);
        add_shortcode('miro_nap', [__CLASS__, 'shortcode_nap']);
        add_shortcode('miro_local', [__CLASS__, 'shortcode_nap']);
    }

    public static function register_setting(): void {
        register_setting('miro_seo_hub_local_group', self::OPT, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
        ]);
    }

    public static function default_settings(): array {
        return [
            'business_name'    => '',
            'business_type'    => 'LocalBusiness',
            'description'      => '',
            'street_address'   => '',
            'address_locality' => '',
            'address_region'   => '',
            'postal_code'      => '',
            'address_country'  => '',
            'telephone'        => '',
            'email'            => '',
            'url'              => '',
            'geo_lat'          => '',
            'geo_lng'          => '',
            'opening_hours'    => '',
            'image'            => '',
            'price_range'      => '',
            'service_area'     => '',
            'same_as'          => '',
            'output_schema'    => 1,
        ];
    }

    public static function get_settings(): array {
        $s = get_option(self::OPT, []);
        return array_merge(self::default_settings(), is_array($s) ? $s : []);
    }

    public static function sanitize($input): array {
        if (!is_array($input)) return self::default_settings();
        $out = self::default_settings();
        $out['business_name']    = sanitize_text_field($input['business_name'] ?? '');
        $out['business_type']    = sanitize_text_field($input['business_type'] ?? 'LocalBusiness');
        $out['description']      = sanitize_textarea_field($input['description'] ?? '');
        $out['street_address']   = sanitize_text_field($input['street_address'] ?? '');
        $out['address_locality'] = sanitize_text_field($input['address_locality'] ?? '');
        $out['address_region']   = sanitize_text_field($input['address_region'] ?? '');
        $out['postal_code']      = sanitize_text_field($input['postal_code'] ?? '');
        $out['address_country']  = sanitize_text_field($input['address_country'] ?? '');
        $out['telephone']        = sanitize_text_field($input['telephone'] ?? '');
        $out['email']            = sanitize_email($input['email'] ?? '');
        $out['url']              = esc_url_raw($input['url'] ?? '');
        $out['geo_lat']          = sanitize_text_field($input['geo_lat'] ?? '');
        $out['geo_lng']          = sanitize_text_field($input['geo_lng'] ?? '');
        $out['opening_hours']    = sanitize_textarea_field($input['opening_hours'] ?? '');
        $out['image']            = esc_url_raw($input['image'] ?? '');
        $out['price_range']      = sanitize_text_field($input['price_range'] ?? '');
        $out['service_area']     = sanitize_textarea_field($input['service_area'] ?? '');
        $out['same_as']          = sanitize_textarea_field($input['same_as'] ?? '');
        $out['output_schema']    = !empty($input['output_schema']) ? 1 : 0;
        return $out;
    }

    public static function handle_save(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SAVE_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Unslashed via wp_unslash, sanitized in self::sanitize().
        $opts = isset($_POST[self::OPT]) && is_array($_POST[self::OPT]) ? wp_unslash($_POST[self::OPT]) : [];
        update_option(self::OPT, self::sanitize($opts));
        Miro_SEO_Hub::set_tab_enabled('local_seo', !empty(sanitize_text_field(wp_unslash($_POST['section_active'] ?? ''))));
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=local_seo&saved=1'));
        exit;
    }

    /** Business types for schema (subset of Schema.org) */
    public static function business_types(): array {
        return [
            'LocalBusiness'   => __('Local Business', 'miro-ai-seo-free'),
            'Store'           => __('Store', 'miro-ai-seo-free'),
            'Restaurant'      => __('Restaurant', 'miro-ai-seo-free'),
            'HealthClub'      => __('Health Club / Gym', 'miro-ai-seo-free'),
            'LegalService'    => __('Legal Service', 'miro-ai-seo-free'),
            'Dentist'         => __('Dentist', 'miro-ai-seo-free'),
            'Physician'       => __('Physician', 'miro-ai-seo-free'),
            'Plumber'         => __('Plumber', 'miro-ai-seo-free'),
            'Electrician'     => __('Electrician', 'miro-ai-seo-free'),
            'RealEstateAgent' => __('Real Estate Agent', 'miro-ai-seo-free'),
            'AccountingService' => __('Accounting Service', 'miro-ai-seo-free'),
            'AutoRepair'      => __('Auto Repair', 'miro-ai-seo-free'),
            'TravelAgency'    => __('Travel Agency', 'miro-ai-seo-free'),
            'Organization'    => __('Organization (generic)', 'miro-ai-seo-free'),
        ];
    }

    public static function render_form(): void {
        $s = self::get_settings();
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Redirect param for display, not form processing.
        $saved = isset($_GET['saved']) && sanitize_text_field(wp_unslash($_GET['saved'])) === '1';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        if (empty($s['url'])) $s['url'] = get_site_url();
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('Local SEO', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Add your business details for LocalBusiness schema and consistent NAP (Name, Address, Phone) across your site. Use the shortcode [miro_nap] to display your address anywhere. Turn off if you use another local SEO plugin.', 'miro-ai-seo-free'); ?></p>
        <?php if ($saved) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="miro-seo-hub-local-form">
            <input type="hidden" name="action" value="<?php echo esc_attr(self::SAVE_ACTION); ?>" />
            <?php wp_nonce_field(self::SAVE_ACTION, self::NONCE); ?>

            <table class="form-table">
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Status', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Active', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="checkbox" name="section_active" value="1" <?php checked(Miro_SEO_Hub::is_tab_enabled('local_seo')); ?> /> <?php echo esc_html__('Enable this section (output schema, shortcodes)', 'miro-ai-seo-free'); ?></label>
                        <p class="description"><?php echo esc_html__('Leave unchecked if another plugin handles local SEO.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Business info', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="local_business_name"><?php echo esc_html__('Business name', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_business_name" name="<?php echo esc_attr(self::OPT); ?>[business_name]" value="<?php echo esc_attr($s['business_name']); ?>" class="regular-text" placeholder="<?php echo esc_attr(get_bloginfo('name')); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_business_type"><?php echo esc_html__('Business type', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <select id="local_business_type" name="<?php echo esc_attr(self::OPT); ?>[business_type]">
                            <?php foreach (self::business_types() as $value => $label) : ?>
                                <option value="<?php echo esc_attr($value); ?>" <?php selected($s['business_type'], $value); ?>><?php echo esc_html($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('Used in schema for rich results.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_description"><?php echo esc_html__('Short description', 'miro-ai-seo-free'); ?></label></th>
                    <td><textarea id="local_description" name="<?php echo esc_attr(self::OPT); ?>[description]" rows="2" class="large-text"><?php echo esc_textarea($s['description']); ?></textarea></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Address (NAP)', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="local_street"><?php echo esc_html__('Street address', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_street" name="<?php echo esc_attr(self::OPT); ?>[street_address]" value="<?php echo esc_attr($s['street_address']); ?>" class="large-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_locality"><?php echo esc_html__('City', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_locality" name="<?php echo esc_attr(self::OPT); ?>[address_locality]" value="<?php echo esc_attr($s['address_locality']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_region"><?php echo esc_html__('State / Region', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_region" name="<?php echo esc_attr(self::OPT); ?>[address_region]" value="<?php echo esc_attr($s['address_region']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_postal"><?php echo esc_html__('Postal code', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_postal" name="<?php echo esc_attr(self::OPT); ?>[postal_code]" value="<?php echo esc_attr($s['postal_code']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_country"><?php echo esc_html__('Country (e.g. US)', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_country" name="<?php echo esc_attr(self::OPT); ?>[address_country]" value="<?php echo esc_attr($s['address_country']); ?>" class="small-text" maxlength="2" placeholder="US" /></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Contact', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="local_telephone"><?php echo esc_html__('Phone', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_telephone" name="<?php echo esc_attr(self::OPT); ?>[telephone]" value="<?php echo esc_attr($s['telephone']); ?>" class="regular-text" placeholder="+1 234 567 8900" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_email"><?php echo esc_html__('Email', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="email" id="local_email" name="<?php echo esc_attr(self::OPT); ?>[email]" value="<?php echo esc_attr($s['email']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_url"><?php echo esc_html__('Website URL', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="url" id="local_url" name="<?php echo esc_attr(self::OPT); ?>[url]" value="<?php echo esc_attr($s['url']); ?>" class="large-text" /></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Location & hours', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="local_geo_lat"><?php echo esc_html__('Latitude / Longitude', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="local_geo_lat" name="<?php echo esc_attr(self::OPT); ?>[geo_lat]" value="<?php echo esc_attr($s['geo_lat']); ?>" placeholder="40.7128" class="small-text" />
                        <input type="text" id="local_geo_lng" name="<?php echo esc_attr(self::OPT); ?>[geo_lng]" value="<?php echo esc_attr($s['geo_lng']); ?>" placeholder="-74.0060" class="small-text" />
                        <p class="description"><?php echo esc_html__('Optional. For map pins and schema. Find at Google Maps.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_opening_hours"><?php echo esc_html__('Opening hours', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <textarea id="local_opening_hours" name="<?php echo esc_attr(self::OPT); ?>[opening_hours]" rows="4" class="large-text" placeholder="Mo-Fr 09:00-17:00&#10;Sa 10:00-14:00"><?php echo esc_textarea($s['opening_hours']); ?></textarea>
                        <p class="description"><?php echo esc_html__('One line per rule, e.g. "Mo-Fr 09:00-17:00" or "Monday: 9am-5pm".', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Media & extra', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="local_image"><?php echo esc_html__('Image URL', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <div style="display:flex;flex-wrap:wrap;align-items:center;gap:8px;">
                            <input type="url" id="local_image" name="<?php echo esc_attr(self::OPT); ?>[image]" value="<?php echo esc_attr($s['image']); ?>" class="large-text" placeholder="https://..." style="min-width:280px;" />
                            <button type="button" class="button miro-seo-hub-select-image" id="miro-seo-hub-select-image"><?php echo esc_html__('Select from media', 'miro-ai-seo-free'); ?></button>
                        </div>
                        <p class="description"><?php echo esc_html__('Logo or photo for schema.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_price_range"><?php echo esc_html__('Price range', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="local_price_range" name="<?php echo esc_attr(self::OPT); ?>[price_range]" value="<?php echo esc_attr($s['price_range']); ?>" class="small-text" placeholder="$$" maxlength="10" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_service_area"><?php echo esc_html__('Service area', 'miro-ai-seo-free'); ?></label></th>
                    <td><textarea id="local_service_area" name="<?php echo esc_attr(self::OPT); ?>[service_area]" rows="3" class="large-text" placeholder="New York, NY&#10;Brooklyn&#10;Jersey City"><?php echo esc_textarea($s['service_area']); ?></textarea><p class="description"><?php echo esc_html__('Cities or regions you serve, one per line.', 'miro-ai-seo-free'); ?></p></td>
                </tr>
                <tr>
                    <th scope="row"><label for="local_same_as"><?php echo esc_html__('Social / same as', 'miro-ai-seo-free'); ?></label></th>
                    <td><textarea id="local_same_as" name="<?php echo esc_attr(self::OPT); ?>[same_as]" rows="2" class="large-text" placeholder="https://facebook.com/...&#10;https://twitter.com/..."><?php echo esc_textarea($s['same_as']); ?></textarea><p class="description"><?php echo esc_html__('One URL per line (Facebook, Twitter, LinkedIn, etc.).', 'miro-ai-seo-free'); ?></p></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Output', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('LocalBusiness schema', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[output_schema]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[output_schema]" value="1" <?php checked(!empty($s['output_schema'])); ?> /> <?php echo esc_html__('Output LocalBusiness JSON-LD in the page head', 'miro-ai-seo-free'); ?></label>
                    </td>
                </tr>
            </table>
            <p class="submit"><input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save Local SEO', 'miro-ai-seo-free'); ?>" /></p>
        </form>

        <div class="af-card" style="margin-top:20px;">
            <h3 style="margin:0 0 8px;"><?php echo esc_html__('Shortcode', 'miro-ai-seo-free'); ?></h3>
            <p class="description"><?php echo esc_html__('Use <code>[miro_nap]</code> or <code>[miro_local]</code> in any post or page to display your business name, address, and phone. Style it with your theme or custom CSS.', 'miro-ai-seo-free'); ?></p>
        </div>
        <?php
    }

    public static function output_schema(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('local_seo')) return;
        $s = self::get_settings();
        if (empty($s['output_schema']) || (empty($s['business_name']) && empty($s['street_address']))) return;

        $name = $s['business_name'] ?: get_bloginfo('name');
        $data = [
            '@context' => 'https://schema.org',
            '@type'    => $s['business_type'] ?: 'LocalBusiness',
            'name'     => $name,
        ];
        if (!empty($s['description'])) $data['description'] = $s['description'];
        if (!empty($s['url'])) $data['url'] = $s['url'];
        if (!empty($s['telephone'])) $data['telephone'] = $s['telephone'];
        if (!empty($s['email'])) $data['email'] = $s['email'];
        if (!empty($s['image'])) $data['image'] = $s['image'];
        if (!empty($s['price_range'])) $data['priceRange'] = $s['price_range'];

        $addr = [];
        if (!empty($s['street_address'])) $addr['streetAddress'] = $s['street_address'];
        if (!empty($s['address_locality'])) $addr['addressLocality'] = $s['address_locality'];
        if (!empty($s['address_region'])) $addr['addressRegion'] = $s['address_region'];
        if (!empty($s['postal_code'])) $addr['postalCode'] = $s['postal_code'];
        if (!empty($s['address_country'])) $addr['addressCountry'] = $s['address_country'];
        if (!empty($addr)) $data['address'] = array_merge(['@type' => 'PostalAddress'], $addr);

        if (!empty($s['geo_lat']) && !empty($s['geo_lng'])) {
            $data['geo'] = ['@type' => 'GeoCoordinates', 'latitude' => (float)$s['geo_lat'], 'longitude' => (float)$s['geo_lng']];
        }
        if (!empty($s['opening_hours'])) {
            $lines = array_filter(array_map('trim', explode("\n", $s['opening_hours'])));
            if (!empty($lines)) $data['openingHours'] = $lines;
        }
        if (!empty($s['service_area'])) {
            $areas = array_filter(array_map('trim', explode("\n", $s['service_area'])));
            if (!empty($areas)) $data['areaServed'] = array_map(function ($a) {
                return ['@type' => 'City', 'name' => $a];
            }, $areas);
        }
        if (!empty($s['same_as'])) {
            $urls = array_filter(array_map('trim', explode("\n", $s['same_as'])));
            $data['sameAs'] = array_values(array_filter($urls, 'esc_url_raw'));
        }

        echo '<script type="application/ld+json">' . "\n" . wp_json_encode($data) . "\n" . '</script>' . "\n";
    }

    public static function shortcode_nap($atts): string {
        if (!Miro_SEO_Hub::is_tab_enabled('local_seo')) return '';
        $s = self::get_settings();
        $name = $s['business_name'] ?: get_bloginfo('name');
        if (empty($name) && empty($s['street_address']) && empty($s['telephone'])) return '';

        $a = shortcode_atts(['class' => 'miro-nap'], $atts ?? [], 'miro_nap');
        $class = esc_attr($a['class']);
        $out = '<div class="' . $class . '" style="display:block; margin:0.5em 0;">';
        if ($name) $out .= '<span class="miro-nap-name" style="display:block; font-weight:600;">' . esc_html($name) . '</span>';
        $addr_parts = array_filter([$s['street_address'], trim(($s['address_locality'] ?? '') . ', ' . ($s['address_region'] ?? '') . ' ' . ($s['postal_code'] ?? '')), $s['address_country'] ?? '']);
        if (!empty($addr_parts)) $out .= '<span class="miro-nap-address" style="display:block;">' . esc_html(implode(', ', $addr_parts)) . '</span>';
        if (!empty($s['telephone'])) $out .= '<span class="miro-nap-phone" style="display:block;"><a href="tel:' . esc_attr(preg_replace('/\s+/', '', $s['telephone'])) . '">' . esc_html($s['telephone']) . '</a></span>';
        if (!empty($s['email'])) $out .= '<span class="miro-nap-email" style="display:block;"><a href="mailto:' . esc_attr($s['email']) . '">' . esc_html($s['email']) . '</a></span>';
        $out .= '</div>';
        return $out;
    }
}
