<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — OG & Twitter Cards: Open Graph and Twitter Card meta tags for social sharing.
 */
class SEO_Hub_OG_Twitter {

    const OPT = 'miro_seo_hub_og_twitter';
    const SAVE_ACTION = 'miro_seo_hub_save_og_twitter';
    const NONCE = 'miro_seo_hub_og_twitter_nonce';

    /** Recommended min size for OG image (Facebook). */
    const OG_IMAGE_WIDTH = 1200;
    const OG_IMAGE_HEIGHT = 630;

    public static function init(): void {
        add_action('admin_init', [__CLASS__, 'register_setting']);
        add_action('admin_post_' . self::SAVE_ACTION, [__CLASS__, 'handle_save']);
        add_action('wp_head', [__CLASS__, 'output_meta_tags'], 5);
    }

    public static function register_setting(): void {
        register_setting('miro_seo_hub_og_twitter_group', self::OPT, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
        ]);
    }

    public static function default_settings(): array {
        return [
            'default_image'   => '',
            'site_name'       => '',
            'twitter_handle'   => '',
            'twitter_card'    => 'summary_large_image',
            'fb_app_id'       => '',
        ];
    }

    public static function get_settings(): array {
        $s = get_option(self::OPT, []);
        return array_merge(self::default_settings(), is_array($s) ? $s : []);
    }

    public static function sanitize($input): array {
        if (!is_array($input)) return self::default_settings();
        $out = self::default_settings();
        $out['default_image'] = esc_url_raw($input['default_image'] ?? '');
        $out['site_name']     = sanitize_text_field($input['site_name'] ?? '');
        $out['twitter_handle'] = sanitize_text_field($input['twitter_handle'] ?? '');
        if (strpos($out['twitter_handle'], '@') === 0) {
            $out['twitter_handle'] = substr($out['twitter_handle'], 1);
        }
        $out['twitter_card'] = in_array($input['twitter_card'] ?? '', ['summary', 'summary_large_image'], true)
            ? $input['twitter_card'] : 'summary_large_image';
        $out['fb_app_id']    = sanitize_text_field($input['fb_app_id'] ?? '');
        return $out;
    }

    public static function handle_save(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SAVE_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Unslashed via wp_unslash, sanitized in self::sanitize().
        $opts = isset($_POST[self::OPT]) && is_array($_POST[self::OPT]) ? wp_unslash($_POST[self::OPT]) : [];
        update_option(self::OPT, self::sanitize($opts));
        Miro_SEO_Hub::set_tab_enabled('og_twitter', !empty(sanitize_text_field(wp_unslash($_POST['section_active'] ?? ''))));
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=og_twitter&saved=1'));
        exit;
    }

    /** Get title for current page. */
    public static function get_title(): string {
        if (is_singular()) {
            $post = get_post();
            return $post ? get_the_title($post) : '';
        }
        $s = self::get_settings();
        $site_name = $s['site_name'] !== '' ? $s['site_name'] : get_bloginfo('name');
        if (is_front_page() || is_home()) {
            return $site_name . (get_bloginfo('description') ? ' - ' . get_bloginfo('description') : '');
        }
        if (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            return $term && isset($term->name) ? $term->name . ' - ' . $site_name : $site_name;
        }
        if (is_author()) {
            $author = get_queried_object();
            return $author && isset($author->display_name) ? $author->display_name . ' - ' . $site_name : $site_name;
        }
        if (is_post_type_archive()) {
            $obj = get_queried_object();
            $name = $obj && isset($obj->labels->name) ? $obj->labels->name : __('Archive', 'miro-ai-seo-free');
            return $name . ' - ' . $site_name;
        }
        if (is_search()) {
            /* translators: %s: Search query string. */
            return sprintf(__('Search for "%s"', 'miro-ai-seo-free'), get_search_query()) . ' - ' . $site_name;
        }
        if (is_404()) {
            return __('Page not found', 'miro-ai-seo-free') . ' - ' . $site_name;
        }
        return $site_name;
    }

    /** Get description for current page. */
    public static function get_description(): string {
        if (is_singular()) {
            $post = get_post();
            if (!$post) return '';
            $desc = get_post_meta($post->ID, '_miro_og_description', true);
            if (is_string($desc) && $desc !== '') return $desc;
            $desc = has_excerpt($post) ? get_the_excerpt($post) : wp_trim_words(wp_strip_all_tags($post->post_content), 30);
            return $desc ?: get_bloginfo('description');
        }
        return get_bloginfo('description') ?: '';
    }

    /** Get absolute image URL for current page. Prefer featured image, then default. */
    public static function get_image(): string {
        if (is_singular()) {
            $post = get_post();
            if ($post) {
                $thumb_id = get_post_thumbnail_id($post);
                if ($thumb_id) {
                    $img = wp_get_attachment_image_src($thumb_id, [self::OG_IMAGE_WIDTH, self::OG_IMAGE_HEIGHT]);
                    if (!empty($img[0])) return $img[0];
                    $img = wp_get_attachment_image_src($thumb_id, 'full');
                    if (!empty($img[0])) return $img[0];
                }
            }
        }
        $s = self::get_settings();
        if ($s['default_image'] !== '') return $s['default_image'];
        return '';
    }

    /** Get canonical URL for current request. */
    public static function get_url(): string {
        if (is_singular()) {
            $post = get_post();
            return $post ? get_permalink($post) : '';
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- REQUEST_URI is server-provided; esc_url output used in meta tags.
        return esc_url(home_url(wp_unslash($_SERVER['REQUEST_URI'] ?? '')));
    }

    /** Output Open Graph and Twitter Card meta tags. */
    public static function output_meta_tags(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('og_twitter')) return;

        $s = self::get_settings();
        $title = self::get_title();
        $description = self::get_description();
        $image = self::get_image();
        $url = self::get_url();
        $site_name = $s['site_name'] !== '' ? $s['site_name'] : get_bloginfo('name');
        $type = is_singular() ? 'article' : 'website';

        if ($title === '') $title = $site_name;

        $tags = [
            ['property' => 'og:type', 'content' => $type],
            ['property' => 'og:title', 'content' => $title],
            ['property' => 'og:url', 'content' => $url],
            ['property' => 'og:site_name', 'content' => $site_name],
        ];
        if ($description !== '') {
            $tags[] = ['property' => 'og:description', 'content' => $description];
        }
        if ($image !== '') {
            $tags[] = ['property' => 'og:image', 'content' => $image];
            $tags[] = ['property' => 'og:image:width', 'content' => (string) self::OG_IMAGE_WIDTH];
            $tags[] = ['property' => 'og:image:height', 'content' => (string) self::OG_IMAGE_HEIGHT];
        }
        if ($s['fb_app_id'] !== '') {
            $tags[] = ['property' => 'fb:app_id', 'content' => $s['fb_app_id']];
        }
        if (is_singular()) {
            $post = get_post();
            if ($post) {
                $tags[] = ['property' => 'article:published_time', 'content' => get_the_date('c', $post)];
                $tags[] = ['property' => 'article:modified_time', 'content' => get_the_modified_date('c', $post)];
            }
        }

        foreach ($tags as $t) {
            echo '<meta property="' . esc_attr($t['property']) . '" content="' . esc_attr($t['content']) . '" />' . "\n";
        }

        $twitter = [
            ['name' => 'twitter:card', 'content' => $s['twitter_card']],
            ['name' => 'twitter:title', 'content' => $title],
            ['name' => 'twitter:description', 'content' => $description],
        ];
        if ($image !== '') {
            $twitter[] = ['name' => 'twitter:image', 'content' => $image];
        }
        if ($s['twitter_handle'] !== '') {
            $twitter[] = ['name' => 'twitter:site', 'content' => '@' . $s['twitter_handle']];
        }
        foreach ($twitter as $t) {
            if ($t['content'] !== '') {
                echo '<meta name="' . esc_attr($t['name']) . '" content="' . esc_attr($t['content']) . '" />' . "\n";
            }
        }
    }

    public static function render_form(): void {
        $s = self::get_settings();
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Redirect param for display, not form processing.
        $saved = isset($_GET['saved']) && sanitize_text_field(wp_unslash($_GET['saved'])) === '1';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('OG & Twitter Cards', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Open Graph and Twitter Card meta tags for social sharing. Turn off if another plugin (e.g. Yoast, Rank Math) already handles this.', 'miro-ai-seo-free'); ?></p>
        <?php if ($saved) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="miro-seo-hub-og-twitter-form">
            <input type="hidden" name="action" value="<?php echo esc_attr(self::SAVE_ACTION); ?>" />
            <?php wp_nonce_field(self::SAVE_ACTION, self::NONCE); ?>

            <table class="form-table">
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Status', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Active', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="checkbox" name="section_active" value="1" <?php checked(Miro_SEO_Hub::is_tab_enabled('og_twitter')); ?> /> <?php echo esc_html__('Enable this section (output OG & Twitter meta)', 'miro-ai-seo-free'); ?></label>
                        <p class="description"><?php echo esc_html__('Leave unchecked if another plugin handles this.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Defaults', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="og_site_name"><?php echo esc_html__('Site name', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="og_site_name" name="<?php echo esc_attr(self::OPT); ?>[site_name]" value="<?php echo esc_attr($s['site_name']); ?>" class="regular-text" placeholder="<?php echo esc_attr(get_bloginfo('name')); ?>" />
                        <p class="description"><?php echo esc_html__('Used for og:site_name. Leave blank to use site title.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="og_default_image"><?php echo esc_html__('Default image', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <div style="display:flex;flex-wrap:wrap;align-items:center;gap:8px;">
                            <input type="url" id="og_default_image" name="<?php echo esc_attr(self::OPT); ?>[default_image]" value="<?php echo esc_attr($s['default_image']); ?>" class="large-text" style="min-width:280px;" />
                            <button type="button" class="button miro-seo-hub-select-schema-image" data-target="og_default_image"><?php echo esc_html__('Select from media', 'miro-ai-seo-free'); ?></button>
                        </div>
                        <p class="description"><?php echo esc_html__('Fallback when a post has no featured image. Recommended: 1200×630 px.', 'miro-ai-seo-free'); ?></p>
                        <?php if ($s['default_image']) : ?>
                            <p style="margin-top:6px;"><img src="<?php echo esc_url($s['default_image']); ?>" alt="" style="max-width:200px; height:auto; border:1px solid #ddd;" /></p>
                        <?php endif; ?>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Twitter', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="og_twitter_handle"><?php echo esc_html__('Twitter handle', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="og_twitter_handle" name="<?php echo esc_attr(self::OPT); ?>[twitter_handle]" value="<?php echo esc_attr($s['twitter_handle']); ?>" class="regular-text" placeholder="username" />
                        <span class="description"><?php echo esc_html__('Without @. Used for twitter:site.', 'miro-ai-seo-free'); ?></span>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="og_twitter_card"><?php echo esc_html__('Twitter card type', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <select id="og_twitter_card" name="<?php echo esc_attr(self::OPT); ?>[twitter_card]">
                            <option value="summary_large_image" <?php selected($s['twitter_card'], 'summary_large_image'); ?>><?php echo esc_html__('Summary large image', 'miro-ai-seo-free'); ?></option>
                            <option value="summary" <?php selected($s['twitter_card'], 'summary'); ?>><?php echo esc_html__('Summary', 'miro-ai-seo-free'); ?></option>
                        </select>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Facebook (optional)', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><label for="og_fb_app_id"><?php echo esc_html__('Facebook App ID', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="og_fb_app_id" name="<?php echo esc_attr(self::OPT); ?>[fb_app_id]" value="<?php echo esc_attr($s['fb_app_id']); ?>" class="regular-text" />
                        <p class="description"><?php echo esc_html__('Optional. For Facebook Insights.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
            </table>
            <p class="submit"><input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save', 'miro-ai-seo-free'); ?>" /></p>
        </form>

        <div class="af-card" style="margin-top:20px;">
            <h3 style="margin:0 0 8px;"><?php echo esc_html__('Per-post description', 'miro-ai-seo-free'); ?></h3>
            <p class="description"><?php echo esc_html__('By default we use the post excerpt or trimmed content for og:description. To override for a single post, add a custom field:', 'miro-ai-seo-free'); ?></p>
            <code style="display:inline-block; margin:8px 0; padding:8px 12px; background:#f1f5f9; border-radius:4px;">_miro_og_description</code>
            <p class="description"><?php echo esc_html__('Set its value to the description text you want for social sharing.', 'miro-ai-seo-free'); ?></p>
        </div>
        <?php
    }
}
