<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — Schema / Structured Data tab: Organization, WebSite, Article, FAQ, HowTo, BreadcrumbList.
 * Each type can be enabled/disabled so it plays nice with Rank Math, Yoast, etc.
 */
class SEO_Hub_Schema {

    const OPT = 'miro_seo_hub_schema';
    const SAVE_ACTION = 'miro_seo_hub_save_schema';
    const NONCE = 'miro_seo_hub_schema_nonce';

    public static function init(): void {
        add_action('admin_init', [__CLASS__, 'register_setting']);
        add_action('admin_post_' . self::SAVE_ACTION, [__CLASS__, 'handle_save']);
        add_action('wp_head', [__CLASS__, 'output_global_schema'], 6);
        add_action('wp_head', [__CLASS__, 'output_article_schema'], 7);
        add_action('wp_head', [__CLASS__, 'output_faq_schema'], 8);
        add_action('wp_head', [__CLASS__, 'output_breadcrumb_schema'], 9);
        add_shortcode('miro_howto', [__CLASS__, 'shortcode_howto']);
        add_shortcode('miro_faq_schema', [__CLASS__, 'shortcode_faq_schema']);
    }

    public static function register_setting(): void {
        register_setting('miro_seo_hub_schema_group', self::OPT, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
        ]);
    }

    public static function default_settings(): array {
        return [
            'org_enabled'       => 0,
            'org_name'          => '',
            'org_url'           => '',
            'org_logo'          => '',
            'website_enabled'   => 0,
            'website_name'      => '',
            'website_url'       => '',
            'search_action_url' => '',
            'article_enabled'   => 0,
            'article_author_show' => 1,
            'faq_enabled'       => 0,
            'howto_enabled'     => 0,
            'breadcrumb_enabled'=> 0,
        ];
    }

    public static function get_settings(): array {
        $s = get_option(self::OPT, []);
        return array_merge(self::default_settings(), is_array($s) ? $s : []);
    }

    public static function sanitize($input): array {
        if (!is_array($input)) return self::default_settings();
        $out = self::default_settings();
        $out['org_enabled']       = !empty($input['org_enabled']) ? 1 : 0;
        $out['org_name']          = sanitize_text_field($input['org_name'] ?? '');
        $out['org_url']           = esc_url_raw($input['org_url'] ?? '');
        $out['org_logo']          = esc_url_raw($input['org_logo'] ?? '');
        $out['website_enabled']   = !empty($input['website_enabled']) ? 1 : 0;
        $out['website_name']      = sanitize_text_field($input['website_name'] ?? '');
        $out['website_url']       = esc_url_raw($input['website_url'] ?? '');
        $out['search_action_url'] = esc_url_raw($input['search_action_url'] ?? '');
        $out['article_enabled']   = !empty($input['article_enabled']) ? 1 : 0;
        $out['article_author_show'] = !empty($input['article_author_show']) ? 1 : 0;
        $out['faq_enabled']       = !empty($input['faq_enabled']) ? 1 : 0;
        $out['howto_enabled']     = !empty($input['howto_enabled']) ? 1 : 0;
        $out['breadcrumb_enabled']= !empty($input['breadcrumb_enabled']) ? 1 : 0;
        return $out;
    }

    public static function handle_save(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SAVE_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passed to self::sanitize() which sanitizes all fields.
        $opts = isset($_POST[self::OPT]) && is_array($_POST[self::OPT]) ? wp_unslash($_POST[self::OPT]) : [];
        update_option(self::OPT, self::sanitize($opts));
        Miro_SEO_Hub::set_tab_enabled('schema', !empty(sanitize_text_field(wp_unslash($_POST['section_active'] ?? ''))));
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=schema&saved=1'));
        exit;
    }

    public static function render_form(): void {
        $s = self::get_settings();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Redirect param for success message display, not form processing.
        $saved = isset($_GET['saved']) && sanitize_text_field(wp_unslash($_GET['saved'])) === '1';
        if (empty($s['org_url'])) $s['org_url'] = get_site_url();
        if (empty($s['website_url'])) $s['website_url'] = get_site_url();
        if (empty($s['website_name'])) $s['website_name'] = get_bloginfo('name');
        if (empty($s['org_name'])) $s['org_name'] = get_bloginfo('name');
        $search_placeholder = home_url('/?s={search_term_string}');
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('Schema / Structured Data', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Output JSON-LD for rich results. Enable only the types you need. Turn off if Rank Math or Yoast already outputs schema.', 'miro-ai-seo-free'); ?></p>
        <?php if ($saved) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="miro-seo-hub-schema-form">
            <input type="hidden" name="action" value="<?php echo esc_attr(self::SAVE_ACTION); ?>" />
            <?php wp_nonce_field(self::SAVE_ACTION, self::NONCE); ?>

            <table class="form-table">
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Status', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Active', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="checkbox" name="section_active" value="1" <?php checked(Miro_SEO_Hub::is_tab_enabled('schema')); ?> /> <?php echo esc_html__('Enable this section (output JSON-LD)', 'miro-ai-seo-free'); ?></label>
                        <p class="description"><?php echo esc_html__('Leave unchecked if another plugin handles schema.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Organization', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Enable', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[org_enabled]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[org_enabled]" value="1" <?php checked(!empty($s['org_enabled'])); ?> /> <?php echo esc_html__('Output Organization schema', 'miro-ai-seo-free'); ?></label></td>
                </tr>
                <tr>
                    <th scope="row"><label for="schema_org_name"><?php echo esc_html__('Name', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="schema_org_name" name="<?php echo esc_attr(self::OPT); ?>[org_name]" value="<?php echo esc_attr($s['org_name']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="schema_org_url"><?php echo esc_html__('URL', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="url" id="schema_org_url" name="<?php echo esc_attr(self::OPT); ?>[org_url]" value="<?php echo esc_attr($s['org_url']); ?>" class="large-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="schema_org_logo"><?php echo esc_html__('Logo URL', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <div style="display:flex;flex-wrap:wrap;align-items:center;gap:8px;">
                            <input type="url" id="schema_org_logo" name="<?php echo esc_attr(self::OPT); ?>[org_logo]" value="<?php echo esc_attr($s['org_logo']); ?>" class="large-text" style="min-width:280px;" />
                            <button type="button" class="button miro-seo-hub-select-schema-image" data-target="schema_org_logo"><?php echo esc_html__('Select from media', 'miro-ai-seo-free'); ?></button>
                        </div>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('WebSite', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Enable', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[website_enabled]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[website_enabled]" value="1" <?php checked(!empty($s['website_enabled'])); ?> /> <?php echo esc_html__('Output WebSite schema', 'miro-ai-seo-free'); ?></label></td>
                </tr>
                <tr>
                    <th scope="row"><label for="schema_website_name"><?php echo esc_html__('Name', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="text" id="schema_website_name" name="<?php echo esc_attr(self::OPT); ?>[website_name]" value="<?php echo esc_attr($s['website_name']); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="schema_website_url"><?php echo esc_html__('URL', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="url" id="schema_website_url" name="<?php echo esc_attr(self::OPT); ?>[website_url]" value="<?php echo esc_attr($s['website_url']); ?>" class="large-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="schema_search_action"><?php echo esc_html__('Search action URL', 'miro-ai-seo-free'); ?></label></th>
                    <td><input type="url" id="schema_search_action" name="<?php echo esc_attr(self::OPT); ?>[search_action_url]" value="<?php echo esc_attr($s['search_action_url']); ?>" class="large-text" placeholder="<?php echo esc_attr($search_placeholder); ?>" /><p class="description"><?php echo esc_html__('Optional. Use {search_term_string} for the query. Enables sitelinks search box.', 'miro-ai-seo-free'); ?></p></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Article (posts)', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Enable', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[article_enabled]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[article_enabled]" value="1" <?php checked(!empty($s['article_enabled'])); ?> /> <?php echo esc_html__('Output Article schema on single posts', 'miro-ai-seo-free'); ?></label></td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Author in schema', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[article_author_show]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[article_author_show]" value="1" <?php checked(!empty($s['article_author_show'])); ?> /> <?php echo esc_html__('Include author (Person) in Article', 'miro-ai-seo-free'); ?></label></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('FAQ & HowTo', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('FAQPage', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[faq_enabled]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[faq_enabled]" value="1" <?php checked(!empty($s['faq_enabled'])); ?> /> <?php echo esc_html__('Output FAQ schema (from AI FAQ meta or shortcode [miro_faq_schema])', 'miro-ai-seo-free'); ?></label></td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('HowTo', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[howto_enabled]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[howto_enabled]" value="1" <?php checked(!empty($s['howto_enabled'])); ?> /> <?php echo esc_html__('Allow HowTo schema via shortcode [miro_howto]', 'miro-ai-seo-free'); ?></label></td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('BreadcrumbList', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Enable', 'miro-ai-seo-free'); ?></th>
                    <td><label><input type="hidden" name="<?php echo esc_attr(self::OPT); ?>[breadcrumb_enabled]" value="0" /><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[breadcrumb_enabled]" value="1" <?php checked(!empty($s['breadcrumb_enabled'])); ?> /> <?php echo esc_html__('Output BreadcrumbList schema on singular posts/pages', 'miro-ai-seo-free'); ?></label></td>
                </tr>
            </table>
            <p class="submit"><input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save Schema', 'miro-ai-seo-free'); ?>" /></p>
        </form>

        <div class="af-card" style="margin-top:20px;">
            <h3 style="margin:0 0 8px;"><?php echo esc_html__('Shortcodes', 'miro-ai-seo-free'); ?></h3>
            <p><code>[miro_faq_schema questions="Question 1?|Question 2?" answers="Answer 1|Answer 2"]</code> — <?php echo esc_html__('Outputs FAQPage schema (pipe-separated). Enable FAQ above.', 'miro-ai-seo-free'); ?></p>
            <p><code>[miro_howto title="How to do X" steps="Step 1|Step 2|Step 3"]</code> — <?php echo esc_html__('Outputs HowTo schema and an ordered list. Enable HowTo above.', 'miro-ai-seo-free'); ?></p>
        </div>
        <?php
    }

    public static function output_global_schema(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('schema')) return;
        $s = self::get_settings();

        if (!empty($s['org_enabled']) && (!empty($s['org_name']) || !empty($s['org_logo']))) {
            $org = [
                '@context' => 'https://schema.org',
                '@type'    => 'Organization',
                'name'     => $s['org_name'] ?: get_bloginfo('name'),
            ];
            if (!empty($s['org_url'])) $org['url'] = $s['org_url'];
            if (!empty($s['org_logo'])) $org['logo'] = $s['org_logo'];
            echo '<script type="application/ld+json">' . "\n" . wp_json_encode($org) . "\n" . '</script>' . "\n";
        }

        if (!empty($s['website_enabled'])) {
            $site = [
                '@context' => 'https://schema.org',
                '@type'    => 'WebSite',
                'name'     => $s['website_name'] ?: get_bloginfo('name'),
                'url'      => $s['website_url'] ?: get_site_url(),
            ];
            if (!empty($s['search_action_url'])) {
                $template = $s['search_action_url'];
                if (strpos($template, '{search_term_string}') === false) {
                    $template = add_query_arg('s', '{search_term_string}', $template);
                }
                $site['potentialAction'] = [
                    '@type'       => 'SearchAction',
                    'target'      => ['@type' => 'EntryPoint', 'urlTemplate' => $template],
                    'query-input' => 'required name=search_term_string',
                ];
            }
            echo '<script type="application/ld+json">' . "\n" . wp_json_encode($site) . "\n" . '</script>' . "\n";
        }
    }

    public static function output_article_schema(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('schema')) return;
        $s = self::get_settings();
        if (empty($s['article_enabled']) || !is_singular('post')) return;

        $post = get_post();
        if (!$post || $post->post_status !== 'publish') return;

        $article = [
            '@context'    => 'https://schema.org',
            '@type'       => 'Article',
            'headline'    => get_the_title($post),
            'datePublished' => get_the_date('c', $post),
            'dateModified'  => get_the_modified_date('c', $post),
            'mainEntityOfPage' => ['@type' => 'WebPage', '@id' => get_permalink($post)],
        ];
        if (!empty($s['article_author_show'])) {
            $author_id = (int) $post->post_author;
            if ($author_id) {
                $article['author'] = ['@type' => 'Person', 'name' => get_the_author_meta('display_name', $author_id)];
            }
        }
        $s2 = self::get_settings();
        if (!empty($s2['org_enabled']) && (!empty($s2['org_name']) || !empty($s2['org_logo']))) {
            $article['publisher'] = [
                '@type' => 'Organization',
                'name'  => $s2['org_name'] ?: get_bloginfo('name'),
            ];
            if (!empty($s2['org_logo'])) $article['publisher']['logo'] = ['@type' => 'ImageObject', 'url' => $s2['org_logo']];
        } else {
            $article['publisher'] = ['@type' => 'Organization', 'name' => get_bloginfo('name')];
        }
        echo '<script type="application/ld+json">' . "\n" . wp_json_encode($article) . "\n" . '</script>' . "\n";
    }

    public static function output_faq_schema(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('schema')) return;
        $s = self::get_settings();
        if (empty($s['faq_enabled']) || !is_singular()) return;

        $post = get_post();
        if (!$post) return;

        $jsonld = get_post_meta($post->ID, '_miro_schema_jsonld', true);
        if (is_string($jsonld) && $jsonld !== '') {
            $decoded = json_decode($jsonld, true);
            if (is_array($decoded) && !empty($decoded['@type']) && $decoded['@type'] === 'FAQPage') {
                echo '<script type="application/ld+json">' . "\n" . wp_json_encode($decoded) . "\n" . '</script>' . "\n";
            }
        }
    }

    public static function output_breadcrumb_schema(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('schema')) return;
        $s = self::get_settings();
        if (empty($s['breadcrumb_enabled']) || !is_singular()) return;

        $post = get_post();
        if (!$post) return;

        $items = [];
        $pos = 1;
        $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => get_bloginfo('name'), 'item' => get_site_url()];

        if (is_singular('post')) {
            $cats = get_the_category($post->ID);
            if (!empty($cats)) {
                $cat = reset($cats);
                $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => $cat->name, 'item' => get_category_link($cat->term_id)];
            }
        }
        $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => get_the_title($post), 'item' => get_permalink($post)];

        $data = ['@context' => 'https://schema.org', '@type' => 'BreadcrumbList', 'itemListElement' => $items];
        echo '<script type="application/ld+json">' . "\n" . wp_json_encode($data) . "\n" . '</script>' . "\n";
    }

    public static function shortcode_faq_schema($atts): string {
        if (!Miro_SEO_Hub::is_tab_enabled('schema')) return '';
        $s = self::get_settings();
        if (empty($s['faq_enabled'])) return '';

        $a = shortcode_atts([
            'questions' => '',
            'answers'   => '',
        ], $atts ?? [], 'miro_faq_schema');

        $questions = array_filter(array_map('trim', explode('|', $a['questions'])));
        $answers   = array_filter(array_map('trim', explode('|', $a['answers'])));
        if (empty($questions) || count($questions) !== count($answers)) return '';

        $mainEntity = [];
        foreach ($questions as $i => $q) {
            $mainEntity[] = [
                '@type'          => 'Question',
                'name'           => $q,
                'acceptedAnswer' => ['@type' => 'Answer', 'text' => $answers[$i] ?? ''],
            ];
        }
        $data = [
            '@context'   => 'https://schema.org',
            '@type'      => 'FAQPage',
            'mainEntity' => $mainEntity,
        ];
        add_action('wp_footer', function () use ($data) {
            echo '<script type="application/ld+json">' . "\n" . wp_json_encode($data) . "\n" . '</script>' . "\n";
        }, 5);
        return '';
    }

    public static function shortcode_howto($atts): string {
        if (!Miro_SEO_Hub::is_tab_enabled('schema')) return '';
        $s = self::get_settings();
        if (empty($s['howto_enabled'])) return '';

        $a = shortcode_atts([
            'title'       => '',
            'description' => '',
            'steps'       => '',
        ], $atts ?? [], 'miro_howto');

        $steps_str = $a['steps'];
        $steps_list = array_filter(array_map('trim', explode('|', $steps_str)));
        if (empty($a['title']) || empty($steps_list)) return '';

        $schema_steps = [];
        foreach ($steps_list as $idx => $text) {
            $schema_steps[] = ['@type' => 'HowToStep', 'position' => $idx + 1, 'text' => $text];
        }
        $data = [
            '@context'    => 'https://schema.org',
            '@type'       => 'HowTo',
            'name'        => $a['title'],
            'step'        => $schema_steps,
        ];
        if (!empty($a['description'])) $data['description'] = $a['description'];

        add_action('wp_footer', function () use ($data) {
            echo '<script type="application/ld+json">' . "\n" . wp_json_encode($data) . "\n" . '</script>' . "\n";
        }, 5);

        $html = '<div class="miro-howto">';
        $html .= '<h3>' . esc_html($a['title']) . '</h3>';
        if (!empty($a['description'])) $html .= '<p>' . esc_html($a['description']) . '</p>';
        $html .= '<ol class="miro-howto-steps">';
        foreach ($steps_list as $step) {
            $html .= '<li>' . esc_html($step) . '</li>';
        }
        $html .= '</ol></div>';
        return $html;
    }
}
