<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * SEO Hub — Sitemap: customize WordPress core XML sitemap (WP 5.5+).
 * Exclude post types, taxonomies, users, and specific post IDs.
 */
class SEO_Hub_Sitemap {

    const OPT = 'miro_seo_hub_sitemap';
    const SAVE_ACTION = 'miro_seo_hub_save_sitemap';
    const NONCE = 'miro_seo_hub_sitemap_nonce';

    public static function init(): void {
        add_action('admin_init', [__CLASS__, 'register_setting']);
        add_action('admin_post_' . self::SAVE_ACTION, [__CLASS__, 'handle_save']);
        add_action('init', [__CLASS__, 'register_sitemap_filters'], 20);
    }

    public static function register_setting(): void {
        register_setting('miro_seo_hub_sitemap_group', self::OPT, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
        ]);
    }

    public static function default_settings(): array {
        return [
            'post_types'       => ['post', 'page'],
            'taxonomies'       => ['category', 'post_tag'],
            'exclude_users'    => 1,
            'exclude_post_ids' => '',
        ];
    }

    public static function get_settings(): array {
        $s = get_option(self::OPT, []);
        return array_merge(self::default_settings(), is_array($s) ? $s : []);
    }

    public static function sanitize($input): array {
        if (!is_array($input)) return self::default_settings();
        $out = self::default_settings();
        $out['post_types'] = isset($input['post_types']) && is_array($input['post_types'])
            ? array_map('sanitize_key', $input['post_types'])
            : ['post', 'page'];
        if (empty($out['post_types'])) $out['post_types'] = ['post', 'page'];
        $out['taxonomies'] = isset($input['taxonomies']) && is_array($input['taxonomies'])
            ? array_map('sanitize_key', $input['taxonomies'])
            : ['category', 'post_tag'];
        $out['exclude_users'] = !empty($input['exclude_users']) ? 1 : 0;
        $ids = isset($input['exclude_post_ids']) ? $input['exclude_post_ids'] : '';
        $out['exclude_post_ids'] = is_string($ids) ? $ids : '';
        return $out;
    }

    public static function handle_save(): void {
        if (!current_user_can(function_exists('miro_ai_cap') ? \miro_ai_cap() : 'manage_options')) {
            wp_die(esc_html__('Unauthorized', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        if (!isset($_POST[self::NONCE]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE])), self::SAVE_ACTION)) {
            wp_die(esc_html__('Invalid nonce', 'miro-ai-seo-free'), '', ['response' => 403]);
        }
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Unslashed via wp_unslash, sanitized in self::sanitize().
        $opts = isset($_POST[self::OPT]) && is_array($_POST[self::OPT]) ? wp_unslash($_POST[self::OPT]) : [];
        update_option(self::OPT, self::sanitize($opts));
        Miro_SEO_Hub::set_tab_enabled('sitemap', !empty(sanitize_text_field(wp_unslash($_POST['section_active'] ?? ''))));
        wp_safe_redirect(admin_url('admin.php?page=' . Miro_SEO_Hub::MENU_SLUG . '&tab=sitemap&saved=1'));
        exit;
    }

    public static function register_sitemap_filters(): void {
        if (!Miro_SEO_Hub::is_tab_enabled('sitemap')) return;
        if (version_compare(get_bloginfo('version'), '5.5', '<')) return;
        add_filter('wp_sitemaps_post_types', [__CLASS__, 'filter_post_types'], 10, 1);
        add_filter('wp_sitemaps_taxonomies', [__CLASS__, 'filter_taxonomies'], 10, 1);
        add_filter('wp_sitemaps_add_provider', [__CLASS__, 'filter_provider'], 10, 2);
        add_filter('wp_sitemaps_posts_query_args', [__CLASS__, 'filter_posts_query_args'], 10, 2);
    }

    public static function filter_post_types($post_types): array {
        $s = self::get_settings();
        $allowed = array_flip($s['post_types']);
        return array_filter($post_types, function ($key) use ($allowed) {
            return isset($allowed[$key]);
        }, ARRAY_FILTER_USE_KEY);
    }

    public static function filter_taxonomies($taxonomies): array {
        $s = self::get_settings();
        $allowed = array_flip($s['taxonomies']);
        return array_filter($taxonomies, function ($key) use ($allowed) {
            return isset($allowed[$key]);
        }, ARRAY_FILTER_USE_KEY);
    }

    public static function filter_provider($provider, $name) {
        $s = self::get_settings();
        if (!empty($s['exclude_users']) && $name === 'users') {
            return false;
        }
        return $provider;
    }

    public static function filter_posts_query_args(array $args, string $post_type): array {
        $s = self::get_settings();
        if (trim($s['exclude_post_ids']) === '') return $args;
        $ids = array_filter(array_map('intval', array_map('trim', explode(',', $s['exclude_post_ids']))));
        if (empty($ids)) return $args;
        // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- User-configured exclusions for sitemap.
        $args['post__not_in'] = isset($args['post__not_in']) && is_array($args['post__not_in'])
            ? array_merge($args['post__not_in'], $ids)
            : $ids;
        return $args;
    }

    public static function render_form(): void {
        $s = self::get_settings();
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Redirect param for display, not form processing.
        $saved = isset($_GET['saved']) && sanitize_text_field(wp_unslash($_GET['saved'])) === '1';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        $has_wp_sitemap = version_compare(get_bloginfo('version'), '5.5', '>=');
        $post_types = get_post_types(['public' => true], 'objects');
        $taxonomies = get_taxonomies(['public' => true], 'objects');
        ?>
        <h2 class="miro-seo-hub-tab-title"><?php echo esc_html__('Sitemap', 'miro-ai-seo-free'); ?></h2>
        <p class="description"><?php echo esc_html__('Customize the WordPress core XML sitemap: choose which post types and taxonomies to include, exclude authors, and exclude specific post IDs. Turn off if you use Yoast, Rank Math or another plugin for sitemaps.', 'miro-ai-seo-free'); ?></p>
        <?php if ($saved) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'miro-ai-seo-free'); ?></p></div>
        <?php endif; ?>
        <?php if (!$has_wp_sitemap) : ?>
            <div class="notice notice-warning inline"><p><?php echo esc_html__('WordPress 5.5+ is required for the core sitemap. Your version may use a different sitemap system.', 'miro-ai-seo-free'); ?></p></div>
        <?php else : ?>
            <p class="description"><a href="<?php echo esc_url(home_url('/wp-sitemap.xml')); ?>" target="_blank" rel="noopener"><?php echo esc_html__('View your sitemap', 'miro-ai-seo-free'); ?></a> (wp-sitemap.xml)</p>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="miro-seo-hub-sitemap-form">
            <input type="hidden" name="action" value="<?php echo esc_attr(self::SAVE_ACTION); ?>" />
            <?php wp_nonce_field(self::SAVE_ACTION, self::NONCE); ?>

            <table class="form-table">
                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Status', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Active', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="checkbox" name="section_active" value="1" <?php checked(Miro_SEO_Hub::is_tab_enabled('sitemap')); ?> /> <?php echo esc_html__('Enable this section (customize WordPress sitemap)', 'miro-ai-seo-free'); ?></label>
                        <p class="description"><?php echo esc_html__('Leave unchecked if another plugin handles sitemaps.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Post types', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Include in sitemap', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <?php foreach ($post_types as $pt) :
                            if (in_array($pt->name, ['attachment'], true)) continue;
                            $checked = in_array($pt->name, $s['post_types'], true);
                            ?>
                            <label style="display:block; margin-bottom:4px;"><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[post_types][]" value="<?php echo esc_attr($pt->name); ?>" <?php checked($checked); ?> /> <?php echo esc_html($pt->labels->name); ?></label>
                        <?php endforeach; ?>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Taxonomies', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Include in sitemap', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <?php foreach ($taxonomies as $tax) :
                            $checked = in_array($tax->name, $s['taxonomies'], true);
                            ?>
                            <label style="display:block; margin-bottom:4px;"><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[taxonomies][]" value="<?php echo esc_attr($tax->name); ?>" <?php checked($checked); ?> /> <?php echo esc_html($tax->labels->name); ?></label>
                        <?php endforeach; ?>
                    </td>
                </tr>

                <tr><th colspan="2"><h3 style="margin:16px 0 8px;"><?php echo esc_html__('Exclusions', 'miro-ai-seo-free'); ?></h3></th></tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Exclude authors', 'miro-ai-seo-free'); ?></th>
                    <td>
                        <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[exclude_users]" value="1" <?php checked(!empty($s['exclude_users'])); ?> /> <?php echo esc_html__('Remove author pages from sitemap', 'miro-ai-seo-free'); ?></label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="sitemap_exclude_ids"><?php echo esc_html__('Exclude post IDs', 'miro-ai-seo-free'); ?></label></th>
                    <td>
                        <input type="text" id="sitemap_exclude_ids" name="<?php echo esc_attr(self::OPT); ?>[exclude_post_ids]" value="<?php echo esc_attr($s['exclude_post_ids']); ?>" class="large-text" placeholder="123, 456, 789" />
                        <p class="description"><?php echo esc_html__('Comma-separated post or page IDs to exclude from the sitemap.', 'miro-ai-seo-free'); ?></p>
                    </td>
                </tr>
            </table>
            <p class="submit"><input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save', 'miro-ai-seo-free'); ?>" /></p>
        </form>
        <?php
    }
}
