<?php
if (!defined('ABSPATH')) exit;

/**
 * Miro_REST_Alt_Fix — Scan first, then bulk generate ALTs.
 *
 * Routes:
 *  - GET/POST /miro/v1/altfix/scan
 *      Params: post_type (default post), include_featured (bool), page (int, 1-based), per_page (<=50)
 *      Returns: {
 *        items: [{
 *          post_id, title, focus, edit_link, view_link,
 *          images:[{id, has_alt, current_alt, is_featured}]
 *        }],
 *        done, next_page
 *      }
 *
 *  - POST /miro/v1/altfix/apply
 *      Params:
 *        mode: 'fill' | 'overwrite' (default 'fill')
 *        apply_to: 'content' | 'library' (default 'content')
 *        set_title: bool (default true)
 *        include_featured: bool (default false)
 *        items: [{ post_id, image_ids: [attachmentId,...] }]  // from scan
 */
class Miro_REST_Alt_Fix {
    const REST_NS = 'miro/v1';
    const R_SCAN  = '/altfix/scan';
    const R_APPLY = '/altfix/apply';

    public static function init() {
        add_action('rest_api_init', function() {
            register_rest_route(self::REST_NS, self::R_SCAN, [
                'methods'  => ['GET','POST'],
                'callback' => [__CLASS__, 'scan'],
                'permission_callback' => function() { return current_user_can('manage_options'); }
            ]);
            register_rest_route(self::REST_NS, self::R_APPLY, [
                'methods'  => 'POST',
                'callback' => [__CLASS__, 'apply'],
                'permission_callback' => function() { return current_user_can('manage_options'); }
            ]);
        });
    }

    /** -------- SCAN -------- */
    public static function scan(\WP_REST_Request $req) {
        $post_type        = sanitize_text_field($req->get_param('post_type') ?: 'post');
        $include_featured = (bool) $req->get_param('include_featured');
        $page             = max(1, (int)$req->get_param('page'));
        $per_page         = min(50, max(1, (int)($req->get_param('per_page') ?? 25)));

        $q = new \WP_Query([
            'post_type'      => $post_type,
            'post_status'    => 'publish',
            'posts_per_page' => $per_page,
            'paged'          => $page,
            'fields'         => 'ids',
            'no_found_rows'  => false,
        ]);

        $items = [];
        foreach ($q->posts as $post_id) {
            $focus   = self::get_focus_keyword($post_id);
            $title   = get_the_title($post_id);
            $content = get_post_field('post_content', $post_id);

            $attachment_ids = self::extract_attachment_ids_from_content($content, $post_id, $include_featured);
            $images = [];
            $feat_id = $include_featured ? (int)get_post_thumbnail_id($post_id) : 0;

            foreach ($attachment_ids as $aid) {
                if (get_post_type($aid) !== 'attachment') continue;
                $alt = get_post_meta($aid, '_wp_attachment_image_alt', true);
                $images[] = [
                    'id'          => (int)$aid,
                    'has_alt'     => strlen(trim((string)$alt)) > 0,
                    'current_alt' => (string)$alt,
                    'is_featured' => ($feat_id && $aid === $feat_id),
                ];
            }

            $items[] = [
                'post_id'   => $post_id,
                'title'     => $title,
                'focus'     => $focus,
                'edit_link' => get_edit_post_link($post_id, 'raw'),
                'view_link' => get_permalink($post_id),
                'images'    => $images,
            ];
        }

        $done = ($q->max_num_pages <= $page);
        return new \WP_REST_Response([
            'items'     => $items,
            'done'      => $done,
            'next_page' => $done ? null : $page + 1,
        ], 200);
    }

    /** -------- APPLY (BULK) -------- */
    public static function apply(\WP_REST_Request $req) {
        $mode             = in_array($req->get_param('mode'), ['fill','overwrite'], true) ? $req->get_param('mode') : 'fill';
        $apply_to         = in_array($req->get_param('apply_to'), ['content','library'], true) ? $req->get_param('apply_to') : 'content';
        $set_title        = (bool) ($req->get_param('set_title') ?? true);
        $include_featured = (bool) $req->get_param('include_featured');

        $items = $req->get_param('items');
        if (!is_array($items)) $items = [];

        $changed   = 0;
        $processed = 0;
        $logs      = [];

        foreach ($items as $it) {
            $post_id     = (int) ($it['post_id'] ?? 0);
            $image_ids   = array_values(array_unique(array_map('intval', (array)($it['image_ids'] ?? []))));
            if ($post_id <= 0 || empty($image_ids)) continue;

            $processed++;
            $focus   = self::get_focus_keyword($post_id);
            $title   = get_the_title($post_id);
            if (!$focus) $focus = self::fallback_focus($post_id, $title);

            $alts = self::build_batch_alts($post_id, $image_ids, $focus, $include_featured);

            if ($apply_to === 'library') {
                foreach ($image_ids as $aid) {
                    if (get_post_type($aid) !== 'attachment') continue;
                    $current_alt = get_post_meta($aid, '_wp_attachment_image_alt', true);
                    $new_alt     = $alts[$aid] ?? $focus;

                    if ($mode === 'fill' && strlen(trim((string)$current_alt)) > 0) {
                        // keep existing
                    } else {
                        update_post_meta($aid, '_wp_attachment_image_alt', $new_alt);
                        $changed++;
                    }

                    if ($set_title) {
                        wp_update_post(['ID' => $aid, 'post_title' => self::build_title_text($focus)]);
                    }
                }
                $logs[] = "Post #$post_id: updated library ALTs for ".count($image_ids)." image(s).";
            } else {
                $content  = get_post_field('post_content', $post_id);
                $updated  = self::update_alts_in_post_content($content, $image_ids, $alts, $mode);
                if ($updated !== $content) {
                    wp_update_post(['ID' => $post_id, 'post_content' => $updated]);
                    $changed += count($image_ids);
                }
                foreach ($image_ids as $aid) {
                    $current_alt = get_post_meta($aid, '_wp_attachment_image_alt', true);
                    $new_alt     = $alts[$aid] ?? $focus;
                    if ($mode === 'overwrite' || strlen(trim((string)$current_alt)) === 0) {
                        update_post_meta($aid, '_wp_attachment_image_alt', $new_alt);
                    }
                    if ($set_title) {
                        wp_update_post(['ID' => $aid, 'post_title' => self::build_title_text($focus)]);
                    }
                }
                $logs[] = "Post #$post_id: updated content & library ALTs for ".count($image_ids)." image(s).";
            }
        }

        return new \WP_REST_Response([
            'processed_posts' => $processed,
            'alt_changed'     => $changed,
            'messages'        => $logs,
        ], 200);
    }

    /* ---------------- Focus + ALT builders ---------------- */

    private static function get_focus_keyword($post_id) {
        $keys = ['miro_focus_keyword', '_rank_math_focus_keyword', '_yoast_wpseo_focuskw'];
        foreach ($keys as $k) {
            $v = get_post_meta($post_id, $k, true);
            if ($v) {
                if (is_array($v)) $v = reset($v);
                $v = trim((string)$v);
                if ($v !== '') return $v;
            }
        }
        return '';
    }

    private static function fallback_focus($post_id, $title) {
        $title = trim((string)$title);
        if ($title !== '') return $title;
        $taxes = ['post_tag', 'category'];
        foreach ($taxes as $tx) {
            $terms = get_the_terms($post_id, $tx);
            if (is_array($terms) && !empty($terms)) {
                return trim($terms[0]->name);
            }
        }
        return get_bloginfo('name');
    }

    private static function build_alt_text($focus, $is_featured = false) {
        $focus = trim((string)$focus);
        if ($focus === '') $focus = 'Image';
        $alt = $is_featured ? "{$focus} – featured image" : $focus;
        $alt = wp_strip_all_tags($alt);
        $alt = preg_replace('/\s+/', ' ', $alt);
        $alt = trim($alt);
        if (strlen($alt) > 125) $alt = mb_substr($alt, 0, 122) . '...';
        return $alt;
    }

    private static function build_title_text($focus) {
        $t = trim((string)$focus);
        if ($t === '') $t = 'Image';
        return $t;
    }

    private static function build_batch_alts($post_id, array $image_ids, $focus, $include_featured) {
        $alts = [];
        $feat_id = $include_featured ? (int)get_post_thumbnail_id($post_id) : 0;

        $base = [];
        foreach ($image_ids as $aid) {
            $is_feat = ($feat_id && $aid === $feat_id);
            $txt = self::build_alt_text($focus, $is_feat);
            $base[$aid] = $txt;
        }
        $counter = [];
        foreach ($image_ids as $aid) {
            $txt = $base[$aid];
            if (!isset($counter[$txt])) {
                $counter[$txt] = 1;
                $alts[$aid] = $txt;
            } else {
                $counter[$txt]++;
                $alts[$aid] = $txt . ' (' . $counter[$txt] . ')';
            }
        }
        return $alts;
    }

    /* ---------------- Extraction ---------------- */

    private static function extract_attachment_ids_from_content($content, $post_id = 0, $include_featured = false) {
        $ids = [];

        if ($include_featured && $post_id) {
            $thumb_id = get_post_thumbnail_id($post_id);
            if ($thumb_id) $ids[] = (int)$thumb_id;
        }

        if (function_exists('parse_blocks')) {
            $ids = array_merge($ids, self::collect_from_blocks(parse_blocks($content)));
        }

        if (preg_match_all('/wp-image-([0-9]+)/', $content, $m)) {
            foreach ($m[1] as $id) $ids[] = (int)$id;
        }
        if (preg_match_all('/(?:data-id|data-attachment-id)\s*=\s*"(\d+)"/i', $content, $m2)) {
            foreach ($m2[1] as $id) $ids[] = (int)$id;
        }
        if (preg_match_all('/<img[^>]+src="([^"]+)"/i', $content, $m3)) {
            foreach ($m3[1] as $url) {
                $aid = attachment_url_to_postid($url);
                if ($aid) $ids[] = (int)$aid;
            }
        }
        if (preg_match_all('/"id"\s*:\s*([0-9]+)/', $content, $m4)) {
            foreach ($m4[1] as $id) $ids[] = (int)$id;
        }

        $ids = array_values(array_unique(array_filter(array_map('intval', $ids))));
        return $ids;
    }

    private static function collect_from_blocks(array $blocks) {
        $found = [];
        foreach ($blocks as $block) {
            $name  = $block['blockName'] ?? '';
            $attrs = $block['attrs'] ?? [];

            if ($name === 'core/image' && !empty($attrs['id'])) $found[] = (int)$attrs['id'];
            if ($name === 'core/gallery') {
                if (!empty($attrs['ids']) && is_array($attrs['ids'])) {
                    foreach ($attrs['ids'] as $id) $found[] = (int)$id;
                }
                if (!empty($block['innerBlocks'])) {
                    foreach ($block['innerBlocks'] as $ib) {
                        if (!empty($ib['attrs']['id'])) $found[] = (int)$ib['attrs']['id'];
                    }
                }
            }
            if ($name === 'core/media-text' && !empty($attrs['mediaId'])) $found[] = (int)$attrs['mediaId'];
            if ($name === 'core/cover' && !empty($attrs['id'])) $found[] = (int)$attrs['id'];

            if (!empty($block['innerBlocks']) && is_array($block['innerBlocks'])) {
                $found = array_merge($found, self::collect_from_blocks($block['innerBlocks']));
            }
        }
        return $found;
    }

    /* ---------------- Rewriters ---------------- */

    private static function update_alts_in_post_content($content, array $attachment_ids, array $alts, $mode) {
        if (function_exists('parse_blocks') && function_exists('serialize_blocks')) {
            $touched = false;
            $blocks  = parse_blocks($content);
            $blocks  = self::rewrite_blocks_alts($blocks, $attachment_ids, $alts, $mode, $touched);
            if ($touched) {
                return serialize_blocks($blocks);
            }
        }

        $updated = $content;

        foreach ($attachment_ids as $aid) {
            $new_alt = esc_attr($alts[$aid] ?? 'Image');

            if ($mode === 'overwrite') {
                $updated = preg_replace(
                    '#(<img[^>]*class="[^"]*wp-image-' . $aid . '[^"]*"[^>]*)(alt="[^"]*")([^>]*>)#i',
                    '$1alt="' . $new_alt . '"$3',
                    $updated
                );
            }
            $updated = preg_replace(
                '#(<img[^>]*class="[^"]*wp-image-' . $aid . '[^"]*"(?:(?!alt=).)*)(/?>)#i',
                '$1 alt="' . $new_alt . '"$2',
                $updated
            );

            if (preg_match_all('/<img[^>]+src="([^"]+)"/i', $updated, $m)) {
                foreach ($m[1] as $url) {
                    $res_id = attachment_url_to_postid($url);
                    if ($res_id && $res_id === $aid) {
                        if ($mode === 'overwrite') {
                            $updated = preg_replace(
                                '#(<img[^>]*src="' . preg_quote($url, '#') . '"[^>]*)(alt="[^"]*")([^>]*>)#i',
                                '$1alt="' . $new_alt . '"$3',
                                $updated
                            );
                        }
                        $updated = preg_replace(
                            '#(<img[^>]*src="' . preg_quote($url, '#') . '"(?:(?!alt=).)*)(/?>)#i',
                            '$1 alt="' . $new_alt . '"$2',
                            $updated
                        );
                    }
                }
            }
        }

        return $updated;
    }

    private static function rewrite_blocks_alts(array $blocks, array $ids, array $alts, $mode, &$touched) {
        $idset = array_flip($ids);

        foreach ($blocks as &$block) {
            $name  = $block['blockName'] ?? '';
            $attrs = $block['attrs'] ?? [];

            if ($name === 'core/image' && !empty($attrs['id']) && isset($idset[(int)$attrs['id']])) {
                $aid = (int)$attrs['id'];
                $alt = (string)($attrs['alt'] ?? '');
                $new = (string)($alts[$aid] ?? 'Image');
                if ($mode === 'overwrite' || strlen(trim($alt)) === 0) {
                    $block['attrs']['alt'] = $new;
                    $touched = true;
                }
            }

            if (!empty($block['innerBlocks']) && is_array($block['innerBlocks'])) {
                $block['innerBlocks'] = self::rewrite_blocks_alts($block['innerBlocks'], $ids, $alts, $mode, $touched);
            }
        }
        return $blocks;
    }
}

Miro_REST_Alt_Fix::init();
