<?php
namespace Miro_AI_SEO;

if (!defined('ABSPATH')) exit;

/**
 * REST: /miro/v1/titles/ai
 * Generates AI SEO titles or meta descriptions with ideal SEO length (green range)
 */
class Miro_REST_Title_AI {
    const NS = 'miro/v1';

    public static function init() {
        add_action('rest_api_init', [__CLASS__, 'routes']);
    }

    public static function routes() {
        register_rest_route(self::NS, '/titles/ai', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'generate'],
            'permission_callback' => function() {
                return current_user_can('edit_posts');
            },
        ]);
    }

    public static function generate($req) {
        $type    = sanitize_text_field($req['type'] ?? 'title');
        $context = sanitize_text_field($req['context'] ?? '');
        $focus   = sanitize_text_field($req['focus'] ?? '');

        // Use homepage or site info as fallback
        if ($context === '') {
            $site  = get_bloginfo('name');
            $desc  = get_bloginfo('description');
            $context = "Website: {$site}. Description: {$desc}. Create the best SEO {$type} possible.";
        }

        // ===== Build highly optimized prompt =====
        if ($type === 'title') {
            $prompt = "You are an SEO expert. Write a perfect homepage SEO title between 50 and 60 characters.\n\n"
                . "Topic: {$context}\nFocus keyword: {$focus}\n\n"
                . "Rules:\n"
                . "- MUST be between 50–60 characters (never shorter or longer)\n"
                . "- Include focus keyword naturally\n"
                . "- Make it catchy, brand-friendly, and click-worthy\n"
                . "- Avoid symbols, quotes, brackets, or emojis\n"
                . "- Return ONLY the plain title text.";
            $ai_context = 'seo_title';
        } else {
            $prompt = "You are an SEO expert. Write a meta description between 140 and 160 characters.\n\n"
                . "Topic: {$context}\nFocus keyword: {$focus}\n\n"
                . "Rules:\n"
                . "- MUST be between 140–160 characters (never shorter or longer)\n"
                . "- Include focus keyword naturally\n"
                . "- Write in a helpful, human, and persuasive tone\n"
                . "- Return ONLY the plain description text.";
            $ai_context = 'seo_meta';
        }

        // ===== Use Miro AI bridge (OpenAI / Gemini) =====
        $result = apply_filters('miro/ai/complete', '', [
            'prompt'  => $prompt,
            'context' => $ai_context,
            'opts'    => [
                'temperature' => 0.4,
                'max_tokens'  => 180,
                'model'       => 'gpt-4o-mini',
            ],
        ]);

        // ===== Validate AI response =====
        if (empty($result)) {
            return new \WP_Error('ai_empty', 'AI returned no response. Check your OpenAI key.', ['status' => 500]);
        }

        if (stripos($result, 'AI error:') !== false) {
            return new \WP_Error('ai_fail', $result, ['status' => 500]);
        }

        $clean = trim($result, " \t\n\r\0\x0B\"'");

        // ===== Length check to ensure “green range” =====
        $len = mb_strlen($clean);
        if ($type === 'title' && ($len < 50 || $len > 60)) {
            $clean .= ' | ' . get_bloginfo('name');
        } elseif ($type === 'desc' && ($len < 140 || $len > 160)) {
            $clean = mb_substr($clean, 0, 157) . '...';
        }

        return rest_ensure_response([
            'success' => true,
            'result'  => $clean,
            'length'  => $len,
        ]);
    }
}
